/*
 * clipper.c: 3D clipping filter for quad lists
 *
 * usage: clipper < infile > outfile
 *
 *  infile should be a quad list
 *  outfile will be another quad list
 *
 * this version always clips to coord sys [-10,10] X [-10,10] X [-10,10]
 */

/* Note: this program uses ungetc on stdin, hence it will only
   work if stdin is buffered (this is usually the case) */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

#define YES 1
#define NO  0

#define BITSPERBYTE 8

typedef struct Point_s {
  double coord[3];
} Point;

/* Our Polygon type can have at most 10 vertices; this is because the
   intersection of a quadrilateral with a cube can have no more than
   10 edges (probably no more than 8 ??) */
#define MAXVERTICES 10

typedef struct Polygon_s {
  Point vertex[MAXVERTICES];
  int n;
} Polygon;

typedef struct Quad_s {
  Point vertex[4];
} Quad;

typedef unsigned short int Outcode_int;

typedef struct outcode_s {
  unsigned bit1	: 1;
  unsigned bit2	: 1;
  unsigned bit3	: 1;
  unsigned bit4	: 1;
  unsigned bit5	: 1;
  unsigned bit6	: 1;
  unsigned pad	: BITSPERBYTE*sizeof(Outcode_int) - 6;
} Outcode_struct;

typedef union outcode_u {
  Outcode_int		I;
  Outcode_struct	S;
} Outcode;

#define TRIVIAL_REJECT(c1,c2)	(c1.I&c2.I)
#define TRIVIAL_ACCEPT(c1,c2)	(!(c1.I|c2.I))

#define SWAP(x1,y1,z1,x2,y2,z2,c1,c2,dtemp,ctemp)   \
  dtemp   = x1;   x1   = x2;   x2   = dtemp;  \
  dtemp   = y1;   y1   = y2;   y2   = dtemp;  \
  dtemp   = z1;   z1   = z2;   z2   = dtemp;  \
  ctemp.I = c1.I; c1.I = c2.I; c2.I = ctemp.I;

double xmin = -10;
double ymin = -10;
double zmin = -10;
double xmax =  10;
double ymax =  10;
double zmax =  10;

static char *c1 = "-.0123456789";
static char *c2 = ".0123456789+Ee";

main()
{
  Quad q;
  Polygon poly;

  printf("QUAD\n");
  skip_chars(c1);
  while (!feof(stdin)) {
    if (read_quad(&q)) {
      clip_quad(&poly, &q);
      print_poly(&poly);
    }
    skip_chars(c1);
  }
}

/*-----------------------------------------------------------------------
 * Function:	read_quad
 * Description:	read a quad from stdin
 * Args OUT:	*q: the quad that was read
 * Returns:	1 if a complete quad was read
 *		0 if EOF or other problems found before complete
 *		  quad is read
 */
read_quad(q)
     Quad *q;
{
  int v,i;

  if (!feof(stdin))
    for (v=0; v<4; ++v)
      for (i=0; i<3; ++i)
	if (scanf("%lf", &(q->vertex[v].coord[i])) == 0) {
	  fprintf(stderr, "incomplete quadrilateral (ignored)\n");
	  return(0);
	}
  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	skip_chars
 * Description:	read chars from stdin until a char from *s
 *		  is read or EOF; then unread that char if not EOF.
 * Args  IN:	*s: chars to stop for
 * Returns:	nothing
 */
skip_chars(s)
     char *s;
{
  int c;

  while (((c=getchar()) != EOF) && (strchr(s, c) == NULL));
  if (c != EOF) ungetc(c, stdin);
}

/*-----------------------------------------------------------------------
 * Function:	clip_quad
 * Description:	clip a quadrilateral, producing a polygon
 * Args  IN:	*q: the quadrilateral to be clipped
 *      OUT:	*poly: the clipped figure
 * Returns:	nothing
 * Notes:	Clips to [xmin,xmax] X [ymin,ymax] X [zmin,zmax]
 */
clip_quad(poly, q)
     Polygon *poly;
     Quad *q;
{
  
  Point newa, newb;
  int qi, clipa, visible;

  /* Construct the new polygon */
  poly->n = 0;
  if (inbox(&(q->vertex[0])))
    add_vertex_to_poly(&(q->vertex[0]), poly);
  for (qi=0; qi<4; ++qi) {
    clipit(q->vertex[qi].coord,
	   q->vertex[(qi+1)%4].coord,
	   newa.coord,
	   newb.coord,
	   &clipa,
	   &visible);
    if (visible) {
      if (clipa) add_vertex_to_poly(&newa, poly);
      /* Last vertex is same as first, so if qi is 3, don't add it */
      if (qi != 3) add_vertex_to_poly(&newb, poly);
    }
  }
}

/*-----------------------------------------------------------------------
 * Function:	print_poly
 * Description:	Print a 'polygon' on standard output
 * Args  IN:	*poly: the polygon to print
 * Returns:	nothing
 * Notes:	We only print quadrilaterals. If the polygon is not a
 *		quadrilateral, it is broken up into a sequence of
 *		quadrilaterals.
 *
 *		If the number of vertices in the polygon is less than
 *		  3, we return immediately without printing anything.
 *
 */
print_poly(poly)
     Polygon *poly;
{
  int i,nsave;
  Quad q;

  if ((nsave=poly->n) < 3) return;
  while (poly->n) {
    if (strip_quad(&q, poly))
      printf("%f %f %f  %f %f %f\n\t%f %f %f  %f %f %f\n",
	     q.vertex[0].coord[0],
	     q.vertex[0].coord[1],
	     q.vertex[0].coord[2],

	     q.vertex[1].coord[0],
	     q.vertex[1].coord[1],
	     q.vertex[1].coord[2],

	     q.vertex[2].coord[0],
	     q.vertex[2].coord[1],
	     q.vertex[2].coord[2],

	     q.vertex[3].coord[0],
	     q.vertex[3].coord[1],
	     q.vertex[3].coord[2]);
  }
  poly->n = nsave;
}

strip_quad(q, poly)
     Quad *q;
     Polygon *poly;
{
  int i;

  if (poly->n < 3) {
    fprintf(stderr, "strip_quad: n = %1d\n", poly->n);
    return;
  }

  if (poly->n > 4) {
    q->vertex[0] = poly->vertex[0];
    q->vertex[1] = poly->vertex[poly->n-3];
    q->vertex[2] = poly->vertex[poly->n-2];
    q->vertex[3] = poly->vertex[poly->n-1];
    poly->n -= 2;
  }
  else {
    for (i=0; i<poly->n; ++i)
      q->vertex[i] = poly->vertex[i];
    if (poly->n == 3)
      q->vertex[3] = poly->vertex[2];
    poly->n = 0;
  }
}

/*-----------------------------------------------------------------------
 * Function:	add_vertex_to_poly
 * Description:	add a vertex to a polygon
 * Args  IN:	*p: the vertex to be added
 *		*poly: the polygon to add it to
 * Returns:	nothing
 * Notes:	Does nothing if polygon is full
 */
add_vertex_to_poly(p, poly)
     Point *p;
     Polygon *poly;
{
  if (poly->n<MAXVERTICES) {
    poly->vertex[poly->n] = *p;
    ++(poly->n);
  }
}

/*-----------------------------------------------------------------------
 * Function:     outcodes
 * Description:  compute the Outcode of a point
 * Arguments IN: x,y,z: the coords of the point
 * Returns:      the Outcode (as its "Outcode_int" member)
 */
Outcode_int
  outcode(x, y, z)
double x,y,z;
{
  Outcode code;

  code.I = 0;
  code.S.bit1 = (z > zmax);
  code.S.bit2 = (z < zmin);
  code.S.bit3 = (y > ymax);
  code.S.bit4 = (y < ymin);
  code.S.bit5 = (x > xmax);
  code.S.bit6 = (x < xmin);
  return(code.I);
}

/*-----------------------------------------------------------------------
 * Function:     clipit
 * Description:  clips line segment
 * Arguments IN: a: initial point of segment
 *               b: terminal point of segment
 *          OUT: a: initial point of clipped segment
 *               b: terminal point of clipped segment
 *               *clipa: flag telling whether point a was clipped
 *               *accept: flag telling whether segment is visible
 *                  at all.
 * Returns:      nothing
 * Notes:        If visible==NO upon return, then the values of the
 *               other 3 parameters are meaningless.
 */
clipit(a, b, newa, newb, clipa, accept)
     double a[3], b[3], newa[3], newb[3];
     int *clipa, *accept;
{
  double  x1, y1, z1, x2, y2, z2;
  double dtemp, factor;
  Outcode c1,c2,ctemp;
  int done, swapped, first_pass;

  x1 = a[0]; y1 = a[1]; z1 = a[2];
  x2 = b[0]; y2 = b[1]; z2 = b[2];

  done = NO;
  *accept = NO;
  swapped = NO;
  first_pass = YES;
  do {
    /* compute Outcodes */
    c1.I = outcode(x1, y1, z1);
    if (first_pass) {
      *clipa = (c1.I!=0);	/* (Outcode nonzero --> pt is OUT) */
      first_pass = NO;
    }
    c2.I = outcode(x2, y2, z2);
    /* check for trivial rejection */
    if (TRIVIAL_REJECT(c1, c2))
      done = YES;
    else {
      /* check for trivial acceptance */
      if (*accept=TRIVIAL_ACCEPT(c1,c2))
	done = YES;
      else {	/* Subdivide the line */
        /* First, swap if necessary to guarantee that point 1 is OUT */
        if (!c1.I) {
	  SWAP(x1, y1, z1, x2, y2, z2, c1, c2, dtemp, ctemp);
	  swapped = !swapped;
        }
	/* Now do the subdivision */
	if (c1.S.bit1) {		/* z > zmax */
          factor = (zmax - z1) / (z2 - z1);
	  x1 = x1 + (x2 - x1) * factor;
	  y1 = y1 + (y2 - y1) * factor;
	  z1 = zmax;
	}
	else if (c1.S.bit2) {		/* z < zmin */
          factor = (zmin - z1) / (z2 - z1);
	  x1 = x1 + (x2 - x1) * factor;
	  y1 = y1 + (y2 - y1) * factor;
	  z1 = zmin;
	}
	else if (c1.S.bit3) {		/* y > ymax */
          factor = (ymax - y1) / (y2 - y1);
	  x1 = x1 + (x2 - x1) * factor;
	  z1 = z1 + (z2 - z1) * factor;
	  y1 = ymax;
	}
	else if (c1.S.bit4) {		/* y < ymin */
          factor = (ymin - y1) / (y2 - y1);
	  x1 = x1 + (x2 - x1) * factor;
	  z1 = z1 + (z2 - z1) * factor;
	  y1 = ymin;
	}
	else if (c1.S.bit5) {		/* x > xmax */
          factor = (xmax - x1) / (x2 - x1);
	  y1 = y1 + (y2 - y1) * factor;
	  z1 = z1 + (z2 - z1) * factor;
	  x1 = xmax;
	}
	else if (c1.S.bit6) {		/* x < xmin */
          factor = (xmin - x1) / (x2 - x1);
	  y1 = y1 + (y2 - y1) * factor;
	  z1 = z1 + (z2 - z1) * factor;
	  x1 = xmin;
	}
      }
    }
  }  while (!done);

  if (accept) {
    if (swapped) {
      newa[0] = x2; newa[1] = y2; newa[2] = z2;
      newb[0] = x1; newb[1] = y1; newb[2] = z1;
    }
    else {
      newa[0] = x1; newa[1] = y1; newa[2] = z1;
      newb[0] = x2; newb[1] = y2; newb[2] = z2;
    }
  }
}

inbox(p)
     Point *p;
{
  return(   (p->coord[0] >= xmin) && (p->coord[0] <= xmax)
	 && (p->coord[1] >= ymin) && (p->coord[1] <= ymax)
	 && (p->coord[2] >= zmin) && (p->coord[2] <= zmax) );
}
