/* $Id: heisenberg.c,v 4.0 89/06/06 15:38:41 mbp Exp $
 *
 * heisenberg.c: Heisenberg graphics displayer; primarily for use
 * 	with Mathematica package.
 */

/***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/


/*
 * usage: heisenberg [-p] [-l file]
 *
 *    options:
 * 	-p (pipe mode): causes prompts and other non-essential output
 * 	    to be suppressed.  Intended for use when program is
 * 	    connected to another program (such as Mathematica) via a
 * 	    pipe.
 *
 * 	-l (log mode): writes a record to file of all commands
 * 	    received and output produced.  Useful for debugging.
 *
 *    This program read commands from stdin and write output to stdout.
 *    Each command is should be terminated by a newline.  Each line must
 *    contain exactly one complete command (multiline commands or
 *    multicommand lines are not allowed).
 *
 *    The general syntax for a command is
 * 			  COMMAND [args ...]
 *    where COMMAND is a short command word and args are the arguments
 *    to the command.
 *
 *    All commands return a value, which heisenberg prints to stdout
 *    upon completion of the command.  Sometimes this value indicates
 *    something about their result, such as a segment number, and
 *    sometimes it is just an error status.  A return value of -1
 *    always indicates an error.  Commands which return only an error
 *    status return 0 to indicate success.
 *
 *    COMMANDS:
 *
 * 	g
 * 	   Draw the grid.  Returns segment number of grid.
 * 	p x y z
 * 	   Draw hpoint (x,y,z).	 Returns segment number of hpoint.
 * 	c x y z r
 * 	   Draw chain with center (x,y,z) and radius r.	 Returns
 * 	   segment number of chain.
 * 	r  x y z r1 r2 f
 * 	   Draw the rcircle with center (x,y,z) and radius r1 + I r2
 * 	   and finiteness flag f (f=1 -> finite, f!=1 -> infinite).
 * 	   Returns the segment number of the rcircle.
 * 	s x1 y1 z1 x2 y2 z2
 * 	   Draw the spinal with vertices (x1,y1,z1) and (x2,y2,z2).
 * 	   The spinal is drawn by chains, rcircles, or both, depending
 * 	   on the setting of the parameter spinaldrawtype.  Returns
 * 	   the segment number of the new spinal.
 * 	e n
 * 	   Erases segment number n.  Returns error status.
 * 	h n
 * 	   Hide segment n (make it invisibile) if it is not already
 * 	   hidden. Returns error status.
 * 	d n
 * 	   Display segment n (make it visible) if it is not already
 * 	   on display. Returns error status.
 * 	k
 * 	   Clear entire screen.	  Returns error status.
 * 	q
 * 	   Quit program.  Does not return.
 *	< file
 *	   Instructs heisenberg to read commands from file instead of
 *	   stdin.  Commands are read until end of file is encountered,
 *	   at which time we resume reading commands from stdin.  Returns
 *	   -1 if can't read file, 0 otherwise.
 *
 *	ts x1 y1 z1 x2 y2 z2 file
 *	   Write a triangulation of the spinal with vertices (x1,y1,z1)
 *	   and (x2,y2,z2) to file.
 *
 *	ms x1 y1 z1 x2 y2 z2 file
 *	   Write a mesh of the spinal with vertices (x1,y1,z1) and
 *	   (x2,y2,z2) to file.
 *
 * 	NOTE: the following are the parameter setting commands.  Each
 * 	of them returns the new value(s) of the parameter if the setting
 * 	is successful, or -1 if not:
 *
 * 	cres n
 * 	   Set the "Chain Resolution" to n.
 *
 * 	vcl min max
 * 	   Sets the "Vertical Chain Limits" to min and max.
 *
 * 	ssc n
 * 	   Sets the "Spinal Slice Count" to n.
 *
 * 	smc n
 * 	   Sets the "Spinal Meridian Count" to n.
 *
 * 	rres n
 * 	   Set the "Rcircle Resolution" to n.
 *
 * 	fud x
 * 	   Sets the "FUDge factor" to x.
 *
 * 	col c
 * 	   Sets the current drawing color to c.  The colors are:
 * 		BLACK	0
 * 		WHITE	1
 * 		RED	2
 * 		GREEN	3
 * 		YELLOW	4
 * 		BLUE	5
 * 		MAGENTA	6
 * 		CYAN	7
 *
 * 	sdr
 * 	   Sets the "Spinal DRaw type".  The values are:
 * 		0 chains
 * 		1 rcircles
 * 		2 both
 *
 * 	NOTE: the following are the parameter inquiry commands. Each
 * 	of them returns the value(s) of the corresponding parameter:
 *
 * 	    ?cres
 * 	    ?vcl
 * 	    ?ssc
 * 	    ?rres
 * 	    ?smc
 * 	    ?fud
 * 	    ?col
 * 	    ?sdr
 */

#include "heisenberg.h"
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include "lgd.h"

#define UNKNOWN			0
#define GRID   			1
#define POINT   		2
#define CHAIN			3
#define RCIRCLE			4
#define SPINAL			5
#define ERASE			6
#define HIDE			7
#define DISPLAY			8
#define CLEAR			9
#define QUIT			10
#define SET_CHAIN_RESOLUTION	11
#define SET_VERT_CHAIN_LIMITS	12
#define SET_SLICE_COUNT		13
#define SET_MERIDIAN_COUNT	14
#define SET_RCIRCLE_RESOLUTION	15
#define SET_FUDGE		16
#define SET_COLOR		17
#define SET_SPINAL_DRAWTYPE	18
#define GET_CHAIN_RESOLUTION	19
#define GET_VERT_CHAIN_LIMITS	20
#define GET_SLICE_COUNT		21
#define GET_MERIDIAN_COUNT	22
#define GET_RCIRCLE_RESOLUTION	23
#define GET_FUDGE		24
#define GET_COLOR		25
#define GET_SPINAL_DRAWTYPE	26
#define READ_FILE		27
#define TRIANGULATE_SPINAL	28
#define MESH_SPINAL		29
#define READY			30

#define INBUF_SIZE 256		/* size of input buffer */
#define MAXARGS 10		/* max # command arguments */

#define NEXTARG --argc; ++argv
#define PPRINT if (!pipe_mode) printf
#define PERROR output_int(-1)

#ifdef DEVICE_SUNVIEW
#include        <suntool/sunview.h>
static short icon_data[] = {
#include "heisenberg.icon"
};
DEFINE_ICON_FROM_IMAGE(heisenberg_icon, icon_data);
#include "gr.h"
#endif

int pipe_mode=0;
int log_mode=0;
FILE *logf;

/* World boundaries: */
double wbox_low[3]  = { -10, -10, -10 };
double wbox_high[3] = {  10,  10,  10 };
double maxlinelen;

/* default values for flags and drawing params */
int chain_resolution = 36;
int rcircle_resolution = 36;
int slice_count = 30;
int meridian_count = 9;
double vert_chain_min = -3.0;
double vert_chain_max =  3.0;
double fudge = FUDGE_DEFAULT;
int spinaldrawtype = CHAINS;

int read_input();

static char *copyright = 
"Copyright (C) 1989 by Mark B. Phillips and William M. Goldman";


main(argc, argv)
     int argc;
     char *argv[];
{
  parse_args(argc, argv);
#ifdef DEVICE_IRIS
  lgd_set_mex_name("heisenberg");
  lgd_set_mex_foreground(!pipe_mode);
#endif /* DEVICE_IRIS */
  if (log_mode) {
    char *ctime();  long t=time(0);
    fprintf(logf, "heisenberg started at %s", ctime(&t)); fflush(logf);
    fprintf(logf, "command line:"); fflush(logf);
    while (argc) {
      fprintf(logf, " %s", *argv); fflush(logf);
      NEXTARG;
    }
    fprintf(logf, "\n");
  } /* end if (log_mode) */

  setlinebuf(stdout);
  lgd_initialize();
  lgd_define_world( 3, wbox_low, wbox_high );
  lgd_set_input_func(read_input, fileno(stdin));
  initialize();
  prompt();
  lgd_main_loop();
}

parse_args(argc, argv)
     int argc;
     char *argv[];
{
  NEXTARG;
  while (argc) {
    if (argv[0][0] == '-')
      switch (argv[0][1]) {
      case 'p':
	pipe_mode = 1;
	break;
      case 'l':
	NEXTARG;
	if (argc == 0) {
	  PPRINT("warning: no log file specified; -l flag ignored\n");
	}
	else {
	  if ((logf=fopen(argv[0], "w")) == NULL) {
	    PPRINT("error: can't open log file; -l flag ignored\n");
	  }
	  else
	    log_mode = 1;
	}
	break;
      default:
	PPRINT("warning: option %s ignored\n", argv[0]);
	break;
      }
    else
      PPRINT("warning: argument %s ignored\n", argv[0]);
    NEXTARG;
  }
}

/*-----------------------------------------------------------------------
 * Function:	read_input
 * Description:	Read and process input from stdin
 * Args:	(none)
 * Returns:	nothing
 * Notes:	This gets called whenever there is input waiting to be
 *		read. 
 */
read_input()
{
  char inbuf[INBUF_SIZE];
  int command, argc;
  char *args[MAXARGS];

  gets(inbuf);
  if (log_mode) {
    fprintf(logf, "heisenberg in: %s\n", inbuf);
    fflush(logf);
  }
  parse_command(inbuf, &command, &argc, args);
  execute_cmd(command, argc, args);
  prompt();
}

/*-----------------------------------------------------------------------
 * Function:	parse_command
 * Description:	parse a command string, breaking it up into the command
 *		part and its arguments
 * Args  IN:	cmd: the command string
 *      OUT:	*command: the command
 *		*argc: number of arguments given in cmd
 *		args: array of arguments
 */
parse_command(cmd, command, argc, args)
     char *cmd;
     int *command, *argc;
     char **args;
{
  static char *seps = " \t";
  static char argtable[512];
  char *argptr;
  char *c;

  /* Get the command */
  *command = UNKNOWN;
  if ((c=strtok(cmd, seps)) == NULL) return;
  *command = fsa_parse(c);

  /* Get the args */
  *argc = 0;
  argptr = argtable;
  while ((c=strtok(NULL,seps)) != NULL) {
    strcpy(argptr, c);
    *args = argptr;
    argptr += strlen(argptr) + 1;
    ++args;
    ++(*argc);
  }
}

/*-----------------------------------------------------------------------
 * Function:	execute_cmd
 * Description:	execute a command
 * Args  IN:	command: command to execute
 *		argc: number of arguments for command
 *		args: array of argument values (doubles only)
 */
execute_cmd(command, argc, args)
     int command, argc;
     char **args;
{
  int segnum;

  switch (command) {
  case READ_FILE:
    if (argc != 1) PERROR;
    else output_int(read_file(args[0]));
    break;
  case TRIANGULATE_SPINAL:
    if (argc != 7) PERROR;
    else {
      Spinal s;
      FILE *fp;
      
      s.p1.hor.re  = atof(args[0]);
      s.p1.hor.im  = atof(args[1]);
      s.p1.hor.inf = (char)0;
      s.p1.ver     = atof(args[2]);
      s.p2.hor.re  = atof(args[3]);
      s.p2.hor.im  = atof(args[4]);
      s.p2.hor.inf = (char)0;
      s.p2.ver     = atof(args[5]);
      if ((fp=fopen(args[6], "w")) == NULL) PERROR;
      else {
	output_int(triangulate_spinal(&s, fp));
	fclose(fp);
      }
    }
    break;
  case MESH_SPINAL:
    if (argc != 7) PERROR;
    else {
      Spinal s;
      FILE *fp;
      
      s.p1.hor.re  = atof(args[0]);
      s.p1.hor.im  = atof(args[1]);
      s.p1.hor.inf = (char)0;
      s.p1.ver     = atof(args[2]);
      s.p2.hor.re  = atof(args[3]);
      s.p2.hor.im  = atof(args[4]);
      s.p2.hor.inf = (char)0;
      s.p2.ver     = atof(args[5]);
      if ((fp=fopen(args[6], "w")) == NULL) PERROR;
      else {
	output_int(mesh_spinal(&s, fp));
	fclose(fp);
      }
    }
    break;
  case UNKNOWN:
    PERROR;
    break;
  case CHAIN:
   if (argc != 4) PERROR;
    else {
      Chain c;
      c.cen.hor.re  = atof(args[0]);
      c.cen.hor.im  = atof(args[1]);
      c.cen.hor.inf = (char)0;
      c.cen.ver     = atof(args[2]);
      c.rad         = atof(args[3]);
      lgd_begin_segment(&segnum);
      draw_chain(&c);
      lgd_end_segment();
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case POINT:
    if (argc != 3) PERROR;
    else {
      Hpoint p;
      p.hor.re  = atof(args[0]);
      p.hor.im  = atof(args[1]);
      p.hor.inf = (char)0;
      p.ver     = atof(args[2]);
      lgd_begin_segment(&segnum);
      draw_hpoint(&p);
      lgd_end_segment();
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case RCIRCLE:
    if (argc != 6) PERROR;
    else {
      Rcircle r;
      r.cen.hor.re   = atof(args[0]);
      r.cen.hor.im   = atof(args[1]);
      r.cen.hor.inf  = (char)0;
      r.cen.ver      = atof(args[2]);
      r.c_rad.re     = atof(args[3]);
      r.c_rad.im     = atof(args[4]);
      r.c_rad.inf    = (char)0;
      r.finite       = ( atoi(args[5]) == 1 );
      lgd_begin_segment(&segnum);
      draw_rcircle(&r);
      lgd_end_segment();
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case SPINAL:
    if (argc != 6) PERROR;
    else {
      Spinal s;
      s.p1.hor.re  = atof(args[0]);
      s.p1.hor.im  = atof(args[1]);
      s.p1.hor.inf = (char)0;
      s.p1.ver     = atof(args[2]);
      s.p2.hor.re  = atof(args[3]);
      s.p2.hor.im  = atof(args[4]);
      s.p2.hor.inf = (char)0;
      s.p2.ver     = atof(args[5]);
      lgd_begin_segment(&segnum);
      switch(spinaldrawtype) {
      case CHAINS:
	draw_spinal_by_chains(&s); break;
      case RCIRCLES:
	draw_spinal_by_rcircles(&s); break;
      case BOTH:
	draw_spinal_by_chains(&s); draw_spinal_by_rcircles(&s); break;
      }
      lgd_end_segment();
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case GRID:
    if (argc != 0) PERROR;
    else {
      lgd_begin_segment(&segnum);
      draw_grid();
      lgd_end_segment();
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case ERASE:
    if (argc != 1) PERROR;
    else {
      segnum = atoi(args[0]);
      lgd_delete_segment( segnum );
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case HIDE:
    if (argc != 1) PERROR;
    else {
      segnum = atoi(args[0]);
      lgd_set_segment_visibility(segnum, 0);
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case DISPLAY:
    if (argc != 1) PERROR;
    else {
      segnum = atoi(args[0]);
      lgd_set_segment_visibility(segnum, 1);
      lgd_update_display();
      output_int(segnum);
    }
    break;
  case CLEAR:
    if (argc != 0) PERROR;
    else {
      int color;
      lgd_inquire_color(&color);	/* save current color */
      lgd_delete_all();
      output_int(0);
      lgd_set_color(color);		/* restore current color */
    }
    break;
  case QUIT:
    output_int(0);
    exit(0);
    break;
  case SET_CHAIN_RESOLUTION:
    if (argc != 1) PERROR;
    else {
      chain_resolution = atoi(args[0]);
      output_int(chain_resolution);
    }
    break;
  case SET_VERT_CHAIN_LIMITS:
    if (argc != 2) PERROR;
    else {
      vert_chain_min = atof(args[0]);
      vert_chain_max = atof(args[1]);
      printf("{%f,%f}\n", vert_chain_min, vert_chain_max);
      if (log_mode) {
	fprintf(logf, "heisenberg out: {%f,%f}\n", vert_chain_min, vert_chain_max);
	fflush(logf);
      }
    }
    break;
  case SET_SLICE_COUNT:
    if (argc != 1) PERROR;
    else {
      slice_count = atoi(args[0]);
      output_int(slice_count);
    }
    break;
  case SET_MERIDIAN_COUNT:
    if (argc != 1) PERROR;
    else {
      meridian_count = atoi(args[0]);
      output_int(meridian_count);
    }
    break;
  case SET_RCIRCLE_RESOLUTION:
    if (argc != 1) PERROR;
    else {
      rcircle_resolution = atoi(args[0]);
      output_int(rcircle_resolution);
    }
    break;
  case SET_FUDGE:
    if (argc != 1) PERROR;
    else {
      fudge = atof(args[0]);
      output_double(fudge);
    }
    break;
  case SET_COLOR:
    if (argc != 1) PERROR;
    else {
      lgd_set_color(atoi(args[0]));
      output_int(atoi(args[0]));
    }
    break;
  case SET_SPINAL_DRAWTYPE:
    if (argc != 1) PERROR;
    else {
      int n = atoi(args[0]);
      if ((n < 0) || (n > 2)) PERROR;
      else output_int(spinaldrawtype=n);
    }
    break;
  case GET_CHAIN_RESOLUTION:
    if (argc != 0) PERROR;
    else
      output_int(chain_resolution);
    break;
  case GET_VERT_CHAIN_LIMITS:
    if (argc != 0) PERROR;
    else {
      printf("{%f,%f}\n", vert_chain_min, vert_chain_max);
      if (log_mode) {
	fprintf(logf, "heisenberg out: {%f,%f}\n", vert_chain_min, vert_chain_max);
	fflush(logf);
      }
    }
    break;
  case GET_SLICE_COUNT:
    if (argc != 0) PERROR;
    else
      output_int(slice_count);
    break;
  case GET_MERIDIAN_COUNT:
    if (argc != 0) PERROR;
    else
      output_int(meridian_count);
    break;
  case GET_RCIRCLE_RESOLUTION:
    if (argc != 0) PERROR;
    else
      output_int(rcircle_resolution);
    break;
  case GET_FUDGE:
    if (argc != 0) PERROR;
    else
      output_double(fudge);
    break;
  case GET_COLOR:
    if (argc != 0) PERROR;
    else {
      int color;
      lgd_inquire_color(&color);
      output_int(color);
    }
    break;
  case GET_SPINAL_DRAWTYPE:
    if (argc != 0) PERROR;
    else
      output_int(spinaldrawtype);
    break;
  case READY:
    printf("ready!\n");
    break;
  }
}

prompt()
{
  PPRINT("heisenberg>");
  fflush(stdout);
}

initialize()
{
  lgd_View3 view;
  double LGD_L2dist_vec();

  /* 1. General variable initialization: */
  maxlinelen = LGD_L2dist_vec( wbox_low, wbox_high);

  /* 2. Graphics initialization: */
  lgd_set_color(LGD_GREEN);	/* default color */
  lgd_inquire_view(&view);
  view.u1 = 1.1 * wbox_low[0];
  view.u2 = 1.1 * wbox_high[0];
  view.v1 = 1.1 * wbox_low[1];
  view.v2 = 1.1 * wbox_high[1];
  lgd_set_view(&view);
  lgd_update_display();

#ifdef DEVICE_SUNVIEW
  {
    extern int gr_redraw();
    char helpfile[256];
    char label[90];
#ifdef COPYRIGHT
    sprintf(label,
	    "Heisenberg graphics                                           %s",
	    copyright);
    gr_set_frame_label( label );
#else
    gr_set_frame_label( "Heisenberg graphics" );
#endif
    gr_set_frame_icon( &heisenberg_icon );
    gr_print_button(1, gr_redraw, "Heisenberg");
    construct_help_file_name(helpfile);
    gr_set_help_file(helpfile);
  }
#endif

  /* 3. Lexical initialization: */
  fsa_initialize(UNKNOWN);
  fsa_install("c",	CHAIN);
  fsa_install("r",	RCIRCLE);
  fsa_install("p",	POINT);
  fsa_install("g",	GRID);
  fsa_install("e",	ERASE);
  fsa_install("d",	DISPLAY);
  fsa_install("h",	HIDE);
  fsa_install("k",	CLEAR);
  fsa_install("q",	QUIT);
  fsa_install("s",	SPINAL);

  fsa_install("cres",	SET_CHAIN_RESOLUTION);
  fsa_install("vcl",	SET_VERT_CHAIN_LIMITS);
  fsa_install("ssc",	SET_SLICE_COUNT);
  fsa_install("smc",	SET_MERIDIAN_COUNT);
  fsa_install("rres",	SET_RCIRCLE_RESOLUTION);
  fsa_install("fud",	SET_FUDGE);
  fsa_install("col",	SET_COLOR);
  fsa_install("sdr",	SET_SPINAL_DRAWTYPE);

  fsa_install("?cres",	GET_CHAIN_RESOLUTION);
  fsa_install("?vcl",	GET_VERT_CHAIN_LIMITS);
  fsa_install("?ssc",	GET_SLICE_COUNT);
  fsa_install("?smc",	GET_MERIDIAN_COUNT);
  fsa_install("?rres",	GET_RCIRCLE_RESOLUTION);
  fsa_install("?fud",	GET_FUDGE);
  fsa_install("?col",	GET_COLOR);
  fsa_install("?sdr",	GET_SPINAL_DRAWTYPE);

  fsa_install("<",      READ_FILE);
  fsa_install("ts",     TRIANGULATE_SPINAL);
  fsa_install("ms",	MESH_SPINAL);

  fsa_install("ready?",	READY);

}

output_int(segnum)
     int segnum;
{
  printf("%1d\n", segnum);
  if (log_mode) {
    fprintf(logf,"heisenberg out: %1d\n", segnum);
    fflush(logf);
  }
}

output_double(x)
     double x;
{
  printf("%1f\n", x);
  if (log_mode) {
    fprintf(logf,"heisenberg out: %f\n", x);
    fflush(logf);
  }
}

#ifdef DEVICE_SUNVIEW
static
  construct_help_file_name(fname)
char *fname;
{
  extern char *getenv();
  char *heisenberg_lib;

  /* First look in HEISENBERG_LIB */
  heisenberg_lib = getenv("HEISENBERG_LIB");
  if (heisenberg_lib != NULL) {
    sprintf(fname,"%s%s%s",
	    heisenberg_lib,
	    (heisenberg_lib[strlen(heisenberg_lib)-1]=='/') ? "" : "/",
	    "heisenberg.hlp");
    if (GR_file_openable(fname, "r")) return;
  }

  /* If not there, then use default */
  strcpy(fname, HELPFILE);
}
#endif

read_file(fname)
  char *fname;
{
  FILE *fp;
  char inbuf[INBUF_SIZE];
  int command, argc;
  char *args[MAXARGS];

  if (strlen(fname)<=0) {
    if (log_mode) fprintf(logf,"read_file error: missing filename\n");
    return(-1);
  }

  if ((fp=fopen(fname,"r")) == NULL) {
    if (log_mode) fprintf(logf,"read_file error: can't open %s\n", fname);
    return(-1);
  }

  while (fgets(inbuf, INBUF_SIZE-1, fp) != NULL) {
  inbuf[strlen(inbuf)-1] = '\0';
    if (log_mode) {
      fprintf(logf, "heisenberg in: %s\n", inbuf);
      fflush(logf);
    }
    parse_command(inbuf, &command, &argc, args);
    execute_cmd(command, argc, args);
    lgd_update_display();
  }
  fclose(fp);

  return(0);
 
}

