/* $Id: draw.c,v 4.0 89/06/06 15:38:39 mbp Exp $
 *
 * draw.c: drawing procedures
 */

/***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/


#include "heisenberg.h"
#include "lgd.h"
#include <stdio.h>

/*
 * NOTE: these procedures perform the actual 'moves' and 'draws' for
 * drawing the various objects.  This is ALL they do.  In particular,
 * any color settings, segment beginnings or endings, or other
 * information must be specified outside of calls to these procedures.
 */

/*-----------------------------------------------------------------------
 * Function:     draw_hpoint
 * Description:  Draw an hpoint
 * Arguments IN: *p: the point to draw
 * Returns:      nothing
 */
draw_hpoint(p)
Hpoint *p;
{
  hpoint_move_abs(p);
  lgd_point();
}

/*-----------------------------------------------------------------------
 * Function:     draw_chain
 * Description:  draw a chain
 * Arguments IN: *c: the chain to draw
 * Returns:      nothing
 * Notes:     1. Uses the global variable chain_resolution as the number
 *               of line segments to use in drawing the chain.
 */
draw_chain(c)
     Chain *c;
{
  double X[3];          /* this is the point which we draw */
  double th, thinc;

  if (c->rad==CHAIN_VERTICAL) {	/* It's a vertical chain ... */
    X[0] = c->cen.hor.re;
    X[1] = c->cen.hor.im;	/* so draw a line. */
    X[2] = vert_chain_min;
    lgd_move_abs( X );
    X[2] = vert_chain_max;
    lgd_draw_abs( X );
  }
  else {   /* It's not a vertical chain, so draw an ellipse. */
    X[0] = c->cen.hor.re + c->rad;
    X[1] = c->cen.hor.im;        
    X[2] = chain_v_coord(&(c->cen),X[0],X[1]);
    lgd_move_abs( X );          /* first point */
    thinc = M_TWO_PI / chain_resolution;
    for(th=thinc; th<2*M_PI; th+=thinc) {
      X[0] = c->cen.hor.re + c->rad * cos(th); 
      X[1] = c->cen.hor.im + c->rad * sin(th); 
      X[2] = chain_v_coord(&(c->cen),X[0],X[1]);
      lgd_draw_abs( X );        /* intermediate points */
    }
    X[0] = c->cen.hor.re + c->rad;
    X[1] = c->cen.hor.im;        
    X[2] = chain_v_coord(&(c->cen),X[0],X[1]);
    lgd_draw_abs( X );      /* last point = first point */
  }
}

/*-----------------------------------------------------------------------
 * Function:     draw_grid
 * Description:  draws an initial configuration
 * Notes:     1. The grid currently consists of a grid in the xy-plane,
 *               and a vertical axis
 */
draw_grid()
{
  int color;

  lgd_inquire_color(&color);	/* save current color */
  draw_xy_plane(LGD_CYAN, LGD_BLUE, LGD_CYAN);
  lgd_set_color(LGD_YELLOW);
  draw_v_axis();
  lgd_set_color(color);		/* restore current color */
}

/*-----------------------------------------------------------------------
 * Function:     draw_xy_plane
 * Description:  draws a grid in the xy-plane
 * Arguments:    c1: color for x and y axes
 *               c2: color for all other lines in grid
 *               c3: color for letters X and Y on axes
 * Returns:      nothing
 * Notes:        Also draws the letters X and Y near the positive end
 *               of these axes.
 */
STATIC draw_xy_plane(c1,c2,c3)
int c1,c2,c3;
{
  int i;
  double s, s_inc, P[3], xmid, ymid, xtenth, ytenth;

  P[2] = 0;
  lgd_set_color(c2);
  /* Draw grid lines perp to x axis: */
  s_inc = (wbox_high[0] - wbox_low[0]) / 10;
  for (i=0; i<=10; ++i) {
    s = wbox_low[0] + i * s_inc;
    if (i==5) lgd_set_color(c1);
    P[0] = s;
    P[1] = wbox_low[1];  lgd_move_abs(P);
    P[1] = wbox_high[1]; lgd_draw_abs(P);
    if (i==5) lgd_set_color(c2);
  }
  /* Draw grid lines perp to y axis: */
  s_inc = (wbox_high[1] - wbox_low[1]) / 10;
  for (i=0; i<=10; ++i) {
    s = wbox_low[1] + i * s_inc;
    if (i==5) lgd_set_color(c1);
    P[1] = s;
    P[0] = wbox_low[0];  lgd_move_abs(P);
    P[0] = wbox_high[0]; lgd_draw_abs(P);
    if (i==5) lgd_set_color(c2);
  }
  /* Draw letter 'X' at positive end of x-axis, 'Y' at positive
     end of y-axis */
  xmid = (wbox_high[0] + wbox_low[0]) / 2;
  xtenth = (wbox_high[0] - wbox_low[0]) / 20;
  ymid = (wbox_high[1] + wbox_low[1]) / 2;
  ytenth = (wbox_high[1] - wbox_low[1]) / 20;
  lgd_set_color(c3);
  draw_x( xmid + 9.2*xtenth, ymid + 0.2*ytenth, 0.6*xtenth, 0.8*ytenth );
  draw_y( xmid + 0.2*xtenth, ymid + 9.0*ytenth, 0.6*ytenth, 0.8*ytenth );
}

/*-----------------------------------------------------------------------
 * Function:     draw_x
 * Description:  draw the letter X in the xy plane
 * Arguments IN: x,y: coords of lower left corner
 *               w: width (parallel to X axis)
 *               h: height (parallel to Y axis)
 * Returns:      nothing
 */
STATIC draw_x(x,y,w,h)
double x,y,w,h;
{
  double P[3];
  P[2] = 0;
  P[0] = x; P[1] = y;   lgd_move_abs(P);
  P[0] += w; P[1] += h; lgd_draw_abs(P);
  P[0] = x;             lgd_move_abs(P);
  P[0] += w; P[1] = y;  lgd_draw_abs(P);
}

/*-----------------------------------------------------------------------
 * Function:     draw_y
 * Description:  draw the letter Y in the xy plane
 * Arguments IN: x,y: coords of lower left corner
 *               w: width (parallel to Y axis)
 *               h: height (parallel to X axis)
 * Returns:      nothing
 */
STATIC draw_y(x,y,w,h)
double x,y,w,h;
{
  double P[3];
  P[2] = 0;
  P[0] = x + w/2; P[1] = y;       lgd_move_abs(P);
  P[1] = y + h/2;                 lgd_draw_abs(P);
  P[0] = x; P[1] = y + h;         lgd_draw_abs(P);
  P[0] = x + w;                   lgd_move_abs(P);
  P[0] = x + w/2; P[1] = y + h/2; lgd_draw_abs(P);
}

/*-----------------------------------------------------------------------
 * Function:     draw_v_axis
 * Description:  Draws the v (vertical) axis
 * Arguments:    (none)
 * Returns:      nothing
 */
STATIC draw_v_axis()
{
  double P[3];
  P[0] = P[1] = 0;
  P[2] = wbox_low[2];  lgd_move_abs(P);
  P[2] = wbox_high[2]; lgd_draw_abs(P);
}

/*-----------------------------------------------------------------------
 * Function:     hpoint_draw_abs
 * Description:  draw abs to a point given as a Hpoint
 * Arguments IN: *p: the Hpoint to move to
 * Returns:      nothing
 */
STATIC hpoint_draw_abs(p)
     Hpoint *p;
{
  double X[3];
  X[0] = p->hor.re;
  X[1] = p->hor.im;
  X[2] = p->ver;
  lgd_draw_abs( X );
}

/*-----------------------------------------------------------------------
 * Function:     hpoint_move_abs
 * Description:  move abs to a point given as a Hpoint
 * Arguments IN: *p: the Hpoint to move to
 * Returns:      nothing
 */
STATIC hpoint_move_abs(p)
     Hpoint *p;
{
  double X[3];
  X[0] = p->hor.re;
  X[1] = p->hor.im;
  X[2] = p->ver;
  lgd_move_abs( X );
}

/*-----------------------------------------------------------------------
 * Function:     draw_spinal_by_chains
 * Description:  draw a spinal by chains
 * Arguments IN: *sp: the spinal to draw
 * Returns:      (nothing)
 * Notes:        Uses the global variable slice_count to determine how
 *               many slices to draw.
 */
draw_spinal_by_chains(sp)
     Spinal *sp;
{
  Cvector Z1,Z2;
  Complex t0;
  Chain slice;
  double latitude, latinc;
  int j, success;
  
  hpoint_to_cvector(Z1,&(sp->p1));
  hpoint_to_cvector(Z2,&(sp->p2));

  /* Normalize Z1 and Z2 to have 3rd coord 1 */
  for (j=0; j<3; j++){  
    C_PUSH_S(Z1[j]); C_PUSH_S(Z1[2]); c_div_s(); C_POP_S(Z1[j]); 
    C_PUSH_S(Z2[j]); C_PUSH_S(Z2[2]); c_div_s(); C_POP_S(Z2[j]); 
  } 

  /* compute t0 */
  compute_spinal_t0(&t0, Z1,Z2);

  /* This loop draws the first vertex (latitude=0) and all
     non-degenerate slices */
  latinc = M_PI / (slice_count + 1);
  for(latitude=0; latitude<M_PI; latitude+=latinc) {
    compute_spinal_slice(&slice,latitude,&t0,Z1,Z2,&success);
    if (success) draw_chain(&slice);
  }

  /* Then we draw the second vertex (latitude=Pi) explicitly */
  compute_spinal_slice(&slice,M_PI,&t0,Z1,Z2,&success);
  if (success) draw_chain(&slice);
}

/*-----------------------------------------------------------------------
 * Function:     draw_spinal_by_rcircles
 * Description:  draw a spinal by R-circles
 * Arguments IN: *sp: the spinal to draw
 * Returns:      (nothing)
 * Notes:        Uses the global variable meridian_count to determine
 *		 how many Rcircles to draw.
 */
draw_spinal_by_rcircles(sp)
     Spinal *sp;
{
  Rcircle rc;
  Cvector Z1,Z2,Z3;
  Complex a, z, M1[3][3], M2[3][3], M3[3][3];
  Complex I12[3][3];	
  int i,success;
  double b, binc;
  
  I12[0][0] = I12[1][1] = C_ONE;
  I12[2][2] = C_NEG_ONE;
  I12[0][1] = I12[0][2] = I12[1][0] = I12[2][0] = C_ZERO;
  I12[1][2] = I12[2][1] = C_ZERO;

  hpoint_to_cvector(Z1,&(sp->p1));
  hpoint_to_cvector(Z2,&(sp->p2));
  Herm(&a,Z1,Z2);
  for(i=0; i<3; i++) {		/* normalize <Z1,Z2>=1 */
    C_PUSH_S(Z1[i]); C_PUSH_S(a); c_div_s(); C_POP_S(Z1[i]);
  }
  cross_product(Z3,Z1,Z2);     /* <Z3,Z1> = <Z3,Z2> = 0 */

  binc = M_TWO_PI / meridian_count;
  for(b=0; b<M_TWO_PI; b+=binc) {
    z.re = cos(b); z.im = sin(b);
    z.inf = NO;

    mult_mat(M1, Z3, Z3,  3,1,1,3);
    mult_mat(M2, M1, I12, 3,3,3,3); 
    scalar_mat(M1, z, M2, 3,3);
    
    mult_mat(M2, Z1, Z2,  3,1,1,3);
    mult_mat(M3, M2, I12, 3,3,3,3); 
    add_mat(M2, M1, M3, 3,3);
 
    mult_mat(M1, Z2, Z1,  3,1,1,3);
    mult_mat(M3, M1, I12, 3,3,3,3);
    add_mat(M1, M2, M3, 3,3);
    
    /* formula: M1 = zeta * Z3*Z3t*I12 + Z1*Z2t*I12 + Z2*Z1t*I12 */
    
    matrix_to_rcircle(&rc,M1,&success);
    if (success) 
      draw_rcircle(&rc);
  }
}

/*-----------------------------------------------------------------------
 * Function:     draw_rcircle
 * Description:  draw R-circle
 * Arguments IN: *c: the R-circle to draw
 * Returns:      nothing
 * Notes:        Uses the global variable rcircle_resolution to determine
 *               how many line segments to draw.
 */
draw_rcircle(rc)
     Rcircle *rc;
{
  double th, thinc;
  Hpoint p1,p2,p3;
  int half_rcircle_resolution,n,sign;
  Complex sq_c_rad, temp;
  double P0[3], D[3], rr;
  
  /* check to see if rcircle is finite */
  
  if (rc->finite) {
    C_SQRT(sq_c_rad,rc->c_rad);
    /* Force rcircle_resolution to be even: */
    half_rcircle_resolution = (rcircle_resolution + 1) / 2;
    thinc = M_PI / (2 * half_rcircle_resolution);
    for(sign= -1;sign < 2; sign += 2) {  /* this for loop goes only twice */
      th = M_PI/4;
      compute_standard_rcircle( &p1, th, sign );
      hpoint_dilate( &p2, &p1, &sq_c_rad);
      hpoint_translate( &p3, &p2, &(rc->cen));
      hpoint_move_abs( &p3 );
      for (n=0, th = M_PI/4 + thinc;
	   n < half_rcircle_resolution;
	   ++n, th += thinc) {
	compute_standard_rcircle( &p1, th, sign );
	hpoint_dilate( &p2, &p1, &sq_c_rad);
	hpoint_translate( &p3, &p2, &(rc->cen));
	hpoint_draw_abs( &p3 );
      }
    }
  }
  else {		/* rcircle is infinite */
#define x0 P0[0]
#define y0 P0[1]
#define v0 P0[2]
#define xd D[0]
#define yd D[1]
#define vd D[2]
    rr = C_ENORM_SQ(rc->cen.hor);
    if (rr < fudge) {		/* closest pt is on V axis */
      x0 = 0.0;
      y0 = 0.0;
      v0 = rc->cen.ver;
      C_SQRT(temp,rc->c_rad);   /* added 10/13/88 wmg */
      xd = temp.re;
      yd = temp.im;
      vd = 0.0; 
    }
    else {			/* closest pt is not on V axis */
      x0 = rc->cen.hor.re;	        /* NOTE: in this case, rr = 0, */
      y0 = rc->cen.hor.im;	        /*  not x0^2 + y0^2 */
      v0 = rc->cen.ver;         /* wmg changed this 10/14/88 */
      xd =  rc->cen.hor.im;
      yd = -rc->cen.hor.re;
      vd = - C_ENORM(rc->cen.hor);
    }
    
    /* 
     *  Now we draw the line (x,y,v) given by the equations:
     *  
     *  rr =    x0 * x + y0 * y
     *  v  =  - y0 * x + x0 * y + v0
     *  
     *  This is the line thru (x0,y0,v0) in the direction of (xd,yd,vd),
     */
    LGD_unit_vec( D );
    LGD_scamul_vec( D, maxlinelen, D );
    lgd_move_abs( P0 );
    lgd_move_rel( D );
    LGD_scamul_vec( D, -2.0, D );
    lgd_draw_rel( D );
#undef x0
#undef y0
#undef v0
#undef xd
#undef yd
#undef vd
  }
  lgd_update_display();    
}
