/* $Id: convert.c,v 4.0 89/06/06 15:38:36 mbp Exp $
 *
 * convert.c: conversion routines
 */

/***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/


#include "heisenberg.h"

#define	SQR(x)	((x)*(x))

/*-----------------------------------------------------------------------
 * Function:     chain_to_cvector
 * Description:  find the pole (a positive vector in C21) of a chain
 * Arguments IN: *c: the chain to find the pole of
 *          OUT: Z: the pole
 * Returns:      nothing
 * Notes:        The pole has positive square-length
 */
chain_to_cvector(Z,c)
     Chain *c;
     Cvector Z;
{
  Z[0].inf = Z[1].inf = Z[2].inf = NO;
  if (c->rad == CHAIN_VERTICAL) {
    Z[0].re = 1;
    Z[0].im = 0;
    Z[1].re = -c->cen.hor.re;
    Z[1].im =  c->cen.hor.im;
    Z[2].re =  c->cen.hor.re;
    Z[2].im = -c->cen.hor.im;
  }
  else {
    Z[0].re = 2 * c->cen.hor.re;
    Z[0].im = 2 * c->cen.hor.im;
    Z[1].re = 1 + (c->rad * c->rad) - 
      (c->cen.hor.re *c->cen.hor.re) - 
        (c->cen.hor.im * c->cen.hor.im);
    Z[1].im =  -2 * c->cen.ver;
    Z[2].re =  2 - Z[1].re;
    Z[2].im = - Z[1].im;
  }
}

/*-----------------------------------------------------------------------
 * Function:     hpoint_to_cvector
 * Description:  convert an Hpoint to a Cvector
 * Arguments IN: *p: the Hpoint
 *          OUT: Z: the resulting Cvector
 * Returns:      nothing
 * Notes:        
 */
hpoint_to_cvector(Z,p)
     Cvector Z;
     Hpoint *p;
{
  Z[0].inf = Z[1].inf = Z[2].inf = NO;
  Z[0].re = 2 * (p->hor.re);
  Z[0].im = 2 * (p->hor.im);
  Z[1].re = 1 - (SQR(p->hor.re) + SQR(p->hor.im));
  Z[1].im =  -2 * (p->ver);
  Z[2].re = 2 - Z[1].re; 
  Z[2].im = - Z[1].im;    
}

/*-----------------------------------------------------------------------
 * Function:     cvector_to_chain
 * Description:  compute a chain from its pole
 * Arguments IN: Z: the pole
 *          OUT: *c: the chain
 *               *success: indicates success
 * Returns:      nothing
 * Notes:        
 */
cvector_to_chain(c,Z,success)
     Chain *c;
     Cvector Z;
     int *success;
{
  Complex w,w2,h;
  
  Herm(&h,Z,Z);                 /* first check to see if Z is positive */ 
  if ( h.re < -fudge ) {
    *success = NO;
    return;
  }
  C_ADD(w,Z[1],Z[2]);
  if( C_ISEQUAL(w,C_ZERO) ) {      /* test to see if chain is vertical */
/* old code:
	C_PUSH_S(Z[0]);
	C_PUSH_S(Z[2]); C_PUSH_S(Z[1]); c_sub_s();
	c_div_s();
	c_sca_mul_s(0.5); */

        C_PUSH_S(Z[2]);	C_PUSH_S(Z[0]); c_div_s(); c_bar_s();
	         /* zeta = conjugate(Z[2]/Z[0]) */

	C_POP_S(c->cen.hor);
        c->cen.ver = 0;
        c->rad   = CHAIN_VERTICAL;
  }
  else {
/* these are the coordinates of the point outside the Siegel domain */
	C_PUSH_S(Z[0]); C_PUSH_S(w); c_div_s();
	C_POP_S(c->cen.hor);
	C_PUSH_S(Z[2]); C_PUSH_S(Z[1]); c_sub_s();
	C_PUSH_S(w); c_div_s();
	c_sca_mul_s(0.5);
	C_POP_S(w2);
	c->cen.ver = w2.im;
	c->rad = sqrt( fabs(C_ENORM_SQ(c->cen.hor) - 2*w2.re) );
  }
  *success = YES;
}

/*-----------------------------------------------------------------------
 * Function:     cvector_to_hpoint
 * Description:  compute an hpoint from null cvector
 * Arguments IN: Z: the null Cvector
 *          OUT: *p: the Hpoint result
 *               *success: indicates success
 * Returns:      nothing
 * Notes:        wmg added this 8/14/88
 *               formula:
 *	       	           Z[0]      
 *	       p.hor = -----------   
 *		       Z[1] + Z[2]   
 *
 *                              Z[2] - Z[1]   	       	     
 *             p.ver = Im { ------------------- }      	     
 *                           2 * (Z[1] + Z[2])  	
 */
cvector_to_hpoint(p,Z,success)
     Hpoint *p;
     Cvector Z;
     int *success;
{
  Complex h,a;
  
  Herm(&h,Z,Z);                 /* first make sure Z is null */ 
  if (C_EDIST(h,C_ZERO)>fudge) {
    *success = NO;
    return; 
  }
  else {
    C_PUSH_S(Z[1]); C_PUSH_S(Z[2]); c_add_s(); C_POP_S(a);
    C_PUSH_S(Z[0]); C_PUSH_S(a);    c_div_s(); C_POP_S(p->hor);        
    C_PUSH_S(Z[2]); C_PUSH_S(Z[1]); c_sub_s(); C_PUSH_S(a);
    c_div_s(); C_POP_S(a);
    p->ver = a.im / 2;
    *success = YES;
  }
}

/*-----------------------------------------------------------------------
 * Function:     htransl_to_matrix
 * Description:  computes the matrix corresponding to a Heisenberg
 *                 translation
 * Arguments IN: *p: the hpoint
 *          OUT: M: the 3x3 complex matrix
 * Returns:      nothing
 * Notes:        
 *               The translation maps 0 to p
 *               0 corresponds to cvector [0,1,1] and 
 *               the point at infinity  is [0,-1,1]
 *
 *                       |  1     z     z  |
 *                       |  _              |
 *        formula:  g =  | -z    1-r   -r  |
 *                       |  _              | 
 *                       |  z     r    1+r |
 *
 *        where     r =  1/2 |z|^2 + iv
 */
htransl_to_matrix(M,p)
     Complex M[3][3];
     Hpoint *p;
{
  Complex rho;

  rho.re = C_ENORM_SQ(p->hor) /2 ;
  rho.im = p->ver;
  rho.inf = NO;
  
  C_COPY(M[0][0],C_ONE);
  C_PUSH_S(p->hor);  C_POP_S(M[0][1]);
  C_COPY(M[0][2], M[0][1]);
  C_PUSH_S(p->hor); c_bar_s();  C_POP_S(M[2][0]);
  C_PUSH_S(M[2][0]);c_neg_s();  C_POP_S(M[1][0]);
  
  C_PUSH_S(C_ONE); C_PUSH_S(rho); c_sub_s(); C_POP_S(M[1][1]); 
  C_PUSH_S(rho); c_neg_s(); C_POP_S(M[1][2]); 
  C_COPY(M[2][1],rho);
  C_PUSH_S(C_ONE); C_PUSH_S(rho); c_add_s(); C_POP_S(M[2][2]); 
}

/*-----------------------------------------------------------------------
 * Function:     matrix_to_rcircle(rc,M)
 * Description:  computes rcircle corresponding to a unitary symmetric
 *               matrix
 * Arguments IN: M  the unitary symmetric matrix
 *          OUT: rc: the rcircle
 * Returns:      nothing
 * Notes:        
 *
 *   Remember that the way that a matrix g in U(2,1) acts on a unsym
 *                                      
 *                                                  2,1
 *   matrix A corresponding to a Real structure on C   is by
 *
 *                                    _ -1
 *                    g : A |--> g A (g)
 *                                      
 *
 *   The unitary symmetric matrix corresponding to an rcircle 
 *   centered at the origin is:
 *
 *                          |  exp(2it)       0            0       |
 *   (unit complex number)* |     0       (R+R^-1)/2   (R-R^-1)/2  |
 *                          |     0       (R^-1-R)/2   (-R-R^-1)/2 |
 *
 *   where c_rad = R^2 * exp(2it) is the complex radius.
 *
 */
matrix_to_rcircle(rc,M,success)
     Rcircle *rc;
     Cmatrix M;
     int *success;
{
  Cmatrix A, B, C, D;
  Complex a, vv;
  Cvector V;  /* corresponds to center of rcircle */
  Hpoint p;   /* opposite of center of rcircle    */
  Cvector V0; 
  Hpoint p0;   
  Complex b, temp;

	/* cvector corresponding to zero in Heisenberg space: */
  Cvector Z_H_ZERO;
	/* cvector corresponding to infinity in Heisenberg space: */
  Cvector Z_H_INF;
  
  Z_H_INF[0] = C_ZERO;
  Z_H_INF[1] = C_NEG_ONE;
  Z_H_INF[2] = C_ONE;
  
  Z_H_ZERO[0] = C_ZERO;
  Z_H_ZERO[1] = C_ONE;
  Z_H_ZERO[2] = C_ONE;
  
  /* First find center of rcircle by conjugating Z_H_INF in M */
  mult_mat(V,M,Z_H_INF,3,3,3,1);
  C_ADD(vv,V[1],V[2]);
  
  if((C_ENORM_SQ(vv)+C_ENORM_SQ(V[0])) < fudge) {
    /* rcircle is infinite (center = infinity) */ 
    rc->finite = 0;	             /* Set the flag */
    mult_mat(V0,M,Z_H_ZERO,3,3,3,1); /* V0 corresponds to image
					of ZERO under conjugation in rc */
    C_SUB(vv,V0[1],V0[2]);          /* vv=0  <=> rc crosses vert axis*/
    if( fabs(vv.re) < fudge)     /* test to see if rc crosses  */
      { 			    /* vertical axis */ 
	/*    infinite rcircle:
	 *	Im(exp(it)) = v - v0 = 0
	 *	corresponds to matrix
	 *	     |	exp(2it)    0         0        |
	 *	     |  0           1-2i v0   -2i v0   |
	 *	     |  0           2i v0     1+2i v0  |  
	 *	"center" member of rc is (0,0,v0)
	 *	"c_rad"  member of rc is exp(2it)
	 */
	rc->cen.hor = C_ZERO;
	C_ADD(temp, M[1][1], M[2][2]);
	C_PUSH_S(M[2][2]);C_PUSH_S(M[1][1]);c_sub_s(); 
		/* stack: M22 - M11 */
        c_sca_mul_s(0.5); C_PUSH_S(temp);        
		/* stack: temp = M22 + M11, (M22 - M11)/2  */
	c_div_s(); C_POP_S(b); 
		/* stack clear; b = (M22 + M11)/2(M22-M11) */
	rc->cen.ver = b.im;

	C_PUSH_S(M[0][0]);C_PUSH_S(temp);  c_div_s(); c_sca_mul_s(2.0);
      		/* stack: M00/(M22+M11) */
	C_POP_S(rc->c_rad); 
		/* c_rad is unit complex number */
      }
    else {	/* now we do the generic case; rcircle doesn't */
                                           /* cross vert axis: */
      cvector_to_hpoint(&p0,V0, success);  /* conjugate origin in rc */
      C_SCA_MUL(rc->cen.hor, 0.5, p0.hor);
      rc->cen.ver = p0.ver * 0.5;	   /* "center" of rcircle is */
      				/* midpoint of origin and its conjugate; */
				/* c_rad ignored; */
    }
  }
  else {				/* rcircle is finite */ 
    rc->finite = 1;	             	/* Set the flag */
    cvector_to_hpoint(&(rc->cen),V,success);
    if (!(*success)) return;
      				/* translate to get rcircle centered at 0 */
    htransl_to_matrix(B,&(rc->cen));
    hpoint_neg(&p,&(rc->cen));
    htransl_to_matrix(A,&p);  	/* B and A are inverses */
    cconj_matrix(C, B, 3, 3);
    mult_mat(D,M,C,3,3,3,3);
    mult_mat(C,A,D,3,3,3,3);
    				/* C is now the unitary symmetric 
				 * matrix  corresponding to an
				 * rcircle centered at 0 	 */
    C_ADD(a, C[1][1], C[1][2]);
    if (C_ENORM(a)<fudge) {	/* i.e. if a is 0 */
      *success = NO;
      return;
    }
    C_PUSH_S(C[0][0]);
    C_PUSH_S(a); c_div_s();
    c_sca_mul_s(C_ENORM_SQ(a));
    C_POP_S(rc->c_rad);
  }
  
  *success = YES;
  return;
}

/*-----------------------------------------------------------------------
 * Function:     rcircle_to_matrix(rc,M)
 * Description:  computes a unitary symmetric matrix
 * 		 corresponding to an rcircle
 * Arguments IN: rc the rcircle
 *          OUT: M  the unitary symmetric matrix
 * Returns:      nothing
 * Notes:        bug found 9/27/88 in translation routine	
 */
rcircle_to_matrix(M,rc)
     Rcircle *rc;
     Cmatrix M;
{
  Cmatrix A,B,C,D,E,F;
  double a,r;
  Complex R1,R2,R3,R4, z;
  Hpoint p;
  
  if (rc->finite) {		/* rcircle finite */
    r = C_ENORM(rc->c_rad);
    R1.re = (r + (1 / r))/2; 
    R2.re = (r - (1 / r))/2;
    R1.im = 0;  R1.inf = NO;
    R2.im = 0;  R2.inf = NO;
    C_NEG(R3,R2);
    C_NEG(R4,R1);
    A[0][0].re = (rc->c_rad.re)/r;
    A[0][0].im = (rc->c_rad.im)/r;
    A[0][0].inf = NO;
    A[0][1] = A[0][2] = C_ZERO;
    A[1][0] = C_ZERO;
    A[1][1] = R1;
    A[1][2] = R2;
    A[2][0] = C_ZERO;
    A[2][1] = R3;
    A[2][2] = R4;
    
    htransl_to_matrix(B,&(rc->cen));
    hpoint_neg(&p,&(rc->cen));
    htransl_to_matrix(C,&p); 
    
    cconj_matrix(D,C,3,3);	/* D is now Cbar */
    mult_mat(E,A,D,3,3,3,3);	/* E = AD = A Cbar */
    mult_mat(M,B,E,3,3,3,3);	/* M = BE = B A Cbar = B A Bbar^(-1) */
  }				/* END OF FINITE RCIRCLE CASE */
  
  else {			/* rcircle infinite */
    a = C_ENORM_SQ(rc->cen.hor);
    if (a < fudge) {		/* rcircle crosses vert axis? */
      M[0][0] = rc->c_rad;
      M[0][1] = C_ZERO;
      M[0][1] = C_ZERO;	
      M[0][2] = C_ZERO;
      M[1][0] = C_ZERO;
      M[2][0] = C_ZERO;
      z.re = 0; z.im = 2 * rc->cen.ver; /* z  = 2i v0 */
      C_SUB(M[1][1], C_ONE, z);
      C_NEG(M[1][2], z);
      M[2][1] = z;
      C_ADD(M[2][2], C_ONE, z);
    }
    else {
      /*
       * now we do the case that the rcircle misses vert axis the matrix
       * corresponding to infinite rcircle with rc->cen = (1+0i,0) is the
       * line with Heisenberg coordinates (z,v) Re(z)=1, Im(z) = v and
       * corresponds to the matrix:
       *
       *	| -1  2  2 |
       *	|  2 -1 -2 |
       *	| -2  2  3 | 
       * with conjugation given by
       *                _
       * (z,v) |-> (2 - z, -v + 2 Im(z) ) in Heisenberg coordinates
       */
      A[0][0].re = -1.0; A[0][0].im = 0.0; A[0][0].inf = NO;
      A[0][1].re =  2.0; A[0][1].im = 0.0; A[0][1].inf = NO;
      A[0][2].re =  2.0; A[0][2].im = 0.0; A[0][2].inf = NO;
      
      A[1][0].re =  2.0; A[1][0].im = 0.0; A[1][0].inf = NO;
      A[1][1].re = -1.0; A[1][1].im = 0.0; A[1][1].inf = NO;
      A[1][2].re = -2.0; A[1][2].im = 0.0; A[1][2].inf = NO;
      
      A[2][0].re = -2.0; A[2][0].im = 0.0; A[2][0].inf = NO;
      A[2][1].re =  2.0; A[2][1].im = 0.0; A[2][1].inf = NO;
      A[2][2].re =  3.0; A[2][2].im = 0.0; A[2][2].inf = NO;
      
      p.hor = C_ZERO;
      p.ver = rc->cen.ver; 
      htransl_to_matrix(B,&p);	/* B corresponds to vertical translation
                                 *        _	
                                 *  and B B = I. */
      r = C_ENORM(rc->cen.hor);
      C_PUSH_S(z); c_sca_mul_s(1/r);C_POP_S(z);	/* make z unit length */
      
      C[0][0] = z; 
      C[0][1] = C_ZERO;
      C[0][2] = C_ZERO;
      C[1][0] = C_ZERO;
      C[2][0] = C_ZERO;
      C[1][1].re = (1/r + r)/2; C[1][1].im = 0.0; C[1][1].inf = NO;
      C[1][2].re = (1/r - r)/2; C[1][2].im = 0.0; C[1][2].inf = NO;
      C[2][1] = C[1][2];
      C[2][2] = C[1][1];
      
      D[0][0] = C[0][0];
      D[0][1] = C_ZERO;
      D[0][2] = C_ZERO;
      D[1][0] = C_ZERO;
      D[2][0] = C_ZERO;
      D[1][1] = C[1][1];
      C_NEG(D[1][2], C[1][2]);
      D[2][1] = D[1][2];
      D[2][2] = D[1][1];
      /*
       *	C corresponds to complex dilation 
       *                	                  2
       *        	(z,v) |-> (r exp(it) z,  r  v )
       *
       *	r exp(it) and is given by matrix
       * 
       *		|  exp(it)     0             0       |  
       *		|     0   (1/r + r)/2   (1/r - r)/2  |
       *		|     0   (1/r - r)/2   (1/r + r)/2  |  
       *
       *	and D is its conjugate-inverse
       */
      
      mult_mat(E,B,C,3,3,3,3); 	/* compose vert trans and dilation */
      /* they should commmute! */
      mult_mat(F,B,D,3,3,3,3); 	/* F = conj-inverse of E */
      mult_mat(C,E,A,3,3,3,3); 	/* Now C = E A */
      mult_mat(M,C,F,3,3,3,3);
      /*
       * M = C F = E A F = 
       *            _  -1
       *          T A T
       * where T is the transl-dilation E
       */
    }
  }
}
