/* $Id: complex_math.c,v 4.0 89/06/06 15:38:34 mbp Exp $
 *
 * complex_math.c: high level complex number math procedures
 */

/***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/


#include "heisenberg.h"

/*-----------------------------------------------------------------------
 * Function:     left_half_plane_to_disk
 * Description:  maps left half plane to unit disk
 * Arguments IN: z a complex number
 * Returns:      a complex number
 * Notes:        z |--> (1+z) / (1-z)
 *		 0 |--> 1, i |--> i, infinity |--> -1
 */
left_half_plane_to_disk(ans, z)
    Complex *ans,*z;
{
  C_PUSH_S(C_ONE); c_push_s(z); c_add_s();
  C_PUSH_S(C_ONE); c_push_s(z); c_sub_s();
  c_div_s();
  c_pop_s(ans);
}

/*-----------------------------------------------------------------------
 * Function:     disk_to_t0_upper_half_plane
 * Description:  maps unit disk to {Im(z-t0) > 0}
 * Arguments IN: *z,*t0 complex numbers
 *          OUT: *ans = answer
 * Returns:      nothing
 * Notes:        t0 should be purely imaginary
 *
 * 		       z (t0 -i) + (t0 +i)
 *		z |--> -------------------
 *		               z + 1
 *
 *		1 |--> t0, i |--> t0 + 1, -1 |--> infinity
 */
disk_to_t0_upper_half_plane(ans, t0, z)
    Complex *ans,*z,*t0;
{
  c_push_s(t0); C_PUSH_S(C_I); c_sub_s(); 
  c_push_s(z); c_mul_s();
  c_push_s(t0); C_PUSH_S(C_I); c_add_s(); c_add_s();
  C_PUSH_S(C_ONE); c_push_s(z); c_add_s();
  c_div_s();
  c_pop_s(ans);
}
  
/*-----------------------------------------------------------------------
 * Function:	disk_to_C2_ball
 * Description:	maps Poincare disk to intersection of B2 with a
 *		complex line.
 * Args	IN:	*z: a point in complex plane of the Poincare disk
 *		t0: magic number (computed with compute_spinal_t0)
 *	OUT:	s: coordinates of answer
 * Returns:     nothing	
 * Notes:       If Z1,Z2 are the points which gave t0, the map takes
 *
 *		        (1 + z)(1 - t0)       [ (1 + z)(1 + t0)    ]
 *		z |-->  --------------- Z1  + [ --------------- - z] Z2
 *		              2		      [        2           ]
 *
 *		This procedure sets s[0] and s[1] to the coefficients
 *		of Z1 and Z2, respectively.
 */
disk_to_C2_ball(s, t0, z)
    Complex s[2], *t0, *z;
{
  C_PUSH_S(C_ONE); c_push_s(t0); c_sub_s(); 
  C_PUSH_S(C_ONE); c_push_s(z); c_add_s(); 
  c_mul_s(); c_sca_mul_s(0.5);
  C_POP_S(s[0]);

  C_PUSH_S(C_ONE); c_push_s(t0); c_add_s(); 
  C_PUSH_S(C_ONE); c_push_s(z); c_add_s(); 
  c_mul_s(); c_sca_mul_s(0.5);
  c_push_s(z); c_sub_s();
  C_POP_S(s[1]);
}

/*-----------------------------------------------------------------------
 * Function:     Herm
 * Description:  Hermitian inner product of two Cvectors in C(2,1)
 * Arguments IN: Z1,Z2: the two Cvectors
 *           OUT: *z: their product
 * Returns:      nothing
 * Notes:        This is the INDEFINITE Hermition inner prod.
 */
Herm(z,Z1,Z2)
     Complex *z;
     Cvector Z1,Z2;
{
  C_PUSH_S(Z1[0]); C_PUSH_S(Z2[0]); c_bar_s(); c_mul_s();
  C_PUSH_S(Z1[1]); C_PUSH_S(Z2[1]); c_bar_s(); c_mul_s();
  c_add_s();
  C_PUSH_S(Z1[2]); C_PUSH_S(Z2[2]); c_bar_s(); c_mul_s();
  c_sub_s();
  c_pop_s(z);
}

/*-----------------------------------------------------------------------
 * Function:     cross_product
 * Description:  computes a Cvector Hermitian-orthogonal to two Cvectors
 * Arguments IN: Z2,Z3: two Cvectors in C21
 *           OUT: Z1: a Cvector such that <Z2,Z1> = <Z3,Z1> = 0
 * Returns:      nothing
 * Notes:        
 */
cross_product(Z1,Z2,Z3)
Cvector Z1,Z2,Z3;
{
  C_PUSH_S(Z2[2]); C_PUSH_S(Z3[1]); c_mul_s();
  C_PUSH_S(Z2[1]); C_PUSH_S(Z3[2]); c_mul_s();
  c_sub_s();  c_bar_s();  C_POP_S(Z1[0]);

  C_PUSH_S(Z2[0]); C_PUSH_S(Z3[2]); c_mul_s();
  C_PUSH_S(Z2[2]); C_PUSH_S(Z3[0]); c_mul_s();
  c_sub_s();  c_bar_s();  C_POP_S(Z1[1]);

  C_PUSH_S(Z2[0]); C_PUSH_S(Z3[1]); c_mul_s();
  C_PUSH_S(Z2[1]); C_PUSH_S(Z3[0]); c_mul_s();
  c_sub_s();  c_bar_s();  C_POP_S(Z1[2]);
}

/*-----------------------------------------------------------------------
 * Function:     cconj_matrix
 * Description:  take complex conjugate of a matrix
 * Arguments IN: 
 *          OUT: 
 * Returns:      
 * Notes:        A[IND(i,j)] = *( A + IND(i,j) )
 */
cconj_matrix(A, B, r, c)
Complex *A, *B;
int r,c;
{
#define IND(i,j) (r*i+j)

  int i,j;

  for (i=0; i<r; ++i)
    for (j=0; j<c; ++j)
      C_BAR(A[IND(i,j)],B[IND(i,j)]);
}
