/* $Id: window.c,v 2.3 89/09/20 17:01:47 mbp Exp $
 *
 * window.c: window definitions and procedures
 *
 * This module is much too long and should really be broken up into
 * smaller pieces.  But for now, just so I can pack everything up in
 * shar files of 50K or smaller, this module consists of two pieces.
 * This is part 1.  Part 2 is the file 'window2.c' and is #included
 * below.
 */

/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include "gr.h"
#include "internal.h"
#include "images.h"
#include <sunwindow/defaults.h>

/************************************************************************
 *	  DECLARATIONS COMMON TO 2D VERSION AND 3D VERSION:             *
 ************************************************************************/

#define MOUSE_MESSAGE_LENGTH			12

#define REGULAR_FONT_FILE "/usr/lib/fonts/fixedwidthfonts/cour.r.14"
#define BOLD_FONT_FILE    "/usr/lib/fonts/fixedwidthfonts/cour.b.14"

Frame		GR_base_frame = NULL;
Panel		GR_control_panel;
Canvas		GR_canvas;
Panel		GR_app_panel;
Panel		GR_error_panel;
Pixwin		*GR_canvas_pw;

/* Gadgets for application panel: */
static Panel_item app_message[2], mouse_image[7], mouse_message[7];
static int	  mouse_image_x[7]   = {  2, 121, 239, 358, 477, 596, 714};
static int	  mouse_image_y[7]   = {  3,   3,   3,   3,   3,   3,   3};
static int	  mouse_message_x[7] = { 19, 137, 257, 375, 494, 613, 731};
static int	  mouse_message_y[7] = {  3,   3,   3,   3,   3,   3,   3};
static int	  app_message_x[4]   = { 10, 10};
static int	  app_message_y[4]   = { 27, 46};
static Panel_item app_text;
static Panel_item app_coordinate_message;

/* Gadgets for control panel: */
static Panel_item quit_button, help_button, print_button=NULL;

/* Gadgets for error panel: */
static Panel_item	error_message;

/* Button menus: */
static Menu     help_button_menu, quit_button_menu, print_button_menu=NULL;

/* Application menu: */
static Menu	app_menu;
/* Procedure to do operations with app_menu on the fly */
static Menu	app_menu_operation();

/* Notification procedures: */
static int	quit_button_proc(), help_button_proc();

/* Event procedures: */
static int	button_event_proc();
static int	canvas_event_proc();

/* Default icon: */
static short default_icon_image[] = {
#include "images/gr.image"
};
DEFINE_ICON_FROM_IMAGE(default_icon, default_icon_image);

/* Miscellaneous Global Variables: */
Pixfont		*GR_regular_font=NULL, *GR_bold_font=NULL;
static int	error_message_present = 0;
static Notify_value interposer();
static int	register_interposer_window();
static int	hold_error_message = 0;
int		(*GR_redraw_proc)() = NULL;

/************************************************************************
 *		 DECLARATIONS SPECIFIC TO 2D VERSION:                   *
 ************************************************************************/

#ifndef THREE_D

/* 2D version:
 * Main frame (GR_base_frame) has 4 subwindows:
 * ___________________________________________________________
 * |                                          |              |
 * |            application                   |   control    |
 * |               panel                      |    panel     |
 * |                                          |              |
 * |---------------------------------------------------------|
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |                       canvas                            |
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |                                                         |
 * |---------------------------------------------------------|
 * |                     error panel                         |
 * |---------------------------------------------------------|
 */

int		GR_canvas_width  = 990;
int		GR_canvas_height = 692;

#endif  /* end of 2D specific declarations */

/************************************************************************
 *		 DECLARATIONS SPECIFIC TO 3D VERSION:                   *
 ************************************************************************/

#ifdef THREE_D

int		GR_canvas_width  = 835;
int		GR_canvas_height = 692;

#define VIEW_CONTROL_PANEL_MESSAGE_LENGTH	15

#define VIEW_DATA_TEXT_LENGTH			30

#define ROTATION_ANGLE_SLIDER_DEFAULT		12
#define ZOOM_FACTOR_SLIDER_DEFAULT		15
#define PERSPECTIVE_FACTOR_SLIDER_DEFAULT	15
#define PAN_FACTOR_DEFAULT			0.1
#define DRAWING_MODE_DEFAULT			1

/* 3D version:
 * Main frame (GR_base_frame) has 5 subwindows:
 * ___________________________________________________________
 * |                                          |              |
 * |            application                   |   control    |
 * |               panel                      |    panel     |
 * |                                          |              |
 * |---------------------------------------------------------|
 * |                                          |              |
 * |                                          |              |
 * |                                          |              |
 * |                                          |    view      |
 * |              canvas                      |   control    |
 * |                                          |    panel     |
 * |                                          |              |
 * |                                          |              |
 * |                                          |              |
 * |                                          |              |
 * |                                          |              |
 * |                                          |              |
 * |---------------------------------------------------------|
 * |                     error panel                         |
 * |---------------------------------------------------------|
 */

Panel		GR_view_control_panel;

/* Gadgets for view control panel: */
static Panel_item horizontal_sweep_button, vertical_sweep_button, tilt_button;
static Panel_item horizontal_pan_button, vertical_pan_button;
static Panel_item perspective_button;
static Panel_item zoom_button, redraw_button, reset_button;
static Panel_item anim_button;
static Panel_item view_data_button;
static Panel_item rotation_angle_label, rotation_angle_value;
static Panel_item rotation_angle_slider;
static Panel_item zoom_factor_label, zoom_factor_value;
static Panel_item zoom_factor_slider;
static Panel_item perspective_factor_label, perspective_factor_value;
static Panel_item perspective_factor_slider;
static Panel_item drawing_mode_cycle;
static Panel_item view_control_panel_message[4];
static int	  view_control_panel_message_x[4] = { 14,  14,  14,  14};
static int	  view_control_panel_message_y[4] = {550, 570, 590, 610};

/* View data subframe has only one panel in it, for displaying
 * current viewing transformation data */
static Frame		view_data_frame = NULL;
static Panel		view_data_panel = NULL;

/* Gadgets for view data panel: */
static Panel_item view_data_redraw_button, view_data_done_button;
static Panel_item	eye_text, focus_text, up_text;
static Panel_item	vpw_message, vpw_horizontal_text, vpw_vertical_text;

/* Button menus: */
static Menu	horizontal_pan_button_menu;
static Menu	horizontal_sweep_button_menu, perspective_button_menu;
static Menu	anim_button_menu,reset_button_menu;
static Menu	tilt_button_menu, vertical_pan_button_menu;
static Menu	vertical_sweep_button_menu, view_data_button_menu;
static Menu	zoom_button_menu;
static Menu	redraw_button_menu;

/* Notification procedures: */
static int	view_data_button_proc(), view_data_done_button_proc();
static int	rotation_angle_slider_proc(), zoom_factor_slider_proc();
static int	perspective_factor_slider_proc();
static int	drawing_mode_cycle_proc();
static int	view_button_proc();
static int	anim_button_proc();

/* Miscellaneous procedures: */
static char	*rotation_angle_string(), *zoom_factor_string();
static char	*perspective_factor_string();
extern int	gr_redraw();

#define SLIDER_VALUE_TO_ROTATION_ANGLE(x) ((double)(x))
#define SLIDER_VALUE_TO_ZOOM_FACTOR(x) ((double)(x)/10.0)
#define SLIDER_VALUE_TO_PERSPECTIVE_FACTOR(x) ((double)(x)/10.0)

int	GR_rotation_angle_slider_value = ROTATION_ANGLE_SLIDER_DEFAULT;
int	GR_zoom_factor_slider_value = ZOOM_FACTOR_SLIDER_DEFAULT;
int	GR_perspective_factor_slider_value = PERSPECTIVE_FACTOR_SLIDER_DEFAULT;
double	GR_pan_factor = PAN_FACTOR_DEFAULT;
int	GR_drawing_mode_value = DRAWING_MODE_DEFAULT;

#endif /* end of 3D specific declarations */

/************************************************************************
 *	PUBLIC PROCEDURES COMMON TO 2D VERSION AND 3D VERSION:          *
 ************************************************************************/

/*-----------------------------------------------------------------------
 * Function:     GR_initialize_windows
 * Description:  initializes the SunView window setup
 * Arguments:	 (none)
 * Returns:      0 for success, 1 for failure
 */
int
GR_initialize_windows(argc, argv)
     int argc;
     char **argv;
{
  int i;
  char *nullargv=NULL;

  GR_regular_font = pf_open(REGULAR_FONT_FILE);
  if (GR_regular_font==NULL) GR_regular_font = pf_default();
  GR_bold_font = pf_open(BOLD_FONT_FILE);
  if (GR_bold_font==NULL) GR_bold_font = pf_default();

  GR_base_frame =
    window_create(NULL, FRAME,
		  WIN_X, 80,
		  WIN_Y, 30,
		  WIN_FONT, GR_regular_font,
		  FRAME_ICON, &default_icon,
		  FRAME_LABEL, "GR",
		  FRAME_ARGS, argc, (argc==0 ? &nullargv : argv),
		  0);
  if (GR_base_frame == NULL) goto BAD_WINDOW_ERROR;
  
  GR_app_panel =
    window_create(GR_base_frame, PANEL,
		  WIN_X, 0,
		  WIN_Y, 0,
		  WIN_WIDTH, 835,
		  WIN_HEIGHT, 80,
		  WIN_FONT, GR_regular_font,
		  0);
  if (GR_app_panel == NULL) goto BAD_WINDOW_ERROR;

  for (i=0; i<2; ++i) {
    app_message[i] =
      panel_create_item(GR_app_panel, PANEL_MESSAGE, 
			PANEL_ITEM_X, app_message_x[i],
			PANEL_ITEM_Y, app_message_y[i],
			PANEL_LABEL_STRING, "",
			0);
  }

  for (i=0; i<7; ++i) {
    mouse_image[i] =
      panel_create_item(GR_app_panel, PANEL_MESSAGE, 
			PANEL_ITEM_X, mouse_image_x[i],
			PANEL_ITEM_Y, mouse_image_y[i],
			PANEL_LABEL_IMAGE, GR_mouse_image_pr_ptr(i),
			PANEL_SHOW_ITEM, FALSE,
			0);
   mouse_message[i] =
     panel_create_item(GR_app_panel, PANEL_MESSAGE, 
		       PANEL_ITEM_X, mouse_message_x[i],
		       PANEL_ITEM_Y, mouse_message_y[i],
		       PANEL_LABEL_STRING, "",
		       PANEL_LABEL_FONT, GR_bold_font,
		       0);

  }

  app_text =
    panel_create_item(GR_app_panel, PANEL_TEXT,
		      PANEL_ITEM_X,		10,
		      PANEL_ITEM_Y,		64,
		      PANEL_LABEL_STRING,	":",
		      PANEL_BLINK_CARET,	TRUE,
		      PANEL_VALUE_STORED_LENGTH,  GR_INPUT_TEXT_LENGTH,
		      PANEL_VALUE_DISPLAY_LENGTH, GR_INPUT_TEXT_LENGTH,
		      PANEL_SHOW_ITEM,		FALSE,
		      0);

  app_coordinate_message =
    panel_create_item(GR_app_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		10,
		      PANEL_ITEM_Y,		64,
		      PANEL_LABEL_STRING,	"",
		      0);

  GR_canvas =
    window_create(GR_base_frame, CANVAS,
		  WIN_X, 0,
		  WIN_Y, 85,
		  WIN_WIDTH, GR_canvas_width,
		  WIN_HEIGHT, GR_canvas_height,
		  WIN_EVENT_PROC,	canvas_event_proc,
		  WIN_IGNORE_KBD_EVENTS, KBD_USE, KBD_DONE, 0,
		  WIN_CONSUME_KBD_EVENT, 27, /* <ESC> */
		  0);
  if (GR_canvas == NULL) goto BAD_WINDOW_ERROR;
  GR_canvas_pw = canvas_pixwin(GR_canvas);

  app_menu =
    menu_create(MENU_GEN_PROC, app_menu_operation, 0);

  GR_control_panel =
    window_create(GR_base_frame, PANEL,
		  WIN_X, 840,
		  WIN_Y, 0,
		  WIN_WIDTH, 150,
		  WIN_HEIGHT, 80,
		  WIN_FONT, GR_regular_font,
		  0);
  if (GR_control_panel == NULL) goto BAD_WINDOW_ERROR;
  
  help_button =
    panel_create_item(GR_control_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 16,
		      PANEL_ITEM_Y, 16,
		      PANEL_LABEL_IMAGE, &help_pr,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, help_button_proc,
		      0);
  
  help_button_menu =
    menu_create(MENU_STRINGS, "Help", 0, 0);

  quit_button =
    panel_create_item(GR_control_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 86,
		      PANEL_ITEM_Y, 16,
		      PANEL_LABEL_IMAGE, &quit_pr,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, quit_button_proc,
		      0);
  quit_button_menu =
    menu_create(MENU_STRINGS, "Exit Program", 0, 0);

  GR_error_panel =
    window_create(GR_base_frame, PANEL,
		  WIN_X,      0,
		  WIN_BELOW,  GR_canvas,
		  WIN_FONT,   GR_bold_font,
		  WIN_WIDTH,  990,
		  WIN_HEIGHT, ATTR_ROW(1),
		  0);

  error_message =
    panel_create_item(GR_error_panel, PANEL_MESSAGE,
		      WIN_X, 10,
		      WIN_Y, 5,
		      PANEL_LABEL_STRING, "",
		      0);

#ifdef THREE_D
  if (initialize_view_control_panel() != 0) return(1);
  if (GR_print_button(1, gr_redraw, "GR") != 0) return(1);
#endif
  
  window_fit(GR_base_frame);
  GR_register_interposer(GR_base_frame);
  if (GR_initialize_confirmer() != 0) return(1);

  return(0);

  BAD_WINDOW_ERROR:
  GR_error("Can't create any more windows !");
  return(1);
} /* End of GR_initialize_windows() */

/* -----------------------------------------------------------------------
 * Function:     GR_show_app_message
 * Description:  display a string in one of the 2 application panel
 * 		   message lines
 * Arguments IN: n: the number of the message line (0 or 1)
 * 		 s: the string to be displayed
 * Notes:        The message must be less than or equal to
 * 	         GR_MESSAGE_LENGTH chars long (not including
 * 	         the terminal null char).  If it is longer than this, on
 * 	         the first GR_MESSAGE_LENGTH chars are
 * 	         displayed, and 1 is returned.  Otherwise 0 is returned.
 *
 *		 If s is the null string (not the NULL pointer!) the
 *		 message line is cleared.
 */
int
GR_show_app_message(n,s)
     int n;
     char *s;
{
  if ( (n<0) || (n>1) || (s==NULL) ) return(1);
  return( GR_show_message(app_message[n], s, GR_MESSAGE_LENGTH) );
}

/* -----------------------------------------------------------------------
 * Function:     GR_show_app_coordinate_message
 * Description:  display a string in the application coordinate
 * 		   message line
 * Arguments IN: s: the string to be displayed
 * Notes:        The message must be less than or equal to
 * 	         GR_COORDINATE_MESSAGE_LENGTH chars long (not including
 * 	         the terminal null char).  If it is longer than this, on
 * 	         the first GR_COORDINATE_MESSAGE_LENGTH chars are
 * 	         displayed, and 1 is returned.  Otherwise 0 is returned.
 *
 *		 If s is the null string (not the NULL pointer!) the
 *		 message is cleared.
 */
int
GR_show_app_coordinate_message(s)
     char *s;
{
  if (s == NULL) return(1);
  return( GR_show_message(app_coordinate_message, s,
		       GR_COORDINATE_MESSAGE_LENGTH) );
}

/* -----------------------------------------------------------------------
 * Function:     GR_show_mouse_message
 * Description:  display a string in one of the 7 mouse button
 * 		   message lines
 * Arguments IN: n: the number of the mouse button (0,1,2,3,4,5,6)
 * 		 s: the string to be displayed
 * Returns:      0 for success, 1 otherwise
 * Notes:        The message must be less than or equal to
 * 	         MOUSE_MESSAGE_LENGTH chars long (not including
 * 	         the terminal null char).  If it is longer than this, on
 * 	         the first MOUSE_MESSAGE_LENGTH chars are
 * 	         displayed, and 1 is returned.  Otherwise 0 is returned.
 *
 *		 If s is the null string (not the NULL pointer!) the
 *		 n-th mouse message is cleared and the corresponding
 *		 image turned off.
 */
int
GR_show_mouse_message(n,s)
     int n;
     char *s;
{
  if ( (n<0) || (n>6) || (s==NULL) ) return(1);
  if (*s=='\0')
    panel_set(mouse_image[n], PANEL_SHOW_ITEM, FALSE, 0);
  else
    panel_set(mouse_image[n], PANEL_SHOW_ITEM, TRUE, 0);
  return( GR_show_message(mouse_message[n], s, MOUSE_MESSAGE_LENGTH) );
}

/* -----------------------------------------------------------------------
 * Function:     GR_show_error_message
 * Description:  display a string in the error message panel
 * Arguments IN: s: the string to be displayed
 * Notes:        The message must be less than or equal to
 * 	         GR_ERROR_MESSAGE_LENGTH chars long (not including
 * 	         the terminal null char).  If it is longer than this, on
 * 	         the first GR_ERROR_MESSAGE_LENGTH chars are
 * 	         displayed, and 1 is returned.  Otherwise 0 is returned.
 *
 *		 If s is the null string (not the NULL pointer!) the
 *		 message is cleared.
 */
int
GR_show_error_message(s)
     char *s;
{
  if (s == NULL) return(1);
  return( GR_show_message(error_message, s, GR_ERROR_MESSAGE_LENGTH) );
}

/*-----------------------------------------------------------------------
 * Function:	GR_app_panel_get_string
 * Description:	Get a string from the user
 * Args  IN:	s: place to store string
 * Notes:	The maximum allowable length of a string gotten by this
 *		procedure is GR_INPUT_TEXT_LENGTH.
 */
int
GR_app_panel_get_string(s)
     char *s;
{
  Event event;
  int done;
  
  /* Make sure all ascii events from canvas get passed to panel */
  window_set(GR_canvas,
	     WIN_INPUT_DESIGNEE,
	     	(int)window_get(GR_app_panel,WIN_DEVICE_NUMBER),
	     0);

  /* Enable display of text item in panel */
  panel_set(app_text, PANEL_SHOW_ITEM, TRUE, 0 );

  /* Read panel events and update display of text item until
   * user hits return key */
  done = 0;
  do {
    window_read_event(GR_app_panel, &event);
    if (event_is_ascii(&event))
      done = process_ascii_event(&event);
  } while (!done);

  /* Now copy text string to s, clear text item, and return
   * to normal operation */
  strcpy(s, (char*)panel_get(app_text,PANEL_VALUE) );
  panel_set(app_text, PANEL_VALUE, "", 0 );
  panel_set(app_text, PANEL_SHOW_ITEM, FALSE, 0 );
  window_set(GR_canvas,
	     WIN_INPUT_DESIGNEE,
	     	(int)window_get(GR_base_frame, WIN_DEVICE_NUMBER),
	     0 );
}    

/* -----------------------------------------------------------------------
 * Function:	   GR_get_canvas_xy
 * Description:	   Get a point in canvas integer coords from the user
 * Args	 IN:	   bmask: tells which buttons to recognize
 *	OUT:	   x,y: coords of point selected by user
 *		   button: button pressed by user
 * Notes:	   bmask should be a logical OR (|) of GR_LEFT,
 *		   GR_MIDDLE, GR_RIGHT, GR_SHIFT_LEFT, GR_SHIFT_MIDDLE,
 *		   GR_SHIFT_RIGHT, or GR_ESC.  It must include at least
 *		   one of these, or else an error occurrs.
 */
GR_get_canvas_xy(x,y,button,bmask)
     int *x, *y;
     gr_Button *button, bmask;
{
  Event event[1], *event_in_canvas_space;
  int done;
  Cursor cursor;
  
  /* Validate bmask value to make sure we can return */
  if ( ! (  (bmask&GR_LEFT)
          | (bmask&GR_MIDDLE)
          | (bmask&GR_RIGHT)
          | (bmask&GR_SHIFT_LEFT)
          | (bmask&GR_SHIFT_MIDDLE)
          | (bmask&GR_SHIFT_RIGHT)
          | (bmask&GR_ESC) ) )
    return(1);

  /* Change the cursor to a crosshair, and enable acceptance
   * of ascii events for canvas so we can tell if user presses
   * <ESC> key */
  cursor = window_get( GR_canvas, WIN_CURSOR );
  cursor_set(cursor,
	     CURSOR_SHOW_CURSOR,		FALSE,
	     CURSOR_SHOW_CROSSHAIRS,		TRUE,
	     CURSOR_CROSSHAIR_THICKNESS,	1,
	     CURSOR_CROSSHAIR_COLOR,		3,
	     CURSOR_CROSSHAIR_LENGTH, 		CURSOR_TO_EDGE,
	     CURSOR_CROSSHAIR_OP, 		PIX_SRC | PIX_DST,
	     0);
  window_set(GR_canvas,
	     WIN_CONSUME_KBD_EVENT, WIN_ASCII_EVENTS,
	     WIN_CURSOR, cursor,
	     0 );

  /* Read events in canvas until user presses appropriate key: */
  do {
    window_read_event(GR_canvas, event);

    /* Translate coords to canvas coords */
    event_in_canvas_space=canvas_event(GR_canvas, event);
    *x = event_x(event_in_canvas_space);
    *y = event_y(event_in_canvas_space);

    /* If a coord display proc has been specified, call it */
    if (GR_coordinate_display_proc!=NULL) {
      GR_show_app_coordinate_message(
		 (*GR_coordinate_display_proc)(*x, *y));
    }

    /* Now determine if we are done */
    if (!event_is_down(event))
      done = 0;
    else
      switch (event_id(event)) {
      case MS_LEFT:
	*button = event_shift_is_down(event) ?
	  GR_SHIFT_LEFT : GR_LEFT;
	done = ( (*button&bmask)!=0 );
	break;
      case MS_MIDDLE:
	*button = event_shift_is_down(event) ?
	  GR_SHIFT_MIDDLE : GR_MIDDLE;
	done = ( (*button&bmask)!=0 );
	break;
      case MS_RIGHT:
	*button = event_shift_is_down(event) ?
	  GR_SHIFT_RIGHT : GR_RIGHT;
	done = ( (*button&bmask)!=0 );
	break;
      case 27: /* <ESC> */
	*button = GR_ESC;
	done = ( (*button&bmask)!=0 );
	break;
      default:
	done = 0;
	break;
      }
  }  while ( !done );

  /* Return the cursor to normal, and disable acceptance of
   * ascii events in canvas: */
  cursor = window_get( GR_canvas, WIN_CURSOR );
  cursor_set(cursor,
	     CURSOR_SHOW_CURSOR,	TRUE,
	     CURSOR_SHOW_CROSSHAIRS,	FALSE,
	     0);
  window_set(GR_canvas,
	     WIN_IGNORE_KBD_EVENT,	WIN_ASCII_EVENTS,
	     WIN_CURSOR,		cursor,
	     0);

  /* Clear the coordinate display, if we're using it */
  if (GR_coordinate_display_proc!=NULL)
      GR_show_app_coordinate_message("");

  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	GR_print_button
 * Description:	turn the "print" button in the control panel on or off
 * Args  IN:	onoff: 1 to turn on, 0 to turn off
 *		redraw_proc: procedure to be called to redraw screen
 *		logo: string to use as program logo
 * Notes:	redraw_proc specifies a procedure to be called when
 *		generating PostScript.  It should redraw the entire
 *		canvas, and should return 0 to indicate success, 1 to
 *		indicate failure.  logo may be NULL if no logo is
 *		desired.  Both redraw_proc and logo are used only when
 *		onoff=1; their values are ignored when onoff=0.
 */
int
  GR_print_button(onoff, redraw_proc, logo)
int onoff;
int (*redraw_proc)();
char *logo;
{
  static int print_button_initialized = 0;

  if (onoff == 1) {		/* turn print button on */
    if (redraw_proc == NULL) return(1);
    GR_redraw_proc = redraw_proc;
    GR_logo = logo;
    if (print_button_initialized)
      /* If already initialized, just make it visible */
      panel_set(print_button, PANEL_SHOW_ITEM, TRUE, 0);
    else {
      /* Otherwize, initialize it */
      print_button =
	panel_create_item(GR_control_panel, PANEL_BUTTON, 
			  PANEL_ITEM_X, 51,
			  PANEL_ITEM_Y, 16,
			  PANEL_LABEL_IMAGE, &print_pr,
			  PANEL_EVENT_PROC, button_event_proc,
			  PANEL_NOTIFY_PROC, GR_show_ps_frame,
			  0);
      print_button_menu =
	menu_create(MENU_STRINGS, "Printer", 0, 0);
      panel_set(help_button, PANEL_ITEM_X, 1, PANEL_ITEM_Y, 16, 0);
      panel_set(quit_button, PANEL_ITEM_X, 101, PANEL_ITEM_Y, 16, 0);
      panel_paint(print_button, PANEL_CLEAR);
      print_button_initialized = 1;
    }
  }
  else {			/* turn print button off */
    if (print_button_initialized)
      panel_set(print_button, PANEL_SHOW_ITEM, FALSE, 0);
  }
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	GR_error
 * Description:	beep and display a message in the error panel
 * Args  IN:	s: the message to display
 * Notes:	s should be at most GR_ERROR_MESSAGE_LENGTH chars
 */
int
GR_error(s)
     char *s;
{
  int r;

  if (GR_window_open) {
    window_bell(GR_base_frame);
    r = GR_show_error_message(s);
    error_message_present = 1;
  }
  else {
    fprintf(stderr,"%s\n", s);
    r = 0;
  }
  return(r);
}

/*-----------------------------------------------------------------------
 * Function:	GR_message
 * Description:	display a message in the error panel
 * Args  IN:	s: the message to display
 * Notes:	s should be at most GR_ERROR_MESSAGE_LENGTH chars.
 *		This is just like GR_error, but does not beep.
 */
int
GR_message(s)
     char *s;
{
  if (GR_window_open) {
    GR_show_error_message(s);
    error_message_present = 1;
  }
  else
    fprintf(stderr,"%s\n", s);
}

/*-----------------------------------------------------------------------
 * Function:	GR_hold_error_message
 * Description:	hold error message for specified number of mouse clicks
 * Args  IN:	n: number of mouse clicks
 * Notes:	Normally, the message in the error panel is cleared with
 *		the first mouse click after it is displayed.  A call to
 *		this procedure makes keeps it around for n additional
 *		clicks.
 */
int
GR_hold_error_message(n)
     int n;
{
  hold_error_message = ( n >= 0 ? n : -n );
}

/*-----------------------------------------------------------------------
 * Function:	GR_release_error_message
 * Description:	release an error message from being held
 * Args:	(none)
 * Notes:	This undoes the last call to GR_hold_error_message.  Any
 *		error message will be cleared by the next mouse click.
 */
int
GR_release_error_message()
{
  hold_error_message = 0;
}

/*-----------------------------------------------------------------------
 * Function:	GR_register_interposer
 * Description:	register our interposer (procedure "interposer" above)
 *		  for every window in a window tree
 * Args  IN:	frame: root of window tree
 */
int
GR_register_interposer(frame)
     Frame frame;
{
  GR_apply_function_all_windows(frame, register_interposer_window);
}

/*-----------------------------------------------------------------------
 * Function:	GR_mouse_image_pr_ptr
 * Description:	get a pointer to a mouse button image Pixrect
 * Args  IN:	n: number of mouse image (0,1,2,3,4,5,6)
 * Returns:	the pointer to the Pixrect storing that mouse button image
 */
Pixrect *
GR_mouse_image_pr_ptr(n)
int n;
{
  switch (n) {
  case 0: return(&lmouse_pr);
  case 1: return(&mmouse_pr);
  case 2: return(&rmouse_pr);
  case 3: return(&lsmouse_pr);
  case 4: return(&msmouse_pr);
  case 5: return(&rsmouse_pr);
  case 6: return(&esc_pr);
  default: return(NULL);
  }
}

/*-----------------------------------------------------------------------
 * Function:	GR_apply_function_all_windows
 * Description:	apply a function to every window in a hierarchy
 * Args  IN:	window: root of hierarchy
 *		function: ptr to function to apply
 * Returns:	nothing
 * Notes:	calling syntax for function should be:
 *			function(window)
 *			  Window window;
 */
int
GR_apply_function_all_windows(window, function)
     Window window;
     int (*function)();
{
  Window subwindow;
  int winno;

  if (window == NULL) return;

  /* Do it to this window */
  (*function)(window);

  /* Now do it to all subwindows */
  winno = 0;
  do {
    subwindow=(Window)window_get(window, FRAME_NTH_WINDOW, winno);
    if (subwindow != NULL) {
      if ((Window_type)(window_get(subwindow,WIN_TYPE)) == FRAME_TYPE)
	GR_apply_function_all_windows(subwindow, function);
      else
	(*function)(subwindow);
    }
  ++winno;
  } while (subwindow != NULL);

  return;
}
/*-----------------------------------------------------------------------
 * Function:     GR_show_message
 * Description:  display a string in a message item
 * Arguments IN: item: handle of message item to display string in
 *		 s: the string to be displayed
 *		 m: maximum number of chars that can be displayed
 *		   in this message item
 * Returns:      0 for success, 1 for failure
 * Notes:        If s has more than max chars, only the first m are
 *		 displayed and 1 is returned.
 */
int
GR_show_message(item, s, m)
     Panel_item item;
     char *s;
     int m;
{
  int toobig;
  char temp;

  if (toobig = (strlen(s)>m)) {
    temp = s[m];
    s[m] = '\0';
    panel_set(item, PANEL_LABEL_STRING, s, 0);
    s[m] = temp;
  }
  else
    panel_set(item, PANEL_LABEL_STRING, s, 0);
  return(toobig);
}

/************************************************************************
 *	      PUBLIC PROCEDURES SPECIFIC TO 3D VERSION:                 *
 ************************************************************************/

#ifdef THREE_D

/* -----------------------------------------------------------------------
 * Function:     GR_show_view_control_panel_message
 * Description:  display a string in one of the four view control
 * 		   panel message lines
 * Arguments IN: n: the number of the message line to use (0,1,2,3)
 * 		 s: the string to be displayed
 * Returns:      0 for success, 1 otherwise
 * Notes:        The message must be less than or equal to
 * 	         VIEW_CONTROL_PANEL_MESSAGE_LENGTH chars long (not including
 * 	         the terminal null char).  If it is longer than this, on
 * 	         the first VIEW_CONTROL_PANEL_MESSAGE_LENGTH chars are
 * 	         displayed, and 1 is returned.  Otherwise 0 is returned.
 */
int
GR_show_view_control_panel_message(n,s)
     int n;
     char *s;
{
  if ( (n<0) || (n>3) ) return(1);
  return(
	 GR_show_message(view_control_panel_message[n], s,
		      VIEW_CONTROL_PANEL_MESSAGE_LENGTH)
	 );
}

int
GR_set_view_display(itemflag, values)
     int itemflag;
     double *values;
{
  char buf[VIEW_DATA_TEXT_LENGTH+1];
  Panel_item item;

  switch (itemflag) {
  case GR_EYE:
    item = eye_text;
    break;
  case GR_FOCUS:
    item = focus_text;
    break;
  case GR_UP:
    item = up_text;
    break;
  case GR_VPW:
    break;
  default:
    return(1);
    break;
  }

  if (itemflag == GR_VPW) {
    sprintf(buf, "[%5.2f,%5.2f]", values[0], values[1]);
    panel_set_value(vpw_horizontal_text, buf);
    sprintf(buf, "[%5.2f,%5.2f]", values[2], values[3]);
    panel_set_value(vpw_vertical_text, buf);
  }
  else {
    sprintf(buf, "(%5.2f,%5.2f,%5.2f)", values[0], values[1], values[2]);
    panel_set_value(item, buf);
  }
  return(0);
}

int
GR_get_view_display(itemflag, values)
     int itemflag;
     double *values;
{
  char buf[VIEW_DATA_TEXT_LENGTH+1], *c;
  static char *seps = " \t,()[]{};";
  Panel_item item;

  if (itemflag == GR_VPW) {
    strcpy(buf, (char*)panel_get_value(vpw_horizontal_text));
    if ((c=strtok(buf,seps)) == NULL) return(1);
    values[0] = atof(c);
    if ((c=strtok(NULL,seps)) == NULL) return(1);
    values[1] = atof(c);
    strcpy(buf, (char*)panel_get_value(vpw_vertical_text));
    if ((c=strtok(buf,seps)) == NULL) return(1);
    values[2] = atof(c);
    if ((c=strtok(NULL,seps)) == NULL) return(1);
    values[3] = atof(c);
  }
  else {
    switch (itemflag) {
    case GR_EYE:
      item = eye_text;
      break;
    case GR_FOCUS:
      item = focus_text;
      break;
    case GR_UP:
      item = up_text;
      break;
    default:
      return(1);
      break;
    }
    strcpy(buf, (char*)panel_get_value(item));
    if ((c=strtok(buf,seps)) == NULL) return(1);
    values[0] = atof(c);
    if ((c=strtok(NULL,seps)) == NULL) return(1);
    values[1] = atof(c);
    if ((c=strtok(NULL,seps)) == NULL) return(1);
    values[2] = atof(c);
  }
  return(0);
}


#endif  /* End of public 3D version procedures */

#include "window2.c"
