(* $Id: CH2.m,v 2.0 89/10/03 21:38:27 mbp Exp $
 *
 * CH2.m: package for basic computations in CH2
 *)

(***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************)

BeginPackage["CH2`", "Complex`"]

Cvector::usage = "Cvector is both the term used to denote a vector in
C(2,1), and a function which converts objects into Cvectors.  A
Cvector is represented by a list of three complex numbers (the symbol
'Cvector' does not appear in the expression).  When used as a
function, as in Cvector[x], it converts the object x to a Cvector.  x
may be an Hpoint, a Chain, or a Cvector (in which case x is returned
unchanged). A Cvector v is said to be postive/null/negative if
C21Herm[v,v] is greater/equal/less than 0."

CvectorQ::usage = "CvectorQ[z] yields True iff z is a Cvector
(actually it just checks to make sure that z is a vector of length
3)."

Hpoint::usage = "Hpoint is both the head used for points in Heisenberg
space, and a function which null converts Cvectors to Hpoints.
Hpoint[z,v] denotes the point with Heisenberg coordinates (z,v) (z is
complex and v is real); Hpoint[Infinity] denotes the point at
infinity.  If v is a null Cvector, Hpoint[v] yields the Hpoint
corresponding to v."

ComplexGauge::usage = "ComplexGauge[x] gives the complex number
corresponding to the Hpoint x."

Chain::usage = "Chain is both the head used for chains in Heisenberg
space, and a function which converts positive Cvectors to chains.
Chain[x,r] denotes the chain with center at the Hpoint x and real
radius.  Chain[x,Vertical] denotes the vertical chain through the
Hpoint x.  If v is a positive Cvector, Chain[v] yields the chain
corresponding to v."

Vertical::usage = "Vertical is the value used for the radius of a
vertical Chain."

Spinal::usage = "Spinal[x1,x2] denotes the spinal hypersurface with
vertices x1 and x2.  If either of the xi is a null Cvector, it is
converted to an Hpoint."

AngularInvariant::usage = "AngularInvariant[x1,x2,x3] gives Cartan's
angular invariant of the triple x1,x2,x3; the xi may be either Hpoints
or Cvectors."

C21Herm::usage = "C21Herm[u,v] gives the (2,1) Hermitian product of
the Cvectors u and v.  C21Herm[u] gives C21Herm[u,u]."

I21Matrix::usage = "I21Matrix is the 3x3 matrix which defines the
(2,1) Hermitian product on C(2,1)."

C2Herm::usage = "C2Herm[u,v] gives the positive definite Hermitian
scalar product of two vectors u and v in C2. C2Herm[u] gives 
C2Herm[u,u]."

(*??X*)
C21Contrajection::usage = "usage message not yet written"

(*??X*)
C21Projection::usage = "C21Projection[v] returns the matrix which
corresponds to orthogonal projection of C21 onto the complex line
spanned by v. This function is used in C21Reflection."

C21Reflection::usage = "C21Reflection[v, z] gives the matrix in
U(2,1) which reflects about the Cvector v with complex reflection
factor z.  If z is an n-th root of unit, then this reflection has
order n; take n = -1 to get the usual symmetry in v.  If v is a
negative Cvector, this is the symmetry in the point in CH2
corresponding to v.  If v is a positive Cvector, this is the inversion
with respect to the complex geodesic corresponding to v.  If x is a
Chain or Hpoint, C21Reflection[x, z] is converted to
C21Reflection[Cvector[x], z]."

C21Inversion::usage = "C21Inversion[v] gives the matrix in SU(2,1)
which reflects about the Cvector v with complex reflection factor -1
(i.e. of order 2).If x is a Chain or Hpoint, C21Inversion[x] is
converted to C21Inversion[Cvector[x], z]."

HTranslation::usage = "HTranslation[x] gives the matrix in SU(2,1)
corresponding to Heisenberg translation by x.  x may be an Hpoint or a
null Cvector."

ComplexGeodesic::usage = "ComplexGeodesic[x1,x2] gives the Cvector
corresponding the the complex geodesic through x1 and x2; the xi may
be either Hpoints or Cvectors."

ChainThrough::usage = "ChainThrough[x1,x2] gives the chain through x1
and x2; the xi may be either Hpoints or Cvectors."

C21Cross::usage = "C21Cross[u,v] gives the C(2,1)-cross product of the
Cvectors u and v; this is a vector w such that C21Herm[w,u] =
C21Herm[w,v] = 0."

Affine::usage = "Affine[v] converts the vector v to affine coordinates
by dividing through by its last component, which must be nonzero.
Affine works on vectors of any dimension --- it returns a vector of
one less dimension."

Projective::usage = "Projective[v] converts the vector v to projective
coordinates by tacking on a 1.  Projective works on vectors of any
dimension; it returns a vector of one greater dimension."

EnormSquared::usage = "EnormSquared[z] is the square of the Euclidean
(Cn) norm of the vector v, which may be of any dimension."

Bergman::usage = "Bergman[x1,x2] gives the Bergman distance between x1
and x2, which should be negative Cvectors."

BergmanRatio::usage = "BergmanRatio[x1,x2] gives the Bergman ratio of
x1 and x2, which should be negative Cvectors."

(*??X*)
HApply::usage = "HApply[m,object] applies the transformation
determined by the matrix m to the object. m should be an element of
U(2,1) and object can be a Cvector, Chain, Hpoint, Rcircle, or
Spinal." 

CmatrixQ::usage = "CmatrixQ[m] yields True iff m is a complex 3x3 matrix."

InfiniteHpointQ::usage = "InfiniteHpointQ[x] returns True if x is
either the Hpoint at infinity or a Cvector corresponding to the Hpoint
at infinity."

VerticalHpointQ::usage = "VerticalHpointQ[x] returns True if x is
either an Hpoint lying on the vertical axis, or a Cvector
corresponding to an Hpoint on the vertical axis."

Rcircle::usage = "Rcircle is both the head used for R-circles in
Heisenberg space, and a function which converts a unitary symmetric
matrix in U(2,1) (symmetric with respect to the indefinite bilinear
form) to an R-circle.  Rcircle[x, r] denotes the finite rcircle with
center at the Hpoint x and complex radius r.  
Rcircle[x, Infinity] or Rcircle[x, DirectedInfinity[]] denotes 
the infinite rcircle with center x not lying on the vertical axis.  
Rcircle[x, DirectedInfinity[z]] denotes the infinite rcircle with
center x on the vertical axis and direction given by the complex
number z.  Rcircle[m] returns the R-circle corresponding to the
unitary symmetric matrix m."

RealStructure::usage = "RealStructure[r] returns the real structure (a
self-adjoint matrix in U(2,1)) corresponding to the Rcircle r."

RealStructure10::usage = "RealStructure10 is the real structure
corresponding to Rcircle x + v - 1 = y = 0."

RealStructureO::usage = "RealStructureO[radius] returns the real
structure corresponding to the Rcircle centered at the origin with
given complex radius."

TransformRealStructure::usage = "TransformRealStructure[a,m]
transforms the real structure m (a self-adjoint matrix in U(2,1)) by
the element a of (2,1)."

Rconjugate::usage = "Rconjugate[r,u] conjugates u in the Rcircle r.
Rconjugate[m,u] applies the real structure r to the object u.  u may
be a Cvector, Hpoint, Chain, Rcircle, or Spinal."

(*??X*)
DilationMappingToPoint::usage = "DilationMappingToPoint[x] gives
matrix in U(2,1) corresponding to (complex) dilation which takes
Hpoint[1,0] to x. This routine is used to find the real structure
corresponding to an Rcircle."

RealTripleQ::usage = "RealTripleQ[x1,x2,x3] returns true if x1,x2 and
x3 lie on an Rcircle or Rplane; the xi may be either Hpoints or
negative Cvectors."

(*??X*)
HDilation::usage = "HDilation[r,t] gives the element of
U(2,1) corresponding to Heisenberg dilation by r Exp[I t]."

SpinalContaining::usage = "SpinalContaining[c1,c2] returns the Spinal
containing the two chains c1 and c2 as slices."

RealStructureFixing::usage = "RealStructureFixing[z1,z2,z3] gives the real
structure containing the 3 null Cvectors z1,z2,z3 of angular invariant
zero."

RcircleThrough::usage = "RcircleThrough[x1,x2,x3] gives the Rcircle
containing the 3 Hpoints x1,x2,x3 of angular invariant zero."

IncidentQ::usage = "IncidentQ[x,y] tests to see if x and y
are incident objects in Heisenberg geometry.
If x is an Hpoint and y is a Chain, Rcircle or Spinal, 
then IncidentQ[x,y] returns True if y passes through x; 
if x is a Chain or Rcircle, and y is a Spinal, then IncidentQ returns
True if x is a slice or meridian of y respectively."

C21OrthogonalQ::usage = "C21OrthogonalQ[x,y] returns True if x and y are
orthogonal Cvectors (in the indefinite Hermitian structure)"

ChainOrthogonal::usage = "ChainOrthogonal[c_1,c_2] returns the common
orthogonal Chain to c1 and c2, it it exists."

Midpoint::usage = "Midpoint[u,v] gives the midpoint (as a Cvector)
between the points represented by Cvectors u,v"

NullEigenvectors::usage = "NullEigenvectors[m] returns a list of the
null (w.r.t. C21Herm) eigenvectors of the matrix m."

GeodesicEndpoints::usage = "GeodesicEndpoints[u,v] gives the endpoints
(as Cvectors) of the real geodesic through the points corresponding to
the Cvectors u and v."

EndpointsEquidistant::usage = "EndpointsEquidistant[u,v] gives
Cvectors corresponding to the vertices of the Spinal surface
equidistant from the points corresponding to Cvectors u,v"

EquidistantSurface::usage = "EquidistantSurface[u,v] gives the Spinal
surface equidistant from the points corresponding to Cvectors u,v."


CartanTripleProduct::usage = "Returns the product of three Hermitian
products."

(************************************************************************)

Begin["`private`"]

Cvector[v_?CvectorQ] := v

Cvector[Hpoint[Infinity]] := {0, -1, 1}
Cvector[x_Hpoint] := HpointDataToCvector[x[[1]], ComplexGauge[x]]
HpointDataToCvector[z_, g_] := {2 z, 1 - g, 1 + g}

Cvector[Chain[x_, Vertical]] :=
  VerticalChainDataToCvector[ Conjugate[ x[[1]] ] ]
VerticalChainDataToCvector[z_] := {1, -z, z}
Cvector[c_Chain] :=
  ChainDataToCvector[ c[[1,1]], c[[2]]^2, ComplexGauge[c[[1]]] ]
ChainDataToCvector[z_,r2_,g_] := {2 z, 1 + r2 - g, 1 - r2 + g}

CvectorQ[_Cvector] := True;
CvectorQ[z_] := (VectorQ[z] && (Length[z]==3))

CmatrixQ[m_] := (MatrixQ[m] && Dimensions[m] == {3,3})

ComplexGauge[x_Hpoint] := ComplexNormSquared[x[[1]]] + 2 I x[[2]]

Hpoint[x_Hpoint] := x
Hpoint[z_] := Hpoint[Infinity] /; CvectorQ[z] && InfiniteHpointQ[z]
Hpoint[z_?CvectorQ] :=
  Hpoint[ z[[1]] / ( z[[2]] + z[[3]] ),
    Im[ (z[[3]] - z[[2]]) / (2 (z[[3]] + z[[2]]))  ] ]
Hpoint/: Times[a_Real, x_Hpoint] := Map[ #*a&, x ]
Hpoint/: Times[a_Rational, x_Hpoint] := Map[ #*a&, x ]
Hpoint/: Times[a_Integer, x_Hpoint] := Map[ #*a&, x ]

InfiniteHpointQ[Hpoint[Infinity]] := True
InfiniteHpointQ[z_?CvectorQ] := Chop[N[z[[2]]] + N[z[[3]]]] == 0
VerticalHpointQ[Hpoint[Infinity]] := True
VerticalHpointQ[x_Hpoint] := Chop[Abs[ N[x[[1]]] ]] == 0
VerticalHpointQ[z_?CvectorQ] := VerticalHpointQ[Hpoint[z]]  

Chain[c_Chain] := c
Chain[z_?CvectorQ] :=
  CvectorDataToChain[z, C21Herm[z,z], z[[2]]+z[[3]]]
CvectorDataToChain[z_, h_, w_?(N[#]==0.0&)] :=
  Chain[ Hpoint[Conjugate[z[[3]]/z[[1]]], 0], Vertical ]
CvectorDataToChain[z_, h_, w_] :=
  CvectorDataToNonverticalChain[z[[1]]/w, (z[[3]]-z[[2]])/(2 w)]
CvectorDataToNonverticalChain[c_, w2_] :=
  Chain[ Hpoint[c,Im[w2]], Sqrt[ComplexNormSquared[c] - 2 Re[w2]] ]

(*************************************************************
 * Old version:
 *   
 *Rcircle[m_?CmatrixQ] :=
 *  Block[{a,b,center,radius,V,V0},
 *    (* First let V = conjugation of Hpoint at infinity in m: *)
 *    V = m . Cvector[Hpoint[Infinity]];
 *
 *    (* Check to see if V ~ Hpoint at infinity *)
 *    If[ InfiniteHpointQ[V],
 *          
 *        (* Case 1: V corresponds to the Hpoint at infinity, so
 *         * the Rcircle is infinite. *)
 *
 *        (* Now let V0 = conjugation of origin in m: *)
 *        V0 = m . Cvector[Hpoint[0,0]];
 *
 *        (* Check to see if V0 lies on vertical axis: *)
 *        If[Chop[Abs[N[Hpoint[V0][[1]]]]] == 0,
 *                
 *          (* Case 1.1: V0 lies on vertical axis; the corresponding
 *           * matrix is:
 *           *                                                 
 *           *   |z      0       0    |     center is v;
 *           *   |0      1-2iv   -2iv |     (complex) radius is z
 *           *   |0      2iv     1+2iv|
 *           *)                                         
 *          a = 1/(m[[2,2]] + m[[3,2]]);
 *            center = Hpoint[0, Im[a m[[3,2]] / 2]];
 *            radius = DirectedInfinity[a m[[1,1]]],
 *
 * ?? *          (* Case 1.2: V0 does not lie on vertical axis *)
 *          *      |-1  2  2|  _ -1
 *          *   T  | 2 -1 -2|  T
 *          *      |-2  2  3|
 *          * where T = DilationMappingToPoint[x]
 *            center = Hpoint[V0] / 2;
 *            radius = Infinity;
 *          ],
 *
 *        (* Case 2: V !~ Hpoint at infinity; finite Rcircle
 *         *)
 *        center = Hpoint[V];
 *          b = TransformRealStructure[Inverse[HTranslation[center]],m];
 *          a = b[[2,2]]/Abs[b[[2,2]]];
 *          radius = b[[1,1]] (b[[2,2]] + b[[2,3]]) /(a^2);
 *           ];
 *
 *    Return[Rcircle[center, radius]]
 *    ]
 * End of old version
 *****************************************************************)

Rcircle[r_Rcircle] := r

Rcircle[m_?CmatrixQ] :=
  RealStructureToRcircle[m, Rconjugate[m, Hpoint[Infinity]]]

RealStructureToRcircle::usage = "RealStructureToRcircle[m,x] returns
the Rcircle corresponding to the real structure m.  x should be the
image of the Hpoint at infinity under conjugation in m.  This function
is intended for internal use only."

(* real structure fixing infinity --> infinite Rcircle: *)
RealStructureToRcircle[m_, Hpoint[Infinity]] :=
  RealStructureToInfiniteRcircle[m, Rconjugate[m, Hpoint[0,0]]]

(* real structure not fixing infinity --> finite Rcircle *)
RealStructureToRcircle[m_?CmatrixQ, center_Hpoint] :=
  RealStructureToFiniteRcircle[
    center,
    TransformRealStructure[Inverse[HTranslation[center]],m] ]
    
RealStructureToInfiniteRcircle::usage =
"RealStructureToInfiniteRcircle[m,v0] returns the infinite Rcircle
corresponding to the real structure m.  v0 should be the image of the
origin (Hpoint[0,0]) under conjugation in m.  m should be a real
structure which fixes the Hpoint at infinity.  This function is
intended for internal use only."

(* infinite Rcircle thru vertical axis  *)
RealStructureToInfiniteRcircle[m_, v0_?VerticalHpointQ] :=
  Rcircle[ Hpoint[0, Im[ m[[3,2]] / 2 (m[[2,2]] + m[[3,2]]) ]],
           DirectedInfinity[ m[[1,1]] / (m[[2,2]] + m[[3,2]]) ] ]

(* infinite Rcircle not thru vertical axis: *)
RealStructureToInfiniteRcircle[m_, v0_] :=
  Rcircle[ v0/2, Infinity]

RealStructureToFiniteRcircle::usage =
"RealStructureToFiniteRcircle[center, b] returns the finite Rcircle
with given center corresponding to the matrix b.  This function is
intended for internal use only."

RealStructureToFiniteRcircle[center_, b_] :=
  Rcircle[center, b[[1,1]] (b[[2,2]] + b[[2,3]]) /(b[[2,2]]/Abs[b[[2,2]]])^2]

(*X*)
SetAttributes[Spinal,Orderless]
Spinal[v1_?CvectorQ, v2_?CvectorQ] :=
  Spinal[Hpoint[v1], Hpoint[v2]]   (**********************************)
Spinal[v1_?CvectorQ, x2_Hpoint] := (* this line and/or next line is  *)
  Spinal[Hpoint[v1], x2]           (* superfluous, once we have the  *)
Spinal[x1_Hpoint, v2_?CvectorQ] := (* orderless attribute for Spinal *)
  Spinal[x1, Hpoint[v2]]           (**********************************)

(*??*)
(* Warning: If AngularInvariant is called with an argument that
   cannot be converted to a Cvector, this yields an infinite loop.
   Since it is an error to call AngularInvariant in this way, we
   won't try to guard against it for now.  But there must be a better
   way of doing this! *)

CartanTripleProduct[ v1_?CvectorQ, v2_?CvectorQ, v3_?CvectorQ ] :=
  - C21Herm[v1,v2] C21Herm[v2,v3] C21Herm[v3,v1]
AngularInvariant[ v1_?CvectorQ, v2_?CvectorQ, v3_?CvectorQ ] :=
  Arg[CartanTripleProduct[v1,v2,v3]]
CartanTripleProduct[ x1_, x2_, x3_ ] :=
  CartanTripleProduct[ Cvector[x1], Cvector[x2], Cvector[x3] ]
AngularInvariant[ x1_, x2_, x3_ ] :=
  AngularInvariant[ Cvector[x1], Cvector[x2], Cvector[x3] ]
(*??X*)
RealTripleQ[x1_,x2_,x3_] := (RRealQ[CartanTripleProduct[x1,x2,x3]])
RRealQ[x_Complex] := (Chop[ Im[ N[ x]]] == 0)

C21Herm[z_,w_] := Dot[ z * {1, 1, -1},  Conjugate[w] ]
C21Herm[z_] := C21Herm[z,z]

C2Herm[z_,w_] := Dot[ z, Conjugate[w] ]
C2Herm[z_] := C2Herm[z,z]

I21Matrix = DiagonalMatrix[ {1, 1, -1} ]

C21Contrajection[v_?CvectorQ] := 
	Outer[Times, v, Conjugate[v]] . I21Matrix / C21Herm[v,v]
C21Reflection[v_?CvectorQ, z_] :=
  IdentityMatrix[3] +  (z - 1 ) C21Contrajection[v] 
C21Reflection[c_Chain, z_] := C21Reflection[Cvector[c], z]
C21Reflection[x_Hpoint, z_] := C21Reflection[Cvector[x], z]
C21Projection[v_?CvectorQ] := C21Reflection[v, 0]
C21Inversion[v_?CvectorQ] := - C21Reflection[v, -1] (* wmg added -1 10/2/89*)
C21Inversion[c_Chain] := C21Inversion[Cvector[c]]

HTranslation[x_Hpoint] :=
  HpointDataToHTranslation[x[[1]], Conjugate[x[[1]]], ComplexGauge[x]/2]
HpointDataToHTranslation[a_, abar_, b_] :=
  {{    1,   a,   a},
   {-abar, 1-b,  -b},
   { abar,   b, 1+b}}
HTranslation[v_?CvectorQ] := HTranslation[Hpoint[v]]

ComplexGeodesic[v1_?CvectorQ, v2_?CvectorQ] := C21Cross[v1,v2]
ComplexGeodesic[x1_, x2_] := ComplexGeodesic[Cvector[x1], Cvector[x2]]

ChainThrough[x_,y_] := Chain[ComplexGeodesic[x,y]]

C21Cross[u_?CvectorQ, v_?CvectorQ] :=
  Conjugate[ Det[ { I21Matrix, u, v } ] ]

Affine[v_?VectorQ] :=
  Reverse[ Rest[ Reverse[v] ] ] / Last[v]  /; Length[v]>1

Projective[v_?VectorQ] := Append[v, 1]

EnormSquared[v_?VectorQ] := Dot[ v, Conjugate[v] ]

Bergman[x1_?CvectorQ, x2_?CvectorQ] :=
  2 ArcCosh[ Sqrt[ BergmanRatio[x1,x2] ] ]
BergmanRatio[x1_?CvectorQ, x2_?CvectorQ] :=
 ( C21Herm[x1,x2] C21Herm[x2, x1] ) / ( C21Herm[x1,x1] C21Herm[x2,x2] )

HApply[m_?CmatrixQ, v_?CvectorQ] := m . v
HApply[m_?CmatrixQ, c_Chain] := Chain[ m . Cvector[c] ]
HApply[m_?CmatrixQ, x_Hpoint] := Hpoint[ m . Cvector[x] ]
HApply[m_?CmatrixQ, r_Rcircle] := 
  Rcircle[TransformRealStructure[m, RealStructure[r]]]
HApply[m_?CmatrixQ, s_Spinal] :=
  Map[ HApply[m,#]&, s ]

RealStructure[c_Rcircle] :=
  If[ Head[c[[2]]] === DirectedInfinity, 

      (* Infinite case *)
(*??X*)     If[ VerticalHpointQ[c[[1]]],
		
          (* infinite Rcircle meets vertical axis *)
(*XX*)          DilationMappingToPoint[
			Hpoint[c[[2,1]]/Abs[c[[2,1]]],c[[1,2]]]],

(* c has the form Rcircle[Hpoint[0,v],z Infinity] where v = c[[1,2]]
is the vertical component of the "center" of the Rcircle and z =
c[[2,1]]/Abs[c[[2,1]]] is a unit complex number in the direction of
the Rcircle. *) 

          (* infinite Rcircle doesn't meet vertical axis *)
          TransformRealStructure[DilationMappingToPoint[c[[1]]],RealStructure10]
          ],
	
      (* Finite case *)
      TransformRealStructure[HTranslation[c[[1]]],RealStructureO[c[[2]]]],
      ]

RealStructure10 =
  {{-1,  2,  2},
   { 2, -1, -2},
   {-2,  2,  3}}

RealStructureO[radius_] := RadiusDataToRealStructureO[radius, Abs[radius]]
RadiusDataToRealStructureO[radius_, mod_] :=
  {{radius/mod, 0,               0               },
   {0,          (mod + 1/mod)/2, (mod - 1/mod)/2 },
   {0,          (1/mod - mod)/2, (-mod - 1/mod)/2}}

TransformRealStructure[a_?CmatrixQ,m_?CmatrixQ] :=
	a . m . Conjugate[Inverse[a]]

Rconjugate[r_Rcircle, u_] :=
  Rconjugate[RealStructure[r], u]
Rconjugate[m_?CmatrixQ, u_?CvectorQ] :=
  m . Conjugate[u]
Rconjugate[m_?CmatrixQ, x_Hpoint] :=
  Hpoint[ Rconjugate[m, Cvector[x]] ]
Rconjugate[m_?CmatrixQ, c_Chain] :=
  Chain[ Rconjugate[m, Cvector[c]] ]
Rconjugate[m_?CmatrixQ, s_Spinal] :=
  Map[ Rconjugate, s ]
Rconjugate[m_?CmatrixQ, r_Rcircle] :=
  Rcircle[ TransformRealStructure[m, Conjugate[RealStructure[r]]] ]
	     
(*??X*)
DilationMappingToPoint[Hpoint[z_,v_]] := 
  HpointDataToDilationMatrix[ z,v,Abs[z]]
HpointDataToDilationMatrix[z_, v_, r_] :=
  Dot[ {{1,    0,          0        },
        {0,    1 - 2 I v, -2 I v    },
        {0,    2 I v,      1 + 2 I v}},
	
	{{z/r, 0,           0          },
         {0,   (1/r + r)/2, (1/r - r)/2},
	 {0,   (1/r - r)/2, (1/r + r)/2}} ]



HDilation[r_,t_] :=
  {{Exp[I t], 0,          0         },
   {0,        (r+1/r)/2,  (1/r -r)/2},
   {0,        (1/r -r)/2, (r+1/r)/2 }}

SpinalContaining[c1_Chain,c2_Chain] :=
  Apply[ Spinal, GeodesicEndpoints[Cvector[c1],Cvector[c2]] ]

RealStructureFixing::none = "Real structure does not exist; nonzero
angular invariant"
RealStructureFixing[v1_?CvectorQ, v2_?CvectorQ, v3_?CvectorQ] :=
(*??X*)  If[ RealTripleQ[v1,v2,v3],
      (* Case 1: 0 angular invariant; real structure exists *)
      NormalizedDataToRealStructure[
	v1 C21Herm[v2,v1] C21Herm[v3,v2],
	v2 C21Herm[v3,v2],
	v3 ],
      (* Case 2: Nonzero angular invariant; no real structure exists *)
      Message[RealStructureFixing::none]; Return[Null] ]

(* NormalizedDataToRealStructure takes 3 normalized Cvectors (null
   Cvectors with real C21Herm products) and returns the real
   structure fixing them *)

NormalizedDataToRealStructure[v1_?CvectorQ, v2_?CvectorQ, v3_?CvectorQ] :=
  CrossDataToRealStructure[
    {v1,v2,v3},
    {C21Cross[v2,v3], C21Cross[v3,v1], C21Cross[v1,v2]} ]

CrossDataToRealStructure[v_List, c_List] :=
  Sum[ Outer[Times, v[[i]], c[[i]]] / C21Herm[c[[i]],v[[i]]],
       {i,1,3} ] . I21Matrix

RcircleThrough[x1_Hpoint,x2_Hpoint,x3_Hpoint] :=
  Block[{Z1,Z2,Z3},
    Z1 = Cvector[x1];
      Z2 = Cvector[x2];
      Z3 = Cvector[x3];
      Return[Rcircle[RealStructureFixing[Z1,Z2,Z3]]]
    ]

IncidentQ[x_,x_]	      := True	
C21OrthogonalQ[x_Cvector,y_Cvector] := (C21Herm[x,y] == 0)
IncidentQ[x_Hpoint,c_Chain]   := C21OrthogonalQ[Cvector[x],Cvector[c]]
IncidentQ[x_Hpoint,r_Rcircle] := (Rconjugate[r,x] == x )
IncidentQ[x_Hpoint,s_Spinal]  := RealTripleQ[s[[1]],s[[2]],Cvector[x]]
IncidentQ[c_Chain,s_Spinal]   := (HApply[C21Inversion[c],s] == s)
IncidentQ[r_Rcircle,s_Spinal] := (Rconjugate[r,s] == s)
SetAttributes[IncidentQ,Orderless]

ChainOrthogonal[x_Hpoint,c_Chain] :=
  Chain[ C21Cross[ Cvector[x], Cvector[c] ] ]
ChainOrthogonal[c1_Chain,c2_Chain] :=
  Chain[ C21Cross[ Cvector[c1], Cvector[c2]]]
ChainOrthogonal[r1_Rcircle,r2_Rcircle] :=
  Apply[ ChainThrough,
         NullEigenvectors[RealStructure[r1].RealStructure[r2]]
       ]
SetAttributes[ChainOrthogonal,Orderless]

Midpoint[u_?CvectorQ, v_?CvectorQ] :=
  u - Sqrt[ C21Herm[u,u]/C21Herm[v,v] ] C21Herm[u,v] / Abs[C21Herm[u,v]] v

GeodesicEndpoints[u_?CvectorQ,v_?CvectorQ] :=
  NullEigenvectors[C21Inversion[u] . C21Inversion[v]]
NullEigenvectors[m_?CmatrixQ] :=
  Select[ Eigenvectors[m], (Chop[C21Herm[#,#]]==0)& ]
EndpointsEquidistant[u_?CvectorQ, v_?CvectorQ] :=
  Map[ (C21Reflection[Midpoint[u,v], #] . GeodesicEndpoints[u,v][[1]])&,
       {I, -I}
     ]
EquidistantSurface[u_?CvectorQ, v_?CvectorQ] :=
  Apply[ Spinal,
         Map[ Hpoint, EndpointsEquidistant[u, v] ]
       ]

End[]

EndPackage[]

InU21Q[m_] := (m. I21Matrix . Conjugate[Transpose[m] ] == I21Matrix)
