/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*    file:      utility.c
*
*    Purpose:   Various utility routines.
*/

#include "include.h"

/**************************************************************
*  
*  Function: catcher()
*
*  Purpose:  User interrupt catcher.  Sets breakflag so 
*            anybody in a long loop can test to see if
*            user wants to break out.
*/

void catcher(sig)
int sig;
{
  if ( sig == SIGINT ) 
    { signal(SIGINT,catcher);
      if ( iterate_flag == 1 )
        { outstring("Will break after iteration.\n");
          breakflag = 1;
          iterate_flag = 0;
        }
      else if ( iterate_flag == 2 ) /* graphing */
	{ breakflag = 1;
          iterate_flag = 0;
	}
      else error("Aborting operation.\n",RECOVERABLE);
    }
}

/*********************************************************************
*
*  function: mycalloc
*
*  purpose: memory allocation with error detection
*/

char *mycalloc(num,size)
int num,size;
{ char *ptr;

  if ( size <= 0 ) size = 1;  /* so won't bomb on zero allocation */
  ptr = (char *)malloc(num*size);
  if ( ptr == NULL )
    error("Cannot allocate memory.\n",RECOVERABLE);
  memset(ptr,0,num*size);
  return ptr;
}  


/*********************************************************************
*
*  function: kb_realloc
*
*  purpose: memory re-allocation with error detection
*/

char *kb_realloc(ptr,size)
char *ptr;
int size;
{
  ptr = (char *)realloc(ptr,size);
  if ( ptr == NULL )
    error("Cannot reallocate memory.\n",RECOVERABLE);
  return ptr;
}  


/********************************************************************
*
* function:  temp_calloc()
*
* purpose:  memory allocation that can be undone by longjmp at error.
*
*/

#define MAXMEMS 20
static char *memlist[MAXMEMS];  /* allocated blocks, packed list */
static int   memcount;          /* number allocated */

char *temp_calloc(num,size)
int num,size;
{
  char *ptr;

  if ( memcount > (MAXMEMS - 1) )
    error("Too many temporary memory blocks.",RECOVERABLE);

  ptr = mycalloc(num,size);

  memlist[memcount++] = ptr;
  return ptr;
}  

/********************************************************************
*
* function:  temp_free()
*
* purpose:  Usual freeing of temporary memory allocation.
*
*/

void temp_free(ptr)
char *ptr;
{
  int k;

  for ( k = 0 ; k < memcount ; k++ )
    if ( ptr == memlist[k] ) break;
  if ( k == memcount )
    { error("Cannot find address in temporary memory list.",WARNING);
      return;
    }
  else
    { free(ptr);
      memlist[k] = memlist[--memcount];  /* pack list */
      memlist[memcount] = NULL;
    }
}

/********************************************************************
*
* function:  temp_free_all()
*
* purpose:  Freeing of all temporary memory in case of error.
*
*/

void temp_free_all()
{
  int k;

  for ( k = 0 ; k < memcount ; k++ )
    { free(memlist[k]);
      memlist[k] = NULL;
    }
  memcount = 0;
  oldcoord = NULL;
}
  
/**************************************************************
*
*  Function: calc_edge()
*
*  Purpose:  Calculate the length of an edge from its endpoint
*            coordinates.  Puts result into edge structure.
*
*/

void calc_edge(e_id)
edge_id e_id;
{
  REAL len;
  REAL s[MAXCOORD];
  REAL midx[MAXCOORD];
  int i,j,k;
  vertex_id tv = get_edge_tailv(e_id);
  REAL *xt=get_coord(tv);
  double euclidean;

  get_edge_side(e_id,s);
  euclidean = dot(s,s,web.sdim);
  if ( web.metric_flag )
  {
    /* energy due to linear tension, metric evaluated at midpoint */
    for ( k = 0, len = 0.0 ; k < web.gauss1D_order ; k++ )
     { REAL sum;
       for ( i = 0 ; i < web.sdim ; i++ )
         midx[i] = gauss1Dpt[k]*s[i] + xt[i];
       if ( web.conformal_flag )
       { double gg = eval(&web.metric[0][0],midx);
         len += sqrt(gg*euclidean);
       }
       else
        for ( sum = 0.0, i = 0 ; i < web.sdim ; i++ )
         for ( j = 0 ; j < web.sdim ; j++ )
          sum += s[i]*s[j]*eval(&web.metric[i][j],midx);
       len += gauss1Dwt[k]*sqrt(sum);
      }
  }
  else
  {
    len = sqrt(dot(s,s,web.sdim));
  }
  set_edge_length(e_id,len);
}

/************************************************************
*
*  get_edge_side()
*
*  Purpose:  calculate the vector of an edge
*
*  Input:    edge_id e_id - specifies edge
*            REAL  *x;  - 3 components of vector
*
*  Output:   x[] has side components
*/

void get_edge_side(e_id,x)
edge_id e_id;
REAL *x;
{
  REAL *y,*z;
  int i;
  REAL w[MAXCOORD];
  
  y = get_coord(get_edge_tailv(e_id));
  z = get_coord(get_edge_headv(e_id));
  if ( web.symmetry_flag ) 
    {
      (*sym_wrap)(z,w,get_edge_wrap(e_id));
      z = w;
    }
  for ( i = 0 ; i < web.sdim ; i++ ) 
     x[i] = z[i] - y[i];
}

/*************************************************************
*
*  function: get_edge_adjust()
*
*  purpose:  adjust a side vector for torus wrap.
*            used only in torvol.c.
*
*/
  
void get_edge_adjust(e_id,w)
edge_id e_id;
REAL *w;
{
  REAL x[MAXCOORD];
  int i ;

  for ( i = 0 ; i < web.sdim ; i++ ) x[i] = 0.0;
  (*sym_wrap)(x,w,get_edge_wrap(e_id));
}
                

/***************************************************************
*
*  Function:  get_edge_valence
*
*  Purpose:  return number of facets around edge
*
*/

int get_edge_valence(e_id)
edge_id e_id;
{
  int n = 0;
  facetedge_id fe = get_edge_fe(e_id);
  facetedge_id next_fe = fe;

  if ( !valid_id(fe) ) return 0;
  do
    { n++;
      next_fe = get_next_facet(next_fe);
    }
  while ( next_fe != fe );
  return n;
}




/**********************************************************8
*
*  Function: calc_vertex_normal
*
*  Purpose: Calculate average normal of facets around a vertex
*           for Pixar normal interpolation.  Only does the facets
*           in the same face as given facet; will not cross
*           multiple edges.
*
*  Input:   vertex id
*           facet-edge id - with tail at vertex, for defining face
*           norm -  pointer to place for normal vector
*
*
*  Output:  Average normal, normalized to unit length.
*/

void calc_vertex_normal(fe,norm)
facetedge_id fe;
REAL *norm;
{
  int i;
  REAL y[MAXCOORD],z[MAXCOORD];
  REAL fnorm[MAXCOORD];
  facetedge_id sidea;
  struct boundary *bdry;  /* to check staying on same boundary */
  REAL size;

  for ( i = 0 ; i < web.sdim ; i++ ) norm[i] = 0.0;
  bdry = get_facet_boundary(get_fe_facet(fe));  /* original boundary */

  /* go around one way to edge of face */
  sidea = fe;
  do
    {

      get_fe_side(sidea,z);
      get_fe_side(get_prev_edge(sidea),y);
      cross_prod(y,z,fnorm);
      for ( i = 0 ; i < web.sdim ; i++ ) norm[i] += fnorm[i];

      /* go to next facet */
      sidea = get_prev_edge(sidea);
      if ( equal_id(sidea,get_next_facet(sidea)) ) break;  /* edge */
      sidea = fe_inverse(get_next_facet(sidea));
      if ( bdry != get_facet_boundary(get_fe_facet(sidea)) ) break;
      if ( equal_id(sidea,fe) ) goto normalize; /* went all the way around */
    }
  while ( equal_id(get_next_facet(sidea),get_prev_facet(sidea)) );

  /* go around the other way */
  sidea = fe;
  while ( equal_id(get_next_facet(sidea),get_prev_facet(sidea)) )
    {
      if ( equal_id(sidea,get_next_facet(sidea)) ) break;  /* edge */
      sidea = fe_inverse(get_next_facet(sidea));
      sidea = get_next_edge(sidea);
      if ( bdry != get_facet_boundary(get_fe_facet(sidea)) ) break;

      get_fe_side(sidea,z);
      get_fe_side(get_prev_edge(sidea),y);
      cross_prod(y,z,fnorm);
      for ( i = 0 ; i < web.sdim ; i++ ) norm[i] += fnorm[i];
    }

  normalize:
  size = sqrt(dot(norm,norm,web.sdim));
  for ( i = 0 ; i < web.sdim ; i++ ) norm[i] /= size;
}


/***************************************************************
*
*  Function: get_facet_verts()
*
*  Purpose: to get all three vertices of a facet, nicely 
*           displaced to fundamental cell in case of symmetry group.
*
*/

void get_facet_verts(f_id,verts,wraps)
facet_id f_id;
REAL **verts;
WRAPTYPE *wraps;  /* NULL if not wanted */
{
  facetedge_id fe;
  int i,j,ii;
  WRAPTYPE wrap;
  REAL *x;

  if ( web.simplex_flag )
   { vertex_id *v = get_facet_vertices(f_id);
     for ( i = 0 ; i <= web.dimension ; i++ )
       { x = get_coord(v[i]);
         for ( j = 0 ; j < web.sdim ; j++ )
           verts[i][j] = x[j];
       }
     return;
   }

  fe = get_facet_fe(f_id);
  if ( web.symmetry_flag ) /* try to get unwrapped edge */
   {
     for ( i = 0 ; i < FACET_VERTS ; i++, fe = get_next_edge(fe) )
       if ( get_fe_wrap(fe) == 0 ) break;
     wrap = 0;
     for ( ii = 0 ; ii < FACET_VERTS ; ii++ )
      { int jj = (i+ii)%FACET_VERTS;
        x = get_coord(get_fe_tailv(fe));
	if ( wraps ) wraps[i] = wrap;
	(*sym_wrap)(x,verts[jj],wrap);
        wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
        fe = get_next_edge(fe);
      }
   }
  else for ( i = 0 ; i < FACET_VERTS ; i++ )
   { x = get_coord(get_fe_tailv(fe));
     for ( j = 0 ; j < web.sdim ; j++ )
         verts[i][j] = x[j];
     fe = get_next_edge(fe);
   }
}

/*****************************************************************
*
* Function: path_open()
*
*  Purpose: Open file for reading on EVOLVERPATH
*/

#ifdef TCPP
FILE *path_open(char *name)   // else TC++ has conniptions
#else
FILE *path_open(name)
char *name;
#endif
{
  char *env;
  char path[200];
  char newname[100];
  int len;
  FILE *fd;
    

  for (;;)
  {
  env = getenv("EVOLVERPATH");
  strncpy(path,name,sizeof(path));
  while ( (fd = fopen(path,"r")) == NULL)
    { /* try paths in EVOLVERPATH */
      if ( env == NULL ) break;
      len = strcspn(env,ENVPATHCHAR);
      if ( len == 0 ) break;
      strncpy(path,env,len);
      path[len] = PATHCHAR;
      strncpy(path+len+1,name,sizeof(path)-len-2);
      env += len+1;
    } 
  
  if ( fd == NULL)
    { sprintf(errmsg,"Cannot open %s.  Enter new filename: ",name);
      prompt(errmsg,newname);
      if ( strlen(newname) == 0 )
	break; /* return NULL */
      name = newname;
    }
  else break;
  }
  return fd;
}

/***************************************************
* 
*  Handy for graphics function pointers for 
*  functions that have nothing to do.
*/

void null_function() {}

/************************************************************
*
*  For uninitialized function pointers.
*
*/

void bad_function()
{ error("Internal error.  Using uninitialzed function pointer.\n",
    WARNING);
}

/*****************************************************************
*
* Function: distance()
*
*  Purpose: Finds distance between two vertices.
*/

REAL distance(v1,v2)
vertex_id v1,v2;
{
  REAL *c1,*c2;
  REAL sum;
  int i;

  c1 = get_coord(v1);
  c2 = get_coord(v2);
  sum = 0.0;
  for ( i = 0 ; i < web.sdim ; i++ )
    sum += (c1[i] - c2[i])*(c1[i] - c2[i]);
  return sqrt(sum);
}

/***************************************************************************
*
* function: grule()
*
* purpose: Calculate abscissas and weights for gaussian quadrature
*          on [0,1].
*
* Adapted from a FORTRAN routine GRULE in a book on numerical integration 
*/

void grule(n,x,w)
int n;  /* number of points */
double *x;   /* for abscissas  x[0] ... x[n-1] */
double *w;   /* for weights    w[0] ... w[n-1] */
{
  double pkm1,pk,t,t1,pkp1,den,d1,dpn,d2pn,d3pn,d4pn,u,v,h,p,dp,fx,x0;
  int m,e1,i,k;
  
  if ( n < -1 )
    { /* rectangle rule */
      for ( k = 0 ; k < abs(n) ; k++ )
	{ x[k] = k/(double)(abs(n)-1);
	  w[k] = 1/(double)abs(n);
        }
    }
  m = (n+1)/2;
  e1 = n*(n+1);
  for ( i = 1 ; i <= m ; i++ )
   {
     /* original calculation on [-1,1] */
     t = (4*i - 1)*M_PI/(4*n+2);
     x0 = (1 - (1 - 1.0/n)/(8*n*n))*cos(t);
     pkm1 = 1.0;
     pk = x0;
     for ( k = 2 ; k <= n ; k++ )
       {
         t1 = x0*pk;
         pkp1 = t1 - pkm1 - (t1 - pkm1)/k + t1;
         pkm1 = pk;
         pk = pkp1;
       }
     den = 1 - x0*x0;
     d1 = n*(pkm1 - x0*pk);
     dpn = d1/den;
     d2pn = (2*x0*dpn - e1*pk)/den;
     d3pn = (4*x0*d2pn + (2 - e1)*dpn)/den;
     d4pn = (6*x0*d3pn + (6 - e1)*d2pn)/den;
     u = pk/dpn;
     v = d2pn/dpn;
     h = -u*(1 + 0.5*u*(v + u*(v*v - d3pn/3/dpn)));
     p = pk + h*(dpn + 0.5*h*(d2pn + h/3*(d3pn + 0.25*h*d4pn)));
     dp = dpn + h*(d2pn + .5*h*(d3pn + h*d4pn/3));
     h = h - p/dp;
     x[i-1] = x0 + h;
     fx = d1 - h*e1*(pk + 0.5*h*(dpn + h/3*(d2pn + 0.25*h*(d3pn + 0.2*h*d4pn))));
     w[n-i] = w[i-1] = (1 - x[i-1]*x[i-1])/fx/fx;

     /* normalization to [0,1] */
     x[i-1] = (1 + x[i-1])/2;
     x[n-i] =  1 - x[i-1];
   }
  if ( 2*m > n ) x[m-1] = 0.5;
}


/****************************************************************
*
*  function: binom_coeff()
*
*  purpose: calculate binomial coefficient (small numbers only)
*
*/

int binom_coeff(n,k)
int n,k;
{ int i, c;
  for ( i = 0, c = 1 ; i < k ; i++ )
    c *= n - i;
  for ( i = 1 ; i <= k ; i++ )
    c /= i;
  return c;
}

/**********************************************************************
*
*  function: set_e_phase_density()
*
*  purpose: calculate edge density from phases of neighboring facets.
*/
void set_e_phase_density(e_id)
edge_id e_id;
{ facetedge_id fe = get_edge_fe(e_id);
  int i = get_f_phase(get_fe_facet(fe));	
  int j = get_f_phase(get_fe_facet(get_next_facet(fe)));	
  set_edge_density(e_id,phase_data[i][j]);
}

/**********************************************************************
*
*  function: set_f_phase_density()
*
*  purpose: calculate edge density from phases of neighboring bodies.
*/
void set_f_phase_density(f_id)
facet_id f_id;
{ 
  int i = get_b_phase(get_facet_body(f_id));	
  int j = get_b_phase(get_facet_body(inverse_id(f_id)));	
  set_facet_density(f_id,phase_data[i][j]);
}

