/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************************
*
*  File: userio.c
*
*  Purpose: Handles interaction with user.  All interaction goes
*           through this file for easy porting.
*/

#include "include.h"

/***********************************************************************
*
*  Function: outstring()
*
*  Purpose: Prints string for user.  Does not wait for response.
*           Does not add newline.
*/

void outstring(outmsg)
char *outmsg;
{
  fputs(outmsg,outfd);
}

/***********************************************************************
*
*  Function: erroutstring()
*
*  Purpose: Prints error string for user.  Does not wait for response.
*           Does not add newline.
*/

void erroutstring(outmsg)
char *outmsg;
{
  fputs(outmsg,stderr);
}
/***********************************************************************
*
*  Function: getstring()
*
*  Purpose: Gets string from user. 
*/

void getstring(inmsg)
char *inmsg;
{
  if ( commandfd )
    {  /* from command input file */
       if ( (fgets(inmsg,100,commandfd) == NULL) 
               || strstr(inmsg,"REVERT") )
         { fclose(commandfd);
           commandfd = NULL;
           /* fall through to stdin */
         }
       else 
         { char *c;
           printf(inmsg); /* echo */
           if ( (c = strchr(inmsg,'\n')) != NULL ) *c = '\0';
           return;
         }
     }

  /* from stdin */
#ifdef NeXTapp
  NeXT_prompt(NULL,inmsg);
#endif
  if ( gets(inmsg) == NULL )
    exit(0);
}

/***********************************************************************
*
*  Function: prompt()
*
*  Purpose: Displays prompt and gets response from user.
*/

void prompt(promptmsg,inmsg)
char *promptmsg;
char *inmsg;
{
  if ( commandfd )
    {  /* from command input file */
       if ( (fgets(inmsg,100,commandfd) == NULL) 
               || strstr(inmsg,"REVERT") )
         { fclose(commandfd);
           commandfd = NULL;
           /* fall through to stdin */
         }
       else 
         { char *c;
	   printf(promptmsg);  /* so appears on screen also */
           printf(inmsg); /* echo */
           if ( (c = strchr(inmsg,'\n')) != NULL ) *c = '\0';
           return;
         }
     }

#ifdef NeXTapp
  /* from attention panel */
  NeXT_prompt(promptmsg,inmsg);
#else
  /* from stdin */
  fputs(promptmsg,outfd);
  if ( gets(inmsg) == NULL )
    exit(0);
#endif
}

/***********************************************************************
*
*  Purpose: Error handling.  Prints error message.  If error is
*    recoverable, longjmps to pre-set spot.  If not, exits.
*
*  May be system dependent for message display.
*/

void error(emsg,mode)
char *emsg;
int mode;
{
  extern int line_no;

#ifndef NOPIPE
  if ( outfd != stdout )
   { pclose(outfd);
     outfd = stdout;   /* in case of previous piping */
   }
#endif

  if ( data_fd ) fprintf(stderr,"Line %d - ",line_no);

  switch ( mode )
    {
      case UNRECOVERABLE:
      fprintf(stderr,"FATAL ERROR: %s\n",emsg);
/*      exit(1);  */

      case RECOVERABLE:
      fprintf(stderr,"ERROR: %s\n",emsg);
/*      fprintf(stderr,"Operation aborted. Hit any key to continue.");  */
/*      getchar();    */
      temp_free_all();
      longjmp(jumpbuf,1);

      case WARNING:
      fprintf(stderr,"WARNING: %s\n",emsg);
      break;

      case EXPRESSION_ERROR:
      fprintf(stderr,"EXPRESSION ERROR: %s",emsg);
      dump_buff();
      if ( ++parse_errors >= 5 ) 
	{
	  fprintf(stderr,"Too many errors. Aborting.\n");
          temp_free_all();
          longjmp(jumpbuf,2);
        }
      parse_error_flag = 1;
      recovery_flag = 1;
      break;

      case COMMAND_ERROR:
      fprintf(stderr,"COMMAND ERROR: %s\n",emsg);
      dump_buff();
      parse_error_flag = 1;
/*
      temp_free_all();
      longjmp(cmdbuf,1);
*/
      break;


      case PARSE_ERROR:
      fprintf(stderr,"PARSING ERROR: %s",emsg);
      dump_buff();
      if ( ++parse_errors >= 5 ) 
	{
	  fprintf(stderr,"Too many errors in datafile. Aborting.\n");
          temp_free_all();
          longjmp(jumpbuf,2);
        }
      parse_error_flag = 1;
      recovery_flag = 1;
      break;
      
      case DATAFILE_ERROR:
      fprintf(stderr,"DATAFILE ERROR: %s",emsg);
      if ( ++parse_errors >= 5 ) 
	{
	  fprintf(stderr,"Too many errors in datafile. Aborting.\n");
          temp_free_all();
          longjmp(jumpbuf,2);
        }
      parse_error_flag = 1;
      recovery_flag = 1;
      break;
    }
}
 
