/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************************
*
*  File: skeleton.h
*
*  Header file for evolver.  Defines skeleton structures.
*
*/

/************************************************************************
*
*  Union skeleton element structure for common operations.
*
*/

typedef int MAP; /* constraint, etc, bitmap type */
typedef unsigned short int ATTR;   /* attribute bitmap type */
typedef short int tagtype;         /* element tag type */
typedef short ETYPE;                /* element type type */
typedef long int WRAPTYPE;         /* symmetry group element */

extern int sizes[NUMELEMENTS];

    /* common fields; added trto each element struct type */
#define COMMON_STUFF \
    element_id   forechain;      /* for element and free list forechain */\
    element_id   backchain;      /* for element and free list backchain */\
    ATTR         attr;         /* attribute bits */\
    ORDTYPE      ordinal;        /* for numbering elements */\
    ETYPE        type;           /* element type */\
    int		 original;       /* datafile original number */

struct element {
  COMMON_STUFF
  };

/* element attr bits */
#define ALL_ATTR    (ATTR)0xFFFF
#define ALLOCATED   (ATTR)0x0001
#define NODISPLAY   (ATTR)0x0002
#define NEWELEMENT  (ATTR)0x0004
#define NEWVERTEX   (ATTR)0x0004
#define NEWEDGE     (ATTR)0x0004
#define NEWFACET    (ATTR)0x0004
#define NEWBODY     (ATTR)0x0004
#define WRAPPED     (ATTR)0x0008
#define DENSITY     (ATTR)0x0010
#define FIXEDVOL    (ATTR)0x0020
#define FIXED       (ATTR)0x0040
#define BOUNDARY    (ATTR)0x0080
#define NEGBOUNDARY (ATTR)0x0100
#define BDRY_ENERGY  (ATTR) 0x0200
#define BDRY_CONTENT (ATTR)0x1000
#define CONSTRAINT    (ATTR)0x0400
#define SURF_ENERGY   (ATTR)0x8000
#define SURF_QUANTITY (ATTR)0x1000
#define PRESSURE    (ATTR)0x0800
#define NEGCONSTRAINT (ATTR)0x4000
#define HIT_WALL      (ATTR)0x2000

/*****************************************************************
*
*   Structures peculiar to each dimension of skeleton.
*/


/****************************************************
*
*  facetedge structure 
*/

struct facetedge
  { 
    COMMON_STUFF
    edge_id      fe_edge_id;  /* oriented edge of base pair */
    facet_id     fe_facet_id; /* oriented face of base pair */
    facetedge_id nextedge[2];  /* 0 previous, 1 next */
    facetedge_id nextfacet[2];  /* 0 previous, 1 next */
  };

 
/****************************************************
*
*  vertex structure
*/

struct vertex
  { 
    COMMON_STUFF
    MAP  constraint_map;       /* constraint bit map */
    MAP  constraint_status;    /* whether on constraint bit map */
    struct boundary *boundary; /* which boundary */
    REAL coord[MAXCOORD];     /* coordinates */
    REAL force[MAXCOORD];     /* net force   */
    REAL param[MAXPARAM];     /* boundary parameters */
    REAL restore[MAXCOORD];   /* volume restoring motion */
    REAL star;         /* area of surrounding facets */
    facetedge_id fe_id;  /* link to global structure */
    int valence;        /* number of edges incident */
  };

/* constraint_status flags */
#define OFF_CONSTRAINT 0
#define ON_CONSTRAINT  1

/****************************************************
*
*  edge structure
*/
  
struct edge
  { 
    COMMON_STUFF
    MAP  constraint_map;       /* constraint bit map */
    struct boundary *boundary; /* which boundary */
    vertex_id vertices[(MAXCOORD>3)?(MAXCOORD-1):3];
     /* endpts and midpt for quadratic patches, or simplex edges */
    facetedge_id fe_id;     /* link to global structure */
    REAL density;          /* energy per unit length */
    REAL length;          /* edge length; be careful of validity */
    REAL star;            /* total area of adjacent facets */
    WRAPTYPE wrap;        /* symmetry group element, tail to head */
    short color;          /* for display */
  };

/******************************************************
*
*  facet structure
*/

struct facet
  { 
    COMMON_STUFF
    tagtype      tag;          /* inherited optional tag */
    MAP  constraint_map;       /* constraint bit map */
    MAP  surfen_map;           /* surface energy map */
    MAP  quantity_map;         /* quantity integrands in effect */
    struct boundary *boundary; /* which boundary */
    body_id body[2];   /* the two bodies the facet belongs to */
    facetedge_id fe_id;  /* link to global structure */
    REAL        density;      /* for doing real currents and varifolds */
    REAL   area;              /* for diffusion or whatever */
    REAL   flux;              /* for lower bound estimate */
    vertex_id    vertices[MAXCOORD];  /* vertices of facet */
    short phase;                /* for phase-dependent boundaries */
    short color;              /* for display */
};

/*********************************************************
*
*  body structure
*/

struct body
  { 
    COMMON_STUFF
    REAL fixvol;     /* volume constraint */
    REAL volume;     /* actual volume     */
    REAL pressure;   /* internal pressure */
    facetedge_id fe_id;  /* link to global structure */
    REAL volconst;   /* for body volume constant componenet */
    REAL density;    /* for gravitational potential energy */
    short phase;                /* for phase-dependent boundaries */
  }; 

/*********************************************************************
*
*   Boundaries and constraints
*/

typedef short NTYPE; /* for node indexes, types, etc. */

/* tree node for expression trees */
struct treenode { NTYPE left;   /* left subexpression index offset */
                 NTYPE right;  /* right subexpression index offset */
                 NTYPE type;   /* see defines above */
                 NTYPE inx;    /* index of params or integer power */
		 REAL  value;  /* constant value */
               };

/* node for expression evaluation list */
struct evalnode {
                 NTYPE type;   /* see defines above */
                 NTYPE inx;    /* index of params or integer power */
		 REAL  value;  /* constant value */
               };

/* structure for expression */
struct expnode { 
		  struct treenode *root;
		  struct evalnode *evallist;
               };

#define BDRYMAX 20
#define CONSTRMAX (8*sizeof(MAP))
#define SURFENMAX (8*sizeof(MAP))
#define QUANTMAX  (8*sizeof(MAP))

/* for specifying free boundaries in parametric form */
struct boundary
  {
    ATTR attr;             /* attribute flags */
    int pcount;           /* number of parameters */
    struct expnode *coordf[MAXCOORD];   /* coordinate functions for eval() */
    struct expnode *envect[MAXPARAM];  /* functions for boundary energy */
    struct expnode *convect[MAXPARAM]; /* functions for boundary interior content */
  };


/* for specifying constraints */
#define MAXCONCOMP (MAXCOORD*(MAXCOORD-1)/2)
struct constraint
  {
    ATTR attr;             /* attribute flags */
    struct expnode *formula;   /* function expression for eval() */
    int compcount;      /* number of components for integrands */
    struct expnode *envect[MAXCONCOMP]; /* functions for energy integral */
    struct expnode *convect[MAXCONCOMP]; /* functions for content integral */
    MAP    quantity_map;     /* quantities have integrals on edge */
    struct expnode *quanvect[QUANTMAX][MAXCONCOMP];
  };

/* for specifying facet energy integrands */
struct surf_energy
  {
    ATTR attr;             /* attribute flags */
    struct expnode *envect[MAXCOORD]; /* functions for energy integral */
  };

/* for specifying facet quantity integrands */
struct quantity
  {
    ATTR attr;      /* attribute flags, esp. QFIXED for constrained quantity */
    struct expnode *quanvect[MAXCOORD]; /* functions for facet integral */
    REAL  value;    /* actual total value */
    REAL  target;   /* goal if QFIXED */
    REAL  pressure;  /* Lagrange multiplier */
  };


/* boundary and constraint attributes */
#define B_CONVEX     0x0001
#define NONPOSITIVE  0x0400
#define NONNEGATIVE  0x0800
#define GLOBAL       0x0004
#define QFIXED       0x0010
#define IN_USE       0x0020
#define CON_ENERGY   0x0040
#define CON_CONTENT  0x0080

/* adjustable constant list */
struct param

  { char name[32];  /* 31 significant characters */
    REAL value; 
    int  flags;    /* see defines below */
    double *values;   /* for parameter file values */
    char value_file[60];  /* parameter file name */
  };

/* defines for parameter flags */
#define FILE_VALUES 1

/**********************************************************************
*
* The ultimate structure for a whole surface 
*/

struct web {
    struct skeleton skel[NUMELEMENTS];
    int sdim;  /* dimension of ambient space */
    int dimension;   /* where tension resides */
    int modeltype;   /* quadratic or linear, see defines below */
    struct constraint constraints[CONSTRMAX]; /* for free boundaries */
    int concount;    /* number of constraints */
    MAP con_global_map; /* bitmap for global constraints */
    REAL tolerance;     /* constraint error tolerance */
    struct boundary boundaries[BDRYMAX]; /* for free boundaries */
    struct surf_energy surfen[SURFENMAX]; /* facet energy integrands */
    MAP surf_global_map; /* bitmap for global surface energies */
    int surfen_count;    /* number of surface energy integrands */
    struct quantity quants[QUANTMAX]; /* facet quantity integrands */
    MAP quant_global_map; /* bitmap for global quantities */
    int quantity_count;    /* number of quantities */
    struct param *params;  /* adjustable parameter list */
    int paramcount;       /* number of adjustable parameters */
    int diffusion_flag;  /* whether diffusion in effect */
    REAL diffusion_const;  /* coefficient for diffusion */
    int simplex_flag;    /* 1 if facets represented directly with vertices */
    int torus_clip_flag;
    int torus_body_flag;
    int symmetric_content; /* 1 if volumes use symmetric divergence */
    REAL meritfactor;   /* for multiplying figure of merit */
    int gravflag;       /* whether gravity is on */
    REAL grav_const;     /* multiplier for gravitational force */
    int convex_flag;    /* whether any convex boundaries present */
    int pressflag;      /* whether prescribed pressures present */
    int constr_flag;    /* set if there are any one-sided constraints */
    int hide_flag;      /* set for hidden surface removal */
    int motion_flag;    /* set for fixed scale of motion;
                                otherwise seek minimum. */
    int symmetry_flag;  /* whether symmetry group in effect */
    int torus_flag;    /* whether working in toroidal domain */
    int full_flag;    /* whether torus solidly packed with bodies */
    int pressure_flag;  /* whether pressure used dynamically */
    int projection_flag; /* whether to project */
    int area_norm_flag; /* whether to normalize force by area surrounding vertex */
    int norm_check_flag;  /* whether area normalization checks normal deviation */
    REAL norm_check_max;  /* maximum allowable deviation */
    int vol_flag;       /* whether body volumes up to date */
    int jiggle_flag;    /* whether to jiggle vertices at each move */
    int homothety;      /* flag for homothety adjustment each iteration */
    int wulff_flag;     /* whether we are using wulff shapes for energy */
    int wulff_count;    /* number of Wulff vectors read in */
    char wulff_name[60]; /* Wulff file or keyword */
    vertex_id  zoom_v;   /* vertex to zoom on */
    REAL zoom_radius;    /* current zoom radius */
    REAL total_area;
    REAL total_energy;
    REAL spring_energy;
    int total_facets;
    int bodycount;  /* number of bodies */
    body_id outside_body;  /* a body surrounding all others */
    REAL scale;    /* force to motion scale factor */
    REAL maxscale;    /* upper limit on scale factor */
    REAL pressure;   /* ambient pressure */
    REAL min_area;      /* criterion on weeding out small triangles */
    REAL min_length;    /* criterion on weeding out small triangles */
    REAL max_len;       /* criterion for dividing long edges */
    REAL max_angle;     /* max allowed deviation from parallelism */
    REAL temperature;  /* "temperature" for jiggling */
    REAL spring_constant;  /* for forcing edges to conform to boundary */
    int  gauss_order;      /* order for gaussian integration on facets */
    int  gauss1D_order;      /* order for gaussian 1D integration */
    int  gauss2D_order;      /* order for gaussian 2D integration */
    REAL torusv;             /* unit cell volume or area */
    REAL torus_period[MAXCOORD][MAXCOORD];
    REAL **inverse_periods;/* inverse matrix of torus periods */
    int  metric_flag;     /* set if background metric in force */
    int  conformal_flag;  /* set for conformal metrics */
    struct expnode metric[MAXCOORD][MAXCOORD]; /* metric component functions */
     
  };

extern struct web web;

extern element_id
  NULLVERTEX,
  NULLEDGE,
  NULLFACET,
  NULLBODY,
  NULLFACETEDGE;

/* vertex volume gradient storage */

/* structures chained from each vertex */
typedef struct volgrad 
  {
     body_id   b_id;   /* body involved */
     struct volgrad  *chain;  /* next body for this vertex */
     REAL grad[MAXCOORD];        /* volume gradient of body for this vertex */
     REAL normal[MAXCOORD];      /* approx normal of body at this vertex */
  } volgrad;

extern volgrad huge *vgradbase;   /* allocated list pointer */
extern int      vgradtop;    /* number of first free  structure */
extern long     vgradmax;    /* number allocated */

/* loop macros, for use when lists are not modified in loop */
#define FOR_ALL_VERTICES(v_id) \
for (v_id = web.skel[VERTEX].used ; valid_id(v_id) ; \
v_id = vptr(v_id)->forechain)

#define FOR_ALL_EDGES(e_id) \
for (e_id = web.skel[EDGE].used ; valid_id(e_id) ; \
e_id = eptr(e_id)->forechain)

#define FOR_ALL_FACETS(f_id) \
for (f_id = web.skel[FACET].used ; valid_id(f_id) ; \
f_id = fptr(f_id)->forechain)

#define FOR_ALL_BODIES(b_id) \
for (b_id = web.skel[BODY].used ; valid_id(b_id) ; \
b_id = bptr(b_id)->forechain)

#define FOR_ALL_FACETEDGES(fe_id) \
for (fe_id = web.skel[FACETEDGE].used ; valid_id(fe_id) ; \
fe_id = feptr(fe_id)->forechain)

/* These access functions are defined as macros, since they involve
   only one evaluation of the arguments */

#define valid_element(id)  (get_attr(id) & ALLOCATED)

#define get_edge_midv(e_id)         (eptr(e_id)->vertices[2])
#define set_edge_midv(e_id,v_id)    (eptr(e_id)->vertices[2] = (v_id))

#define get_fe_side(fe_id,x)  get_edge_side(get_fe_edge(fe_id),x)
#define get_fe_edge(fe_id)   \
   (x1_id=(fe_id),same_sign(feptr(x1_id)->fe_edge_id,x1_id))
#define get_edge_headv(e_id) \
   (x2_id=(e_id),eptr(x2_id)->vertices[inverted(x2_id) ? 0 : 1])
#define get_edge_tailv(e_id) \
   (x3_id=(e_id),eptr(x3_id)->vertices[inverted(x3_id) ? 1 : 0])
#define get_next_edge(fe_id)  (x4_id = (fe_id),\
   same_sign(feptr(x4_id)->nextedge[inverted(x4_id) ? 0 : 1],x4_id))
#define get_prev_edge(fe_id)  (x5_id = (fe_id),\
   same_sign(feptr(x5_id)->nextedge[inverted(x5_id) ? 1 : 0],x5_id))

#define get_coord(v_id)         (vptr(v_id)->coord)
#define get_param(v_id)         (vptr(v_id)->param)
#define get_boundary(v_id)      (vptr(v_id)->boundary)
#define set_boundary(v_id,bdry) (vptr(v_id)->boundary = (bdry))
#define set_boundary_num(v_id,bnum) \
                           (vptr(v_id)->boundary = web.boundaries + (bnum))
#define get_edge_boundary(e_id)      (eptr(e_id)->boundary)
#define set_edge_boundary(e_id,bdry) (eptr(e_id)->boundary = (bdry))
#define set_edge_boundary_num(e_id,bnum) \
                           (eptr(e_id)->boundary = web.boundaries + (bnum))
#define get_facet_boundary(f_id)      (fptr(f_id)->boundary)
#define set_facet_boundary(f_id,bdry) (fptr(f_id)->boundary = (bdry))
#define set_facet_boundary_num(f_id,bnum) \
                           (fptr(f_id)->boundary = web.boundaries + (bnum))

#define get_surfen(n)           (web.surfen + (n))
#define get_f_surfen_map(f_id)  (fptr(f_id)->surfen_map)
#define set_f_surfmap(f_id,map) (fptr(f_id)->surfen_map |= (map))
#define set_f_surfen_map(f_id,n)   (fptr(f_id)->surfen_map |= (1<<(n)))

#define get_quant(n)           (web.quants + (n))
#define get_f_quant_map(f_id)  (fptr(f_id)->quantity_map)
#define set_f_quantmap(f_id,map) (fptr(f_id)->quantity_map |= (map))
#define set_f_quant_map(f_id,n)   (fptr(f_id)->quantity_map |= (1<<(n)))

#define get_constraint(n)      (web.constraints + (n))
#define get_v_constraint_map(v_id)    (vptr(v_id)->constraint_map)
#define get_v_constraint_status(v_id,i) ((vptr(v_id)->constraint_status>>(i))&1)
#define get_v_constraint_state(v_id) (vptr(v_id)->constraint_status)
#define get_e_constraint_map(e_id)    (eptr(e_id)->constraint_map)
#define get_f_constraint_map(f_id)    (fptr(f_id)->constraint_map)

#define set_v_global(v_id)  (vptr(v_id)->constraint_map |= web.con_global_map)
#define set_v_conmap(v_id,map)   (vptr(v_id)->constraint_map |= (map))
#define set_e_conmap(e_id,map)   (eptr(e_id)->constraint_map |= (map))
#define set_f_conmap(f_id,map)   (fptr(f_id)->constraint_map |= (map))
#define set_v_constraint_map(v_id,n)   (vptr(v_id)->constraint_map |= (1<<(n)))
#define unset_v_constraint_map(v_id,n)   \
	       (vptr(v_id)->constraint_map &= ~(1<<(n)))
#define set_v_constraint_status(v_id,n)   \
	       (vptr(v_id)->constraint_status |= (1<<(n)))
#define set_v_constraint_state(v_id,n)   \
	       (vptr(v_id)->constraint_status = (n))
#define unset_v_constraint_status(v_id,n)   \
	       (vptr(v_id)->constraint_status &= ~(1<<(n)))
#define clear_v_constraint_status(v_id)   \
	       (vptr(v_id)->constraint_status = 0)
#define set_e_constraint_map(e_id,n)   (eptr(e_id)->constraint_map |= (1<<(n)))
#define unset_e_constraint_map(e_id,n)   \
	       (eptr(e_id)->constraint_map &= ~(1<<(n)))
#define set_f_constraint_map(f_id,n)   (fptr(f_id)->constraint_map |= (1<<(n)))
#define unset_f_constraint_map(f_id,n)   \
	       (fptr(f_id)->constraint_map &= ~(1<<(n)))

#define get_force(v_id)         (vptr(v_id)->force)
#define get_grad(v_id)          (vptr(v_id)->grad)
#define get_restore(v_id)       (vptr(v_id)->restore)
#define set_vertex_fe(v_id,fe)  (vptr(v_id)->fe_id = (fe))
#define get_vertex_fe(v_id)     (vptr(v_id)->fe_id)
#define get_vertex_star(v_id)   (vptr(v_id)->star)
#define set_vertex_star(v_id,a) (vptr(v_id)->star = (a))
#define add_vertex_star(v_id,a) (vptr(v_id)->star += (a))
#define set_vertex_valence(v_id,n)  (vptr(v_id)->valence = (n))
#define add_vertex_valence(v_id,n)  (vptr(v_id)->valence += (n))
#define get_vertex_valence(v_id)  (vptr(v_id)->valence)

#define get_fe_wrap(fe_id)    get_edge_wrap(MSCBUG get_fe_edge(fe_id)) 
#define get_fe_tailv(fe_id)     get_edge_tailv(get_fe_edge(fe_id))
#define get_fe_headv(fe_id)     get_edge_headv(get_fe_edge(fe_id))
#define get_fe_midv(fe_id)     get_edge_midv(get_fe_edge(fe_id))
#define set_edge_length(e_id,x) (eptr(e_id)->length = (x))
#define get_edge_density(e_id)      (eptr(e_id)->density)
#define set_edge_density(e_id,den)  (eptr(e_id)->density = (den))
#define get_edge_star(e_id)   (eptr(e_id)->star)
#define set_edge_star(e_id,a) (eptr(e_id)->star = (a))
#define add_edge_star(e_id,a) (eptr(e_id)->star += (a))

#define get_facet_vertices(f_id)    (fptr(f_id)->vertices)
#define get_edge_vertices(e_id)    (eptr(e_id)->vertices)

#define get_facet_density(f_id)      (fptr(f_id)->density)
#define set_facet_density(f_id,den)  (fptr(f_id)->density = (den))

#define get_facet_area(f_id)      (fptr(f_id)->area)
#define set_facet_area(f_id,a)  (fptr(f_id)->area = (a))

#define get_facet_flux(f_id)      (fptr(f_id)->flux)
#define set_facet_flux(f_id,a)  (fptr(f_id)->flux = (a))

#define get_body_fe(b_id)  (x6_id=(b_id),\
    ( valid_id(x6_id) ? bptr(x6_id)->fe_id : NULLID ))

#define set_body_fe(b_id,fe)   (x7_id=(b_id),\
            ( valid_id(x7_id) ?  bptr(x7_id)->fe_id = (fe) : NULLID ) )

#define get_body_density(b_id)  (xx_id=(b_id),\
             ( valid_id(xx_id) ?  bptr(xx_id)->density : 0.0 ))

#define get_body_volume(b_id)  (xx_id=(b_id),\
             ( valid_id(xx_id) ?  bptr(xx_id)->volume : 0.0 ))

#define get_body_fixvol(b_id)   (xx_id=(b_id),\
             ( valid_id(xx_id) ?  bptr(xx_id)->fixvol : 0.0 ))

#define get_body_pressure(b_id)  (xx_id=(b_id),\
             ( valid_id(xx_id) ?   bptr(xx_id)->pressure : 0.0 ))

#define get_body_volconst(b_id) (xx_id=(b_id),\
             ( valid_id(xx_id) ?  bptr(xx_id)->volconst : 0.0 ))

#define set_body_volconst(b_id,v) (x8_id=(b_id),\
             ( valid_id(x8_id) ?  bptr(x8_id)->volconst = (v) : 0.0 ) )

#define set_body_density(b_id,v)  (x9_id=(b_id),\
             ( valid_id(x9_id) ?  bptr(x9_id)->density = (v) : 0.0 ) )

#define set_body_volume(b_id,v)  (xa_id=(b_id),\
             ( valid_id(xa_id) ?  bptr(xa_id)->volume = (v) : 0.0 ) )

#define set_body_fixvol(b_id,v)  (xb_id=(b_id),\
             ( valid_id(xb_id) ?  bptr(xb_id)->fixvol = (v) : 0.0 ) )

#define set_body_pressure(b_id,v) (xc_id=(b_id),\
             ( valid_id(xc_id) ?  bptr(xc_id)->pressure = (v) : 0.0 ) )

#define get_attr(id)    (elptr(id)->attr)
#define get_vattr(id)   (vptr(id)->attr)
#define get_eattr(id)   (eptr(id)->attr)
#define get_fattr(id)   (fptr(id)->attr)
#define get_battr(id)   (bptr(id)->attr)
#define get_feattr(id)  (feptr(id)->attr)

#define get_tag(f_id)   (fptr(f_id)->tag)
#define set_tag(f_id,t) (fptr(f_id)->tag = (t))

#define set_f_phase(f_id,p)  (fptr(f_id)->phase = (short)(p))
#define set_b_phase(b_id,p)  (bptr(b_id)->phase = (short)(p))
#define get_f_phase(f_id)  (fptr(f_id)->phase)
#define get_b_phase(b_id)  (bptr(b_id)->phase)

#define set_original(id,num)  (elptr(id)->original = (num) )
#define get_original(id)  (elptr(id)->original)

#define set_edge_color(e_id,col)   (eptr(e_id)->color = (col))
#define get_edge_color(e_id)         (eptr(e_id)->color)
#define set_facet_color(f_id,col)   (fptr(f_id)->color = (col))
#define get_facet_color(f_id)         (fptr(f_id)->color)

#define get_vertex_work(v_id)         (vptr(v_id)->workarea)

