
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*    file:      lowbound.c
*
*    Purpose:   Finding lower bound on surface area by
*               establishing near-maximal flow.  Flow
*               constructed by following surface as volume
*               is increased. Flow is splined through positions.
*/

#include "include.h"

/* spline stuff */
/* uniform cubic B-splines */
#define KNOTS 4
#define SPLINEDEGREE 3
double spline_coeff[KNOTS][SPLINEDEGREE+1] =
    { {-1/6.0,0.5,-0.5,1/6.0}, /* highest pwer coeff first */
      {0.5, -1.0, 0.0, 2/3.0},
      {-0.5,0.5,0.5,1/6.0},
      {1/6.0,0.0,0.0}};


static int shrink_flag; /* for telling when motion gone far enough */
static REAL  (*startlist[KNOTS-1])[MAXCOORD];  /* allocated space for
       coordinates needed for going both ways */
static REAL  (*knotlist[KNOTS])[MAXCOORD];  /* allocated space for old coordinates */
static REAL *lambda;   /* displacement multiples along vol grads */
static REAL *dlambda;   /* differential multiples along vol grads */
static REAL *gg;       /* squares of vol grads */
static int  lambda_flag;  /* whether lambda values decent */
static body_id b_id;

void lowbound()
{
   facet_id f_id;
   REAL dvol = 0.01;
   REAL total_flux = 0.0;
   int i;
   int step;
   int maxsteps = 3;
   double original_volume;
   ATTR old_attr;

   if ( (web.dimension != SOAPFILM) )
     error("Lower bound estimate implemented for SOAPFILM model only.",
        RECOVERABLE);

   if ( (web.modeltype != LINEAR) )
     error("Lower bound estimate implemented for LINEAR model only.",
        RECOVERABLE);

   if ( web.torus_flag )
     error("Lower bound estimate not implemented for TORUS model.",
        RECOVERABLE);

   if ( web.bodycount != 1 )
      { sprintf(errmsg,"Body count is %d, not 1.\n",web.bodycount);
        error(errmsg,RECOVERABLE);
      }


  /* allocate space to hold vertex body volume gradients */
  vgrad_init(0);   /* no fixed quantities */
  lambda = (double *)temp_calloc(web.skel[VERTEX].max_ord,sizeof(double));
  dlambda = (double *)temp_calloc(web.skel[VERTEX].max_ord,sizeof(double));
  gg = (double *)temp_calloc(web.skel[VERTEX].max_ord,sizeof(double));
  lambda_flag = 0;
  for ( i = 0 ; i < web.skel[VERTEX].max_ord ; i++ )
     lambda[i] = 1.0;    /* crude initialization */

   /* get our body */
   b_id = NULLBODY;
   generate_all(BODY,&b_id); 

   /* Starting conditions:  Have surface near minimal. One body
      is defined with no fixed volume with that surface as boundary.
      Facet areas in place.
    */

   /* Initialize flux through each facet as facet area */
   FOR_ALL_FACETS(f_id)
     set_facet_flux(f_id,get_facet_area(f_id));

   /* Set body volume to current volume */
   original_volume = get_body_volume(b_id);
   old_attr = get_battr(b_id);
   set_body_fixvol(b_id,original_volume);
   set_attr(b_id,FIXEDVOL);
  
   /* save original configuration */
   save_coords();
   startlist [1] = oldcoord;
   oldcoord = NULL;
printf("original volume %f\n",original_volume);
command("list vertices where x > .1 and x < 1.9 and y > .1 and y < 1.9");
 
   /* get initial couple of positions */
       /* decrement body volume */
       set_body_fixvol(b_id,original_volume - dvol);

       /* find new configuration */
       puff();

       /* save surface */
       save_coords();
       startlist[0] = oldcoord;
       oldcoord = NULL;

       /* increment body volume */
       set_body_fixvol(b_id,original_volume + dvol);

       /* find new configuration */
       oldcoord = startlist[1]; restore_coords(); oldcoord = NULL;
       lambda_flag = 0;
       puff();

       /* save surface */
       save_coords();
       startlist[2] = oldcoord;
       oldcoord = NULL;
   
   /* going forwards */
   for ( i = 0 ; i < KNOTS-1 ; i++ ) knotlist[i] = startlist[i];

   /* Main loop of slightly moving surface and calculating flux */
   for ( step = (KNOTS+1)/2 ; step < maxsteps ; step++ )
     {
       /* increment body volume */
       set_body_fixvol(b_id,original_volume + step*dvol);

       /* find new configuration */
       lambda_flag = 0;
       puff();
       
       /* save configuration */
       save_coords();
       knotlist[KNOTS-1] = oldcoord;
       oldcoord = NULL;

       /* find fluxes */
       total_flux = 0.0;
       shrink_flag = 0; /* will be set if any fluxes shrink */
       FOR_ALL_FACETS(f_id)
         total_flux += find_flux(f_id);

       /* print result */
       fprintf(outfd,"Flux = %20.15f \n",total_flux);

       /* set up for next loop */
       if ( step - (KNOTS+1)/2 >= KNOTS - 1 )
           temp_free((char *)knotlist[0]);  /* don't need anymore */
       for ( i = 0 ; i < KNOTS-1 ; i++ ) knotlist[i] = knotlist[i+1]; 

       /* see if done */
       if ( shrink_flag == 0 )
         break;
     }
  for ( i = 0 ; i < KNOTS-1 ; i++ )
    if ( step - (KNOTS+1)/2 + i > KNOTS - 2 )
      temp_free((char *)knotlist[i]);
   
   /* going backwards */
   for ( i = 0 ; i < KNOTS-1 ; i++ ) knotlist[i+1] = startlist[i];
   oldcoord = knotlist[0]; restore_coords(); oldcoord = NULL;

   /* Main loop of slightly moving surface and calculating flux */
   for ( step = (KNOTS-1)/2 ; step < maxsteps ; step++ )
     {
       /* increment body volume */
       set_body_fixvol(b_id,original_volume - step*dvol);

       /* find new configuration */
       lambda_flag = 0;
       puff();
       
       /* save configuration */
       save_coords();
       knotlist[0] = oldcoord;
       oldcoord = NULL;

       /* find fluxes */
       total_flux = 0.0;
       shrink_flag = 0; /* will be set if any fluxes shrink */
       FOR_ALL_FACETS(f_id)
         total_flux += find_flux(f_id);

       /* print result */
       fprintf(outfd,"Flux = %20.15f \n",total_flux);

       /* set up for next loop */
       if ( step - (KNOTS-1)/2 >= KNOTS - 1 )
           temp_free((char *)knotlist[KNOTS-1]);  /* don't need anymore */
       for ( i = KNOTS-1 ; i > 0 ; i-- ) knotlist[i] = knotlist[i-1]; 

       /* see if done */
       if ( shrink_flag == 0 )
         break;
     }
  for ( i = 1 ; i < KNOTS ; i++ )
    if ( i <= step - (KNOTS-1)/2 )
      temp_free((char *)knotlist[i]);

  /* clean up */
  unset_attr(b_id,ALL_ATTR);
  set_attr(b_id,old_attr);
  oldcoord = startlist[1];
  restore_coords();
  oldcoord = NULL;
  for ( i = 0 ; i < KNOTS-1 ; i++ )
    temp_free((char *)startlist[i]);
  calc_content();
  calc_pressure();
  calc_energy();  /* energy after motion */
}

/***********************************************************************
*
*  function: puff()
*
*  purpose:  move surface along normals (volume gradients)
*            to get minimum area for new volume.
*/

void puff()
{
  vertex_id v_id;
  double sumgg = 0.0;
  int k;

  /* calculate body volume gradients at all control points 
     due to free surfaces */
  if ( web.dimension == STRING )
    (*string_grad)();
  else /* web.dimension == SOAPFILM */
    (*film_grad)();

  /* get squares of vol grads */
  FOR_ALL_VERTICES(v_id)
    { 
      volgrad *vg = get_bv_vgrad(b_id,v_id);
      int j = ordinal(v_id);
      if ( get_vattr(v_id) & FIXED ) continue;
      if ( vg == NULL )
       { fprintf(stderr,"b_id %x  v_id %d \n",b_id,ordinal(v_id)+1);
	 error("247: No vgrad.",RECOVERABLE);
       }
      gg[j] = dot(vg->grad,vg->grad,web.sdim);
      sumgg += gg[j];
    }

  for ( k = 0 ; k < 10 ; k++ )
    {
      double alpha;
      double sum;
      double dv;
      double energy0,energy1,energy2;
      double scale,minscale;

      /* fix volume */
      calc_content();
      dv = get_body_fixvol(b_id) - get_body_volume(b_id);
      sum = 0.0;
      FOR_ALL_VERTICES(v_id)
	{ int j = ordinal(v_id);
	  if ( get_vattr(v_id) & FIXED ) continue;
	  sum += lambda[j]*gg[j];
	}
      alpha = dv/sum;
      FOR_ALL_VERTICES(v_id)
	{ REAL *c,*g;
	  int m,j;
	  if ( get_vattr(v_id) & FIXED ) continue;
	  c = get_coord(v_id);
	  g = get_bv_vgrad(b_id,v_id)->grad;
	  j = ordinal(v_id);
	  for ( m = 0 ; m < web.sdim ; m++ )
	    c[m] += alpha*lambda[j]*g[m];
          if ( lambda_flag )
            lambda[j] += alpha*lambda[j];
          else
            lambda[j] = alpha*lambda[j];
	}
      lambda_flag = 1;

      /*  minimize area */
      energy0 = web.total_energy;
      calc_force();
      sum = 0.0;
      FOR_ALL_VERTICES(v_id)
	{ double prod; 
	  if ( get_vattr(v_id) & FIXED ) continue;
	  prod = dot(get_force(v_id),get_bv_vgrad(b_id,v_id)->grad,web.sdim);
	  dlambda[ordinal(v_id)] = prod/gg[ordinal(v_id)];
	  sum += prod;
	}
      alpha = sum/sumgg;
      FOR_ALL_VERTICES(v_id)
	dlambda[ordinal(v_id)] -= alpha;
      scale = .1;
      FOR_ALL_VERTICES(v_id)
	{ REAL *c,*g;
	  int m,j;
	  if ( get_vattr(v_id) & FIXED ) continue;
	  c = get_coord(v_id);
	  g = get_bv_vgrad(b_id,v_id)->grad;
	  j = ordinal(v_id);
	  lambda[j] += scale*dlambda[j];
	  for ( m = 0 ; m < web.sdim ; m++ )
	    c[m] += scale*dlambda[j]*g[m];
	}
      calc_energy();
      energy1 = web.total_energy;
      FOR_ALL_VERTICES(v_id)
	{ REAL *c,*g;
	  int m,j;
	  if ( get_vattr(v_id) & FIXED ) continue;
	  c = get_coord(v_id);
	  g = get_bv_vgrad(b_id,v_id)->grad;
	  j = ordinal(v_id);
	  lambda[j] += scale*dlambda[j];
	  for ( m = 0 ; m < web.sdim ; m++ )
	    c[m] += scale*dlambda[j]*g[m];
	}
      calc_energy();
      energy2 = web.total_energy;
      minscale = scale*(energy2 - 4*energy1 + 3*energy0)
		  /(energy2 - 2*energy1 + energy0)/2;
      if ( minscale <= 0.0 ) 
	{ error("minscale <= 0",RECOVERABLE);
	}
      if ( minscale > .4 ) minscale = .4;
      scale = minscale - .2;  /* correct for what already done */
      FOR_ALL_VERTICES(v_id)
	{ REAL *c,*g;
	  int m,j;
	  if ( get_vattr(v_id) & FIXED ) continue;
	  c = get_coord(v_id);
	  g = get_bv_vgrad(b_id,v_id)->grad;
	  j = ordinal(v_id);
	  lambda[j] += scale*dlambda[j];
	  for ( m = 0 ; m < web.sdim ; m++ )
	    c[m] += scale*dlambda[j]*g[m];
	}
      calc_energy();
    }

printf("puffed   volume %f\n",get_body_fixvol(b_id));
command("list vertices where x > .1 and x < 1.9 and y > .1 and y < 1.9");

}

/*****************************************************************************
*
*  Function: splinepoly()
*
*  Purpose:  value of spline
*
*/

double splinepoly(k,t)
int k; /* which part of spline */
double t; /* where in spline */
{
  int i;
  double value = spline_coeff[k][0];

  for ( i = 1 ; i <= SPLINEDEGREE ; i++ ) 
    value = spline_coeff[k][i] + t*value;

  return value;
}


/*****************************************************************************
*
*  Function: splinederiv()
*
*  Purpose:  value of spline derivative
*
*/

double splinederiv(k,t)
int k; /* which part of spline */
double t; /* where in spline */
{
  int i;
  double value = SPLINEDEGREE*spline_coeff[k][0];

  for ( i = 1 ; i < SPLINEDEGREE ; i++ ) 
    value = (SPLINEDEGREE - i)*spline_coeff[k][i] + t*value;

  return value;
}

/****************************************************************************
*
*  Function: spline_partial()
*
*  Purpose:  Find barycentric partials of spline flow in facet.
*            Point at barycentric coords (1,0,0)
*/

void spline_partial(vnum,t,bari,partial)
int *vnum;  /* numbers of vertices around facet in knotlist */
double t;  /* time, 0 < t < 1 */
int bari;  /* which barycentric coordinate, 0 or 1 */
REAL *partial; /* where to put results */
{
  int i,k;

  /* zero out */
  for ( i = 0 ; i < web.sdim ; i++ ) partial[i] = 0.0;

  for ( k = 0 ; k < KNOTS ; k++ )
    { double spline_val = splinepoly(k,t);
      for ( i = 0 ; i < web.sdim ; i++ )
        partial[i] += spline_val*(knotlist[k][vnum[bari]][i]
                                         -knotlist[k][vnum[2]][i]);
    }
}

/****************************************************************************
*
*  Function: spline_partial_t()
*
*  Purpose:  Find t partial of spline flow in facet at vertex 0.
*
*/

void spline_partial_t(vnum,t,partial)
int *vnum;  /* numbers of vertices around facet in knotlist */
double t;  /* time, 0 < t < 1 */
REAL *partial; /* where to put results */
{
  int i,k;

  /* zero out */
  for ( i = 0 ; i < web.sdim ; i++ ) partial[i] = 0.0;

  for ( k = 0 ; k < KNOTS ; k++ )
    { double spline_val = splinederiv(k,t);
      for ( i = 0 ; i < web.sdim ; i++ )
        partial[i] += spline_val*knotlist[k][vnum[0]][i];
    }
}

/*************************************************************************
*
*  function: find_flux()
* 
*  purpose:  find guaranteed flux through moved facet by finding
*            minimum flux factor for any point in facet.
*            Assumes minimum will be at one of the corners, which
*            is a pretty good assumption.
*
*  input:    facet_id f_id  - the facet to check
*
*  output:   flux through new facet 
*
****/

REAL find_flux(f_id)
facet_id f_id;
{
  REAL t;
  REAL flux; /* current fluxes */
  int i;
  facetedge_id fe;
  vertex_id v_id[FACET_VERTS+2];
  int vord[FACET_VERTS+2]; /* +2 so can always go 3 from current */

  /* get info */
  fe = get_facet_fe(f_id);
  for ( i = 0 ; i < FACET_VERTS+2 ; i++, fe = get_next_edge(fe) )
    { 
      v_id[i] = get_fe_tailv(fe);
      vord[i] = ordinal(v_id[i]);
    }

  flux = 999.0;  /* initial ridiculous value */

  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { /* do each of three corners */
      double throat;
      REAL dxda[MAXCOORD],dxdb[MAXCOORD],dxdt[MAXCOORD];

      if ( get_vattr(v_id[i]) & FIXED ) continue;
      for ( t = 0.0 ; t < 1.00001 ; t += 0.2 ) /* crude search */
        { /* get partial */
          spline_partial(vord+i,t,0,dxda); 
          spline_partial(vord+i,t,1,dxdb); 
          spline_partial_t(vord+i,t,dxdt);

          throat = triple_prod(dxda,dxdb,dxdt)/sqrt(dot(dxdt,dxdt,web.sdim));
          if ( throat < flux ) flux = throat; 
        }
    }

  flux /= 2;  /* triangle is half of quadrilateral */

  if ( get_facet_flux(f_id) > flux )
    { set_facet_flux(f_id,flux);
      shrink_flag = 1;
    }

/********/
if ( flux < 0.0 )
  fprintf(stderr,"Facet %d has negative flux %f\n",ordinal(f_id)+1,flux);

  fprintf(stderr,"Facet %d has flux %f\n",ordinal(f_id)+1,flux);
/********/

  return get_facet_flux(f_id);
}

