/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*    file:      iterate.c
*
*    Purpose:   Do one iteration of motion.
*/


#include "include.h"

/*************************************************************
*
*  Function: iterate()
*
*  Purpose:  calculate and make one move of vertices
*/

void iterate()
{
  static REAL energy0,energy1,energy2;
  int seekcount = 0;  /* to prevent infinite loop */
  double denom;  /* denominator of interpolation expression */
  double scale0,scale1,scale2;  /* for interpolation */

  if ( web.skel[VERTEX].count == 0 )
    { error("No surface.\n",WARNING);
      return;
    }


  iterate_flag = 1;  /* for interrupt handler */

  if ( web.diffusion_flag ) 
    diffuse();
  
restart:
  calc_force();
  if ( approx_curve_flag )
    {  
      /* approx_curvature();*/  /* Dzuid and Schmidt version */
      mobility_setup();
      approx_curv_calc();
    }
  if ( web.projection_flag &&
         web.torus_flag && (web.dimension == SOAPFILM) ) 
           torvol_project(SET_PRESSURE);
    else  vol_project(SET_PRESSURE);
  if ( approx_curve_flag ) mobility_cleanup();

  if ( (web.area_norm_flag && web.norm_check_flag) || !web.motion_flag )
      save_coords();
  if ( conj_grad_flag )
    { cg_calc_gamma(); /* find adjustment factor  */
      cg_direction();  /* fix up motion direction */
    }
  if ( !web.motion_flag )  /* want to seek minimum energy along gradient */
    { REAL tempscale = web.scale;
      seekcount++;
      web.scale = 0.0; /* to do restoring force */
      move_vertices();  /* moving by current scale */
      energy0 = web.total_energy;
      scale0 = 0;
      restore_coords(); 
      if ( tempscale < 1e-9 ) web.scale = web.maxscale/100;
      else web.scale = tempscale;
      move_vertices();  /* moving by current scale */
      energy1 = web.total_energy;
      scale1 = web.scale;
      restore_coords();
/*      sprintf(msg,"energy0 %15.10f  energy1 %15.10f  scale %f\n",
                                           energy0,energy1,web.scale);
      outstring(msg);
*/
      if ( energy1 < energy0 )
        {
          while ( (web.scale < web.maxscale) || conj_grad_flag )
          {
           web.scale *= 2;
           move_vertices();
           energy2 = web.total_energy;
           scale2 = web.scale;
           restore_coords();
/*           sprintf(msg,"energy2: %15.10f   new scale: %f\n",energy2,web.scale);
           outstring(msg);
*/
           if ( energy2 > energy1 ) 
             { web.scale /= 2; break; }
           energy1 = energy2;  scale1 = scale2;
          }
        }
      else /* have to come down in scale */
        {
          web.scale /= 2;
          do
           { 
             if ( seekcount > 20 ) /* looks like energy won't decrease */
               { web.scale = 0.0; break; }
             energy2 = energy1; scale2 = scale1;
             web.scale /= 2;
             if  ( web.scale < 1e-12 ) { web.scale = 0.0; break; }
             move_vertices();
             energy1 = web.total_energy;
             scale1 = web.scale;
             restore_coords();
/*             sprintf(msg,"energy1: %15.10f   new scale: %f\n",
                                                   energy1,web.scale);
           outstring(msg);
  */
           }
          while ( energy1 > energy0 );
          web.scale *= 2;
        }

      if ( web.scale > web.maxscale ) 
        web.scale = web.maxscale;
      else if ( web.scale > 0.0 )
       {
/*         sprintf(msg,"energy1 %15.10f  energy2 %15.10f  scale %g\n",
                                        energy1,energy2,web.scale);
       outstring(msg);
  */
         /* now quadratic interpolation for minimum energy */
         denom = energy0*(scale1-scale2)+energy1*(scale2-scale0)
                   + energy2*(scale0 - scale1);
         if ( denom == 0.0 ) web.scale = 0.0;
         else
           web.scale = ((energy0-energy2)*scale1*scale1
               +(energy1-energy0)*scale2*scale2
               +(energy2-energy1)*scale0*scale0)/2/denom;
/*
         if ( 2*energy0 - 3*energy1 + energy2 != 0.0 )
           web.scale *= 0.75*(4*energy0 - 5*energy1 + energy2)
                         /(2*energy0 - 3*energy1 + energy2); 
*/
         /* else leave scale as is */
       }
   }
  else if ( runge_kutta_flag )
    { /* only for fixed scale */
      save_coords();
      runge_kutta(); 
    }


  if ( web.scale > web.maxscale ) 
     web.scale = web.maxscale; /* max on movement */
  if ( web.jiggle_flag )  jiggle(); 
  if ( autopop_flag || autochop_flag ) 
   { autopop_detect(web.scale);
     if ( autopop_count || autochop_count )
       { autopop_pop();
	 if ( run_checks() ) error("Bad surface after autopop.\n",RECOVERABLE);
	 autochop_chop();
	 if ( run_checks() ) error("Bad surface after autochop.\n",RECOVERABLE);
         goto restart; 
       }
    }
  move_vertices();
  total_time += web.scale;
  if ( !web.motion_flag && (web.total_energy > energy0) )
    { error("Optimizing scale did not reduce energy.\n",WARNING);
      goto iterate_error_exit;
    }
  if ( web.area_norm_flag && web.norm_check_flag &&(web.dimension==SOAPFILM) )
    {  REAL delta = normal_change_check();
       if ( delta  > web.norm_check_max )
        { sprintf(msg,"Max normal change: %f. Restoring old coordinates.\n",
                      delta);
          outstring(msg);
          goto iterate_error_exit;
        }
    }
    
/* if chop here, then instabiilties have a chance to get out of control */
/*  if ( autopop_flag ) autopop_pop();
  if ( autochop_flag ) autochop_chop();
/*
/* following good for debugging to see if energy gradients really give
   change in energy.  Estimated decrease should be approx exact for
   scale much less than optimum; at optimum scale, estimate is twice
   actual (if energy shape true parabola )
*/

  if ( estimate_flag )
    {
      sprintf(msg,"Estimated energy change: %20.15g\n",estimate_decrease());
      outstring(msg);
    }


  if ( (fabs(web.total_area) > 1e-4) && (fabs(web.total_energy) > 1e-4) )
    sprintf(msg,"%3d. %s: %17.15f energy: %17.15f  scale: %g\n",gocount,
            areaname,web.total_area,web.total_energy,web.scale);
  else
    sprintf(msg,"%3d. %s: %17.15g energy: %17.15g  scale: %g\n",gocount,
            areaname,web.total_area,web.total_energy,web.scale);
  outstring(msg);

iterate_exit:
  if ( oldcoord ) unsave_coords();
  iterate_flag = 0;  /* for interrupt handler */
  update_display();
  return;

iterate_error_exit:
  /* in case optimizing scale didn't work */
  restore_coords();
  if ( fixed_volumes || web.pressure_flag ) calc_content();
  calc_pressure();
  calc_energy();  /* energy after motion */
  goto iterate_exit;
} 

/***************************************************************
*
*   Function: fix_vertices()
*
*   Purpose:  Fixes vertices by zeroing out force.  Also projects
*             force to be tangent to boundary.
*
*/

void fix_vertices()
{
  vertex_id v_id;
  REAL *force;
  int j;
  int hitcount;

  FOR_ALL_VERTICES(v_id)
    {
      ATTR attr = get_vattr(v_id);

      if ( attr & FIXED )
        { force = get_force(v_id);
          force[0] = force[1] = force[2] = 0.0;
        }
      else if ( attr & CONSTRAINT )
        {
          MAP conmap = get_v_constraint_map(v_id);
          int oncount = 0;
          struct constraint *con[CONSTRMAX];
          int conlist[CONSTRMAX];
          REAL perp[MAXCOORD];

          force = get_force(v_id);
          for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
            { if ( !(conmap & 1) ) continue;
              if ( get_v_constraint_status(v_id,j) == ON_CONSTRAINT )
                { conlist[oncount] = j;
                  con[oncount++] = get_constraint(j);
                }
            }

          hitcount = constr_proj(TANGPROJ,oncount,con,get_coord(v_id),
                         force,perp,conlist,DETECT);
          if ( hitcount != oncount )
            { clear_v_constraint_status(v_id);
              for ( j = 0 ; j < hitcount ; j++ )
                set_v_constraint_status(v_id,conlist[j]);
            }
          for ( j = 0 ; j < web.sdim ; j++ )
            force[j] -= perp[j];
       }
    }
}


/***************************************************************
*
*   Function: move_vertices()
*
*   Purpose:  moves all unfixed vertices by current scale
*             factor and force.
*
*   Input:    uses global variable for scale
*/

/* moves vertices according to force and scale factor */
void move_vertices()
{
  REAL *force;
  REAL *x,*r;
  REAL scale = web.scale;
  int i,j;
  vertex_id v_id;

  /* move by multiple of energy gradient, and volume restoration */
  FOR_ALL_VERTICES(v_id)
    {
      if ( get_vattr(v_id) & FIXED ) continue ;
      force = get_force(v_id);
      x = get_coord(v_id);
      r = get_restore(v_id);
      if ( get_vattr(v_id) & BOUNDARY )
      {
        struct boundary *boundary = get_boundary(v_id);
        int pcount = boundary->pcount;
        REAL *param = get_param(v_id);

        if ( web.pressure_flag )
          for ( i = 0 ; i < pcount ; i++ )
            param[i] += scale*force[i];
        else
          for ( i = 0 ; i < pcount ; i++ )
            param[i] += r[i] + scale*force[i];
        for ( j = 0 ; j < web.sdim ; j++ )
          x[j] = eval(boundary->coordf[j],param);
      }
      else
       {
         if ( web.pressure_flag )
           for ( i = 0 ; i < web.sdim ; i++ )
             x[i] += scale*force[i];
         else
           for ( i = 0 ; i < web.sdim ; i++ )
             x[i] += r[i] + scale*force[i];
       }
    }
  
  if ( web.homothety )
    homothety();

  /* project to constraints */
  FOR_ALL_VERTICES(v_id)
    {
      if ( get_vattr(v_id) & CONSTRAINT )
          project_v_constr(v_id);
    }

  /* enforce volume constraints also; keep conjugate gradient under control */
  if ( conj_grad_flag )
    {
      if ( fixed_volumes  || web.pressure_flag) calc_content();
      if ( web.projection_flag &&
         web.torus_flag && (web.dimension == SOAPFILM) )
           torvol_project(NO_SET_PRESSURE);
      else  vol_project(NO_SET_PRESSURE);
      FOR_ALL_VERTICES(v_id)
        { REAL *r = get_restore(v_id);
          REAL *x = get_coord(v_id);
          if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) continue;
          for ( i = 0 ; i < web.sdim ; i++ )
            x[i] += r[i];
        }
    }

  if ( fixed_volumes || web.pressure_flag ) calc_content();
  calc_pressure();
  calc_energy();  /* energy after motion */
}

/**************************************************************
*
*  Function: move_vertex()
* 
*  Purpose:  Moves one vertex.
*/

void move_vertex(v_id)
vertex_id v_id;
{ 
  REAL *force;
  REAL *x,*r;
  int i,j;

      if ( get_vattr(v_id) & FIXED ) return ;
      force = get_force(v_id);
      x = get_coord(v_id);
      r = get_restore(v_id);
      if ( get_vattr(v_id) & BOUNDARY )
      {
        struct boundary *boundary = get_boundary(v_id);
        int pcount = boundary->pcount;
        REAL *param = get_param(v_id);

        if ( web.pressure_flag )
          for ( i = 0 ; i < pcount ; i++ )
            param[i] +=  web.scale*force[i];
        else
          for ( i = 0 ; i < pcount ; i++ )
            param[i] += r[i] + web.scale*force[i];
        for ( j = 0 ; j < web.sdim ; j++ )
          x[j] = eval(boundary->coordf[j],param);
      }
      else
       {
         if ( web.pressure_flag )
           for ( i = 0 ; i < web.sdim ; i++ )
             x[i] += web.scale*force[i];
         else
           for ( i = 0 ; i < web.sdim ; i++ )
             x[i] += r[i] + web.scale*force[i];
       }

      /* project to constraints */
      if ( get_vattr(v_id) & CONSTRAINT )
          project_v_constr(v_id);
}

/****************************************************************
*
*  Function: save_coords()
*
*  Purpose: Save current coordinates so they can be restored
*           after a trial motion.
*
*/


void save_coords()
{
  vertex_id v_id;

  if ( oldcoord )
     temp_free((char *)oldcoord);   /* in case somebody forgot */
  oldcoord = (REAL (*)[MAXCOORD])temp_calloc(web.skel[VERTEX].max_ord,
                                                sizeof(REAL)*MAXCOORD);

  FOR_ALL_VERTICES(v_id)
    {
      if ( get_vattr(v_id) & BOUNDARY )
       memcpy((char *)(oldcoord+ordinal(v_id)),(char *)get_param(v_id),
                                               sizeof(REAL)*MAXPARAM);
      else
       memcpy((char *)(oldcoord+ordinal(v_id)),(char *)get_coord(v_id),
                                               sizeof(REAL)*MAXCOORD);
    }
}

/****************************************************************
*
*  Function: restore_coords()
*
*  Purpose: Restore current coordinates after a trial motion.
*
*/

void restore_coords()
{
  vertex_id v_id;

  if ( oldcoord == NULL )
    {
      error("Cannot restore old coordinates since there aren't any!\n",
                  RECOVERABLE);
    }

  FOR_ALL_VERTICES(v_id)
    restore_vertex(v_id);

}

/******************************************************************
*
*  Function: restore_vertex()
*
*  Purpose:  Put a vertex back where it was.
*/

void restore_vertex(v_id)
vertex_id v_id;
{
  int i;
  REAL *p,*x;
  struct boundary *bdry;

      if ( get_vattr(v_id) & BOUNDARY )
        { p = get_param(v_id);
          x = get_coord(v_id);
          bdry = get_boundary(v_id);
          memcpy((char *)p,(char *)(oldcoord+ordinal(v_id)),
                                              sizeof(REAL)*MAXPARAM);
          for ( i = 0 ; i < web.sdim ; i++ )
            x[i] = eval(bdry->coordf[i],p);
        }
      else
       memcpy((char *)get_coord(v_id),(char *)(oldcoord+ordinal(v_id)),
                                           sizeof(REAL)*MAXCOORD);
}

/********************************************************************
*
*  Function: unsave_coords()
*
*  Purpose: Clean up after all trial motions done.
*/

void unsave_coords()
{
  if ( oldcoord )
    temp_free( (char *)oldcoord );
  oldcoord = NULL;
}


/****************************************************************
*
*  Function:  jiggle()
*
*  Purpose:   Move each vertex a little bit at random to get
*             away from metastable equilibria and crystalline
*             integrand hangups.
*
*  Input:     The global variable temperature scales the size
*             of the jiggles.  They are taken to be spherically
*             symmetric Gaussian distribution with mean size
*             temperature*max_length.
*/

void jiggle()
{
  vertex_id v_id;
  REAL *x;
  double val;
  char response[100];

  sprintf(msg,"Enter temperature for jiggling (default %f): ",
                         web.temperature);
  outstring(msg);
  getstring(response);
  if ( logfd ) fprintf(logfd,"%s\n",response);
  if ( sscanf(response,"%lf",&val) == 1 )
     web.temperature = val;

  FOR_ALL_VERTICES(v_id)
    { if ( get_vattr(v_id) & FIXED ) continue;
      x = get_coord(v_id);
      x[0] += gaussian()*web.temperature*web.max_len*overall_size;
      x[1] += gaussian()*web.temperature*web.max_len*overall_size;
      if ( web.dimension == SOAPFILM )
        x[2] += gaussian()*web.temperature*web.max_len*overall_size;
    }
  outstring("One jiggle done.\n");
}


/****************************************************************
*
*  Function:  long_jiggle()
*
*  Purpose:   Move each vertex with a long wavelenth perturbation
*             away from metastable equilibria and crystalline
*             integrand hangups.
*
*  Input:     The global variable temperature scales the size
*             of the jiggles.  The perturbation has a random
*             amplitude vector chosen from sphere and random
*             wavevector likewise chosen at random.  Amplitude
*             is multiplied by current temperature.
*/

void long_jiggle()
{
  /* remembered parameters */
  static REAL wavev[MAXCOORD];
  static REAL amp[MAXCOORD];
  static REAL phase;

  REAL mag;
  REAL ww; 
  int j;
  vertex_id v_id;
  REAL *x;
  char response[100];

  /* get wave vector */
get_wv:
  sprintf(msg,"Enter wave vector (%f,%f,%f;r): ",wavev[0],wavev[1],wavev[2]);
  outstring(msg);
  getstring(response);
  if ( logfd ) fprintf(logfd,"%s\n",response);
  if ( response[0] == 'r' )
   { /* random */
     outstring("Enter random number seed: ");
     getstring(response);
     if ( logfd ) fprintf(logfd,"%s\n",response);
     if ( atoi(response) != 0 ) srand(atoi(response));

     /* pick random wavelength in unit sphere */
     do 
       { for ( j = 0 ; j < web.sdim ; j++ )
           wavev[j] = 1 - 2*(double)(rand()&0x7FFF)/0x7FFFL;
           ww = dot(wavev,wavev,web.sdim);
       }
     while ( ww > 1.0 );
     /* invert to wavevector and scale to surface size */
     for ( j = 0 ; j < web.sdim ; j++ )
       wavev[j] /= ww*overall_size;
   }
  else if ( isalpha(response[0]) ) return; /* escape without jiggle */
  else if ( response[0] )
   { double val[3];
     if ( 3 != sscanf(response,"%lf %lf %lf",val,val+1,val+2) )
       { outstring("Need three components.\n");
         goto get_wv;
       }
     for ( j = 0 ; j < web.sdim ; j++ ) wavev[j] = val[j];
   }

  /* pick random phase */
get_phase:
  sprintf(msg,"Enter phase (%f;r): ",phase);
  outstring(msg);
  getstring(response);
  if ( logfd ) fprintf(logfd,"%s\n",response);
  if ( response[0] == 'r' )
    phase = 2*M_PI*(double)(rand()&0x7FFF)/0x7FFFL;
  else if ( isalpha(response[0]) ) return; /* escape without jiggle */
  else if ( response[0] )
    { double val;
      if ( sscanf(response,"%lf",&val) != 1 )
        goto get_phase;
      phase = val;
    }

  /* amplitude */  
get_amp:
  sprintf(msg,"Enter amplitude (%f,%f,%f;r): ",amp[0],amp[1],amp[2]);
  outstring(msg);
  getstring(response);
  if ( logfd ) fprintf(logfd,"%s\n",response);
  if ( response[0] == 'r' )
   { /* random */
     /* pick random amplitude */
     do 
       for ( j = 0 ; j < web.sdim ; j++ )
         amp[j] = 1 - 2*(double)(rand()&0x7FFF)/0x7FFFL;
     while ( dot(amp,amp,web.sdim) > 1.0 );
     for ( j = 0 ; j < web.sdim ; j++ )
       amp[j] *= web.temperature*overall_size;
   }  
  else if ( isalpha(response[0]) ) return; /* escape without jiggle */
  else if ( response[0] )
   { double val[3];
     if ( 3 != sscanf(response,"%lf %lf %lf",val,val+1,val+2) )
       { outstring("Need three components.\n");
         goto get_amp;
       }
     for ( j = 0 ; j < web.sdim ; j++ ) amp[j] = val[j];
   }

  /* move vertices */
  FOR_ALL_VERTICES(v_id)
    { if ( get_vattr(v_id) & FIXED ) continue;
      x = get_coord(v_id);
      mag = sin(dot(wavev,x,web.sdim) + phase);
      for ( j = 0 ; j < web.sdim ; j++ )
        x[j] += amp[j]*mag;
    }
  outstring("One long jiggle done.\n");
}

/****************************************************************
*
*  Function: gaussian()
*
*  Purpose:  generate gaussian random variables with variance 1.
*
*/

REAL gaussian()
{
  int k;
  REAL sum = 0.0;

  for ( k = 0 ; k < 5 ; k++ ) sum += (double)(rand()&0x7FFF);
  return (sum/0x7FFFL - 2.5)/5*sqrt(12.0/5);
}


/***************************************************************
*
*   Function: estimate_decrease()
*
*   Purpose:  Estimates energy decrease from gradients of all 
*             unfixed vertices using current scale factor and force.
*
*   Input:    uses global variable for scale
*/

double estimate_decrease()
{
  vertex_id v_id;
  REAL *force;
  REAL *r;
  REAL area;
  int i;
  REAL change = 0.0;
  double delta;

  FOR_ALL_VERTICES(v_id)
    { if ( get_vattr(v_id) & FIXED ) continue;
      force = get_force(v_id);
      r = get_restore(v_id);
      if ( web.area_norm_flag ) area = get_vertex_star(v_id)/star_fraction;
      else area = 1.0;
      delta = 0.0;
      if ( get_vattr(v_id) & BOUNDARY )
      {
        struct boundary *boundary = get_boundary(v_id);
        int pcount = boundary->pcount;

        if ( web.pressure_flag )
            delta =  web.scale*dot(force,force,pcount)*area;
        else
          for ( i = 0 ; i < pcount ; i++ )
            delta  += force[i]*area*(r[i] + web.scale*force[i]);
      }
      else
       {
         if ( web.pressure_flag )
           for ( i = 0 ; i < web.sdim ; i++ )
             delta += web.scale*force[i]*force[i]*area;
         else
           for ( i = 0 ; i < web.sdim ; i++ )
             delta += force[i]*area*(r[i] + web.scale*force[i]);
       }
     change += delta;
    }
  return -change;  /* negative since forces are opposite gradients */
}

/********************************************************************
*
*  Function: homothety()
*
*  Purpose: homothety to scale to fixed area or volume.
*           Normalizes total volume of all bodies to 1.
*/

void homothety()
{
  body_id b_id;
  vertex_id v_id;
  double vol = 0.0;
  REAL *x;
  REAL scale;
  int i;

  if ( square_curvature_flag )
    scale = 1/pow(web.total_area/homothety_target,1.0/web.dimension);
  else
    {
      /* be sure to have current volumes */
      calc_content(); 
      FOR_ALL_BODIES(b_id)
        vol += get_body_volume(b_id);
      if ( web.dimension == STRING ) scale = 1/sqrt(vol/homothety_target);
      else scale = 1/pow(vol/homothety_target,1/3.0);
    }
  FOR_ALL_VERTICES(v_id)
    {  x = get_coord(v_id);
       for ( i = 0 ; i < web.sdim ; i++ ) x[i] *= scale;
    }
}

/***********************************************************************
*
*  function: cg_calc_gamma()
*
*  purpose:  Calculate conjugate gradient direction adjustment factor.
*
*/

void cg_calc_gamma()
{
  vertex_id v_id;
  double sum = 0.0;

  FOR_ALL_VERTICES(v_id)
    { REAL *f = get_force(v_id);
      if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) continue;
      sum += dot(f,f,web.sdim);
    }
  if ( cg_oldsum != 0.0 )
    cg_gamma = sum/cg_oldsum;

  cg_oldsum = sum; return;
}

/***********************************************************************
*
*  function: cg_direction()
*
*  purpose:  Adjusts direction of motion for conjugate gradient.
*
*/

void cg_direction()
{
  vertex_id v_id;
  REAL *f,*h;
  int i;

  if ( cg_hvector == NULL )
    { cg_hvector = (REAL (*)[MAXCOORD])mycalloc(web.skel[VERTEX].max_ord,
                               sizeof(REAL [MAXCOORD]));
      if ( cg_hvector == NULL )
        error("Cannot allocate memory for conjugate gradient.",RECOVERABLE);
    }

  FOR_ALL_VERTICES(v_id)
    { h = cg_hvector[ordinal(v_id)];
      f = get_force(v_id);
      if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) continue;
      for ( i = 0 ; i < web.sdim ; i++ )
        h[i] = f[i] += cg_gamma*h[i];
    }
}


/*********************************************************************
*
*  function: runge_kutta()
*
*  purpose: do runge-kutta method for calculating motion.
*           Assumes first force evaluation already done.
*           and vertices saved.
*
*/

void runge_kutta()
{
  REAL **k1,**k2,**k3, **k4;  /* saved motions */
  int i,j;
  vertex_id v_id;


  k1 = dmatrix(0,web.skel[VERTEX].max_ord,0,web.sdim-1);
  k2 = dmatrix(0,web.skel[VERTEX].max_ord,0,web.sdim-1);
  k3 = dmatrix(0,web.skel[VERTEX].max_ord,0,web.sdim-1);
  k4 = dmatrix(0,web.skel[VERTEX].max_ord,0,web.sdim-1);

  /* save first motion */
  FOR_ALL_VERTICES(v_id)
    { int k = ordinal(v_id);
      REAL *f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i++ ) k1[k][i] = f[i];
    }

  /* second motion */
  web.scale /= 2;
  move_vertices();
  calc_force();
  FOR_ALL_VERTICES(v_id)
    { int k = ordinal(v_id);
      REAL *f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i++ ) k2[k][i] = f[i];
    }

  /* third motion */
  restore_coords();
  move_vertices();
  calc_force();
  FOR_ALL_VERTICES(v_id)
    { int k = ordinal(v_id);
      REAL *f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i++ ) k3[k][i] = f[i];
    }

  /* fourth motion */
  restore_coords();
  web.scale *= 2;
  move_vertices();
  calc_force();
  FOR_ALL_VERTICES(v_id)
    { int k = ordinal(v_id);
      REAL *f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i++ ) k4[k][i] = f[i];
    }

  /* runge-kutta combination */
  FOR_ALL_VERTICES(v_id)
    { int k = ordinal(v_id);
      REAL *f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i++ )
	f[i] = (k1[k][i] + 2*k2[k][i] + 2*k3[k][i] + k4[k][i])/6;
    }
  restore_coords();

  free_matrix(k1);
  free_matrix(k2);
  free_matrix(k3);
  free_matrix(k4);
}

