/*
/*    file:      evolver.c
/*    level:     control
/*    functions: main()
/*	       a_area()
/*	       a_count()
/*	       a_go()
/*	       a_scale()
/*	       a_pressure()
/*	       a_incompress()
/*	       a_volumes()
/*	       a_refine()
/*	       a_weed()
/*	       a_long()
/*	       a_tiny()
/*	       a_equi()
/*	       a_notch()
/*	       a_motion()
/*	       a_jiggle()
/*	       a_temperature()
/*	       a_view()
/*	       a_load()
/*             a_save()
/*	       a_dump()
*/


/**************************************************************
*
*  File: evolver.c
*
*  Contents: Aargs interface for surface evolver.
*
*/

#include <aargi.h>
#include "include.h"
#include "aaproto.h"

void a_load(), a_restore(), a_save(), a_dump(), old_menu();
void extrapolate();
void resize(),display();
/* int make_movie(); */

void a_area()
{
  if ( aa_cmd.error ) return;
  if ( aa_cmd.vcnt < 1 )
    if ( web.area_norm_flag ) puts("Area normalization ON.");
    else puts("Area normalization OFF.");
  else if (strcmp(aargv[1],"on") == 0 ) web.area_norm_flag = 1;
  else if (strcmp(aargv[1],"off") == 0 ) web.area_norm_flag = 0;
}

void a_count()
{
  if ( aa_cmd.error ) return;
  memory_report();
}

#ifdef PIXAR
a_pixar()
{
  if ( aa_cmd.error ) return;
  graph_start = pixar_start;
  graph_facet = pixar_facet;
  graph_end = pixar_end;
  graphgen();
}
#endif

void a_go()
{
  int gocount;

  if ( aa_cmd.error ) return;
  if ( aa_cmd.vcnt < 1 ) gocount = 1;
  else gocount = atoi(aargv[1]);
  for ( ; gocount > 0 ; gocount-- ) iterate();
}

void a_scale()
{
  if ( aa_cmd.error ) return;
  if ( aa_cmd.vcnt < 1 ) printf("Current scale factor is %f\n",web.scale);
  else web.scale = atof(aargv[1]);
}

void a_pressure()
{
  if ( aa_cmd.vcnt < 1 )
   {
     printf("Pressure is %s\n",web.pressure_flag ? "on" : "off");
     printf("Incompressibility is %f\n",web.pscale);
   }
  else if (strcmp(aargv[1],"on") == 0 ) web.pressure_flag = 1;
  else if (strcmp(aargv[1],"off") == 0 ) web.pressure_flag = 0;
}

void a_incompress()
{
   web.pscale = atof(aargv[1]);
   energy_init = 0;
}

void a_volumes()
{
  double total_volume;
  body_id b_id;

  if ( web.torus_flag ) torvol();
   puts("Body      target volume       actual volume     pressure");
   b_id = NULLBODY;
   total_volume = 0.0;
   while ( generate_all(BODY,&b_id) )
     { printf("%3d     %15.8f     %15.8f   %10.5f\n",
         1+ordinal(b_id),get_body_fixvol(b_id),
         get_body_volume(b_id),get_body_pressure(b_id));
       total_volume += get_body_volume(b_id);
     }
   printf("Area/volume figure of merit: %f\n\n",
     pow(2*web.total_area,3.0)/web.bodycount/total_volume/total_volume);
}

void a_refine()
{
  refine();
  energy_init = 0;
  memory_report();
  web.min_area /= 4;
  web.max_len /= 2;
  web.min_length /= 2;
}
 
void a_weed()
{
  if ( aa_cmd.error ) return;
  web.min_area = atof(aargv[1]);
  printf("New facets from skinny triangles: %d\n",
       areaweed(web.min_area));
  energy_init = 0;
}

void a_long()
{
  if ( aa_cmd.error ) return;
  web.max_len = atof(aargv[1]);
  printf("New facets: %d\n",articulate(web.max_len));
}

void a_tiny()
{
  if ( aa_cmd.error ) return;
  web.min_length = atof(aargv[1]);
  printf("Deleted facets: %d\n",edgeweed(web.min_length));
}

void a_equi()
{
  printf("Edges switched in equiangulation: %d\n", equiangulate() );
  energy_init = 0;
}
 
void a_notch()
{
  if ( aa_cmd.error ) return;
  web.max_angle = atof(aargv[1]);
  printf("Number of edges notched: %d\n",
                    ridge_notcher(web.max_angle));
  energy_init = 0;
}
  
void a_motion()
{
  if ( aa_cmd.error ) return;
  if ( aa_cmd.vcnt < 1 )
      printf("Motion-by-mean-curvature is %s\n",web.motion_flag ? "on" : "off");
  else if ( strcmp(aargv[1],"on") == 0 ) web.motion_flag = 1;
  else if ( strcmp(aargv[1],"off") == 0 ) web.motion_flag = 0;
  energy_init = 0;
}

void a_jiggle()
{
  if ( aa_cmd.error ) return;
  if ( aa_cmd.vcnt < 1 ) printf("Jiggling is %s\n",
                                web.jiggle_flag ? "on" : "off");
  else if ( strcmp(aargv[1],"on") == 0 ) web.jiggle_flag = 1;
  else if ( strcmp(aargv[1],"off") == 0 ) web.jiggle_flag = 0;
  energy_init = 0;
}

void a_temperature()
{
  if ( aa_cmd.error ) return;
  if ( aa_cmd.vcnt < 1 ) printf("Temperature is %f\n",web.temperature);
  else web.temperature = atof(aargv[1]);
}

void a_view()
{
  if ( aa_cmd.error ) return;
  view_transform(aargv[1]);
}


 


struct aarg_cmd cmd[] = {

    AA_BEGIN
      a_load,"[-load %s]","load ASCII geometry file"
    AA_NEXT
      a_restore,"[-restore %s]","restore state from file"
    AA_NEXT
      a_save,"[-save %s]","save state in file"
    AA_NEXT
      a_dump,"[-dump %s]","dump state in a ASCII format"
    AA_NEXT
      old_menu,"[-old]","switch to old torusd control mode"
    AA_NEXT
      a_area,"[-area [%s]]","set area normalization flag on or off"
    AA_NEXT
      a_count,"[-c[ount]]","report counts of allocated structures"
#ifdef PIXAR
    AA_NEXT
      a_pixar,"[-pixar]","create data files for Pixar"
#endif
    AA_NEXT
      a_go,"[-g[o] [%d]]","do given number of iterations (default 1)"
    AA_NEXT
      a_scale,"[-scale [%f]]","set scaling factor for motion"
    AA_NEXT
      a_pressure,"[-pressure [%f]]","set incompressibility"
    AA_NEXT
      a_incompress,"[-incomp[ressibility] %s]","set incompressibility factor"
    AA_NEXT
      a_volumes,"[-vol[umes]]","report body volumes"
    AA_NEXT
      a_refine,"[-r[efine]]","refine triangulation"
    AA_NEXT
      a_weed,"[-weed %f]","subdivide long edges of small area triangles"
    AA_NEXT
      a_long,"[-long %f]","subdivide edges longer than given length"
    AA_NEXT
      a_tiny,"[-tiny %f]","remove edges shorter than given length"
    AA_NEXT
      a_equi,"[-equi[angulate]]","make triangulation more equiangular"
    AA_NEXT
      a_notch,"[-notch %f]","notch triangle joints greater than given radians"
    AA_NEXT
      a_motion,"[-motion %s]","set motion-by-mean-curvature mode on or off"
    AA_NEXT
      a_jiggle,"[-jiggle [%s]]","set jiggling on or off "
    AA_NEXT
      a_temperature,"[-temp[erature] [%f]]","set jiggling temperature"
    AA_NEXT
      extrapolate,"[-extrapolate]","extrapolate results of refinements"
    AA_NEXT
      a_view,"[-view %s]","viewing transformation string; view ? for help"
    AA_NEXT
      resize,"[-resize]","resize and recenter object in viewing window"
/*
    AA_NEXT
      make_movie,"[-movie]","make Cheyenne movie frame"
 */
    AA_NEXT
      display,"[-display]","display Cheyenne picture"
    AA_END,
    AA_STDCMD,
    AA_LASTCMD
  };

main(argc,argv)
int argc;
char **argv;
{ 
  signal(SIGINT,catcher);   /* to catch user interrupt */    
  view = dmatrix(0,3,0,3);
  init_view();
  scoeff_init();
  vcoeff_init();
  aa_prompt = "evolver> ";
  aa_SetBothModes();
  if ( setjmp(jumpbuf) )
    do_aargs(cmd,0,NULL);   /* return here after recoverable errors */
  else
    do_aargs(cmd,argc,argv);

  return 0;
}



/******************************************************************
*
*  Function: a_load()
*
*  Purpose:  load polyhedral complex from file
*/

void a_load()
{
  if ( aa_cmd.error ) return;

  initialize(aargv[1]);
  calc_energy(0.0);  /* to initialize volumes */
  resize();  /* to initialize view transform */
}




void a_save()
{
  if ( aa_cmd.error ) return;
  savefd = fopen(aargv[1],"wb");
  if ( savefd == NULL ) perror(aargv[1]);
  else do_save();
}


void a_restore()
{
  if ( aa_cmd.error ) return;
  savefd = fopen(aargv[1],"rb");
  if ( savefd == NULL ) perror(aargv[1]);
  else do_restore();
}


void a_dump()
{
  if ( aa_cmd.error ) return;
  do_dump(aargv[1]);
}
