/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**************************************************************
*
*  File: boundary.c
*
*  Purpose: Handle free boundary computations.
*/

#include "include.h"

/***************************************************************
*
*  Function: b_proj()
*
*  Purpose:  Construct projection matrices from R^n vectors
*            to parameter tangent space or to boundary tangent
*            space.
*
*  Mathematical background:
*     Let T[i] be the boundary tangent along parameter i, and
*  let V be the vector in R^n.  Let C be the linear combination
*  coefficients that give the projection of V:
*       (proj V)[j] = C[i]*T[i][j]      (sum on repeated indices)
*  The projection is defined by having identical dot products with
*  the tangents as does V:
*       C[i]*T[i][j]*T[k][j] = V[j]*T[k][j]  for all k.
*  Define the matrix A by A[i][k] = T[i][j]*T[k][j], and let B be
*  its inverse.  Note A and B are symmetric. So
*       A[k][i]*C[i] = T[k][j]*V[j]
*  and
*       C[i] = B[i][k]*T[k][j]*V[j].
*  So the coefficient projection matrix is D[i][j] = B[i][k]*T[k][j].
*  And
*      (proj V)[m] = C[i]*T[i][m] = T[i][m]*B[i][k]*T[k][j]*V[j]
*  so the tangent projection matrix is
*       E[m][j] = T[i][m]*B[i][k]*T[k][j].
*/

static REAL **B;  /* both A and B above */
static REAL **T;  /* T above */

void b_proj(bdry,param,a,type)
struct boundary *bdry;  /* boundary involved */
REAL *param;          /* parameter values to use */
REAL **a;             /* returned matrix */
int type;               /* PARAMPROJ or TANGPROJ */
{
  int pcount = bdry->pcount;
  int i,j,k,m;
  REAL dummy;  /* for eval_all function value */
  REAL temp[MAXCOORD];

  if ( B == NULL ) B = dmatrix(0,2,0,2);   /* permanent allocation */
  if ( T == NULL ) T = dmatrix(0,MAXCOORD,0,MAXCOORD);

  for ( j = 0 ; j < web.sdim ; j++ )
   {
    eval_all(bdry->coordf[j],param,pcount,&dummy,temp);
    for ( i = 0 ; i < pcount ; i++ )
      T[i][j] = temp[i];
   }
  for ( i = 0 ; i < pcount ; i++ )
    for ( j = 0 ; j < pcount ; j++ )
      B[i][j] = dot(T[i],T[j],web.sdim);
  mat_inv(B,pcount);

  if ( type == PARAMPROJ )
    {
      for ( i = 0 ; i < pcount ; i++ )
        for ( j = 0 ; j < web.sdim ; j++ )
          {
            a[i][j] = 0.0;
            for ( k = 0 ; k < pcount ; k++ )
              a[i][j] += B[i][k]*T[k][j];
          }
    }
  else if ( type == TANGPROJ )
    {
      for ( m = 0 ; m < web.sdim ; m++ )
        for ( j = 0 ; j < web.sdim ; j++ )
          {
            a[m][j] = 0.0;
            for ( i = 0 ; i < pcount ; i++ )
              for ( k = 0 ; k < pcount ; k++ )
                a[m][j] += T[i][m]*B[i][k]*T[k][j];
          }
    }
  else if ( type == GRADPROJ )
    {
      for ( i = 0 ; i < web.sdim ; i++ )
        for ( j = 0 ; j < pcount ; j++ )
          {
            a[i][j] = 0.0;
            for ( k = 0 ; k < pcount ; k++ )
              a[i][j] += T[k][i]*B[k][j];
          }
    }

  
}

/**********************************************************************
*
*  Function: b_extrapolate()
*
*  Purpose:  Find projection of point on boundary, starting at
*            given boundary location and projecting several
*            times on tangent.
*/

void b_extrapolate(bdry,base_x,point_x,new_x,base_param,new_param)
struct boundary *bdry;   /* boundary involved */
REAL *base_x;     /* coordinates of base point */
REAL *point_x;    /* coordinates of point to project */
REAL *new_x;      /* projected coordinates */
REAL *base_param; /* base point parameters */
REAL *new_param;  /* projected parameters  */
{
  int pcount;
  REAL **a;
  REAL co[MAXCOORD];
  int i,k;
  REAL x[MAXCOORD],s[MAXCOORD];

  pcount = bdry->pcount;
  a = dmatrix(0,pcount-1,0,2);
  for ( k = 0 ; k < web.sdim ; k++ )
    { x[k] = base_x[k];                     /* start at base point */
      new_param[k] = base_param[k];
    }

  /* iterate projection on tangent till get close to desired point */
  for ( i = 0 ; i < 5 ; i++ )
    {
      for ( k = 0 ; k < web.sdim ; k++ )
        s[k] = point_x[k] - x[k];
      b_proj(bdry,new_param,a,PARAMPROJ);
      matvec_mul(a,s,co,pcount,web.sdim);
      for ( k = 0 ; k < pcount ; k++ )
        new_param[k] += co[k];
      for ( k = 0 ; k < web.sdim ; k++ )
         x[k] = eval(bdry->coordf[k],new_param);
    }

  for ( i = 0 ; i < web.sdim ; i++ ) new_x[i] = x[i];
  free_matrix(a);

}

/****************************************************************
*
*  Function: bdry_force()
*
*  Purpose:  Since only vertices are actually confined to boundaries,
*            edges and faces supposedly on boundaries can pull
*            away from convex boundaries, and in fact do, since
*            a long edge short-cuts the boundary.  To prevent
*            this and encourage equal-length boundary edges, an
*            energy penalty is inflicted for an edge angling away
*            from its boundary.  If S is the edge vector and P is the
*            projection matrix on the boundary tangent at the tail
*            of the edge, then the energy is the  area of the right
*            triangle formed by S and P*S,
*                 E = |S x P*S|/2
*            and the force on the head is perpendicular to the base
*            and of magnitude proportional to the length of the base,
*                 F = -|P*S|*(S - P*S)/|S - P*S|.
*            Recall force will later be projected to tangent.
*/

void bdry_force(e_id)
edge_id e_id;
{
  REAL s[MAXCOORD],q[MAXCOORD],u[MAXCOORD],**a,*f;
  struct boundary *bdry;
  vertex_id head,tail;
  REAL b,norm;
  int i; 

  if ( get_eattr(e_id) & FIXED ) return;

  /* see if edge and tail are on same boundary */
  bdry = get_edge_boundary(e_id);
  if ( !bdry ) return;
  if ( !(bdry->attr & B_CONVEX) ) return;
  tail = get_edge_tailv(e_id);
  if ( bdry != get_boundary(tail) )
    { 
      invert(e_id);
      tail = get_edge_tailv(e_id);
      if ( bdry != get_boundary(tail) ) return;
    }
  head = get_edge_headv(e_id);

  /* now the calculation */
  a = dmatrix(0,2,0,2);
  get_edge_side(e_id,s);
  b_proj(bdry,get_param(tail),a,TANGPROJ);
  matvec_mul(a,s,q,web.sdim,web.sdim);
  for ( i = 0 ; i < web.sdim ; i++ )
    u[i] = s[i] - q[i];   /* opposite side */
  norm = -0.5*sqrt(dot(q,q,web.sdim)/dot(u,u,web.sdim));
  f = get_force(head);
  for ( i = 0 ; i < web.sdim ; i++ )
    f[i] += web.spring_constant*norm*u[i];

  /* assume mirror force on tail */
  b = 2*dot(u,s,web.sdim)/dot(s,s,web.sdim);
  f = get_force(tail);
  for ( i = 0 ; i < web.sdim ; i++ ) 
    f[i] += web.spring_constant*norm*(u[i] - b*s[i]);

  free_matrix(a);
}

/*****************************************************************
*
*  Function: bdry_spring_energy()
*
*  Purpose:  Calculate energy of kludge boundary force.
*/

void bdry_spring_energy(e_id)
edge_id e_id;
{
  REAL s[MAXCOORD],q[MAXCOORD],u[MAXCOORD],**a;
  struct boundary *bdry;
  vertex_id tail;

  if ( get_eattr(e_id) & FIXED ) return;

  /* see if edge and tail are on same boundary */
  bdry = get_edge_boundary(e_id);
  if ( !bdry ) return;
  if ( !(bdry->attr & B_CONVEX) ) return;
  tail = get_edge_tailv(e_id);
  if ( bdry != get_boundary(tail) )
    { 
      invert(e_id);
      tail = get_edge_tailv(e_id);
      if ( bdry != get_boundary(tail) ) return;
    }

  /* now the calculation */
  a = dmatrix(0,2,0,2);
  get_edge_side(e_id,s);
  b_proj(bdry,get_param(tail),a,TANGPROJ);
  matvec_mul(a,s,q,web.sdim,web.sdim);
  cross_prod(s,q,u);
  web.total_energy += 0.5*web.spring_constant*sqrt(dot(u,u,web.sdim));
  free_matrix(a);
}

/*****************************************************************
*
*  Function: calc_bdry_force_v()
*
*  Purpose: calculate force on vertex due to boundary energy. (string model)
*           Note: should do result in ambient space.
*/

void calc_bdry_force_v(v_id)
vertex_id v_id;
{
  return;
}

/*****************************************************************
*
*  Function: calc_bdry_force_e()
*
*  Purpose: calculate force on endpoints of edge due to boundary energy.
*           Note: should do result in ambient space.
*/

void calc_bdry_force_e(e_id)
edge_id e_id;
{
}

/*****************************************************************
*
*  Function: calc_bdry_energy_v()
*
*  Purpose: calculate boundary energy due to vertex. (string model)
*/

void calc_bdry_energy_v(v_id)
vertex_id v_id;
{
  REAL e;
  struct boundary *bdry = get_boundary(v_id);
 
  if ( bdry->pcount != 1 ) return;

  e = eval(bdry->envect[0],get_param(v_id));  
  if ( get_vattr(v_id) & NEGBOUNDARY )
    web.total_energy -= e;
  else
    web.total_energy += e;

}

/*****************************************************************
*
*  Function: calc_bdry_energy_e()
*
*  Purpose: calculate energy on boundary due to edge.
*           Stub.
*/

void calc_bdry_energy_e(e_id)
edge_id e_id;
{
  return;
}

/*****************************************************************
*
*  Function: calc_bdry_content_v()
*
*  Purpose: calculate interior content due to vertex. (string model)
*/

void calc_bdry_content_v(v_id)
vertex_id v_id;
{
  REAL e;
  struct boundary *bdry = get_boundary(v_id);
  body_id b_id;
  facetedge_id fe_id;
  facet_id f_id;
 
  if ( bdry->pcount != 1 ) return;

  e = eval(bdry->convect[0],get_param(v_id));  
  if ( get_vattr(v_id) & NEGBOUNDARY )
    e = -e;

  fe_id = get_vertex_fe(v_id);
  if ( !valid_id(fe_id) ) return;

  /* cell on plus side of edge */
  f_id = get_fe_facet(fe_id);
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)+e);
    }

  /* cell on other side of edge */
  f_id = get_fe_facet(inverse_id(fe_id));
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)-e);
    }

}

/*****************************************************************
*
*  Function: calc_bdry_content_e()
*
*  Purpose: calculate content on constraint due to edge.
*           Stub.
*/

void calc_bdry_content_e(e_id)
edge_id e_id;
{
}


/*****************************************************************
*
*  Function: bdry_basis()
*
*  Purpose: calculate basis of boundary tangent plane.
*/

int bdry_basis(v_id,basis)
vertex_id v_id;
REAL **basis;  /* for return */
{
  struct boundary *b = get_boundary(v_id);
  int j,i;
  REAL dummy;  /* for eval_all function value */
  REAL temp[MAXCOORD];

  if ( !b ) return 0;

  for ( j = 0 ; j < web.sdim ; j++ )
   {
    eval_all(b->coordf[j],get_param(v_id),b->pcount,&dummy,temp);
    for ( i = 0 ; i < b->pcount ; i++ )
      basis[i][j] = temp[i];
   }

  return b->pcount;
}

