/******************************************************************************
*
*  The complex math library was originally written in Modula2 by Jesse Jones  *
*                          jesjones@u.washington.edu                 
*                                                  
******************************************************************************/

#include <math.h>

typedef struct { double r,i;} complex; 

complex Add (a,b)
complex a,b;

{
   complex c;
   c.r = a.r + b.r;
   c.i = a.i + b.i;
   return(c);
}

complex Sub (a,b)
complex a,b;  

 { complex c;
   c.r = a.r - b.r;
   c.i = a.i - b.i;
   return(c);
 }

complex Mult (a,b)
complex a,b;  

{  complex c;
   c.r = a.r*b.r - a.i*b.i;
   c.i = a.r*b.i + a.i*b.r;
   return(c);
}

complex Divide (a,b)
complex a,b;  

   { 
      double ratio;
      double denom;
      complex result;
   if (fabs(b.r) >= fabs(b.i))     /* Don't square to avoid over or underflow */
  {   ratio = b.i/b.r; 
      denom = b.r + ratio*b.i; 
      result.r = (a.r + a.i*ratio)/denom; 
      result.i = (a.i - a.r*ratio)/denom; 
  }
   else
   { 
      ratio = b.r/b.i; 
      denom = b.i + ratio*b.r; 
      result.r = (a.r*ratio + a.i)/denom; 
      result.i = (a.i*ratio - a.r)/denom;
   }
   return(result);
 }

double Mag (a)
complex a;

  {
   double x;
   if (a.i == 0.0)
      x = fabs(a.r);
   else
   {
      a.r = a.r * a.r;                  /* Square  */
      a.i = a.i * a.i;
      x = sqrt(a.r + a.i);            /* and Add */
   }
   return(x);
}

complex Conjugate (a)
complex a;   

{
   a.i = -a.i;
   return(a);
}


double GetTheta (a)
complex a;   

{
   double theta;
   if (a.i == 0.0)
      if (a.r >= 0.0) 
	 theta = 0.0;            /* positive real */
      else
	 theta = M_PI;            /* negative real */
   else  
     if (a.r == 0.0)
      if (a.i >= 0.0) 
	 theta = M_PI/2.0;         /* positive imag */
      else
	 theta = -M_PI/2.0;         /* negative imag */
      
   else
      theta = atan(a.i/a.r);
      if (a.r < 0.0)             /* make sure theta is in the correct quadrant */
	 if (a.i > 0.0)
	    theta = M_PI + theta;
	 else
	    theta = theta - M_PI;
	 
   return(theta);
}

complex Sqr (z)
complex z;   

 { complex x;
   x.r = z.r*z.r - z.i*z.i;
   x.i = 2*z.r*z.i;
   return(x);
 }

/* sqrt(z) = sqrt(r)*[cos(theta/2) + i*sin(theta/2)] */
complex Sqrt (z,branch)
complex z;
int branch;

{  complex x;
   double r,t;
   if (z.i == 0.0 && z.r > 0.0) 
      { if (branch == 0) 
	 x.r = sqrt(z.r);
      else
	 x.r = -sqrt(z.r);
      x.i = 0.0;
      }
   else
   {
      r = Mag(z);
      x.r = (r + z.r)/2.0;
      x.i = (r - z.r)/2.0;
      x.r = sqrt(x.r);
      if(z.i>=0) x.i= sqrt(x.i);
       else x.i= -sqrt(x.i);
      
   }
   return(x);
}

/* log(z) = log(z) = log|z| + i(theta + 2*M_PI*n)    where n is any integer
   Ln uses n = 0. */
complex Ln (z,branch)
complex z;
int branch;

{  complex x;
   if (z.i == 0.0 && z.r >= 0.0) 
     { x.r = log(z.r);
       x.i = 0.0;
     }
   else
     { x.r = log(Mag(z));
       x.i = GetTheta(z);
       x.i = x.i + 2*M_PI*branch;
     }
   return(x);
}

complex Log (z,branch)
complex z;
int branch;

{  complex x;
   x = Ln(z, branch);
   x.r = x.r/M_LN10;
   x.i = x.i/M_LN10;
   return(x);
}

/* exp(z) = e^x*[cos(y) + i*sin(y)] */
complex Exp (z)
complex z;
   
 { 
   complex x;
   double a;
   if (z.i == 0.0) 
    {  x.r = exp(z.r);
       x.i = 0.0;
    }
   else
    { a = exp(z.r);
      x.r = a*cos(z.i);
      x.i = a*sin(z.i);
    }
   return(x);
}

/* z^n = r^n*[cos(n*theta) + i*sin(n*theta)] */
complex RaiseInt (z,n)
complex z;
int n;
  
 {
   complex x;
   double r,t;
   if (z.i == 0.0) 
      if (z.r == 0.0) 
	 if (n == 0) 
	    { x.r = 0;   /*WARNING: 0^0 should be undefined */
	      x.i = 0;
	    }
	 else
	    { x.r = 0.0;
	      x.i = 0.0;
	    }
      else if (z.r == 1.0)
	{ x.r = 1.0;
	  x.i = 0.0;
	}
	   else
	    { x.r = pow(z.r,(double)(n));
	     x.i = 0.0;
	    }
    
   else if (n == 0)
      { x.r = 1.0;
       x.i = 0.0; 
       }

	else if (n == 1) 
	     x = z;
	      else
		{  r = Mag(z);
		   t = GetTheta(z);
		   r = pow(r,(double)(n));
		   t = n*t;
		   x.r = r*cos(t);
		   x.i = r*sin(t);
		}
   return(x); 
 }


/* 
   z^c = exp(c*log(z))
   NOTE:
      Because log(z) is a many valued function z^c does not, in general,
      have a unique value. In particular if we let j and k be any integer
      then if:
	 c = k 
	    there is one power
	 c = 1/k 
	    there are k distinct powers
	 c = j/k, with j and k having no common factors
	    there are k distinct powers
	 if c is not an integer or a rational number
	    there are an infinite number of powers
      This can result in some unexpected answers with roots of negative
      reals. For example -8^(1/3) will come out as (1+1.732i) the second
      value (n = 1) yields the expected result: -2.
*/
complex Raise (z,c,branch)
complex z,c;
int branch;
   
 {
   double r,t;
   complex x,y;
   if ((z.r == 0.0) && (z.i == 0.0)) 
      if ((c.i == 0.0) && (c.i == 0.0)) 
	{ x.r = 0;       /*WARNING: 0^0 should be undefined */
	  x.i = 0;
	}
      else
	{ x.r = 0.0;
	  x.i = 0.0;
	 } 
      
   else 
      if ((c.r == 0.0) && (c.i == 0.0)) 
	  { x.r = 1.0;
	    x.i = 0.0; }
       else  if ((c.r == 1.0) && (c.i == 0.0)) 
		 x = z;
	     else  if (c.i == 0.0) 
		       if (((int)(c.r) - c.r) == 0) 
		       x = RaiseInt(z, (int)(c.r));
		       else
			  {  r = Mag(z);
			     t = c.r*GetTheta(z);
			     r = pow(r, c.r);
			     x.r = r*cos(t);
			     x.i = r*sin(t);
			  }
		  else
		     {  y = Ln(z, branch);
			y = Mult(c, y);
			x = Exp(y);
		    }
   return(x);
}




	
	
complex Sin (z)
complex z; 
   
{
   complex x;
   if (z.i == 0.0)
      { x.r = sin(z.r);
	x.i = 0.0;
      }
   else
      { x.r = sin(z.r)*cosh(z.i);
	x.i = cos(z.r)*sinh(z.i);
      }
   return(x);
}

complex Cos (z)
complex z; 
   
{  complex x;
   if (z.i == 0.0)
     { x.r = cos(z.r);
       x.i = 0.0;
     }
   else
     {  x.r = cos(z.r)*cosh(z.i);
	x.i = -sin(z.r)*sinh(z.i);
     }
   return(x);
}

complex Tan (z)
complex z; 
   
{  complex x,y;
   if (z.i == 0.0)
     { x.r = tan(z.r);
       x.i = 0.0;
     }
   else
     { x = Sin(z);
       y = Cos(z);
       x = Divide(x, y);
     }
   
   return(x);
}


/* arcSin(z) = -i*log[i*z + sqrt(1 - z*z)] */
complex ArcSin (z)
complex z; 
  
 {  
   complex x,y;
   if (z.i == 0.0) 
     { y.r =asin(z.r);
       y.i = 0.0;
     }
   else
    { x.r = -z.i;
      x.i = z.r;
      y = Sqr(z);
      y.r = 1.0 - y.r;
      y = Sqrt(y, 0);
      x = Add(x, y);
      x = Ln(x, 0);
      y.r = x.i;
      y.i = -x.r;
    }
   return(y);
}

/* arcCos(z) = -i*log[z + i*sqrt(1 - z*z)] */
complex ArcCos (z)
complex z; 

{  
   complex x,y;
   if (z.i == 0.0) 
     { x.r = acos(z.r);
       x.i = 0.0;
     }
   else
    { y = Sqr(z);
      y.r = 1.0 - y.r;
      y = Sqrt(y, 0);
      x.r = -y.i;
      x.i = y.r;
      y = Add(z, x);
      y = Ln(y, 0);
      x.r = y.i;
      x.i = -y.r;
    }
   
   return(x);
 }


/* arcTan(z) = i*log[(i + z)/(i - z)]/2 */
complex ArcTan (z)
complex z; 

 { complex x,y;
   if (z.i == 0.0)
    {  y.r = atan(z.r);
       y.i = 0.0;
    }
   else
   {  x.r = z.r;
      x.i = 1.0 + z.i;
      y.r = -z.r;
      y.i = 1.0 - z.i;
      x = Divide(x, y);
      x = Ln(x, 0);
      y.r = x.i/2.0;    
      y.i = x.r/2.0;
    }
   return(y);
 }


complex Sinh (z)  
complex z; 
 
{  complex x;
   if (z.i == 0.0) 
   { x.r = sinh(z.r);
     x.i = 0.0;
   }
   else
    {  x.r = sinh(z.r)*cos(z.i);
       x.i = cosh(z.r)*sin(z.i);
    }
   return(x);
}

complex Cosh (z)
complex z; 
   
 {
   complex x;
   if (z.i == 0.0)
     {  x.r = cosh(z.r);
	x.i = 0.0;
     }
   else
    {  x.r = cosh(z.r)*cos(z.i);
       x.i = sinh(z.r)*sin(z.i);
    }
   return(x);
}

complex Tanh (z)
complex z; 
 
 { 
   complex x,y;
   if (z.i == 0.0) 
     { x.r = tanh(z.r);
       x.i = 0.0;
      }
   else
     { x = Sinh(z);
       y = Cosh(z);
       x = Divide(x, y);
     }
   return(x);
  }


/* arcSinh(z) = log[z + sqrt(z*z + 1)] */
complex ArcSinh (z) 
complex z; 
 
 {
   complex x;
   if (z.i == 0.0)
    {  x.r = asinh(z.r);
       x.i = 0.0;
    }
   else
     { x = Sqr(z);
       x.r = 1.0 + x.r;
       x = Sqrt(x, 0);
       x = Add(x, z);
       x = Ln(x, 0);
     }
   return(x);
 }

/* arcCosh(z) = log[z + sqrt(z*z - 1)] */
complex ArcCosh (z)
complex z; 

 {
   complex x;
   if (z.i == 0.0) 
    {  x.r = acosh(z.r);
       x.i = 0.0;
     }
   else
     { x = Sqr(z);
       x.r = x.r - 1.0;
       x = Sqrt(x, 0);
       x = Add(z, x);
       x = Ln(x, 0);
     }
   
   return(x);
}

/* arcTanh(z) = log[(1 + z)/(1 - z)]/2 */
complex ArcTanh (z)
complex z; 

{ 
   complex x,y;
   if (z.i == 0.0)
     { x.r = atanh(z.r);
      x.i = 0.0;
      }
   else
    { x = z;
      x.r = 1.0 + x.r;
      y = z;
      y.r = 1.0 - y.r;
      x = Divide(x, y);
      x = Ln(x, 0);
      x.r = x.r/2.0;
      x.i = x.i/2.0;
    }
   return(x);
 }



