/*
 *  draw a graph
 */

#include "Htdag3_htdag3.h"

static int Moves_per_tick = 1;

static char*	HTDAG_HelpMsg = "htdag3 filename objname";
static char*	HTSET_HelpMsg = "htset objname {rand f|color f}";

initialize_htdag3()
{
	register_client("htdag3", htdag3, HTDAG_HelpMsg);
	register_client("htset", htset, HTSET_HelpMsg);
}

	
htset(argc,argv)
int argc;
char **argv;
{
	if (argc > 1) {
		if (!strcmp(argv[1],"rand")) {
			if (argc > 2) srand48(atol(argv[2]));
			else srand48(getpid());
		}
		else if (!strcmp(argv[1],"rate")) {
			if (argc > 2) Moves_per_tick = MAX(1,atoi(argv[2]));
			else Moves_per_tick = 1;
		}
		else fprintf(stderr,"%s\n",HTSET_HelpMsg);
	}
}

htdag3(argc,argv)
int		argc;
char	**argv;
{
	graph_t		*g;

	if ((argc < 2) || (argc > 3)) fprintf(stderr,"%s\n",HTDAG_HelpMsg);
	else {
		if (g = read_graph(argc,argv)) {
			init_graph(g);
			init_model(g);
			printf("%s: %d nodes %d edges\n",g->obj->name,g->N_nodes,g->N_edges);
		}
	}
}


double fpow32(x)
double x;
{
	x = sqrt(x);
	return x * x * x;
}

static int hash(s)
char	*s;
{
	int		index = 0;
	while (*s) index = ( index << 1 ) ^ *s++;
	return	index % HASHSIZE;
}

/* read_graph:
 *  a new graph object with its tick is created by reading file.
 *  node names are scanf strings.
 *  a node name and three floats (coordinates) is a node.
 *  two node names and an optional float (strength) is an edge.
 */
graph_t	*read_graph(argc, argv)
int		argc;
char	**argv;
{
	int			i,nval;
	node_t		*vi,*vj;
	edge_t		*e;
	Coord		pt[10];
	char		line[128], node0[128], node1[128];
	char		filename[MAXLINE];
	float		length;
	FILE		*fp;
	graph_t		*g = NULL;

	if ((fp = fopen(argv[1],"r")) == NULL)	{
		sprintf(filename,"%s/%s",GRAPHDIR,argv[1]);
		if ((fp = fopen(filename,"r")) == NULL)	{
			fprintf(stderr,"htdag3: can't open %s\n",argv[1]);
			return(g);
			}
		}

	g = new_graph(argc,argv);
	while (fgets(line,BUFSIZ,fp)) {
		length = 1.0;
		if ((nval = sscanf(line,"%s %f %f %f %f %f %f %f %f %f ",node0,
			&pt[0], &pt[1],&pt[2],&pt[3],&pt[4],
			&pt[5], &pt[6],&pt[7],&pt[8],&pt[9])) > 3) {
				vi = bind_node(g,node0,argc,argv);
				if (nval > NDIM) nval = NDIM; 
				for (i = 0; i < nval; i++) vi->pos.x[i] = pt[i];
		}
		else if (sscanf(line,"%s %s %f",node0,node1,&length) >= 2) {
			vi = bind_node(g,node0,argc,argv);
			vj = bind_node(g,node1,argc,argv);
			if (vi == vj) continue;
			e = bind_edge(g,vi,vj,argc,argv);
			e->length = length;
			}
		else { fprintf(stderr,"bad input line: %s\n",line); }
		}

	return g;
}

void init_arrays(g)
graph_t	*g;
{
	int		i,j,k;
	node_t	*vi,*vj;
	double	del[NDIM],dist;
	float	**K = g->K;
	float	**sum_t = g->sum_t;
	float	***t = g->t;
	float	**L = g->L;

	for (i = 0; i < g->N_nodes; i++) 
		for (k = 0 ; k < NDIM ; k++ )
			sum_t[i][k] = 0.0 ;
	for (i = 0; i < g->N_nodes; i++) {
		vi = g->nodelist[i];
		for (j = 0; j < g->N_nodes; j++) {
			if (i == j) continue;
			vj = g->nodelist[j];
			dist = 0.0;
			for (k = 0 ; k < NDIM ; k++ )	{
				del[k] = vi->pos.x[k] - vj->pos.x[k]; 
				dist += SQR(del[k]);
			}
			dist = sqrt(dist);
			for (k = 0 ; k < NDIM ; k++ )	{
				t[i][j][k] = K[i][j]*(del[k] - L[i][j]*del[k]/dist);
				sum_t[i][k] += t[i][j][k];
			}
		}
	}
}

void update_arrays(g,i)
graph_t	*g;
int		i;
{
	int		j,k;
	double	d[NDIM],dist,old;
	node_t	*vi,*vj;
	float	**sum_t = g->sum_t;
	float	***t = g->t;
	float	**L = g->L;
	float	**K = g->K;

	vi = g->nodelist[i];
	for (k = 0 ; k < NDIM ; k++ ) sum_t[i][k] = 0.0;
	for (j = 0; j < g->N_nodes; j++) {
		if (i == j) continue;
		vj = g->nodelist[j];
		dist = 0.0;
		for (k = 0 ; k < NDIM ; k++ )	{
			d[k] = vi->pos.x[k] - vj->pos.x[k]; 
			dist += SQR(d[k]);
		}
		dist = sqrt(dist);
		for (k = 0 ; k < NDIM ; k++ )	{
			old = t[i][j][k];
			t[i][j][k] = K[i][j]*(d[k] - L[i][j]*d[k]/dist);
			sum_t[i][k] += t[i][j][k];
			old = t[j][i][k];
			t[j][i][k] = -t[i][j][k];
			sum_t[j][k] += (t[j][i][k] - old);
		}
	}
}

void D2E(g, n, M)
graph_t	*g;
int		n;
double	M[NDIM][NDIM];
{
	int		i,l,k;
	node_t	*vi,*vn;
	double	scale,sq, t[NDIM];
	float	**K = g->K;
	float	**L = g->L;

	vn = g->nodelist[n];
	for (l = 0 ; l < NDIM ; l++ )
		for (k = 0 ; k < NDIM ; k++)
			M[l][k] = 0.0;
	for (i = 0; i < g->N_nodes; i++) {
		if (n == i) continue;
		vi = g->nodelist[i];
		sq = 0.0;
		for (k = 0 ; k < NDIM ; k++ ) {
			t[k] = vn->pos.x[k] - vi->pos.x[k];
			sq += SQR(t[k]);
		}
		scale = 1 / fpow32(sq);
		for (k = 0 ; k < NDIM ; k++ )	{
			for (l = 0 ; l < k ; l++ )
				M[l][k] += K[n][i] * L[n][i]*t[k]*t[l]*scale;
			M[k][k] += K[n][i] *(1. - L[n][i]*(sq-SQR(t[k]))*scale);
		}
	}
	for (k = 1 ; k < NDIM ; k++ )
		for (l = 0 ; l < k ; l++ )
			M[k][l] = M[l][k];
}

node_t *choose_node(g)
graph_t	*g;
{
	int			i,k;
	double		m,max;
	node_t		*choice;

	max = 0.0;
	choice = NULL;
	for (i = 0; i < g->N_nodes; i++) {
		m = 0.0;
		for (k = 0 ; k < NDIM ; k++ )
			m += SQR(g->sum_t[i][k]);
		m = sqrt(m);
		if (m > max) {choice = g->nodelist[i]; max = m;}
		g->nodelist[i]->energy = m;
	}
	if (max < EPSILON) choice = NULL;
	return choice;
}

void move_node (g,n)
graph_t	*g;
node_t	*n;
{
	int		i,m;
	double	a[NDIM][NDIM],b[NDIM],c[NDIM];
	edge_t	*e;

	m = n->id;
	D2E(g,m,a);
	for (i = 0 ; i < NDIM ; i++) c[i] = - g->sum_t[m][i];
	solve(a,b,c,NDIM);
	for (i = 0 ; i < NDIM ; i++ ) n->pos.x[i] += b[i]; 
	g->Move++;
	update_node_geom(g,n);
	for (e = n->edge_list; e; e = (e->n0 == n? e->next0 : e->next1))
		update_edge_geom(g,e);
	update_arrays(g,m);
}

htdag3_tick(obj) 
object_t *obj;
{
	int		cnt;
	node_t	*node;
	graph_t	*g = (graph_t*) obj->user;;

	for (cnt = 0; cnt < Moves_per_tick; cnt++) {
		node = choose_node(g);
		if (node == NULL) {
			obj->tick = NULL;
			break;
		}
		else move_node(g,node);
	}
}

void init_springs(g)
graph_t	*g;
{
	int i,j,l=1;
	int n_found, otherid;
	node_t	*n;
	edge_t	*e;

	/* init shortest path matrix */
	for (i = 0; i < g->N_nodes; i++)
		for (j = 0; j < g->N_nodes; j++)
			if (g->Adj[i][j])
				g->L[i][j] = 10;
			else
				g->L[i][j] = 20*g->N_nodes;

	/* find shortest paths (which is spring length modulo scaling) */
	do {
		n_found = 0;
		for (i = 0; i < g->N_nodes; i++)
			for (j = 0; j < g->N_nodes; j++)
				if (g->L[i][j] == l*10) {
					n = g->nodelist[j];
					for (e = n->edge_list; e; e = (e->n0 == n? e->next0 : e->next1)) {
						if (e->n0 == n) otherid = e->n1->id;
						else otherid = e->n0->id;
						if (g->L[i][otherid] > l*10) {
							n_found++;
							g->L[i][otherid] = l*10+10;}
						}
				}
		l++;
	} while (n_found > 0);

	/* compute spring strength */
	for (i = 0; i < g->N_nodes - 1; i++) {
		for (j = i + 1; j < g->N_nodes; j++) {
			g->K[i][j] = g->K[j][i] = K_coeff/SQR(g->L[i][j]);
/*
			if (g->Adj[i][j]->length > 1.0)
				g->L[i][j] = g->L[j][i] = g->Adj[i][j]->length;
*/
		}
	}
}

init_model(g)
graph_t	*g;
{
	init_springs(g);
	init_arrays(g);
}

graph_t *new_graph(argc,argv)
int		argc;
char	**argv;
 {
	int		i;

	graph_t	*g = NEW(graph_t);
	g->obj = new_object(argv[argc-1],argc,argv);
	g->N_nodes = g->N_edges = 0;
	g->nodelist = N_NEW(1000,node_t*);
	for (i = 0; i < HASHSIZE; i++) g->hashtable[i] = NULL;
	g->node_color_scale = 1.0;
	g->Move = 0;
	g->obj->tick = htdag3_tick;
	g->obj->user = (char*) g;
	return g;
}

init_graph(g)
graph_t	*g;
{
	int		 	i,j,k;
	node_t		*n;
	edge_t		*e;

	g->Adj = N_NEW(g->N_nodes, edge_t**);
	g->K = N_NEW(g->N_nodes, float*);
	g->L = N_NEW(g->N_nodes, float*);
	g->t = N_NEW(g->N_nodes, float**);
	g->sum_t = N_NEW(g->N_nodes, float*);
	for (i = 0; i < g->N_nodes; i++) {
		g->Adj[i] = N_NEW(g->N_nodes, edge_t*);
		g->K[i] = N_NEW(g->N_nodes, float);
		g->L[i] = N_NEW(g->N_nodes, float);
		g->t[i] = N_NEW(g->N_nodes, float*);
		g->sum_t[i] = N_NEW(NDIM, float);
		for (j = 0; j < g->N_nodes; j++) {
			g->t[i][j] = N_NEW(NDIM, float);
			g->Adj[i][j] = NULL;
		}
	}
	for (i = 0; i < g->N_nodes; i++) {
		for (k = 0; k < NDIM; k++)
			if (g->nodelist[i]->pos.x[k] != 0.0) break;
		/*if (k >= NDIM)*/
			for (k = 0 ; k < NDIM ; k++ )
				g->nodelist[i]->pos.x[k] =  g->N_nodes * (drand48() - .5);
	}

	for (i = 0; i < g->N_nodes; i++) {
		n = g->nodelist[i];
		update_node_geom(g,n);
		for (e = n->edge_list; e; e = (e->n0 == n? e->next0 : e->next1)) {
			g->Adj[e->n0->id][e->n1->id] = g->Adj[e->n1->id][e->n0->id] = e;
			if (e->n0 == n) update_edge_geom(g,e);
		}
	}
}

node_t *bind_node(g,name,argc,argv)
graph_t		*g;
char		*name;
int			argc;
char		**argv;
{
	node_t			*np;

	for (np = g->hashtable[hash(name)]; np; np = np->hash_link)
		if (strcmp(name,np->obj->name) == 0) break;
	if (np == NULL) np = new_node(g,name,argc,argv);
	return np;
}

node_t *new_node(g,name,argc,argv)
graph_t		*g;
char		*name;
int			argc;
char		**argv;
{       
	int			index;
	node_t		*node;

	index = hash(name);
	node = NEW(node_t);
	node->id = g->N_nodes++;
	node->pos = zeropt;
	node->degree = 0;
	node->obj = new_node_obj(g,name,argc,argv);
	node->hash_link = g->hashtable[index];
	g->hashtable[index] = node;
	node->edge_list = NULL;
	node->energy = 0.0;
	g->nodelist[node->id] = node;
	return node;
}

object_t *new_node_obj(g,name,argc, argv)
graph_t		*g;
char		*name;
int			argc;
char		**argv;
{
	return new_subobject(name,g->obj,argc,argv);
}

update_node_geom (gr, node)
graph_t	*gr;
node_t	*node;
{
	int			pi,vi;
	polygon_t	*pp;
	object_t	*obj;
	vertex_t	*vl;
        float           h,r,g,b;

	obj = node->obj;
	if (obj == NULL) return;
	for (pi = 0; pi < obj->n_polygons; pi++) {
		pp = &(obj->polygons[pi]);
		for (vi = 0; vi < pp->n_vertices; vi += 2) {
			h = node->energy/(gr->node_color_scale) ;
			if (h > 1.0 ) h = 0.0 ;
			else h = 0.8 * (1.0 - h); 
			vl = pp->v[vi];
                        hsv2rgb(&r, &g, &b, h, 0.9, 0.9);
                        vl->fastcolor = color_triple(r,g,b);
                        vl->colored = TRUE;
		}
	}
       
	emu_copy_matrix(obj->v,Identity);
	emu_translate(obj->v,node->pos.x[0], node->pos.x[1], node->pos.x[2]);
}

edge_t *bind_edge(g,vi,vj,argc,argv)
graph_t		*g;
node_t		*vi,*vj;
int			argc;
char		**argv;
{
	edge_t	*e;
	for (e = vi->edge_list; e; e = (e->n0 == vi? e->next0 : e->next1))
		if (((e->n0 == vi) && (e->n1 == vj))||((e->n0 == vj)&&(e->n1 == vi)))
			break;
	if (e == NULL) e = new_edge(g,vi,vj,argc,argv);
	return e;
}

edge_t *new_edge(g,vi,vj,argc,argv)
graph_t		*g;
node_t		*vi,*vj;
int			argc;
char		**argv;
{       
	edge_t	*e;

	e = NEW(edge_t);
	g->N_edges++;
	e->n0 = vi;
	e->n1 = vj;
/* someday do this from graph coordinates automagicahly !!!! */
	e->obj = new_edge_obj(g, 5, .5, argc, argv);
	e->length = 1.0;
	e->next0 = vi->edge_list; vi->edge_list = e;
	e->next1 = vj->edge_list; vj->edge_list = e;
	return e;
}

/*  new_edge_obj:
 *  create unit-length edge lying along X axis.
 *  update_geom depends on this property.
 */
object_t *new_edge_obj(g,order,radius, argc, argv)
graph_t		*g;
int			order;
double		radius;
int			argc;
char		**argv;
{
	int		i,j;
	double	ly = radius, lz = 0.0;
	double	a;
	polygon_t	*polygons;
	vertex_t 	*v_list;
	object_t	*obj;
	static int	ctr;
	char		buf[20];

	sprintf(buf,"edge_%d",ctr++);
	obj = new_subobject(buf,g->obj,argc,argv);
	obj->v_list	= (vertex_t*)calloc((unsigned)order * 4,(unsigned)sizeof(vertex_t));
	obj->n_vertices	= 4 * order;
	obj->polygons	= (polygon_t*) calloc((unsigned)order,(unsigned)sizeof(polygon_t));
	obj->n_polygons	= order;
	polygons = obj->polygons;
	v_list = obj->v_list;

	for (i = 0; i < order; i++) {
		polygons[i].n_vertices = 4;
		polygons[i].v = (vertex_t**)calloc((unsigned)4,(unsigned)sizeof(vertex_t*));
		polygons[i].fastcolor = random_face_color();
		polygons[i].colored = TRUE;
		for (j = 0; j < 4; j++) {
			polygons[i].v[j] = (vertex_t*) &(v_list[4 * i + j]);
			polygons[i].v[j]->order++;
			}
		a = (PI / 180.) * (i + 1) * 360.0 / order;
		/* X */
		polygons[i].v[0]->position[0] = polygons[i].v[1]->position[0] = 0.0;
		polygons[i].v[2]->position[0] = polygons[i].v[3]->position[0] = 1.0;
		/* Y */
		polygons[i].v[0]->position[1] = polygons[i].v[3]->position[1] = ly;
		ly = polygons[i].v[1]->position[1] = polygons[i].v[2]->position[1] = radius * cos(a);
		/* Z */
		polygons[i].v[0]->position[2] = polygons[i].v[3]->position[2] = lz;
		lz = polygons[i].v[1]->position[2] = polygons[i].v[2]->position[2] = radius * sin(a);
		/* W */
		polygons[i].v[0]->position[3] = polygons[i].v[3]->position[3] = 1.0;
		polygons[i].v[1]->position[3] = polygons[i].v[2]->position[3] = 1.0;
		make_normal(&polygons[i]);
	}
	make_vertex_colors(obj);
	return obj;
}

update_edge_geom(g, e)
graph_t	*g;
edge_t	*e;
{
	int			k;
	double		a,del[3],d;
	object_t	*obj;
	node_t		*n0,*n1;

	obj = e->obj;
	if (obj == NULL) return;
	d = 0.0; 
	for (k = 0 ; k < 3 ; k++ ) {
		del[k] = e->n1->pos.x[k] - e->n0->pos.x[k];
		d += SQR(del[k]); 
	}
	d = sqrt(d);

	/* n0 is placed at the origin */
	n0 = e->n0; n1 = e->n1;
	emu_copy_matrix(obj->v,Identity);
	emu_translate(obj->v,n0->pos.x[0],n0->pos.x[1], n0->pos.x[2]);

	if (d < MIN_EDGE)		/* degenerate edges */
		emu_scale(obj->v,MIN_EDGE,MIN_EDGE,MIN_EDGE);
	else {
		if ((del[1] != 0.0) || (del[0] != 0.0)) {
			a = atan2(del[1],del[0]);
			emu_rotz(obj->v,a);
		}
		if (del[2] != 0.0) {
			a = atan2(del[2],sqrt(del[0]*del[0] + del[1]*del[1]));
			emu_roty(obj->v,-a);
		}
		emu_scale(obj->v,d,1.0,1.0);
	}
}
