/*
 * ShowDXF.c - Read polygons from a DXF file and display them.
 * The file "view.dat" is processed for the camera view that will be used.
 *
 * Author:  Alexander Enzmann
 *
 * size_factor is ignored.
 *
 *    size_factor       # spheres        # squares
 *	     x               xx                 x
 */

/* Read polygons from a DXF file and display them.  The file "view.dat"
    is processed for the camera view that will be used. */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>     /* atoi */
#include <string.h>     /* strcmp() */
#include "def.h"
#include "drv.h"       /* display_close() */
#include "lib.h"

/* These may be read from the command line */
static int raytracer_format = OUTPUT_VIDEO;

#ifdef OUTPUT_TO_FILE
static FILE * stdout_file = NULL;
#else
#define stdout_file stdout
#endif /* OUTPUT_TO_FILE */

/* This is an outrageous hack to read the polygons from a DXF file.
   No attempt is made to be smart, just to graph 3DFACEs.  If you have
   something better, go for it. There are plenty of DXF files that this
   routine won't work for... */
static void
read_dxf_faces( file )
FILE *file;
{
    char buffer[128];
    int i, j, ind, vcnt, fcnt, lineno, maxvert;
    float x;
    COORD3 face[16];

    fcnt = 0;
    while (!feof(file)) {

	/* Skip over uninteresting stuff */
	while (!feof(file) &&
		 fgets(buffer, 127, file) != NULL &&
		 strcmp(buffer, "3DFACE\n"))
	    /* Void - we are reading lines we can't deal with */
	    lineno++;

	if (!feof(file)) {

#if defined(applec) || defined(THINK_C)
#else
	    /* Test to see if we should stop */
	    if (kbhit()) {
		display_close(0);
		fprintf(stderr, "Draw aborted\n");
		exit(1);
	    }
#endif

	    /* As long as there are more faces, read them in */
	    fgets(buffer, 127, file); /* Skip the "8" */
	    lineno++;

	    fgets(buffer, 127, file); /* Skip the "0main" */
	    lineno++;

	    if (buffer[0] == '0')
		;
	    else if (!strcmp(buffer, "3DFURN\n")) {
		fgets(buffer, 127, file); /* Skip the thing after the 3DFURN */
		fgets(buffer, 127, file); /* Skip over the CONTINUOUS */
		lineno += 2;
		if (!strcmp(buffer, "CONTINUOUS\n") &&
		     fgets(buffer, 127, file) != NULL &&
		     sscanf(buffer, "%d", &ind) != NULL) {
		    lineno++;
		    if (ind == 62) {
			fgets(buffer, 127, file); /* Max # of vertices? */
			sscanf(buffer, "%d", &maxvert);
			fgets(buffer, 127, file);
			sscanf(buffer, "%d", &ind);
			lineno++;
		    } else {
			maxvert = -1;
		    }
		    fgets(buffer, 127, file);
		    sscanf(buffer, "%f", &x);
		    lineno++;
		    goto inside_vertex_loop;
		}
		else
		    break;
	    } else if (!strcmp(buffer, "2\n")) {
		fgets(buffer, 127, file);
		sscanf(buffer, "%d", &ind);
		lineno++;
		if (ind == 62) {
		    fgets(buffer, 127, file); /* Max # of vertices */
		    sscanf(buffer, "%d", &maxvert);
		    fgets(buffer, 127, file);
		    sscanf(buffer, "%d", &ind);
		    lineno++;
		} else {
		    maxvert = -1;
		}
		fgets(buffer, 127, file);
		sscanf(buffer, "%f", &x);
		lineno++;
		goto inside_vertex_loop;
	    } else
		break;

	    lineno += 3;
	    vcnt = 0;
	    maxvert = -1;

	    /* This is a face, read the vertices */
	    while (fgets(buffer, 127, file) != NULL &&
		 sscanf(buffer, "%d", &ind) != NULL &&
		 ind != 0 &&
		 fgets(buffer, 127, file) != NULL &&
		 sscanf(buffer, "%f", &x)) {

		PLATFORM_MULTITASK();

		lineno += 2;
inside_vertex_loop:
		/* Got a vertex value */
		j = ind / 10;
		i = ind % 10;

		if (maxvert > 0 && i > maxvert)
		    break;
		else if (i > vcnt)
		    vcnt = i;

		/* Place the value into the appropriate face */
		if (j == 1)
		    face[i][X] = x;
		else if (j == 2)
		    face[i][Y] = x;
		else if (j == 3)
		    face[i][Z] = x;
		else if (maxvert > 0)
		    break;
		else {
		    break;
		    display_close(1);
		    fprintf(stderr, "Bad vertex component: %d/%d at line: %d\n",
			      i, j, lineno);
		    exit(1);
		}
	    }
	    /* Display the polygon */
	    if (vcnt > 0)
		lib_output_polygon(vcnt+1, face);
	    /*
	    printf("Vert[%d]: ", fcnt);
	    for (i=0;i<vcnt;i++)
		printf("<%g, %g, %g> ", face[i][X], face[i][Y], face[i][Z]);
	    printf("\n");
	    */
	    fcnt++;
	}
    }
}

/* Read in the camera specifics: from, at, up, fov.  Aspect is hard coded
    to 1.  */
static void
setup_view()
{
    char buffer[128];
    COORD3 from, at, up;
    double angle;
    FILE *setup;

    /* output viewpoint */
    if ((setup = fopen("view.dat", "r")) != NULL) {
	if (fgets(buffer, 127, setup) &&
	     sscanf(buffer, "%lf %lf %lf",
		      &from[X], &from[Y], &from[Z]) != NULL &&
	     fgets(buffer, 127, setup) &&
	     sscanf(buffer, "%lf %lf %lf",
		      &at[X], &at[Y], &at[Z]) != NULL &&
	     fgets(buffer, 127, setup) &&
	     sscanf(buffer, "%lf %lf %lf",
		      &up[X], &up[Y], &up[Z]) != NULL &&
	     fgets(buffer, 127, setup) &&
	     sscanf(buffer, "%lf", &angle)) {
	    lib_output_viewpoint(from, at, up, 45.0, 1.0, 1.0, 512, 512);
	} else {
#if defined(applec) || defined(THINK_C)
#else
	    fprintf(stderr, "Invalid 'view.dat' file\n");
#endif
	    exit(1);
	}
	fclose( setup );
    } else {
	SET_COORD3(from, 0, 10, -10);
	SET_COORD3(at, 0, 0, 0);
	SET_COORD3(up, 0, 0, 1);
	lib_output_viewpoint(from, at, up, 45.0, 1.0, 1.0, 512, 512);
    }
}

static void showdxf_usage PARAMS((void))
{
    /* localize the usage strings to be expanded only once for space savings */
#if defined(applec) || defined(THINK_C)
    /* and don't write to stdout on Macs, which don't have console I/O, and  */
    /* won't ever get this error anyway, since parms are auto-generated.     */
#else

    fprintf(stderr, "usage [-r format] filename\n");
    fprintf(stderr, "   filename - DXF file to convert/display\n");
    fprintf(stderr, "-r format - input database format to output\n");
    fprintf(stderr, "	0   Output direct to the screen (sys dependent)\n");
    fprintf(stderr, "	1   NFF - MTV\n");
    fprintf(stderr, "	2   POV-Ray 1.0\n");
    fprintf(stderr, "	3   Polyray v1.4, v1.5\n");
    fprintf(stderr, "	4   Vivid 2.0\n");
    fprintf(stderr, "	5   QRT 1.5\n");
    fprintf(stderr, "	6   Rayshade\n");
    fprintf(stderr, "	7   POV-Ray 2.0 (format is subject to change)\n");
    fprintf(stderr, "	8   RTrace 8.0.0\n");
    fprintf(stderr, "	9   PLG format for use with rend386\n");
    fprintf(stderr, "	10  Raw triangle output\n");
    fprintf(stderr, "	11  art 2.3\n");

#endif
} /* showdxf_usage */

static int
showdxf_get_opts( argc, argv, p_rdr, p_file_name )
int	argc ;
char	*argv[] ;
int	*p_rdr ;
char	**p_file_name ;
{
int num_arg ;
int val ;

    *p_file_name = NULL ;

    num_arg = 0 ;

    while ( ++num_arg < argc ) {
	if ( (*argv[num_arg] == '-') || (*argv[num_arg] == '/') ) {
	    switch( argv[num_arg][1] ) {
		case 'r':	/* renderer selection */
		    if ( ++num_arg < argc ) {
			sscanf( argv[num_arg], "%d", &val ) ;
			if ( val < OUTPUT_VIDEO || val >= OUTPUT_DELAYED ) {
			    fprintf( stderr,
				    "bad renderer value %d given\n",val);
			    showdxf_usage();
			    return( TRUE ) ;
			}
			*p_rdr = val ;
		    } else {
			fprintf( stderr, "not enough args for -r option\n" ) ;
			showdxf_usage();
			return( TRUE ) ;
		    }
		    break ;
		default:
		    fprintf( stderr, "unknown argument -%c\n",
			    argv[num_arg][1] ) ;
		    showdxf_usage();
		    return( TRUE ) ;
	    }
	} else {
	    /* filename selection */
	    if ( *p_file_name == NULL ) {
		/* no other file name found previously */
		*p_file_name = argv[num_arg] ;
				    /* [esp] was "...= &argv[num_arg];" */
	    } else {
		fprintf( stderr, "too many filenames\n" ) ;
		showdxf_usage();
		return( TRUE ) ;
	    }
	}
    }
    if ( *p_file_name == NULL ) {
	fprintf( stderr, "no filename given for display\n" ) ;
	showdxf_usage();
	return( TRUE ) ;
    }
    return( FALSE ) ;
}

/* Read in the camera view, then read in DXF polygons, then display them. */
int
main(argc, argv)
    int argc;
    char *argv[];
{
    COORD3 back_color, dxf_color;
    COORD4 light;
    char *file_name = NULL ;
    FILE *file;

    PLATFORM_INIT(SPD_SHOWDXF);

    /* Start by defining which raytracer we will be using */
    if ( showdxf_get_opts( argc, argv, &raytracer_format, &file_name ) ) {
	return 0;
    }

    if (raytracer_format == OUTPUT_RTRACE ||
	 raytracer_format == OUTPUT_PLG)
	lib_set_raytracer(OUTPUT_DELAYED);
    else
	lib_set_raytracer(raytracer_format);

    file = fopen(file_name, "r");
    if (file == NULL) {
	fprintf(stderr, "Cannot open dxf file: '%s'\n", file_name);
	exit(1);
    }

#ifdef OUTPUT_TO_FILE
    /* no stdout, so write to a file! */
    if (raytracer_format != OUTPUT_VIDEO)
	stdout_file = fopen("ShowDXF.out", "w");
#endif /* OUTPUT_TO_FILE */
    lib_set_output_file(stdout_file);

    lib_set_polygonalization(3, 3);

    /* output background color - dark blue */
    /* NOTE: Do this BEFORE lib_output_viewpoint(), for display_init() */
    SET_COORD3(back_color, 0.1, 0.0, 0.5);
    lib_output_background_color(back_color);

    setup_view();

    /* output object color - light gray */
    SET_COORD3(dxf_color, 0.8, 0.8, 0.8);
    lib_output_color(NULL, dxf_color, 0.1, 0.8, 0.0, 0.2, 5.0, 0.0, 1.0);

    /* output light sources */
    SET_COORD4(light, 40.0, 30.0, 20.0, 1);
    lib_output_light(light);
    SET_COORD4(light, -40, -20, 10, 1);
    lib_output_light(light);

    read_dxf_faces(file);

    /* Make sure everything is cleaned up */
    if (raytracer_format == OUTPUT_RTRACE ||
	 raytracer_format == OUTPUT_PLG) {
	lib_set_raytracer(raytracer_format);
	lib_flush_definitions();
    }

    if (file)
	fclose(file);

#ifdef OUTPUT_TO_FILE
    /* no stdout, so close our output! */
    if (stdout_file)
	fclose(stdout_file);
#endif /* OUTPUT_TO_FILE */
    if (raytracer_format == OUTPUT_VIDEO)
	display_close(1);

    PLATFORM_SHUTDOWN();
    return EXIT_SUCCESS;
}
