/****************************************************************************
 * render_ctrl.c
 * Author Joel Welling and Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
The routines in this module act as the interface between lisp and the
renderer, and between lisp and the controller device.
*/

#include <stdio.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "control.h"
#include "ui.h"
#include "ren.h"
 
/* Needed external definitions */
extern char *malloc(), *realloc();

/*
The routines in this module interface to Lisp, but don't change the
Lisp environment.  Because of this, they should return a nil.
(Nil is an entity in its own right, so a new nil must be created).
The routines to which this applies are those which return a NODE *
and take NODE *'s as arguments.
*/

/* 
This routine causes the renderers to dump their internal
representation of a gob.
*/
NODE *dumpgob(thisgob)
NODE *thisgob;
{
	ger_debug( "Dumping gob number %d",gob_idnum((Gob)thisgob) );

	ren_dump( (Gob)thisgob );

	return( NIL );
}

NODE *do_sphere()
/* This routine causes the renderer to generate a sphere primitive. */
{
	ger_debug("do_sphere");
	ren_sphere();
	return( NIL );
}

NODE *do_cylinder()
/* This routine causes the renderer to generate a cylinder primitive. */
{
	ger_debug("do_cylinder");
	ren_cylinder();
	return( NIL );
}

NODE *do_torus(arglist)
NODE *arglist;
/*
This routine causes the renderer to construct a cylinder object from
the arguments given.  The argument list should contain two floats,
representing the major and minor radii of the torus.
*/
{
	float bigradius, smallradius;

	ger_debug("do_torus");

	if (length_list(arglist) < 2)
		{
		ger_error("ERROR, NEED AT LEAST 2 ARGUMENTS TO DO-TORUS");
		return(NIL);
		}
	if (!floatp(safe_car(arglist)))
		{
		ger_error("ERROR, ARGUMENT 1 OF DO-TORUS NOT OF TYPE FLOAT");
		return(NIL);
		}
	if (!floatp(safe_car(safe_cdr(arglist))))
		{
		ger_error("ERROR, ARGUMENT 2 OF DO-TORUS NOT OF TYPE FLOAT");
		return(NIL);
		}

	bigradius = getflonum(safe_car(arglist));
	smallradius = getflonum(safe_car(safe_cdr(arglist)));
	ren_torus( bigradius, smallradius );

	return(NIL);
}

NODE *do_text(arglist)
NODE *arglist;
/*
This routine causes the renderer to construct a text object from the args 
in the list passed it.  The argument list should contain a point, two vectors 
to define the u and v directions of the text plane, and a text string.
*/
{
	Point txtpoint;
	Vector uvec, vvec;
	char *txtstring;

	ger_debug("do_text");

	txtpoint= (Point)safe_car(arglist);
	arglist= safe_cdr(arglist);
	uvec= (Vector)safe_car(arglist);
	arglist= safe_cdr(arglist);
	vvec= (Vector)safe_car(arglist);
	arglist= safe_cdr(arglist);
	txtstring= getstring( safe_car(arglist) );

	ren_text( txtpoint, uvec, vvec, txtstring );

	return(NIL);
}

NODE *do_polyline(arglist)
NODE *arglist;
/*
This routine causes the renderer to constructs a polyline object from the 
args (all vertices) in the list passed it.
*/
{
	int vertexcount;
 
	ger_debug("do_polyline");
 
	/*
	Find out how many vertices there are.
	*/
	vertexcount= length_list( arglist );
	if ( vertexcount<2 )
		{
		ger_error("do_polyline: %d is not enough vertices!",
			vertexcount);
		return(NIL);
		};
 
	/* Create the polyline. */
	ren_polyline( (Vertex_list)arglist, vertexcount );

	return(NIL);
}
 
NODE *do_polymarker(arglist)
NODE *arglist;
/*
This routine causes the renderer to constructs a polymarker object from the 
args (all vertices) in the list passed it.
*/
{
	int vertexcount;
 
	ger_debug("do_polymarker");
 
	/*
	Find out how many vertices there are.
	*/
	vertexcount= length_list( arglist );
	if ( vertexcount<1 )
		{
		ger_error("do_polymarker: %d is not enough vertices!",
			vertexcount);
		return(NIL);
		};
 
	/* Create the polyline. */
	ren_polymarker( (Vertex_list)arglist, vertexcount );

	return(NIL);
}
 
NODE *do_polygon(arglist)
NODE *arglist;
/*
This routine causes the renderer to constructs a polygon object from the 
args (all vertices) in the list passed it.
*/
{
	int vertexcount;
 
	ger_debug("do_polygon");
 
	/*
	Find out how many vertices there are.
	*/
	vertexcount= length_list( arglist );
	if ( vertexcount<3 )
		{
		ger_error("do_polygon: %d is not enough vertices!",
			vertexcount);
		return(NIL);
		};
	/* Create the polygon. */
	ren_polygon( (Vertex_list)arglist, vertexcount );

	return(NIL);
}

NODE *do_triangle(arglist)
NODE *arglist;
/*
This routine causes the renderer to construct a triangle strip object from 
the args (all vertices) in the list passed it.  The object constructed is a 
triangle strip, with each adjacent triple of vertices specifying one triangle.
*/
{
	int vertexcount;
 
	ger_debug("do_triangle");
 
	/*
	Find out how many vertices there are.
	*/
	vertexcount= length_list( arglist );
	if ( vertexcount<3 )
		{
		ger_error("do_triangle: %d is not enough vertices!",
			vertexcount);
		return(NIL);
		};
 
	/* Create the triangle strip. */
	ren_triangle( (Vertex_list)arglist, vertexcount );

	return(NIL);
}
 
NODE *do_mesh(arglist)
NODE *arglist;
/*
This routine causes the renderer to construct a general mesh object from
the args
*/
{
	Vertex_list vlist;
	Facet_list flist;
	int vertexcount, facetcount;

	if (length_list(arglist) < 2)
		{
		ger_error("ERROR, NEED AT LEAST 2 ARGUMENTS TO DO-MESH");
		return(NIL);
		}
	vlist= (Vertex_list)safe_car( arglist );
	vertexcount= length_list( vlist );
	if (vertexcount==0)
		{
		ger_error("ERROR, EMPTY MESH VERTEX LIST");
		return(NIL);
		}
	flist= (Facet_list)safe_car( safe_cdr(arglist) );
	facetcount= length_list( (NODE *)flist );
	if (vertexcount==0)
		{
		ger_error("ERROR, EMPTY MESH FACET LIST");
		return(NIL);
		}

	ren_mesh( vlist, vertexcount, flist, facetcount );
	return(NIL);
}

NODE *do_bezier(arglist)
NODE *arglist;
/*
This routine causes the renderer to construct a Bezier patch object from 
the args (16 vertices) in the list passed it.
*/
{
	int vertexcount;
 
	ger_debug("do_bezier");
 
	/*
	Find out how many vertices there are.
	*/
	vertexcount= length_list( arglist );
	if ( vertexcount!= 16 )
		{
		ger_error("do_bezier: got %d vertices instead of 16 needed!",
			vertexcount);
		return(NIL);
		};
 
	/* Create the triangle strip. */
	ren_bezier( (Vertex_list)arglist, vertexcount );

	return(NIL);
}
 
NODE *do_gob(thisgob)
NODE *thisgob;
/*
This routine sees all gobs as they are defined.  It extracts appropriate
gob elements and passes them to the renderer.
*/
{
	int current_gob;
 
	current_gob= gob_idnum( (Gob)thisgob );

	ger_debug("do_gob: Defining gob %d.", current_gob);
 
	ren_gob(
		current_gob,
		gob_trans( (Gob)thisgob ),
		gob_attr( (Gob)thisgob ),
		gob_primitive( (Gob)thisgob ),
		gob_children( (Gob)thisgob ) );

 	return( thisgob );
}
 
NODE *do_light(arglist)
NODE *arglist;
/*                                             
This routine causes the renderer to construct a directional light object, 
at the location and with the color in the parameter list.
*/
{
	Point location; 
	Color lightcolor;
 
	ger_debug("do_light");

	/* Set the location if it's given (or complain if it's not) */
	location= (Point)safe_car( arglist );
	arglist= safe_cdr( arglist );
	if (null(location))
		{
		ger_error("do_light: no location given for light!");
		return(NIL);
		}

	/* Set the color if it's given */
	lightcolor= (Color)safe_car( arglist );

	/* Add the light */
	ren_light( location, lightcolor );

	return(NIL);
}

NODE *do_ambient(arglist)
NODE *arglist;
/*                                             
This routine causes the renderer to construct an ambient light object, 
with the color given in the arglist.
*/
{
	Color lightcolor;
 
	ger_debug("do_ambient");

	/* get the color */
	lightcolor= (Color)safe_car( arglist );

	/* Add the light */
	ren_ambient( lightcolor );

	return(NIL);
}

NODE *set_camera(camera)
NODE *camera;
/*
This routine causes the renderer to set it's camera to that given by the
parameters in the argument list.
*/
{
	Point lookat, lookfrom;
	Vector lookup;
	float fov, hither, yon;
	Color background;
 
	ger_debug("set_camera");

 	/* 
	Get the camera lookat, lookfrom, and lookup vectors and set them
	(or complain if any are missing)
	*/
	lookfrom= camera_lookfrom( (Camera)camera );
	lookat= camera_lookat( (Camera)camera );
	lookup= camera_up( (Camera)camera );
	if ( null(lookfrom) || null(lookat) || null(lookup) )
		{
		ger_error(
			"set_camera: lookat, lookfrom, or lookup not defined!");
		return( NIL );
		}

	/* Get the fovea and hither and yon distances */
	fov= camera_fovea( (Camera)camera );
	hither= camera_hither( (Camera)camera ) ;
	yon= camera_yon( (Camera)camera );

	/* Get the background color */
	background= camera_background( (Camera)camera );
		
	/* Tell the user interface about the camera */
	ui_camera( lookat, lookfrom, lookup, fov, hither, yon, background );

      	return(NIL);
}

NODE *render(theArgs)
NODE *theArgs;
/* 
This routine causes the renderer to render the object given, in the 
current rendering environment.
 */
{
	Gob thisgob;
	Transformation thistrans;
	Attribute_list thisattrlist;

	ger_debug("render:");

	if (null( thisgob= (Gob)safe_car(theArgs) ))
		ger_fatal("render: called with no gob!");
	if (null( thistrans= (Transformation)safe_car(safe_cdr(theArgs))))
		ger_fatal("render: called with no transformation!");
	thisattrlist= (Attribute_list)safe_car(safe_cdr(safe_cdr(theArgs))); /* may be null */

	ui_render( thisgob, thistrans, thisattrlist );  

	return(NIL);
}

NODE *traverselights(theArgs)
NODE *theArgs;
/* This routine adds the given object to the lights environment. */
{
	Gob thisgob;
	Transformation thistrans;
	Attribute_list thisattrlist;

	ger_debug("traverselights:");

	if (null( thisgob= (Gob)safe_car(theArgs) ))
		ger_fatal("traverselights: called with no gob!");
	if (null( thistrans= (Transformation)safe_car(safe_cdr(theArgs))))
		ger_fatal("traverselights: called with no transformation!");
	thisattrlist= (Attribute_list)safe_car(safe_cdr(safe_cdr(theArgs))); /* may be null */
	ui_traverselights( thisgob, thistrans, thisattrlist );

	return(NIL);
}

NODE *do_free_gob(thisgob)
NODE *thisgob;
/* 
This routine signals the user interface to cause the renderer to free
memory associated with the given gob on the C side.  If the user interface
does so, any future call referncing this gob or its children may be
impossible unless it is somehow redefined first.  Memory associated
with the gob on the Lisp side need not be freed here, as the lisp
code which calls this routine will do so.
*/
{

	ger_debug("do_free_gob");

	ui_free( (Gob)thisgob );

	return(NIL);
}

NODE *setup(theArgs)
NODE *theArgs;
/* 
This routine causes the user interface to initialize the renderer 
and controller, if they are not already initialized. 
*/
{
	static int initialized=0; /* to hold initialization state */
	static char 
		renderer_default[]= "none",
		device_default[]= "none",
	        controller_default[]= "none",
	        outfile_default[]= "-";
	char 	*renderer= renderer_default, 
		*device= device_default, 
		*controller= controller_default,
	        *outfile= outfile_default;
	int startframe=0, framecopies=0;
	Attribute_list hints;
 
	ger_debug("setup: initializing user interface");

	/* 
	Extract renderer, device, controller, startframe, and
	framecopies from arg list 
	*/
      	if ( !null(theArgs) )
		{
		if ( !null( safe_car(theArgs) ) )
			renderer= getname( safe_car(theArgs) );
		theArgs= safe_cdr(theArgs);
		}

      	if ( !null(theArgs) )
		{
		if ( !null( safe_car(theArgs) ) )
			device= getname( safe_car(theArgs) );
		theArgs= safe_cdr(theArgs);
		}

      	if ( !null(theArgs) )
		{
		if ( !null( safe_car(theArgs) ) )
			controller= getname( safe_car(theArgs) );
		theArgs= safe_cdr(theArgs);
		}

      	if ( !null(theArgs) )
		{
		if ( !null( safe_car(theArgs) ) )
			startframe= getfixnum( safe_car(theArgs) );
		theArgs= safe_cdr(theArgs);
		}

      	if ( !null(theArgs) )
		{
		if ( !null( safe_car(theArgs) ) )
			framecopies= getfixnum( safe_car(theArgs) );
		theArgs= safe_cdr(theArgs);
		}

	if ( !null(theArgs) )
	       {
	       if ( !null( safe_car(theArgs) ) )
	               outfile= getname( safe_car(theArgs) );
	       theArgs= safe_cdr(theArgs);
	       }

	if ( !null(theArgs) ) hints= safe_car(theArgs);
	else hints= theArgs; /* a handy nil */

	if (!initialized)
		{
		/* 
		Initilize the user interface, which will in turn
		initialize the controller and renderer.
		*/
		ui_setup( renderer, device, outfile, hints, controller, 
			 startframe, framecopies );
		initialized= 1;
		}
	else
		{
		ger_debug(
	   "setup: called twice; this call interpreted as a hard reset.");
		ui_reset( 1, startframe, framecopies );
		}
	return(NIL);
}

void terminate()
/* 
This routine causes the user interface to shut down the render and 
controller.
*/
{
      ger_debug("terminate: shutting down user interface\n");

      ui_shutdown();
}

NODE *reset(theArgs)
NODE *theArgs;
/* 
This routine resets certain parameters associated with the renderer and/or
controller.
*/
{
	int startframe, framecopies;
 
	ger_debug("reset: resetting user interface");

	/* Extract controller, startframe, framecopies from arg list */
      	if ( !null(theArgs) )
		{
		if ( !null( safe_car(theArgs) ) ) 
			startframe= getfixnum( safe_car(theArgs) );
		else startframe= 0;

		if ( !null( safe_car(safe_cdr(theArgs)) ) ) 
			framecopies= getfixnum( safe_car(safe_cdr(theArgs)) );
		else framecopies= 0;
		}
	else
		{
		startframe= 0;
		framecopies= 0;
		}

	/* Reset the user interface, which will in turn reset
	 * the controller and renderer.  The 0 means this is
	 * a soft reset, which does not regenerate lisp-side 
	 * variables.
	 */
	ui_reset( 0, startframe, framecopies );

	return(NIL);
}

NODE *do_toggle_debug()
/* This routine turns debugging on if it is off, and vice versa. */
{
	ger_toggledebug();
	return( NIL );
}
