/****************************************************************************
 * ray_ren.c
 * Author Brian Welcker
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This 'renderer' dumps information about structures in a format compatible with
the rayshade ray tracer.
*/
#include <stdio.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "assist.h"
#include "ren.h"
#include "indent.h"

/* Needed external definitions */
extern char *malloc(), *realloc();

/* Symbols defined by ren_setup and used in parsing attribute lists */
static Symbol color_symbol, backcull_symbol, text_height_symbol;

/*
Primitive currently being executed, set by ren_render for the benefit
of primitive funcions which need data about themselves.
*/
static Primitive current_primitive;

/*
Ambient light level
*/
static float ambient_red, ambient_green, ambient_blue;

/* Number of current surface */
static int current_surface = 1;

static void dump_point( point )
Point point;
/* This routine dumps a point */
{
	ger_debug("dump_point: dumping point");
	ind_write("%f %f %f ",
		point_x( point ),
		point_y( point ),
		point_z( point ) );
}

static void dump_vector( vector )
Vector vector;
/* This routine dumps a vector */
{
	ger_debug("dump_vector: dumping vector");
	ind_write("%f %f %f ",
		vector_x( vector ),
		vector_y( vector ),
		vector_z( vector ) );
}

static void dump_color( color )
Color color;
/* This routine dumps a color */
{
	ger_debug("dump_color: dumping color");
	ind_write("%f %f %f ",
		color_red( color ),
		color_green( color ),
		color_blue( color ));
}

void dump_surface ()
/*
This routine dumps a surface representation
*/
{
    int thisid;
    Color color;

    ger_debug("dump_surface");
    thisid = primitive_id( current_primitive );
    color = ast_color_attribute( color_symbol );
    ind_write("surface s%d", current_surface);
    ind_eol();
    ind_push();
    ind_write("%f %f %f", ambient_red, ambient_green, ambient_blue);
    ind_eol();
    dump_color( color );
    ind_eol();
    ind_write("0.9 0.9 0.5 10.0 0.1 ");
    ind_write("%f 1.0", (1.0 - color_alpha( color )));
    ind_eol();
    ind_pop();
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* 
This routine should dump the renderer's internal definition
of an object. 
*/
void ren_dump(thisgob)
Gob thisgob;
{
  int thisid;

  thisid= gob_idnum(thisgob);
  ger_debug("ren_dump: dump requested for gob number %d",thisid);
}

void dump_poly_surface ( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a surface representation of a polygon, averaging
the color at each vertex.
*/
{
  int thisid, i = 0, j;
  Color color;
  float r = 0.0, g = 0.0, b = 0.0, a = 0.0;

  ger_debug("dump_poly_surface");
  thisid = primitive_id( current_primitive );

  for (j = 0; j < count; j++) {
    if ( !null( color = vertex_color( first_vertex( vlist ) ) ) ) {
      r += (float) color_red( color );
      g += (float) color_green( color );
      b += (float) color_blue( color );
      a += (float) color_alpha( color );
      i++;
    }
    vlist = rest_of_vertices( vlist );
  }
  if (i > 0) {
    (void)set_color_red( color, r / (float)i );
    (void)set_color_green( color, g / (float)i );
    (void)set_color_blue( color, b / (float)i );
    (void)set_color_alpha( color, a / (float)i );
  } else color = ast_color_attribute( color_symbol );

  ind_write("surface s%d", current_surface);
  ind_eol();
  ind_push();
  ind_write("%f %f %f", ambient_red, ambient_green, ambient_blue);
  ind_eol();
  dump_color( color );
  ind_eol();
  ind_write("1.0 1.0 1.0 10.0 0.1 ");
  ind_write("%f 1.0", (1.0 - color_alpha( color )));
  ind_eol();
  ind_pop();
}

static void dump_vertex( vertex )
Vertex vertex;
/* 
This routine dumps vertices.
*/
{
 	ind_write("%f %f %f ",
		vertex_x(vertex),
		vertex_y(vertex),
		vertex_z(vertex));
}

void ren_sphere()
/*
This routine dumps a sphere primitive object.
*/
{
	ger_debug("ren_sphere");
	dump_surface();
	ind_write("sphere s%d 1.0 0.0 0.0 0.0", current_surface++);
	ind_eol();
}

void ren_cylinder()
/*
This routine dumps a cylinder primitive object.
*/
{
	ger_debug("ren_cylinder");
	dump_surface();
	ind_write("cylinder s%d	0.0 0.0 0.0 0.0 0.0 1.0 1.0",
		  current_surface++);
	ind_eol();
}

void ren_torus( bigradius, smallradius )
float bigradius, smallradius;
/*
This routine dumps a torus primitive object.
*/
{
	ger_debug("ren_torus");
	ast_torus( bigradius, smallradius, ren_mesh );
}

static void impl_trans( trans )
Transformation trans;
/* This function is used by ast_text to do transformations between letters. */
{

}

void ren_text( txtpoint, uvec, vvec, txtstring )
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine dumps a text primitive object. Text is not supported by art.
*/
{
	ger_debug("ren_text");
/*	ast_text(txtpoint,uvec,vvec,txtstring,impl_trans,ren_polyline); */
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polyline primitive object.
*/
{
	ger_debug("ren_polyline");
	if (count<2) {
		ind_write("***Error***: not enough vertices"); 
		ind_eol();
	      }
	while (!null(rest_of_vertices(vlist)))
	  {
	    dump_surface();
	    ind_write("cylinder s%d", current_surface++);
	    ind_eol();
	    ind_push();
	    dump_vertex( first_vertex( vlist ) );
	    ind_eol();
	    dump_vertex( first_vertex(rest_of_vertices((vlist))));
	    ind_eol();
	    ind_write("0.05");
	    ind_eol();
	    ind_pop();
	    vlist= rest_of_vertices( vlist );
	  }
}

void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polymarker primitive object.
*/
{
	ger_debug("ren_polymarker");
	dump_surface();
	ind_write("sphere s%d 0.1", current_surface++);
	ind_eol();
	ind_push();
	dump_vertex( first_vertex( vlist ) );
	ind_eol();
	ind_pop();
}

void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polygon primitive object.
*/
{
	ger_debug("ren_polygon");
	dump_poly_surface( vlist, count );
	ind_write("poly s%d", current_surface++);
	ind_eol();
	ind_push();
	if (count<3) {
		ind_write("***Error***: not enough vertices"); 
		ind_eol();
		}
	while (!null(vlist)) {
		dump_vertex( first_vertex( vlist ) );
		vlist= rest_of_vertices( vlist );
		ind_eol();
		}
	ind_pop();
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a triangle strip primitive object.
*/
{
  int i, even = 0;
  
  ger_debug("ren_triangle");
  if ( count < 3) {
    fprintf(stderr,"***Error***: not enough vertices"); 
    return;
  }
  for (i = 0; i < (count-2); i++)
    {
      if (even = (1-even))
	{
	  dump_poly_surface( vlist, 3 );
	  ind_write("poly s%d", current_surface++);
	  ind_eol();
	  ind_push();
	  dump_vertex(first_vertex(vlist));
	  ind_eol();
	  dump_vertex(first_vertex(rest_of_vertices(vlist)));
	  ind_eol();
	  dump_vertex(first_vertex(rest_of_vertices(rest_of_vertices(vlist))));
	  ind_eol();
	  ind_pop();
	}
      else
	{
	  dump_poly_surface( vlist, 3 );
	  ind_write("poly s%d", current_surface++);
	  ind_eol();
	  ind_push();
	  dump_vertex(first_vertex(vlist));
	  ind_eol();
	  dump_vertex(first_vertex(rest_of_vertices(rest_of_vertices(vlist))));
	  ind_eol();
	  dump_vertex(first_vertex(rest_of_vertices(vlist)));
	  ind_eol();
	  ind_pop();
	}
      vlist = rest_of_vertices(vlist);
    }
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/*
This routine dumps a general mesh object from the args passed it.
*/
{
	Vertex_list thisfacet;
	int facet_vertices;

	ger_debug("ren_mesh");
	while (fcount--) {
		thisfacet= first_facet( flist );
		flist= rest_of_facets( flist );
		facet_vertices= length_list( thisfacet );
		ren_polygon( thisfacet, facet_vertices );
	      }
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a Bezier patch primitive object.
*/
{
 	ger_debug("ren_bezier");
	ast_bezier( vlist, ren_mesh );
}

static void dump_trans( trans )
Transformation trans;
/* This routine dumps a transformation matrix */
{
	float *matrix;
	register float *m_copy;

	ger_debug("dump_trans: dumping transformation matrix");

	matrix= array2d_to_c( trans );
	ind_write("transform");
	ind_eol();
	ind_push();
	ind_write("%f %f %f", *(matrix+0), *(matrix+1), *(matrix+2));
	ind_eol();
	ind_write("%f %f %f", *(matrix+4), *(matrix+5), *(matrix+6));
	ind_eol();
	ind_write("%f %f %f", *(matrix+8), *(matrix+9), *(matrix+10));
	ind_eol();
	ind_write("%f %f %f", *(matrix+3), *(matrix+7), *(matrix+11));
	ind_eol();
	ind_pop();

	free( (char *)matrix );
}
 
void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It dumps information
about their internal structure.
*/
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
}
 
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine dumps a light primitive object.
*/
{
  float intensity;

  ger_debug("ren_light");
  
  intensity= (color_red( lightcolor ) + color_blue( lightcolor )
	      + color_green( lightcolor)) / 3;
  ind_write("light %f point ", intensity );
  dump_point( location );
  ind_eol();
}

void ren_ambient( lightcolor )
Color lightcolor;
/*                                             
This routine sets the ambient light global variable.
*/
{
  ger_debug("ren_ambient");
  ambient_red = color_red ( lightcolor );
  ambient_green = color_green( lightcolor );
  ambient_blue = color_blue( lightcolor );
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine dumps a camera.
*/
{
	ger_debug("ren_camera");

	ind_write("eyep ");
	dump_point( lookfrom );
	ind_eol();
	ind_write("lookp ");
	dump_point( lookat );
	ind_eol();
	ind_write("up ");
	dump_vector(lookup);
	ind_eol();
	ind_write("fov %f", fov);
	ind_eol();
}

void ren_render(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/* 
This routine reports a request to render the given object.
 */
{
  int thisid; 
  Attribute_list newattr;
  Transformation newtrans;
  Child_list kidlist, newkids;

  thisid= gob_idnum( thisgob );
  kidlist= gob_children( thisgob );
  newtrans= gob_trans(thisgob);
  newattr= gob_attr(thisgob);

  current_primitive= gob_primitive(thisgob);
  
  ger_debug("render: rendering object given by gob %d", thisid);

  /* Push default attributes */
  if ( !null( thisattr ) ) ast_push_attributes( thisattr );

  if ( !null( newattr ) ) ast_push_attributes( newattr );

  if ( !null( gob_children( thisgob ) ) )
    {
      ind_write("define gob%d", thisid);
      ind_eol();
      ind_push();
      while ( !null( kidlist ) )
	{
	  ren_render( first_child( kidlist ), NIL, NIL);
	  kidlist= rest_of_children( kidlist );
	}
      ind_pop();
      ind_write("defend");
      ind_eol();
      ind_write("object gob%d", thisid);
      ind_eol();
    } 
  else
    {
      if ( !null( current_primitive ) ) 
	eval_function( primitive_op( current_primitive ));
    }

  if ( !null( newattr) ) ast_pop_attributes( newattr );

  if ( !null( newtrans ) ) 
    {
      ind_push();
      dump_trans( newtrans );
      ind_pop();
    }
}


void ren_traverselights(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/*
This routine reports a request to traverse the given object in search
of lights.
*/
{
  int thisid; 
  Attribute_list newattr;
  Transformation newtrans;
  Primitive newprim;
  Child_list kidlist, newkids;

  thisid= gob_idnum( thisgob );
  ger_debug("ren_traverselights: setting lights to gob %d",thisid);
  if ( !null( newtrans= gob_trans(thisgob) ) ) {
    dump_trans( newtrans );
    newtrans= matrix_mult( newtrans, thistrans );
  }
  if ( !null( newprim= gob_primitive(thisgob) ) ) 
    eval_function( primitive_op(newprim) );
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      ren_traverselights( first_child( kidlist ), 
			 newtrans, newattr );
      kidlist= rest_of_children( kidlist );
    }
  }
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );
	ger_debug("ren_free:  freeing gob %d",thisid);
}

void ren_setup(renderer,device,outfile,hints)
char *renderer, *device, *outfile;
Attribute_list hints;
/* This routine initializes the renderer, if it is not already initialized. */
{
  Attribute_list initial;

  static int initialized=0; /* to hold initialization state */
 
/* Ultimately renderer and device will be useful;  for the moment
   they are ignored. */
  ger_debug("ren_setup: initializing renderer; renderer %s, device %s",
	    renderer, device);
  
/* Generate some symbols to be used later in attribute list parsing. */
  if (!initialized) {
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    
    fprintf(stderr,"initial push\n");
    ambient_red = 0.3;
    ambient_green = 0.3;
    ambient_blue = 0.3;
    
    ger_debug("ren_setup: initializing dumper");
    ind_setup();
    
    initialized= 1;
  }
  else ger_error("ren_setup: called twice, this call ignored");
}

void ren_reset()
/* This routine resets the renderer */
{
	/* At the moment, this routine does nothing. */
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	/* At the moment, this routine does nothing. */
	ger_debug("ren_shutdown: doing nothing");
}
