/*****************************************************************************
 *                                                                           *
 *	mem.c                                                                *
 *      Copyright 1989, Pittsburgh Supercomputing Center                     *
 *                      All Rights Reserved                                  *
 *			Author Chris Nuuja                                   *
 *									     *
 * Permission use, copy, and modify this software and its documentation      *
 * without fee for personal use or use within your organization is hereby    *
 * granted, provided that the above copyright notice is preserved in all     *
 * copies and that that copyright and this permission notice appear in       *
 * supporting documentation.  Permission to redistribute this software to    *
 * other organizations or individuals is not granted;  that must be          *
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon             *
 * University make any representations about the suitability of this         *
 * software for any purpose.  It is provided "as is" without express or      *
 * implied warranty.     						     *
 *****************************************************************************/

 /*  
     This file defines all the functions used to decrement node pointer counts
     and to reclaim garbage nodes
 */
#include  "alisp.h"
NODE *node_buff;
static NODE *temp;
NODE *memory_record;



/*  This function frees all the memory used in an execution of alisp.  It is
    to be called immediately before leaving alisp.  <i> is simply for debugging
    purposes.
*/

void clean_memory(mem_list,i)
NODE *mem_list;
int i;
{

	if (Type_Of(mem_list) == N_FREE)
		return;
	clean_memory(cdr(mem_list),++i);
	free(cdr(mem_list));
}
 
/* this places a node back on the free list */
int free_node(anode)
NODE *anode;
{
	if (Type_Of(anode) == N_NIL)
		{
		fprintf(stderr,"Severe lisp Error, tried to free null\n");
		anode->ptr_cnt = 300;
		return(1);
		}
	Type_Of(anode) = N_LIST;
	rplaca(anode,NIL);
	rplacd(anode,node_buff);
	node_buff = anode;
	return(1);
}

/* 
   allocates a block of node memory and links it togeather by cdr's.
*/
void alloc_node_mem()
{
	int i;

	node_buff = (NODE *) malloc(16384*sizeof(NODE));
	if (node_buff == 0)
		{
		fprintf(stderr,"ERROR, ran out of memory\n");
		return;
		}
	cdr(memory_record) = node_buff;
	memory_record = node_buff++;
	Type_Of(memory_record) = N_LIST;
	for (i=0; i<16382; i++)
		{
		(node_buff[i]).expr_type = N_LIST;
		rplacd(&(node_buff[i]),&(node_buff[i+1]));
		}
	rplacd(&node_buff[i],(NODE *)0);
}


/*   
    This will also be put into the memory management module
    It takes as input a (assumed) valid NODE type, and returns a pointer to
    a node structure of that type.  The value assigned to the slots of that node
    structure should be appropriate to its enumerated type.  If the slot
    values set are for a different type, this will cause a core dump when
    the interpreter tries to acess the wrong type of data 
*/
NODE *new_node(atype)
EXPR_TYPE atype;
{
	NODE *anode;

	if (!node_buff)
		alloc_node_mem();
	anode = node_buff;
	node_buff = cdr(node_buff);
	Type_Of(anode) = atype;
	anode->ptr_cnt = 0;
	return( anode );
}


void decr_list(anode)
NODE *anode;
{
   int state,i;
   NODE *current, *previous; /* ptrs to where to where we are now, and before */
   extern NODE *DECR_START;  /* Ancor at start of list.  Its a global, so */
			     /* It does not need to be allocated and */
			     /* deallocated each time */
   Type_Of(DECR_START) = N_LIST_R;
   car(DECR_START) = NIL;
   cdr(DECR_START) =DECR_START;  /* Make list circular.  car = cdr is used as */
				 /* an end test  */
   current = anode;
   previous = DECR_START;
   state=1;

   while (current != previous)  /* only true if we are back at DECR_START */
   {
      switch (state)
         {
         case 1:  	/*  Advance */
	  if (current->ptr_cnt != 1)  /* end of decrementable part of list */
	     state=2;
	  else
             switch(Type_Of(current))
	        {
	        case N_LIST:	     /* Haven't been here  before         */
	          if (consp(car(current)))
	   	     {
		     Type_Of(current) = N_LIST_L;
		     temp = previous;
		     previous = current;
		     current = car(current);
		     car(previous) = temp;
		     }
	          else if (consp(cdr(current)))
		     {	
		     decr_elem(car(current));  /* a non list node */
		     Type_Of(current) = N_LIST_R;
		     temp = previous;
		     previous = current;
		     current = cdr(current);
		     cdr(previous) = temp;
		     }
	          else   /* dead end */
		     {
		     decr_elem(car(current));
		     decr_elem(cdr(current));
		     state=2;  /* start retreat */
		     }
	          break;	
	       case N_LIST_R:
		  state=2;     /* retreat      */
		  break;
	       case N_LIST_L:
		  if (consp(cdr(current)))
		     state=2;
		  else
		     {
		     decr_elem(cdr(current));
		     state=2;
		     }
		  break;
	       };
	    break;
         case 2:     /*   Retreating   */
	    Type_Of(current) = N_LIST;
	    switch(Type_Of(previous))
	       {
	       case N_LIST_L:
		  if (consp(cdr(previous)))
		     {
		     Type_Of(previous) = N_LIST_R;
		     temp = car(previous);
		     car(previous) = current;
	             decr_ref(current);
		     current = cdr(previous); 
		     cdr(previous) = temp;
		     state=1;
		     }
		  else
		     {
		     temp = previous;
		     previous = car(previous);
		     car(temp) = current; 
		     current = temp;
		     decr_ref(car(current));
		     }
		  break;
	       case N_LIST:
		  if (previous != current)
		     {
		     if (previous == DECR_START)
		     	current=DECR_START;
		     else if (current == DECR_START)
		        previous=DECR_START;
		     else	
			{
			fprintf(stderr,"Severe internal error:\n");
			fprintf(stderr,"Normal list encountered while backtracking in decr_list \n");
			print_out(current);
			exit(1);		 
		        }
		     }
	          break;	
	       case N_LIST_R:
		  temp = previous;
		  previous = cdr(previous);
		  cdr(temp) = current;
		  current = temp;
		  decr_ref(cdr(current));
		  break;
	       default:
		  fprintf(stderr,"Severe internal error:\n");
		  fprintf(stderr,"Unknown state in decr_list\n");
	          dprint_out(current); dprint_out(previous);
		  exit(1);
		  break;
	       }
	    break;
	default:
	   printf("Error, unknown state %d\n",state);
	   print_out(current);
	   print_out(previous);
	   exit(1);
	}
   }
}

/*
   its constituant parts.  If the ptr_cnt of <anode> is not 1, then it
   is just decremented.  If the ptr_cnt of <anode> is 1, then decr_elem is
   applied to all of its constituant parts (for instance if <anode> is a LIST, 
   then the car and cdr of that LIST).  The decrementing is done in a depth
   first fashion so that the pointers to the other constituant parts are
   still there when one of the constituant parts is done decrementing.
*/
void decr_elem(anode)
NODE *anode;
{
	int i;

	if (anode->ptr_cnt < 1)
		return;
	if (anode->ptr_cnt != 1)
		{
		decr_ref(anode);
		return;
		}

	switch (Type_Of(anode))
		{
		case N_NIL:	return;
		case N_TRUE:	return;
		case N_LIST:
				decr_list(anode);
				return;
		case N_SYMBOL:
		case N_STRING:
				free(symbol_name(anode));
				break;
		case N_ARRAY:	
				anode->ptr_cnt=0;
				if (symbolp(array_ref(anode,0)))
				{
				char *name;

				name = symbol_name(array_ref(anode,0));
				}
				for (i=0;i<array_max(anode);i++)
					{
					decr_elem(array_ref(anode,i));
					}
				free(array_st(anode)); 
				anode->ptr_cnt=1;
				break;
					
		case N_USERDATA:
		  (*(userdef_methods(anode)->destroy))(userdef_data(anode));
		  break;

		case N_INT:	
		case N_REAL:
		default: break;
		}
	decr_ref(anode);
}
