/****************************************************************************
 * interp.c
 * Author Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <string.h>
#include "alisp.h"
#include "parse.h"

NODE *RESULT,*EVAL_ENV,*LAMBDA_KEY;
int debug=0;

int INTERP_ERROR=0;
int ENDSESSION=0;
int PRINT_FLAG;

/*
   PURPOSE:   The top level function, this takes a stream, parses it into a
   lisp format, and evaluates it.
   INPUT:     <stream_name> is the name of the file to read from, - if 
	      the stream is stdio, or null (also stdio).

	      <stInFlag> is non-zero if stream-name is non-null, indicating 
	      that the user did type in a file name (or -), and zero if the
	      user only typed in "alisp"

   OUTPUT:    will return the alisp format NODE pointer that represents the
	      last evaluation of <stream_name>

   EXPL:      Many little control variables keep a loop calling 
	      eval_new_program over and over on the latest parse until an
	      end of session is signaled.
*/
NODE *ilisp(list)
NODE *list;
{
	NODE *result;

	INTERP_ERROR=1;
/*	if (SESSION_END)	return(NIL);  */
	incr_ref(list);
	result = eval_ilisp(list);
	incr_ref(result);
	decr_elem(list);
	if (PRINT_FLAG)
		{
		fprintf(stderr,"\n  ");
		print_out(result);
		if (Infile(f_stack) == stdin) fprintf(stderr,"\n\n:>");
		else fprintf(stderr,"\n");
		};
	decr_elem(result);
	return(result);
}

/*
    Used by apply_setf when the first argument is a primitive.  
    It only knows of a few valid primitive types, however.
    A thorough implementation would add all the primitive symbols to
    the symbol table and put proper setf methods (i.e. pointers to
    functions) in their p-lists.  This would require a more advanced
    symbol table than we currently have
*/
void setf_prim(prim,arglist,value)
NODE *prim,*arglist,*value;
{

	int  index;

/*	incr_ref(value);   */
	arglist = eval_list(arglist);
	incr_ref(arglist);

	switch(symbol_type(prim))
		{
		case K_CAR:
			    decr_elem(car(car(arglist)));
 			    rplaca(car(arglist),value);
	    	    	    break;
			   
		case K_CDR:
			    decr_elem(cdr(car(arglist)));
			    rplacd(car(arglist),value);
			    break;
		           
		case K_CDDR:
			    decr_elem(cdr(cdr(car(arglist))));
			    rplacd(cdr(car(arglist)),value);
			    break;
		case K_CADR:
			    decr_elem(car(cdr(car(arglist))));
			    rplaca(cdr(car(arglist)),value);
			    break;
		case K_CDAR:
			    decr_elem(cdr(car(car(arglist))));
			    rplacd(car(car(arglist)),value);
			    break;
		case K_CAAR:
			    decr_elem(car(car(car(arglist))));
			    rplaca(car(car(arglist)),value);
			    break;
		case K_AREF:
			    index = getfixnum(car(cdr(arglist)));
			    decr_elem(array_ref(car(arglist), index));
			    array_ref(car(arglist),index) = value;
			    break;
		default:
			    fprintf(stderr, "Unknown  Prim setf type \n");
			    break;

		}
	decr_elem(arglist);
}

/* 
   PURPOSE:  acts like the normal add-setf-method macro in common lisp
   INPUT:  <location> should evaluate to the pointer which is to be altered
	   to point at <value>.  This is awkward, since we want to alter the 
	   pointer, not what it points to.  
   OUTPUT: <value>
   GLOBAL VARIABLE ALTERED: the EVAL_ENV is altered.
*/

NODE *apply_setf(location,value)
NODE *location,*value;
{
  NODE *result,*place;
  
  if (debug)
    {
      fprintf(stderr,"Enter apply setf\n");
      print_out(location);
      fprintf(stderr,"::\n");
      print_out(value);
    }
  if ( (Type_Of(location) != N_LIST) )
    {
      fprintf(stderr,"ERROR, bad location for setf:");
      print_out(location);
      return(NIL);
    }
  else if ( is_prim(car(location)) )
    {
      setf_prim(car(location),cdr(location),value);
      result = value;
    }
  else	/*  It should be an accessor created by defstruct  */
    /* or an access function for a user data object. */
    /* if not, it will halt execution */
    {
      if (!symbolp(car(location)))
	{
	  fprintf(stderr,"Error, bad location for setf");
	  print_out(location);
	  exit(0);
	}
      place = car(location);
      location = cdr(location);
      if (!consp(location))
	{
	  fprintf(stderr,
		  "Error, bad argument in setf method");
	  print_out(location);
	  exit(0);
	}
      location = eval_ilisp(car(location));
      if (Type_Of(location) == N_USERDATA) { /* user data structure */
	if (userdef_methods(location)->setf_method)
	  (*(userdef_methods(location)->setf_method))(place,location,value);
	else {
	  fprintf(stderr,"Error, the following function isn't setf-able: ");
	  print_out(place);
	  return(NIL);
	}
      }
      else if (arrayp(location)) {
	place= get_setf_method(place);
	decr_elem(array_ref(location,getfixnum(place)));
	array_ref(location,getfixnum(place)) = value;
	}
      else {
	fprintf(stderr,
		"Error, bad argument in setf method");
	print_out(location);
      }
      result = value;
    }
  
  return(result);
}


/* 
   PURPOSE:  This function evaluates every element of <deflist>

   INPUT: 
	  <deflist> is of the form ( exp-1 exp2 ... exp-n) where each
	  <exp-x> is evaluatable by eval_ilisp
   OUTPUT:
	   the evaluation of the last expression, <exp-n>
*/
NODE *eval_program(deflist)
NODE *deflist;
{
	NODE *defs,*result;

	/* could be an empty list	*/
	if (null(deflist))	return(NIL);
	for (defs=deflist; consp(defs);defs=cdr(defs))
		result = eval_ilisp(car(defs));
	/* could be a non-nil terminated list */
	if (!null(defs))
		result = eval_ilisp(defs);

	return(result);
}

/*
    Called by eval_ilisp when the expression being evaulated is a list
    with it's car being a symbol of symbol_type equal to one of the
    lisp construct names
    <type> is the type of construct and <exp> is the body of the construct.
    Returns the evaluation of the construct
    NOTE:  eval_Construct has a capital C to make sure no conflicts occur
	   with eval_cond.
*/
NODE *eval_Construct(type,exp)
KEY_TYPE type;
NODE *exp;
{
	NODE *arg1,*arg2;

	if (debug)
		{
		fprintf(stderr,"Enter EVAL-CONST:\n");
		print_out(exp);
		fprintf(stderr,"::\n");
		}
	switch(type)
		{
		case K_SETF:
			arg2 = nth1(exp);
			arg1 = eval_ilisp(nth2(exp));
/*
   Must incr_ref arg1 now so it won't be freed when the position of the setf
   is evaluated.  It should be incremented anyway, since it is going to be
   pointed to by the position of the setf 
*/
			incr_ref(arg1);
			arg1 = apply_setf(arg2,arg1);
			break;
		case K_SETQ:
			arg1 = eval_ilisp(nth2(exp));
			arg2 = assoc(nth1(exp),EVAL_ENV);
			if (null(arg2))
				{
				Bind(nth1(exp),arg1);
				}
			else
				{
				incr_ref(arg1);
				decr_elem(cdr(arg2)); 
				rplacd(arg2,arg1);   
				};
			break;
		case K_DEFUN:
			Bind( nth1(exp), cons( LAMBDA_KEY, cdr(exp) ) );
			arg1 = nth1(exp);
			break;
		case K_QUOTE:
		     arg1 = car(exp);
		     break;
		case K_LAMBDA:
		     /* form:	    (lambda vars       body ...   )  args  
		     arg1 = apply_lambda( car(exp), cdr(exp), NIL );  */
		     fprintf(stderr,"Error: unknown primitive or conditional:");
	     	     print_out(car(exp));
		     arg1 = exp;
		     break;
		case K_COND:
	   	     arg1 = eval_cond( exp );
		     break;
		case K_IF:
		     arg1 = eval_if(exp);
		     break;
		case K_LET:
		     /* form: (let      vars      body ...)  */
	   	     arg1 = eval_let( car(exp),cdr(exp),FALSE );
		     break;
		case K_LETSTAR:
		     /* form: (let*     vars      body ...)  */
	   	     arg1 = eval_let( car(exp),cdr(exp),TRUE );
		     break;
	        case K_DO:
		     /* form: (do      vars       end-form   body ...)  */
	   	     arg1 = eval_do( car(exp), nth2(exp), cdr(cdr(exp)),
	                              FALSE );
		     break;
	        case K_DOSTAR:
		     /* form: (do*     vars       end-form   body ...)  */
	   	     arg1 = eval_do( car(exp), nth2(exp), cdr(cdr(exp)),
	                              TRUE );
		     break;
		case K_PROGN:
		     arg1 = eval_program(exp);
		     break;
		case K_AND:
	   	     arg1 = eval_and( exp );
		     break;
		case K_OR:
	   	     arg1 = eval_or( exp );
		     break;
		case K_NOT:
	   	     arg1 = eval_not( car(exp) );
		     break;
/*		K_FUNARG-CLOSURE?:  Defered implementation */
		default:
			fprintf(stderr,"Unknown Construct:");
			print_out(exp);
			fprintf(stderr,":\n");
			arg1 = NIL;
			break;
	    }
	return(arg1);
}


/* 
     Simulates  a package called system:.  It allows the adding of setf
     methods and creation of structures from the c side.  The only 
     symbols/commands in this package are currently make-structure and
     add-setf-method.  It is not intended that the user would use these
     directly.  The parser will put these in the appropriate places
     when it is parsing a defstruct expression
*/
NODE *eval_system(name,arglist)
NODE *name,*arglist;
{
	NODE *result;
	int size,i;

	if (name == lookup_symbol("make-structure"))
		{
		size = (int) getfixnum(car(arglist));
		arglist = cdr(arglist);
		result = new_node(N_ARRAY);
		array_max(result) = size;
		array_st(result) = (NODE **) malloc(size*sizeof(NODE *));

		array_ref(result,0) = car(arglist);
		incr_ref(car(arglist));
		arglist = cdr(arglist);
		for (i=1;i<size;i++)
			{
			array_ref(result,i) = car(arglist);
			incr_ref(array_ref(result,i));
			arglist = cdr(arglist);
			}
		}
	else if (name == lookup_symbol("access"))
		{
		if ( (!arrayp(result=car(arglist))) ||
		     (array_ref(result,0) != car(cdr(cdr(arglist)))) )
		   {
		   fprintf(stderr,"Error:attempt made to access:\n");
		   print_out(result);
		   if (arrayp(result))
		     print_out(array_ref(result,0));
		   fprintf(stderr,"\nAs a structure of type %s\n",
			   symbol_name(car(cdr(cdr(arglist)))));
		   fprintf(stderr,"Returning Zero\n");
		   return(NIL);
		   }
		return(array_ref(result,getfixnum(car(cdr(arglist))))); 
		}
	else if (name == lookup_symbol("add-setf-method"))
		{
		result = car(arglist);
		arglist = cdr(arglist);
		add_setf_method(result,car(arglist));
		}
	else
		{
		fprintf(stderr,"Error:Unknown system call:\n");
		print_out(name);
		fprintf(stderr,"Type:%d   \n",(int)symbol_type(name));
		fprintf(stderr,"\n");
		result = NIL;
		}
	return(result);
}

/*
     The top level evaluation function.  It returns the evaluation of
     <exp>.  It is by far the single most called function
*/
NODE *eval_ilisp(exp)
NODE *exp;
{
	NODE *result;

	if (debug)
		{
		fprintf(stderr,"Enter EVAL-ILISP:\n");
		print_out(exp);
		fprintf(stderr,"\n");
		}
	if (SESSION_END) { /* Don't evaluate the expression */
	  return(NIL);
	}
	switch( Type_Of(exp) )
		{
		case N_TRUE:
		case N_NIL:
		case N_INT:
		case N_REAL:
		case N_STRING:
		case N_ARRAY:
		case N_USERDATA:
		     result = exp;
		     break;
		case N_SYMBOL:
		     if (keywordp(exp))
			result = exp;
		     else
		        result = lookup(exp);				
		     break;
		case N_LIST:
		     {
		     NODE *theArgs,*functor;

		     functor = car(exp);
		     if (!good_functor(functor))
			{
			functor = eval_ilisp(functor);
			incr_ref(functor);
		     	theArgs = eval_list(cdr(exp));
			incr_ref(theArgs);
		     	result = apply_ilisp(functor,theArgs);
			incr_ref(result);
			decr_elem(theArgs);
/*			decr_elem(functor); */
			if (RESULT->ptr_cnt != 1)
				decr_ref(RESULT);
			else
				decr_elem(RESULT);
			RESULT = result;
			return(result);
			}
		     if (is_callout(functor))
			{
		     	theArgs = eval_list(cdr(exp));
			incr_ref(theArgs);
		     	result = apply_ilisp(functor,theArgs);
			incr_ref(result);
			decr_elem(theArgs);
			if (RESULT->ptr_cnt != 1)
				decr_ref(RESULT);
			else
				decr_elem(RESULT);
			RESULT = result;
			return(result);
			}
                     else if (is_lambda(functor))
			result=exp;
		     else if (is_constr(functor))
			result = eval_Construct(symbol_type(functor),cdr(exp));
		     else if (is_systemcall(functor))
			{
		     	theArgs = eval_list(cdr(exp));
			incr_ref(theArgs);
			result = eval_system(functor,theArgs);
			incr_ref(result);
			decr_elem(theArgs);
			if (RESULT->ptr_cnt != 1)
				decr_ref(RESULT);
			else
				decr_elem(RESULT);
			RESULT = result;
			return(result);
			}
		     else
			{
		     	theArgs = eval_list(cdr(exp));
			incr_ref(theArgs);
		     	result = apply_ilisp(functor,theArgs);
			incr_ref(result);
			decr_elem(theArgs);
			if (RESULT->ptr_cnt != 1)
				decr_ref(RESULT);
			else
				decr_elem(RESULT);
			RESULT = result;
			return(result);
			}
		     break;
		     }
		default:
		     handle_error("Unknown s-expression type");
		     print_out(exp);
		     result = NIL;
		     break;
		}
	incr_ref(result);
	if (RESULT->ptr_cnt != 1)
		decr_ref(RESULT);
	else
		decr_elem(RESULT);
	RESULT = result;
	return(result);

}
/*
   PURPOSE:  the top level apply function.  Given a function, a list of 
   arguments, and an environment, it evaluates that function with those
   arguments

   INPUT: <fun> is one of the following:
      a list of the form (lambda name vars body), 
      a symbol whose symbol_type is a primitive, a call_out, or a system-call,
      or an expression that evaluates to one of the above. 

      <arguments> is a nil terminated list of evaluated arguments to <fun>

   OUTPUT: the result of <fun> applied ot <arguments>
*/
NODE *apply_ilisp(fun, arguments)
NODE *fun, *arguments;
{
	NODE *result;

	if (debug)
		{
		fprintf(stderr,"Enter APPLY ILISP:\n");
		print_out(fun);
		fprintf(stderr,":1\n");
		print_out(arguments);
		fprintf(stderr,":\n");
		}
	if (null(fun))
		{
		result = handle_error("UNKNOWN FUNC  "); 
		fprintf(stderr,"A null function\n");
		print_out(arguments);
		}
	else if (is_prim(fun))
		{
		result = apply_prim( symbol_type(fun), arguments );
		}
	else if (is_callout(fun))
		{
		result = apply_callOut( callout_name(fun)+1,arguments );
		}
	else if (symbolp(fun))
		{
		result = apply_ilisp( eval_ilisp(fun), arguments);
		}
	else if ( consp(fun) && is_lambda(car(fun)) )
		{
		/* form:     (lambda   (vars)     body ...)      args  */
		result = apply_lambda( nth2(fun), cdr(cdr(fun)), arguments );
		}

	else if (listp(fun))
		{
		result = apply_ilisp( eval_ilisp(fun), arguments );
		}
	else 
		result = handle_error( "apply_ilisp");

	return( result );
}
