/****************************************************************************
 * dphigs_ren.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
#include <stdio.h>
#include <math.h>
#ifdef ultrix
#include <X11/Intrinsic.h>
#include <PHIGS/phigs.h>
#include <PHIGS/phigs_defs.h>
#endif
#ifdef VMS
#include "SYS$LIBRARY:decw$include/Intrinsic.h"
#include "SYS$LIBRARY:PHIGS.H"
#include "SYS$LIBRARY:PHIGS_DEFS.H"
#endif
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
#include "assist.h"
 
/* Notes-
   -add some status returns
   -some implementations might use phigs_ws_type_create to get double buffering
   -some implementations may offer other HLHSR modes worth trying
*/

/* 
The following definitions are needed to work around variances in 
Phigs implementations between platforms.  The basic version was done
on a Sun under Sun Phigs; this port uses these macros to ease conversion
to DEC Phigs+.
*/
#define PDEFAULT_MEM_SIZE 0
#define Plightsrcbundl PPlightsrcbundl
typedef struct {
	Pint type;
	Pcobundl val;
	} Pgcolr;
#define cbndl_r( thisbundle ) (thisbundle).rgb.r
#define cbndl_g( thisbundle ) (thisbundle).rgb.g
#define cbndl_b( thisbundle ) (thisbundle).rgb.b
#define gcolr_r( thiscolr ) cbndl_r( (thiscolr).val )
#define gcolr_g( thiscolr ) cbndl_g( (thiscolr).val )
#define gcolr_b( thiscolr ) cbndl_b( (thiscolr).val )

/* Needed external definitions */
extern char *malloc(), *realloc();

/* 
Variables to provide external hooks to the phigs workstation.
See ren_setup(...) for usage.  phigs_conn_id is used only there;
phigs_wsid is the workstation id used throughout the renderer.
window_id, top_window_id, and display_ptr are used under the
Motif and Open Look user interfaces, again in ren_setup().
The compile time option controlled by INCLUDE_RLE_SAVER, which
allows frames to be saved as Utah Raster Library (RLE) format
files, also makes use of display_ptr and window_id.
*/
Pconnid phigs_conn_id;
Pint phigs_wsid= 1;
Window window_id= 0, top_window_id= 0;
Display *display_ptr= 0;

/* Phigs identifier constants */
#define ROOT -1
#define HEAD -2
#define VIEWINDEX 1

/* Renderer state */
typedef enum { DEFINING, TRAVERSING } State;
static State current_state= DEFINING;

/* Light source maintenance information */
#define MAX_LIGHTS 16  /* probably more than phigs will use */
static int current_light= 1;
static Pint light_sources[MAX_LIGHTS]=
  { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16 };

/* Symbols defined by setup and used in parsing attribute lists */
static Symbol depth_cue_symbol, color_symbol, backcull_symbol, 
	     text_height_symbol, material_symbol;

static void set_area_properties();

/* Various things needed to support saving of frames as RLE files */
#ifdef INCLUDE_RLE_SAVER
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <rle.h>
#define COLOR_TABLE_SIZE         256
#define MAX_INTENS     65535L
static int save_frames_flag= 0;
static rle_hdr   the_hdr;
static Visual *visual_type;
#endif

#ifdef INCLUDE_RLE_SAVER
/* The following code implements saving of images as Utah Raster Library
 * (RLE) files.  The author is Curt Schroeder, who provided it to network
 * news.
 */

static void init_frame_saver (file_name)
char *file_name;
{
  int primary_screen;
  int depth;
  struct esc_data_struct
    {
    int     num_ints;
    int     num_floats;
    int     num_strings;
    int     *ints;
    float   *floats;
    int     *string_sizes;
    char    **strings;
    } out_esc_data, in_esc_data;
  Pint esc_id = PHIGS$K_ESC_INQ_WINDOW_IDS;
  Pint out_size;
  Pint int_buffer[2];
  Pint in_size;
  int class;
  Plimit window_bound;
#ifdef never
  float xmeters_per_pixel= 0.0002531, ymeters_per_pixel= 0.0002541;

  /* Reduce the window size */
  window_bound.xmin= 0.0;
  window_bound.ymin= 100*ymeters_per_pixel;
  window_bound.xmax= 513*xmeters_per_pixel;
  window_bound.ymax= 613*ymeters_per_pixel;
  psetwsviewport(phigs_wsid,&window_bound);
#endif

  /* init data records for DEC Phigs escape call */
  in_esc_data.num_ints = 1;
  in_esc_data.num_floats = 0;
  in_esc_data.num_strings = 0;
  in_esc_data.ints = &phigs_wsid;
  in_esc_data.floats = NULL;
  in_esc_data.string_sizes = NULL;
  in_esc_data.strings = NULL;
  in_size = sizeof(struct esc_data_struct);
   
  out_esc_data.num_ints = 2;
  out_esc_data.num_floats = 0;
  out_esc_data.num_strings = 0;
  out_esc_data.ints = int_buffer;
  out_esc_data.floats = NULL;
  out_esc_data.string_sizes = NULL;
  out_esc_data.strings = NULL;
  out_size = sizeof(struct esc_data_struct);

  pescape(esc_id,&in_esc_data,in_size,&out_esc_data,&out_size);

  /* initialization */
  display_ptr = (Display *)int_buffer[0];
  primary_screen = DefaultScreen(display_ptr);

  /* display visiual info */
  depth = DefaultDepth(display_ptr,primary_screen);
  visual_type = DefaultVisual(display_ptr,primary_screen);
  class = visual_type->class;
  if (depth == 1)
    { fprintf(stderr,"one-plane monochrome\n"); }
  else 
    if (class == PseudoColor )
      { fprintf(stderr,"multi-plane color\n"); }
    else
      if (class == GrayScale)
        { fprintf(stderr,"multi-plane monochrome\n"); }
      else 
        if (class == DirectColor)
          { fprintf(stderr,"direct color\n"); }
        else
          if (class == TrueColor)
            { fprintf(stderr,"direct color, unchangeable color map\n"); }

  /* define window_id */
  window_id = (Window)int_buffer[1];

  /* Open the RLE file.  An open error will cause an exit. */
  fprintf(stderr,"file name is <%s>\n",file_name);
  the_hdr.rle_file = 
    rle_open_f(gargv[0],file_name,"wb"); /* alisp provides gargv */
}

static void save_rle_frame (display, drawable)
  Display *display;
  Window drawable;
{
  XColor table[COLOR_TABLE_SIZE];
  int i, j, x, y;
  Window root;
  unsigned int width, height, border_width, depth;
  XImage *image;
  long pixel;
  XColor *colorrow;
  XWindowAttributes window_attr;
  rle_pixel **scan_line;
  rle_map   cmap[768];
  float meters_per_pixel= 0.000254;
  float banner_height= 0.00754;

  /* init the RLE header */
  the_hdr.bg_color = NULL;
  the_hdr.alpha = 0;
  the_hdr.background = 0;
  the_hdr.xmin = 0;
  the_hdr.ymin = 0;
  the_hdr.cmaplen = 8;
  the_hdr.comments = NULL;
  RLE_SET_BIT(the_hdr,RLE_RED);
  RLE_SET_BIT(the_hdr,RLE_GREEN);
  RLE_SET_BIT(the_hdr,RLE_BLUE);
  RLE_CLR_BIT(the_hdr,RLE_ALPHA);

  /* Get window attributes */
  XGetWindowAttributes(display,drawable,&window_attr);

  /* init variables */
  if (the_hdr.rle_file != NULL) {
    if (visual_type->class == PseudoColor) {

      /* get the color map */
      for (i = 0; i < COLOR_TABLE_SIZE; i++)
	table[i].pixel = i;
      XQueryColors(display,window_attr.colormap,table,COLOR_TABLE_SIZE);
      
      /* make certain no RGB values are out of range and store in cmap */
      for (i = 0; i < COLOR_TABLE_SIZE; i++)
	{
	  if (table[i].red > MAX_INTENS)
	    table[i].red = 0;
	  cmap[i] = table[i].red;
	  
	  if (table[i].green > MAX_INTENS)
	    table[i].green = 0;
	  cmap[256 + i] = table[i].green;

	  if (table[i].blue > MAX_INTENS)
	    table[i].blue = 0;
	  cmap[512 + i] = table[i].blue;
	}
      
      /* get the image and store it and it's color map */
      if (XGetGeometry(display,drawable,&root,&x,&y,&width,&height,
		       &border_width,&depth))
	{
	  fprintf(stderr,"image at (%d,%d) %d x %d will be fetched\n",
		 x,y,width,height);
	  image = XGetImage(display,drawable,x,y,width,height,-1,ZPixmap);
	  
	  /* finish initializing the RLE header */
	  the_hdr.ncolors = 1;
	  the_hdr.ncmap = 3;
	  the_hdr.cmap = cmap;
	  the_hdr.xmax = width - 1;
	  the_hdr.ymax = height - 1;
	  
	  /* write out the image- alisp provides gargv */
	  rle_addhist( gargv, (rle_hdr *)0, &the_hdr );
	  rle_put_setup(&the_hdr);
	  rle_row_alloc(&the_hdr,&scan_line);
	  
	  for (i = 0; i < height; i++)
	    {
	      for (j = 0; j < width; j++)
		{
		  pixel = XGetPixel(image,j,(height - 1 - i));
		  scan_line[0][j] = (rle_pixel)pixel;
		}  /* load each pixel in the row into RLE row structure */
	      rle_putrow(scan_line,width,&the_hdr);
	    }
	  
	  /* destroy the image structure */
	  XDestroyImage(image);
	  
	  /* free up the row */
	  rle_row_free( &the_hdr, scan_line );
	}
    }
    else if ((visual_type->class==TrueColor) || 
	     (visual_type->class==DirectColor)) {

      /* get the image and store the image */
      if (XGetGeometry(display,drawable,&root,&x,&y,&width,&height,
		       &border_width,&depth))
	{
	  /* Adjust dimensions to compensate for part of image the banner
	   * pushes off screen, then get the image. 
	   */
	  height -= banner_height/meters_per_pixel;
	  width -= banner_height/meters_per_pixel;
	  fprintf(stderr,"image at (%d,%d) %d x %d will be fetched\n",
		 x,y,width,height);
	  image = XGetImage(display,drawable,x,y,width,height,
			    AllPlanes,XYPixmap);
	  
	  /* finish initializing the RLE header */
	  the_hdr.ncolors = 3;
	  the_hdr.ncmap = 0;
	  the_hdr.cmap = NULL;
	  the_hdr.xmax = width - 1;
	  the_hdr.ymax = height - 1;

	  /* Allocate a row of colors */
	  if ( !(colorrow= (XColor *)malloc( width*sizeof(XColor) )) ) 
	    ger_fatal("save_rle_frame: unable to allocate %d XColors!",
		      width);
	  
	  /* write out the image- alisp provides gargv */
	  rle_addhist( gargv, (rle_hdr *)0, &the_hdr );
	  rle_put_setup(&the_hdr);
	  rle_row_alloc(&the_hdr,&scan_line);
	  
	  for (i = 0; i < height; i++)
	    {
	      for (j = 0; j < width; j++)
		{
		  colorrow[j].pixel = XGetPixel(image,j,(height - 1 - i));
		}
	      XQueryColors(display,window_attr.colormap,colorrow,width);
	      for (j = 0; j < width; j++)
		{
		  scan_line[0][j]= 255*colorrow[j].red/MAX_INTENS;
		  scan_line[1][j]= 255*colorrow[j].green/MAX_INTENS;
		  scan_line[2][j]= 255*colorrow[j].blue/MAX_INTENS;
		}  /* load each pixel in the row into RLE row structure */
	      rle_putrow(scan_line,width,&the_hdr);
	    }

	  /* destroy the image structure */
	  XDestroyImage(image);
	  
	  /* free up the row */
	  rle_row_free( &the_hdr, scan_line );
	  free( (char *)colorrow );
	}
    }
  rle_puteof(&the_hdr);
  }
}

static void close_frame_saver()
/* This routine shuts down the RLE file generation machanism */
{
  /* close the image file */
  if ( the_hdr.rle_file ) {
    fclose( the_hdr.rle_file );
    the_hdr.rle_file= NULL;
    fprintf(stderr,"image file closed.\n");
  }
}

#endif /* INCLUDE_RLE_SAVER */

static Pvector3 *get_phigs_vector(avect)
Vector avect;
{
	Pvector3 *result;

	result = (Pvector3 *) malloc(sizeof(Pvector3));
	if (!result) 
		ger_fatal("get_phigs_vector: can't allocate a Pvector3!");
	result->x= (Pfloat)vector_x( avect );
	result->y= (Pfloat)vector_y( avect );
	result->z= (Pfloat)vector_z( avect );

	return(result);
}

static Ppoint3 *get_phigs_point(apoint)
Point apoint;
{
	Ppoint3 *result;

	result = (Ppoint3 *) malloc(sizeof(Ppoint3));
	if (!result) 
		ger_fatal("get_phigs_point: can't allocate a Ppoint3!");
	result->x= (Pfloat)point_x( apoint );
	result->y= (Pfloat)point_y( apoint );
	result->z= (Pfloat)point_z( apoint );

	return(result);
}

static Pgcolr *get_phigs_color(clrstruct)
Color clrstruct;
{
	Pgcolr *result;

	ger_debug("get_phigs_color:  getting color");

	result = (Pgcolr *) malloc(sizeof(Pgcolr));
	if (!result) ger_fatal("get_phigs_color: can't allocate a Pgcolr!");

        result->type= PCM_RGB;
	gcolr_r( *result )= color_red(clrstruct);
	gcolr_g( *result )= color_green(clrstruct);
	gcolr_b( *result )= color_blue(clrstruct);

	return(result);
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* This routine dumps a Phigs object. */
void ren_dump(thisgob)
Gob thisgob;
{
	int thisid;
 
	thisid= gob_idnum( thisgob );
	fprintf(stderr,"ren_dump: Dumping gob number %d\n",thisid);
fprintf(stderr,"Dump routine not yet implemented\n");

}

static Ppoint3 *get_loc_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations to a vertex list.
*/
{
	Vertex vertex;
	Ppoint3 *vertexlist, *vtx_copy;
                              
	ger_debug("get_loc_vtx: getting vertex list, locs only.");
 
	if ( !(vertexlist= (Ppoint3 *)malloc( vertexcount*sizeof(Ppoint3) ) ))
	  ger_fatal( "get_loc_vtx: unable to allocate %d Ppoint3's!",
		    vertexcount );
 
	vtx_copy= vertexlist;
	while ( vertexcount-- )
	  {
	    vertex= first_vertex( vlist );
	    vlist= rest_of_vertices( vlist );
	    vtx_copy->x= (Pfloat)vertex_x(vertex);
	    vtx_copy->y= (Pfloat)vertex_y(vertex);
	    vtx_copy->z= (Pfloat)vertex_z(vertex);
	    vtx_copy++;
	  }

	return( vertexlist );
} 

static Pcobundl *get_vtx_clrs( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine returns a list of colors extracted from the nodes of
a vertex list.
*/
{
	Vertex vertex;
	Color thiscolor;
	Pcobundl *colorlist, *clr_copy;
                              
	ger_debug("get_vtx_clrs: getting vertex list colors.");
 
	if ( null( vertex_color( first_vertex(vlist) ) ) )
	  ger_fatal("get_vtx_clrs: vertex list has no colors!\n");

	if ( !(colorlist= (Pcobundl *)malloc( vertexcount*sizeof(Pcobundl) ) ))
		ger_fatal( "get_vtx_clrs: unable to allocate %d Pcobundl's!",
			vertexcount );
 
	clr_copy= colorlist;
	while ( vertexcount-- )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		if ( null( thiscolor= vertex_color(vertex) ) )
		  ger_fatal("get_vtx_clrs: vertex has no color data!");
		clr_copy->rgb.r= (Pfloat)color_red(thiscolor);
		clr_copy->rgb.g= (Pfloat)color_green(thiscolor);
		clr_copy->rgb.b= (Pfloat)color_blue(thiscolor);
		clr_copy++;
		}
 
	return( colorlist );
} 

static Pvector3 *get_vtx_nrmls( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine returns a list of normals extracted from the nodes of
a vertex list.
*/
{
	Vertex vertex;
	Vector thisnormal;
	Pvector3 *normlist, *nrml_copy;
                              
	ger_debug("get_vtx_nrmls: getting vertex list normals.");
 
	if ( null( vertex_normal( first_vertex(vlist) ) ) )
	  ger_fatal("get_vtx_nrmls: vertex list has no normals!\n");

	if ( !(normlist= (Pvector3 *)malloc( vertexcount*sizeof(Pvector3) ) ))
	     ger_fatal( "get_vtx_nrmls: unable to allocate %d Pvector3's!",
			vertexcount );
 
	nrml_copy= normlist;
	while ( vertexcount-- )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		if ( null(thisnormal= vertex_normal(vertex) ) )
		  ger_fatal("get_vtx_nrmls: vertex has no normal data!");
		nrml_copy->x= (Pfloat)vector_x(thisnormal);
		nrml_copy->y= (Pfloat)vector_y(thisnormal);
		nrml_copy->z= (Pfloat)vector_z(thisnormal);
	        nrml_copy++;
		}
 
	return( normlist );
} 

void ren_sphere()
/*
This routine constructs a sphere object.  The sphere is of radius 1,
and centered at the origin.
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_sphere");

	ast_sphere(ren_mesh);
}

void ren_cylinder()
/*
This routine constructs a cylinder object.  The cylinder is of radius
1, and is aligned with the z axis.  The ends of the cylinder are at z=0.0
and z=1.0 .
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_cylinder");

	ast_cylinder(ren_mesh);
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
This routine constructs a torus object from the args passed it.  The 
arguments should be the major and minor radii of the torus respectively.  
The torus is drawn so that the hole lies along the z axis.
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_torus");

	ast_torus( bigradius, smallradius, ren_mesh );
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine constructs a text object from the args passed it.  
The argument list should contain a point, two vectors to define the 
u and v directions of the text plane, and a text string.
*/
{
        Ppoint3 *text_pt;
	Pvector3 dir[2];

	if (current_state != DEFINING) return;

	ger_debug("ren_text");

	text_pt= get_phigs_point(txtpoint);
	dir[0].x= vector_x(uvec);
	dir[0].y= vector_y(uvec);
	dir[0].z= vector_z(uvec);
	dir[1].x= vector_x(vvec);
	dir[1].y= vector_y(vvec);
	dir[1].z= vector_z(vvec);
	ptext3( text_pt, dir, txtstring );

	free( (char *)text_pt );
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polyline object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polyline, which may be concave but 
should not be self-intersecting.  The attributes (color and normal) at 
all vertices are assumed to be the same as those of the first vertex.  
*/
{
  Vertex thisvertex;
  Ppoint3 *vertices;
  PPcovallst clrdatalst;
  Pint vertextype;

  if (current_state != DEFINING) return;

  ger_debug("ren_polyline");
  
  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  This will determine the vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( !null(vertex_color( thisvertex )) )
    vertextype= PPVERT_COLOUR;
  else {
    clrdatalst.direct_colours= (Pcobundl *)0;
    vertextype= PPVERT_NONE;
  }
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  vertices= get_loc_vtx( vlist, count );
  switch ( (int)vertextype ) {
  case (int)PPVERT_NONE: break;
  case (int)PPVERT_COLOUR:
    clrdatalst.direct_colours= get_vtx_clrs( vlist, count ); break;
  };
 
  /*  Create the polyline.  It goes into the already open structure. */
  pppolylineset3data( vertextype, PCM_RGB, 1, &count, vertices, clrdatalst );
  
  /* Free memory */
  free( (char *)vertices );
  if (clrdatalst.direct_colours) free( (char *)clrdatalst.direct_colours );
}
 
void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polymarker object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polymarker.  The attributes (color 
and normal) at all vertices are assumed to be the same as those of the 
first vertex.  
*/
{
  Vertex thisvertex;
  Ppoint3 *vertices;
  Pcobundl *colors;
  PPcoval thiscolor;
  int i, has_colors= 0;
  
  if (current_state != DEFINING) return;
  
  ger_debug("ren_polymarker");
  
  /* Check for color information */
  if ( !null( vertex_color( first_vertex(vlist) ) ) ) has_colors= 1;
  
  /* Allocate and extract vertex information */
  vertices= get_loc_vtx( vlist, count );
  if (has_colors) colors= get_vtx_clrs( vlist, count );
  
  /* Create the points.  They go into the already open structure,
   * either all at once (if there are no vertex colors) or as a
   * series of single markers.
   */
  if (!has_colors) ppolymarker3( count, vertices );
  else {
    for (i=0; i<count; i++) {
      thiscolor.direct_colour.rgb.r= colors[i].rgb.r;
      thiscolor.direct_colour.rgb.g= colors[i].rgb.g;
      thiscolor.direct_colour.rgb.b= colors[i].rgb.b;
      ppsetmarkercolour( PCM_RGB, &thiscolor );
      ppolymarker3( 1, vertices+i );
    }
  }

  /* Free memory */
  free( (char *)vertices );
  if (has_colors) free( (char *)colors );
}
 
void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polygon object from the vertices
in the list passed it.  The parameter count contains the number of
vertices in the list.  The object constructed is a single
polygon, which may be concave but should not be self-intersecting.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.   
*/
{
  Vertex thisvertex;
  Ppoint3 *vertices;
  PPcovallst clrdatalst;
  Pvector3 *normals= (Pvector3 *)0;
  Pint vertextype;
  
  if (current_state != DEFINING) return;

  ger_debug("ren_polygon");

  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  If there are no normals, we add them.  The color determines
   *  what vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( null(vertex_normal( thisvertex )) ) ast_polygon_normals( vlist, count );
  if ( !null(vertex_color( thisvertex )) )
    vertextype= PPVERT_COLOURNORMAL;
  else {
    clrdatalst.direct_colours= (Pcobundl *)0;
    vertextype= PPVERT_NORMAL;
  }
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  vertices= get_loc_vtx( vlist, count );
  switch ( (int)vertextype ) {
  case (int)PPVERT_NORMAL:
    normals= get_vtx_nrmls( vlist, count ); break;
  case (int)PPVERT_COLOURNORMAL:
    normals= get_vtx_nrmls( vlist, count );
    clrdatalst.direct_colours= get_vtx_clrs( vlist, count ); break;
  };
 
  /* Create the polygon.  It goes into the already open structure. */
  ppfillarea3data( PPSHAPE_NONCONVEX, PPFA_NONE, vertextype, (Pvector3 *)0,
		 PCM_RGB, count, (PPcoval *)0, vertices, clrdatalst, 
		 normals );
  
  /* Free memory */
  free( (char *)vertices );
  if (normals) free( (char *)normals );
  if (clrdatalst.direct_colours) free( (char *)clrdatalst.direct_colours );
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a triangle mesh object from 
the args (all vertices) in the list passed it.  The object constructed is a 
triangle strip, with each adjacent triple of vertices specifying one triangle.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.
*/
{
  Vertex thisvertex;
  Ppoint3 *vertices;
  PPcovallst clrdatalst, dummylist;
  Pvector3 *normals= (Pvector3 *)0;
  Pint vertextype;

  if (current_state != DEFINING) return;

  ger_debug("ren_triangle");

  if (count<3) {
    ger_error("ren_triangle: %d is not enough vertices; call ignored.",
	      count);
    return;
  }

  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  If there are no normals, we add them.  The color determines
   *  what vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( null(vertex_normal( thisvertex )) ) ast_triangle_normals(vlist,count);
  if ( !null(vertex_color( thisvertex )) )
    vertextype= PPVERT_COLOURNORMAL;
  else {
    clrdatalst.direct_colours= (Pcobundl *)0;
    vertextype= PPVERT_NORMAL;
  }
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  vertices= get_loc_vtx( vlist, count );
  switch ( (int)vertextype ) {
  case (int)PPVERT_NORMAL:
    normals= get_vtx_nrmls( vlist, count ); break;
  case (int)PPVERT_COLOURNORMAL:
    normals= get_vtx_nrmls( vlist, count );
    clrdatalst.direct_colours= get_vtx_clrs( vlist, count ); break;
  };
 
  /* Create the triangle strip.  It goes into the already open structure. */
  dummylist.direct_colours= (Pcobundl *)0;
  pptri3data( PPFA_NONE, vertextype, PCM_RGB, count-2, (Pvector3 *)0,
	    dummylist, vertices, clrdatalst, normals );
    
  /* Free memory */
  free( (char *)vertices );
  if (normals) free( (char *)normals );
  if (clrdatalst.direct_colours) free( (char *)clrdatalst.direct_colours );
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/* 
This routine causes the renderer to construct a general mesh object from 
the args passed it.  The attributes (color and normal) at all vertices 
are assumed to be the same as those of the first vertex.
*/
{
  Vertex thisvertex;
  Vertex_list thisfacet, vlist_copy;
  Facet_list flist_copy;
  Ppoint3 *vertices;
  PPcovallst clrdatalst, dummylist;
  Pvector3 *normals= (Pvector3 *)0;
  Pint vertextype;
  Pint *bounds, *indices, *ind_copy;
  int iloop, facet_vertices, total_references= 0;

  if (current_state != DEFINING) return;

  ger_debug("ren_mesh: %d vertices, %d facets", vcount, fcount);

  if (fcount<0) {
    ger_error("ren_mesh: invalid facet count!");
    return;
  }

  /* Just quit if there are no facets, as the DEC Phigs renderer doesn't
   * seem to handle that special case correctly.
   */
  if (fcount==0) return;

  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  If there are no normals, we add them.  The color determines
   *  what vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( null(vertex_normal( thisvertex )) ) 
    ast_mesh_normals( vlist, vcount, flist, fcount );
  if ( !null(vertex_color( thisvertex )) )
    vertextype= PPVERT_COLOURNORMAL;
  else {
    clrdatalst.direct_colours= (Pcobundl *)0;
    vertextype= PPVERT_NORMAL;
  }

  /*
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  vertices= get_loc_vtx( vlist, vcount );
  switch ( (int)vertextype ) {
  case (int)PPVERT_NORMAL:
    normals= get_vtx_nrmls( vlist, vcount ); break;
  case (int)PPVERT_COLOURNORMAL:
    normals= get_vtx_nrmls( vlist, vcount );
    clrdatalst.direct_colours= get_vtx_clrs( vlist, vcount ); break;
  };

  /* Traverse the vertex list, assigning an index to each vertex. */
  vlist_copy= vlist;
  for (iloop=0; iloop<vcount; iloop++) {
    (void)set_vertex_index( first_vertex(vlist_copy), iloop );
    vlist_copy= rest_of_vertices( vlist_copy );
  }

  /* Create the vertex index list and facet length list.  This involves 
   * walking the facets of the facet list one at a time, and building 
   * appropriate data structures for each as we go.  Unfortunately
   * we have to walk the facet list twice, once to get its length
   * and the second time to get the actual index data.
   */
  if ( !( bounds= (Pint *)malloc( fcount*sizeof(Pint) ) ) )
    ger_fatal("ren_mesh: cannot allocate %d Pint's for bounds list", fcount);
  flist_copy= flist;
  for (iloop=0; iloop<fcount; iloop++) {
    thisfacet= first_facet( flist_copy );
    flist_copy= rest_of_facets( flist_copy );
    bounds[iloop]= list_length( thisfacet );
    total_references += bounds[iloop];
  }

  if ( !( indices= (Pint *)malloc( total_references*sizeof(Pint) ) ) )
    ger_fatal("ren_mesh: cannot allocate %d Pint's for indices list", fcount);
  ind_copy= indices;
  flist_copy= flist;
  for (iloop=0; iloop<fcount; iloop++) {
    thisfacet= first_facet( flist_copy );
    flist_copy= rest_of_facets( flist_copy );
    while ( !null(thisfacet) ) {
      *ind_copy++= vertex_index( first_vertex( thisfacet ) );
      thisfacet= rest_of_vertices( thisfacet );
    }
  }

  /* Generate the mesh.  It goes into the currently open structure. */
  dummylist.direct_colours= (Pcobundl *)0;
  ppindexpolygonsdata( PPSHAPE_NONCONVEX, PPFA_NONE, vertextype,
		      PPE_NONE, PINVIS_EDGE, PCM_RGB, vcount, 
		      (Pvector3 *)0, dummylist, vertices, 
		      clrdatalst, normals, fcount, bounds, indices );
  
  /* Free memory */
  free( (char *)vertices );
  free( (char *)bounds );
  free( (char *)indices );
  if (normals) free( (char *)normals );
  if (clrdatalst.direct_colours) free( (char *)clrdatalst.direct_colours );
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a Bezier patch object from 
the args (16 vertices) in the list passed it.  The object constructed is a 
4 by 4 bicubic Bezier patch.  The attributes (color and normal) at all 
vertices are assumed to be the same as those of the first vertex.
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_bezier");
	ast_bezier( vlist, ren_mesh );
}

static void add_trans( trans )
Transformation trans;
/* Add the transformation in this gob to the current structure. */
{
	float *matrix;
 
	ger_debug("add_trans: adding transformation matrix");

	matrix= array2d_to_c( trans );
	psetlocaltran3( (Pmatrix3 *)matrix, PPOSTCONCATENATE );
	free( (char *)matrix );
}
 
static void add_attr( attrlist )
Attribute_list attrlist;
/* Add the attributes in this gob to the current structure. */
{
        Pgcolr *color;
	float tempfloat;
	Pair thispair;
	Material new_material;
 
	ger_debug("add_attr:");

	/* check for 'material attribute */ 
	if ( !null( thispair= symbol_attr( material_symbol, attrlist ) ) ) {
		new_material= pair_material(thispair);
		set_area_properties(new_material);
		};
	/* check for 'color attribute */
 	if ( !null( thispair= symbol_attr( color_symbol, attrlist ) ) ) {
		color= get_phigs_color( pair_color( thispair ) );
		ger_debug("           color RGB= (%f %f %f)",
			  gcolr_r( *color ), gcolr_g( *color ),
			  gcolr_b( *color ) );
		ppsetlinecolour( color->type, &(color->val) );
		ppsettextcolour( color->type, &(color->val) );
		ppsetmarkercolour( color->type, &(color->val) );
		ppsetintcolour( color->type, &(color->val) );
		free( (char *)color );
		};
	/* check for 'backcull attribute */
 	if ( !null( thispair= symbol_attr( backcull_symbol, attrlist ) ) ) {
		if ( pair_boolean( thispair ) ) {
			ger_debug("               backface culling on");
			ppsetfacecullmode( PPCULL_BACK );
			}
		else {
			ger_debug("               backface culling off");
			ppsetfacecullmode( PPCULL_NONE );
			}
		};
	/* check for 'text-height attribute */
 	if ( !null( thispair= symbol_attr( text_height_symbol, attrlist ) ) ) {
		tempfloat= pair_float( thispair );
		ger_debug("           Text height= %f", tempfloat);
		psetcharheight( tempfloat );
		};
}

static void add_kids( kidlist )
Child_list kidlist;
/*  
This routine adds the gobs in the 'children' slot of the given gob
to the current structure.
*/
{
	int thisid;
	Gob thiskid;
 
	ger_debug("add_kids: adding children");
 
 	while ( !null(kidlist) )
		{
		thiskid= first_child( kidlist );
		kidlist= rest_of_children( kidlist );
	    	thisid= gob_idnum( thiskid );
		ger_debug("            adding gob %d as child.",thisid);
		pexecutestruct( thisid );
		};
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It adds them to
the Phigs database, and constructs the gob dag as appropriate.
*/
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
 
	/*   Open the new structure.  The struct id is the gob id. */
	popenstruct( current_gob ); 

	/* If a transform is present, add it. */
	if ( !null(trans) ) add_trans(trans);
	
	/* If attribute modifiers are present, add them. */
	if ( !null(attr) ) add_attr(attr);

	/* 
	If this gob has primitive operators, add them.  Otherwise,
	add the children of this gob to the current structure.
	*/
	if ( !null(primitive) ) eval_function( primitive_op(primitive) );
	else add_kids( children );
 
	/* Close the new structure, adding it to the gob list. */
	pclosestruct();
}

static void create_light( lightrep )
Plightsrcbundl *lightrep;
/* This routine adds the given light to the light source rep table,
 * so that it is there to be activated by add_lights.
 */
{
  ger_debug("create_light");

  if (current_light <= MAX_LIGHTS)
    ppsetlightsrcrep( phigs_wsid, current_light++, lightrep );
  else ger_error(
       "create_light: implementation only supports %d simultaneous lights",
		MAX_LIGHTS);
}

static void clear_lights()
/* This routine resets the light source maintenance system */
{
  ger_debug("clear_lights");

  current_light= 1;
}

static void add_lights()
/* This routine takes the lighting information saved up by activate_
 * and deactivate_light, and adds an appropriate element to the
 * currently open structure.
 */
{
  Pintlst activate, deactivate;

  ger_debug("add_lights");

  activate.number= current_light - 1;
  activate.integers= light_sources;
  deactivate.number= 0;
  deactivate.integers= (Pint *)0;

  ppsetlightsrcstate( current_light-1, light_sources, 0, (Pint *)0 );
}

void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine constructs a light object, first adding a translation
and a color if necessary.  All new objects are added to the already
open structure.
*/
{
  Plightsrcbundl lightrep;
  float lightpos[4], *transformed_pos;

  if (current_state != TRAVERSING) return;

  ger_debug("ren_light");

  /* Set the location, and transform to world coordinates */
  lightpos[0]= point_x( location );
  lightpos[1]= point_y( location );
  lightpos[2]= point_z( location );
  lightpos[3]= 1.0;
  transformed_pos= matrix_vector_c( ast_get_matrix(), lightpos ); 

  lightrep.type= PPLIGHT_TYPE_INFINITE;
  lightrep.record.infinite.colour_type= PCM_RGB;
  lightrep.record.infinite.colour_value.direct_colour.rgb.r= 
    color_red( lightcolor );
  lightrep.record.infinite.colour_value.direct_colour.rgb.g= 
    color_green( lightcolor );
  lightrep.record.infinite.colour_value.direct_colour.rgb.b= 
    color_blue( lightcolor );
  lightrep.record.infinite.direction.x= -transformed_pos[0];
  lightrep.record.infinite.direction.y= -transformed_pos[1];
  lightrep.record.infinite.direction.z= -transformed_pos[2];

  create_light( &lightrep );

  free( (char *)transformed_pos );
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
This routine constructs a ambient light object, first adding a 
color if necessary.  All new objects are added to the already
open structure.
*/
{
  Plightsrcbundl lightrep;

  if (current_state != TRAVERSING) return;

  ger_debug("ren_ambient");

  lightrep.type= PPLIGHT_TYPE_AMBIENT;
  lightrep.record.ambient.colour_type= PCM_RGB;
  lightrep.record.ambient.colour_value.direct_colour.rgb.r= 
    color_red( lightcolor );
  lightrep.record.ambient.colour_value.direct_colour.rgb.g= 
    color_green( lightcolor );
  lightrep.record.ambient.colour_value.direct_colour.rgb.b= 
    color_blue( lightcolor );

  create_light( &lightrep );
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine defines a camera.  The camera goes into its own structure.
That structure is not a gob, so it doesn't go in the gob table.
*/
{
        Pviewmapping3 map;
	Pvector3 vpn;
	Pvector3 *vup;
	Ppoint3  *vrp;
	Pviewrep3 rep;
	Pint error= 0;
	float range_x, range_y, range_z, range;
	Pcobundl color_rep;
 
	ger_debug("ren_camera");

	range_x= point_x(lookat) - point_x(lookfrom);
	range_y= point_y(lookat) - point_y(lookfrom);
	range_z= point_z(lookat) - point_z(lookfrom);
	range= sqrt( range_x*range_x + range_y*range_y + range_z*range_z );

	/* Set up viewing parameters */
	map.proj= PPERSPECTIVE;
	map.prp.x= 0.0;
	map.prp.y= 0.0;
	map.prp.z= range;
	map.window.xmin= -range*tan( DegtoRad * fov/2.0 );
	map.window.xmax= -map.window.xmin;
	map.window.ymin= map.window.xmin;
	map.window.ymax= map.window.xmax;
	map.front_plane= map.prp.z + hither;
	map.back_plane= map.prp.z + yon;
	map.view_plane= 0.0;

	rep.clip_limit.xmin= map.viewport.xmin= 0.0;
	rep.clip_limit.xmax= map.viewport.xmax= 1.0;
	rep.clip_limit.ymin= map.viewport.ymin= 0.0;
	rep.clip_limit.ymax= map.viewport.ymax= 1.0;
	rep.clip_limit.zmin= map.viewport.zmin= 0.0;
	rep.clip_limit.zmax= map.viewport.zmax= 1.0;
	rep.clip_xy= rep.clip_back= rep.clip_front= PCLIP;

	vpn.x= -range_x;
	vpn.y= -range_y;
	vpn.z= -range_z;
	vup= get_phigs_vector( lookup );
	vrp= get_phigs_point( lookat );

	/* Calculate view orientation matrix and view mapping matrix */
	pevalvieworientationmatrix3( vrp, &vpn, vup, &error,
				    rep.orientation_matrix );
	if (error) {
	  ger_error("ren_camera: pevalvieworientationmatrix3 error %d", error);
	  error= 0;
	}
	pevalviewmappingmatrix3( &map, &error, rep.mapping_matrix );
	if (error) {
	  ger_error("ren_camera: pevalviewmappingmatrix3 error %d",error);
	  error= 0;
	}

	/* set display update state to PWAIT, to prevent re-rendering
	 * until ren_render is called.
	 */
	psetdisplayupdatest(phigs_wsid, PWAIT, PNIVE);

	/* Set the view representation */
	psetviewrep3(phigs_wsid, VIEWINDEX, &rep);

	/* set background color to the desired background color */
	cbndl_r( color_rep )= color_red( background );
	cbndl_g( color_rep )= color_green( background );
	cbndl_b( color_rep )= color_blue( background );
	psetcolourrep( phigs_wsid, 0, &color_rep );

	/* Clean up */
	free( (char *)vup );
	free( (char *)vrp );
}

void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine renders the object given, in the current rendering environment.
 */
{
	int thisid;
 
	thisid= gob_idnum( thisgob );

	ger_debug("ren_render: rendering object given by gob %d", thisid);

	/* set display update state to PWAIT */
	psetdisplayupdatest(phigs_wsid, PWAIT, PNIVE);

	/* Open and empty HEAD structure, then refill and close it. */
	popenstruct( HEAD );
	pemptystruct( HEAD );
	if (!null(thistrans)) add_trans(thistrans);
	if (!null(thisattrlist)) add_attr(thisattrlist);
	add_lights();
	pexecutestruct( thisid );
	pclosestruct();

	/* set display update state to PASAP, causing redraw */
	psetdisplayupdatest(phigs_wsid, PASAP, PNIVE);

	/* Wait to make sure the redraw is complete */
	ppupdatewssynch(phigs_wsid, PPERFORM);

#ifdef INCLUDE_RLE_SAVER
	if (save_frames_flag) {
	  /* Grab the frame */
	  save_rle_frame (display_ptr, window_id);
	}
#endif
}

static void fast_traverse(thisgob)
Gob thisgob;
/* This routine traverses a gob DAG, by handling its transformation 
 * and attributes and either triggering execution of its primitive 
 * or rendering its children.  It is used to traverse the lighting DAG.
 */
{
  int thisid;
  Attribute_list newattr;
  Transformation newtrans;
  float *transmatrix;
  Primitive newprim;
  Child_list kidlist;

  thisid= gob_idnum( thisgob );
  ger_debug("fast-traverse: traversing object given by gob %d", thisid);

  /* If there is a transformation, add it to the transformation stack */
  if ( !null( newtrans=gob_trans(thisgob) ) ) {
    transmatrix= array2d_to_c( newtrans );
    ast_push_matrix(transmatrix);
  }
  /* Check for and if necessary handle new attributes */
  if ( !null( newattr=gob_attr(thisgob) ) )
    ast_push_attributes( newattr );

  /* Either execute the primitive, or render the children */
  if ( !null( newprim= gob_primitive(thisgob) ) ) {
    eval_function( primitive_op(newprim) );
  }
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      fast_traverse( first_child( kidlist ) );
      kidlist= rest_of_children( kidlist );
    }
  }

  /* Pop attributes and transformations, and clean up. */
  if ( !null(newattr) ) ast_pop_attributes( newattr );
  if ( !null(newtrans) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
    }
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine adds the given object to the lights environment. */
{
  int thisid;
  static int lastid= -1; /* impossible id number */
  float *transmatrix;

  thisid= gob_idnum( thisgob );
  ger_debug("ren_traverselights: setting lights to gob %d",thisid);
 
  if (thisid != lastid ) { /* Changed since last render */

    /* Set up for traversal */
    current_state= TRAVERSING;
    clear_lights();

    /* If there is a transformation, add it to the transformation stack */
    if ( !null( thistrans ) ) {
      transmatrix= array2d_to_c( thistrans );
      ast_push_matrix(transmatrix);
    }

    /* Check for and if necessary handle new attributes */
    if ( !null( thisattrlist ) ) ast_push_attributes( thisattrlist );

    fast_traverse( thisgob );

    /* Pop attributes and transformations, and clean up. */
    if ( !null(thisattrlist) ) ast_pop_attributes( thisattrlist );
    if ( !null(thistrans) ) {
      ast_pop_matrix();
      free( (char *)transmatrix );
    }

    lastid= thisid;
    current_state= DEFINING;
  }
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );

	ger_debug("ren_free:  freeing gob %d",thisid);

	/* set display update state to PWAIT, to prevent re-rendering
	 * as the deletion occurs.
	 */
	psetdisplayupdatest(phigs_wsid, PWAIT, PNIVE);

	pdelstruct( thisid );
}

static void set_area_properties(material)
Material material;
/* This routine sets up the 'usual' area properties. */
{
	PPsurfprop area_props;
	
	ger_debug("set_area_properties:");

	area_props.amb= material_ka(material);
	if (area_props.amb < 0.35) area_props.amb= 0.35;
	area_props.dif= material_kd(material);
	if (area_props.dif < 0.35) area_props.dif= 0.35;
	area_props.spec= material_ks(material);
	area_props.colour_type= PCM_RGB;
	area_props.colour_value.direct_colour.rgb.r= 1.0;
	area_props.colour_value.direct_colour.rgb.g= 1.0;
	area_props.colour_value.direct_colour.rgb.b= 1.0;
	area_props.exp= material_exp(material);
	area_props.trans= 1.0;
	ppsetsurfprop( &area_props );
}

void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes, if it is not already initialized. */
{
  static int initialized=0; /* to hold initialization state */
  Pair thispair;
  Pwstype ws_type;
  Pconnid conn_id;
  Pcobundl color_rep;
  int drawable_device= 0;
  char conn_string[128];
  
  ger_debug("ren_setup: initializing phigs; renderer %s, device %s",
	    renderer, device);
  
  /* If the renderer was already initialized, return */
  if (initialized) {
    ger_debug("ren_setup: called twice; this call ignored.");
    return;
  }
  else initialized= 1;
  
  /* Initialize needed assist modules */
  ast_init_matrix();
  
  /* 
    Generate some symbols to be used later in attribute list
    parsing.
    */
  depth_cue_symbol= create_symbol("depth-cue");
  color_symbol= create_symbol("color");
  backcull_symbol= create_symbol("backcull");
  text_height_symbol= create_symbol("text-height");
  material_symbol= create_symbol("material");
  
  /* 
    If open_device is false and the output device is not a drawable,
    someone else has set everything up and we are done.  Otherwise,
    initialize Phigs.  In the case of a drawable device, this carries
    out the process of turning the X Window System drawable into a
    Phigs workstation.
    */
  if ( !strcmp(device,"dec_decwindows_drawable")
      || !strcmp(device,"dec_pex_drawable")
      || ( !open_device && !strcmp(device,"xws") ) /* Painter compat. */ 
      ) drawable_device= 1;
  if (open_device || drawable_device) {
    /* set up phigs and workstation */
    popenphigs( (Pchar*)NULL, PDEFAULT_MEM_SIZE );
    if (drawable_device) {
      if (phigs_conn_id) /* global variable; may be externally set */
	conn_id= phigs_conn_id;
      else {
	sprintf(conn_string,"%d!%d",(int)display_ptr,(int)window_id);
	conn_id= conn_string;
      }
      if ( !strcmp(device,"dec_decwindows_drawable") )
	ws_type= PHIGS$K_DECWINDOWS_DRAWABLE;
      else if ( !strcmp(device,"dec_pex_drawable") 
	       || !strcmp(device,"xws") ) /* Painter compat. */
	ws_type= PHIGS$K_PEX_DRAWABLE;
      else ger_fatal("ren_setup: unknown phigs device %s",device);
    }
    else if ( !device || !strcmp(device,"dec_defaults") ) {
      ws_type= PHIGS$K_WSTYPE_DEFAULT;
      conn_id= PHIGS$K_CONID_DEFAULT;
    }
    else if ( !strcmp(device,"dec_decwindows_output") ) { 
      /* DecWindows device comes from 'outfile' for user convenience */
      ws_type= PHIGS$K_DECWINDOWS_OUTPUT;
      conn_id= (outfile && strcmp(outfile,"-")) ? 
               outfile : PHIGS$K_CONID_DEFAULT;
    }
    else if ( !strcmp(device,"dec_pex_output") 
	     || !strcmp(device,"xws") ) { /* Painter compat. */ 
      /* DecWindows device comes from 'outfile' for user convenience */
      ws_type= PHIGS$K_PEX_OUTPUT;
      conn_id= (outfile && strcmp(outfile,"-")) ? 
               outfile : PHIGS$K_CONID_DEFAULT;
    }
#ifdef INCLUDE_RLE_SAVER
    else if ( !strcmp(device,"dec_decwindows_rle_save") 
	     || !strcmp(device,"xws") ) { /* Painter compat. */ 
      ws_type= PHIGS$K_DECWINDOWS_OUTPUT;
      conn_id= PHIGS$K_CONID_DEFAULT;
      save_frames_flag= 1;
    }
    else if ( !strcmp(device,"dec_pex_rle_save") ) {
      /* DecWindows device comes from 'outfile' for user convenience */
      ws_type= PHIGS$K_PEX_OUTPUT;
      conn_id= PHIGS$K_CONID_DEFAULT;
      save_frames_flag= 1;
    }
#endif /* INCLUDE_RLE_SAVER */
    else ger_fatal("ren_setup: unknown phigs device %s",device);
    popenws( phigs_wsid, conn_id, ws_type );
    psetdisplayupdatest( phigs_wsid, PWAIT, PNIVE );
    psethlhsrmode( phigs_wsid, PPHLHSR_MODE_ZBUFFER );
#ifdef INCLUDE_RLE_SAVER
    if (save_frames_flag)
      init_frame_saver( outfile ? outfile : "phigs_ren.rle" );
#endif /* INCLUDE_RLE_SAVER */
    
    /* set background color to black */
    cbndl_r( color_rep )= 0.0;
    cbndl_g( color_rep )= 0.0;
    cbndl_b( color_rep )= 0.0;
    psetcolourrep( phigs_wsid, 0, &color_rep );
  }
  
  /* create and post root structure */
  popenstruct( ROOT );
  psetviewind( VIEWINDEX );
  psethlhsrid( PPHLHSR_INDEX_ZBUFFER );
  psetlinetype(PLN_SOLID);
  psetlinewidth(1.0);
  psetmarkertype(PMK_POINT);
  psetintstyle(PSOLID);
  ppsetintlightmethod( PPLIGHTING_SPECULAR );
  ppsetintshadmethod( PPSHADE_COLOUR );
  pexecutestruct( HEAD ); /* implicitly creates HEAD struct */
  pclosestruct();
  ppoststruct( phigs_wsid, ROOT, 1.0 );
}

void ren_reset( hard )
/* This routine resets the renderer */
{
	ger_debug("ren_reset: resetting renderer; hard= %d",hard);

	/* Hard resets require recreation of lisp-side variables */
	if (hard) {
		depth_cue_symbol= create_symbol("depth-cue");
		color_symbol= create_symbol("color");
		backcull_symbol= create_symbol("backcull");
		text_height_symbol= create_symbol("text-height");
		material_symbol= create_symbol("material");
	}
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	ger_debug("ren_shutdown");

	/* Close workstation and phigs */
	pclosews( phigs_wsid );
	pclosephigs();

#ifdef INCLUDE_RLE_SAVER
	if (save_frames_flag) {
	  close_frame_saver();
	  save_frames_flag= 0;
	}
#endif
}

