/****************************************************************************
 * assist_mtrx.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides support for renderers which do not have the
capability to do certain operations.  This module provides utilities
to simulate a transformation matrix stack, useful in handling hierarchical
coordinate transformations.  The matrix stack is preallocated, so calling
functions should *never* free the matrices returned by these functions.
*/

#include <math.h>
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "assist.h"

/* Maximum depth of stack */
#define MAX_DEPTH 16

typedef float cmatrix[16];

static cmatrix identity= {
  1.0, 0.0, 0.0, 0.0, 
  0.0, 1.0, 0.0, 0.0, 
  0.0, 0.0, 1.0, 0.0,
  0.0, 0.0, 0.0, 1.0 };

static cmatrix *stack, *current, scratch;

static void copy_matrix( target, source )
cmatrix target, source;
/* This routine copies source to target */
{
  int i;
  /* no debugging; called too often */
  for (i=0; i<16; i++) *target++= *source++;
}

static void multiply_matrix( target, factor1, factor2 )
cmatrix target, factor1, factor2;
/* This routine multiplies factor1 by factor2, putting the result in target. 
 * Target must point to different storage than either factor.
 */
{
  int i, j, k;
  float *ftarget,*ffactor1,*ffactor2;

  /* no debugging; called too often */

  ftarget= (float *)target;
  for (i=0; i<16; i++) *ftarget++= 0.0;

  for (i=0; i<4; i++) {
    for (j=0; j<4; j++) {
      ffactor1= (float *)factor1 + 4*i;
      ffactor2= (float *)factor2 + j;
      for (k=0; k<4; k++) {
	*target += *ffactor1++ * *ffactor2;
	ffactor2 += 4;
      }
      target++;
    }
  }

}

void ast_init_matrix()
/* This routine initializes the package. */
{
  ger_debug("ast_init_matrix:");

  if ( !( stack= (cmatrix *)malloc( MAX_DEPTH*sizeof(cmatrix) ) ) )
    ger_fatal( "ast_init_init: cannot allocate stack of %d 4x4 matrices!",
	      MAX_DEPTH );

  current= stack;
  copy_matrix( *current, identity );
}

float *ast_clear_matrix()
/* This routine pops back to the top of the stack, leaving the identity
 * matrix as the current matrix.  A pointer to the current matrix is returned.
 */
{
  ger_debug("ast_clear_matrix:");

  current= stack;
  copy_matrix( *current, identity );

  return( (float *)current );
}

float *ast_load_matrix( thismatrix )
float *thismatrix;
/* This routine loads the given matrix into the current matrix in the
 * stack.  A pointer to the current matrix is returned.
 */
{
  ger_debug("ast_load_matrix:");

  copy_matrix(*current, thismatrix);

  return( (float *)current );
}

float *ast_push_matrix( thismatrix )
float *thismatrix;
/* This routine pushes down the matrix stack, copying the top matrix into
 * the new slot.  That matrix is then multiplied by the parameter matrix,
 * and the result goes into the new slot.  A pointer to this slot (the
 * new current matrix) is returned.  New matrices are right multiplied,
 * as per the P3D convention.
 */
{
  ger_debug("ast_push_matrix:");

  if ( current-stack >= MAX_DEPTH-1 )
    ger_error("ast_push_matrix: stack overflow (max depth %d); not pushed.",
	      MAX_DEPTH);
  else {
    copy_matrix( scratch, *current );
    ++current;
    multiply_matrix( *current, scratch, thismatrix );
  }

  return( (float *)current );
}

float *ast_pop_matrix()
/* This routine pops up the matrix stack, returning a pointer to the new
 * current matrix.
 */
{
  ger_debug("ast_pop_matrix:");

  if ( current <= stack )
    ger_error("ast_pop_matrix: stack underflow; not popped. ");
  else current--;

  return( (float *)current );
}

float *ast_get_matrix()
/* This routine returns a pointer to the current matrix. */
{
  ger_debug("ast_get_matrix:");

  return( (float *)current );
}
