/****************************************************************************
 * assist_attr.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides support for renderers which do not have the
capability to manage all necessary attributes.  It provides a fast 
mechanism to manage attributes during traversal of the lisp-side
geometry database.  
*/
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "assist.h"

/*
 * The mechanism is to hash the symbols of each
 * symbol-value pair into a table, and then essentially accumulate
 * a stack of values at that table location.  We do no memory management
 * on the Pairs themselves, because it is assumed that they will be
 * bound into attribute lists on the lisp side for the entire interval
 * when they are bound into stacks in this module.
 *
 * The attributes are stored in a hash table.  We assume that the things
 * begin passed in are pointers to word-aligned objects.  We thus guess
 * that a fair value to hash from is the value passed in, right shifted
 * two bits (to throw away the bits which will always match due to word
 * alignment) and masked to 8 bits.  This allows us to use a hash table
 * size which is a power of two, and generate a hash index without a
 * modulo operation.  This operation is encoded by the following 
 * definitions.
 */
#define hashsize 256

#define BAD_HASH_LOOKUP 0 

typedef struct hash_cell_struct {
	Pair result;
	struct hash_cell_struct *next, *last;
	} Hash_cell;

/* Hash table itself- static allocation assures all the elements will start
 * out null.
 */
static Hash_cell *hashtable[ hashsize ];

int make_id(val)
Symbol val;
{
  return( ((int)val >> 2) & (hashsize-1) );
}

static Pair hash_lookup(val)
Symbol val;
/* Hash table lookup */
{
  int id;
  Hash_cell *thiscell;

  id= make_id( val );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( symbol_equality( val, pair_symbol(thiscell->result) ) )
      return(thiscell->result);
  return( (Pair) BAD_HASH_LOOKUP );
}

static Pair hash_add( result )
Pair result;
/* Add a value-result pair to hash table */
{
  int id;
  Hash_cell *thiscell;

  id= make_id( pair_symbol(result) );
  if (!(thiscell= (Hash_cell *)malloc( sizeof(Hash_cell) )))
	ger_fatal("hash_add: unable to allocate %d bytes!\n",
		sizeof(Hash_cell));
  thiscell->result= result;
  thiscell->next= hashtable[id];
  if (hashtable[id]) hashtable[id]->last= thiscell;
  thiscell->last= NULL;
  hashtable[id]= thiscell;

  return( result );
}

static void hash_free(val)
Symbol val;
/* Free a value-result pair from the hash table.  Does nothing if the
 * value is not found.
 */
{
  int id;
  Hash_cell *thiscell;

  id= make_id( val );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( symbol_equality( val, pair_symbol(thiscell->result) ) ) {
      if (thiscell->next) thiscell->next->last= thiscell->last;
      if (thiscell->last) thiscell->last->next= thiscell->next;
      else hashtable[id]= thiscell->next;
      free( (char *)thiscell );
      break;
    }
}

void ast_push_attributes(attrlist)
Attribute_list attrlist;
/* This routine scans an attribute list, and pushes those values found
 * onto the appropriate stacks.
 */
{
  ger_debug("ast_push_attributes");

  while ( !null(attrlist) ) {
    if ( (!consp(attrlist)) || (!consp(first_pair(attrlist))) )
	{
	fprintf(stderr,"ERROR:  Bad attribute list.\n");
	print_out(attrlist);
	fprintf(stderr," \n");
	return;
	}
    hash_add( first_pair(attrlist) );
    attrlist= rest_of_pairs(attrlist);
  }
}

void ast_pop_attributes(attrlist)
Attribute_list attrlist;
/* This routine scans an attribute list, and pops the values found off
 * the appropriate stacks.
 */
{
  ger_debug("ast_pop_attributes");

  while ( !null(attrlist) ) {
    hash_free( pair_symbol( first_pair(attrlist) ) );
    attrlist= rest_of_pairs(attrlist);
  }
}

int ast_bool_attribute(symbol)
Symbol symbol;
/* This routine returns the current value of a boolean attribute.  It
 * returns 0 for false, 1 for true.
 */
{
  Pair thispair;

  ger_debug("ast_bool_attribute");

  thispair= hash_lookup(symbol);
  if ( !thispair ) ger_fatal(
    "ast_bool_attribute: requested color-type attribute not defined!");
  else return( pair_boolean(thispair) );
}

Color ast_color_attribute(symbol)
Symbol symbol;
/* This routine returns the current value of a color attribute.
 */
{
  Pair thispair;

  ger_debug("ast_color_attribute");

  thispair= hash_lookup(symbol);
  if ( !thispair ) ger_fatal(
    "ast_color_attribute: requested color-type attribute not defined!");
  else return( pair_color(thispair) );
}

Material ast_material_attribute(symbol)
Symbol symbol;
/* This routine returns the current value of a material attribute. */
{
  Pair thispair;

  ger_debug("ast_material_attribute");

  thispair= hash_lookup(symbol);
  if ( !thispair ) ger_fatal(
    "ast_material_attribute: requested material-type attribute not defined!");
  else return( pair_material(thispair) );
}

float ast_float_attribute(symbol)
Symbol symbol;
/* This routine returns the current value of a floating point attribute. 
 */
{
  Pair thispair;

  ger_debug("ast_float_attribute");

  thispair= hash_lookup(symbol);
  if ( !thispair ) ger_fatal(
    "ast_float_attribute: requested floating point attribute not defined!"); 
  else return( pair_float(thispair) );
}

int ast_int_attribute(symbol)
Symbol symbol;
/* This routine returns the current value of an integer attribute.
 */
{
  Pair thispair;

  ger_debug("ast_int_attribute");

  thispair= hash_lookup(symbol);
  if ( !thispair ) ger_fatal(
    "ast_int_attribute: requested integer attributed not defined!");
  else return( pair_int(thispair) );
}

char *ast_string_attribute(symbol)
Symbol symbol;
/* This routine returns the current value of a string attribute.
 */
{
  Pair thispair;

  ger_debug("ast_int_attribute");

  thispair= hash_lookup(symbol);
  if ( !thispair ) ger_fatal(
    "ast_string_attribute: requested string attributed not defined!");
  else return( pair_string(thispair) );
}
