/****************************************************************************
 * marchp3d.c
 * Authors Julie Roskies, Robert Earhart
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "ge_error.h"
#include <p3dgen.h>
#include <drawp3d.h>

#ifdef VMS

#include <string.h>
#include climsgdef
#include descrip
#include ctype

#define c_length 128

#else /* ifdef VMS */

#ifdef SYSV
#ifndef CRAY
#include <getopt.h>
#endif
#endif
extern char *optarg;
extern int optind, opterr;

#endif /* ifdef VMS */

#define MAXSTRING 256
#define CMAXINP 80 	/* maximum length of color input line */
#define CTABSIZ 256	/* size of c_table */
#define FOVEA 45.0
#define HITHER -1.0
#define YON -6.0

static FILE *fsetup();   /* safely opens input file, returns pointer */

typedef struct color_struct { float r, g, b, a; } Color;
void colorfun();	/* assigns a color to a value using the color table */

static P_Point lookfrom = { 0.0, 0.0, 2.5 };
static P_Point lookat = { 0.0, 0.0, 0.0 };
static P_Vector up = {0.0, 1.0, 0.0};
static P_Point lightloc = {20.0, 20.0, 0.0};
static P_Color lightcolor = {P3D_RGB, .8, .8, .8, 1.0};
static P_Color amblightcolor = {P3D_RGB, .3, .3, .3, 1.0};
static P_Vector axis = { 0.0, 1.0, 0.0 }; /*Axis to rotate about*/
static int loindex, hiindex;
static Color *this_color_table;

/* Note that some global variables are also handled as parameters */
typedef struct param_struct {
  int nx, ny, nz;
  int xq, yq, zq;
  float min, max, step;
  int vq, gq, iq;
  int iterations, forever;
  char renderer[MAXSTRING], driver[MAXSTRING], datastr[MAXSTRING];
  int rq, dq, sq;
  int inner_surface, bbox;
  float angle;
  int rotq;
  int c_scheme;
  int c_tableq, c_schemeq;
  int c_dataq;
  float loscale, hiscale;
  int loq, hiq;
  int rotate_single_model;
  FILE *clr_data, *clr_table, *arr_data;
} Params;

static Params params= {
  0, 0, 0,          /* nx, ny, nz */
  0, 0, 0,          /* xq, yq, zq */
  0.0, 0.0, 1.0,    /* min, max, step */
  0, 0, 0,          /* vq, gq, iq */
  1, 0,             /* iterations, forever */
  "p3d","-","-",    /* renderer, driver, datastring */
  0, 0, 0,          /* rq, dq, sq */
  0, 0,             /* inner_surface, bbox */
  0.0,              /* angle */
  0,                /* rotq */
  0,                /* c_scheme */
  0, 0,             /* c_tableq, c_schemeq */
  0,                /* c_dataq, c_tableq */
  0.0, 1.0,         /* loscale, hiscale */
  0, 0,             /* loq, hiq */
  0,                /* rotate_single_model */
  (FILE *)0, (FILE *)0, (FILE *)0, /* clr_data, clr_table, arr_data */
};

#ifdef VMS

int vms_present(in_string)
     char *in_string;
{
  int  ret=0;
  
  struct dsc$descriptor_s name_desc;
  name_desc.dsc$w_length = strlen(in_string);  
  name_desc.dsc$a_pointer = in_string;         
  name_desc.dsc$b_class = DSC$K_CLASS_S;       
  name_desc.dsc$b_dtype = DSC$K_DTYPE_T;       
  ret = CLI$PRESENT(&name_desc);  
  return(ret==CLI$_PRESENT);
}

char *vms_string(in_string)
     char *in_string;
{
  char *out_string = (char *)malloc(c_length);
  struct dsc$descriptor_s name_desc, out_desc;
  short str_l;
  int  ret, ret_value, i;
  
  for (i=0; i<c_length; ++i) out_string[i] = ' ';
  name_desc.dsc$w_length = strlen(in_string);    
  name_desc.dsc$a_pointer = in_string;         
  name_desc.dsc$b_class = DSC$K_CLASS_S;       
  name_desc.dsc$b_dtype = DSC$K_DTYPE_T;         
  
  out_desc.dsc$w_length = c_length;    
  out_desc.dsc$a_pointer = out_string;         
  out_desc.dsc$b_class = DSC$K_CLASS_S;       
  out_desc.dsc$b_dtype = DSC$K_DTYPE_T;    
  
  ret = CLI$GET_VALUE(&name_desc, &out_desc, &str_l);
  out_string[str_l] = '\0';
  
  return(out_string);
}

static char *mylower(string)
char *string;
{
  char *orig= string;
  while (*string) {
    *string= tolower(*string);
    string++;
  }
  return(orig);
}

static void parse_params(argc, argv, params)
     int argc;
     char *argv[];
     Params *params;
     /* This is the VMS  parameter parsing mechanism */
{
  char *fsetuptemp, *string;
  /* VMS parsing of command line */

  /* Setting up the main array data file and other parameters */	
  fsetuptemp = vms_string("array_data");
  params->arr_data= fsetup(fsetuptemp, "r");
  (void)sscanf(vms_string("xdim"), "%d", &(params->nx));
  (void)sscanf(vms_string("ydim"), "%d", &(params->ny));
  (void)sscanf(vms_string("zdim"), "%d", &(params->nz));
  (void)sscanf(vms_string("val"), "%f", &(params->min));

  /* Isosurface max and step */
  if (vms_present("isomax")) {
    params->gq= sscanf(vms_string("isomax"), "%f", &(params->max));
    if (vms_present("isostep"))
      (void)sscanf(vms_string("isostep"), "%f", &(params->step));
    else params->step= params->max - params->min;
  }
  else params->max= params->min;

  /* Color table */
  if (vms_present("min")) {
    params->loq= sscanf(vms_string("min"), "%f", &(params->loscale));
  };
  if (vms_present("max")) {
    params->hiq= sscanf(vms_string("max"), "%f", &(params->hiscale));
  };
  if (vms_present("c_table")) {
    fsetuptemp = vms_string("c_table");
    params->clr_table = fsetup (fsetuptemp, "r");
    params->c_tableq = 1;
  };

  /* Color scheme (exclusivity with respect to color table is handled
   * by the .cld file)
   */
  if (vms_present("c_scheme")) {
    params->c_schemeq= 
      sscanf(vms_string("c_scheme"), "%d", &(params->c_scheme));
  };

  /* Color data */
  if (vms_present("clr_data")) {
    fsetuptemp = vms_string("clr_data");
    params->clr_data = fsetup (fsetuptemp, "r");
    params->c_dataq = 1;
  }

  /* Iterations */
  if (vms_present("iterations")) {
    params->iq = sscanf(vms_string("iterations"), "%d", &(params->iterations));
    params->forever = (params->iq && (! params->iterations));
  }

  /* Look direction */
  if (vms_present("lookdirection")) {
    string= vms_string("lookdirection");
    if (!strcmp(string,"X")) {
      lookfrom.x= 2.5; lookfrom.y= 0.0; lookfrom.z= 0.0;
    }
    else if (!strcmp(string,"Y")) {
      lookfrom.x= 0.0; lookfrom.y= 2.5; lookfrom.z= 0.0;
      up.x= 1.0; up.y= 0.0; up.z= 0.0;
    }
    else if (!strcmp(string,"Z")) {
      lookfrom.x= 0.0; lookfrom.y= 0.0; lookfrom.z= 2.5;
    }
    else if (!strcmp(string,"-X")) {
      lookfrom.x= -2.5; lookfrom.y= 0.0; lookfrom.z= 0.0;
    }
    else if (!strcmp(string,"-Y")) {
      lookfrom.x= 0.0; lookfrom.y= -2.5; lookfrom.z= 0.0;
      up.x= 1.0; up.y= 0.0; up.z= 0.0;
    }
    else if (!strcmp(string,"-Z")) {
      lookfrom.x= 0.0; lookfrom.y= 0.0; lookfrom.z= -2.5;
    }
  }

  /* Rotation information */
  if (vms_present("rotangle"))
    params->rotq= sscanf(vms_string("rotangle"),"%f",&(params->angle));
  if (vms_present("rotx"))
    (void)sscanf(vms_string("rotx"),"%f",&(axis.x));
  if (vms_present("roty"))
    (void)sscanf(vms_string("roty"),"%f",&(axis.y));
  if (vms_present("rotz"))
    (void)sscanf(vms_string("rotz"),"%f",&(axis.z));

  /* Renderer information */
  if (vms_present("renderer")) {
    strcpy(params->renderer, mylower(vms_string("renderer")));
    params->rq= 1;
  }
  if (vms_present("device")) {
    strcpy(params->driver, mylower(vms_string("device")));
    params->dq= 1;
  }
  if (vms_present("datastring")) {
    strcpy(params->datastr, mylower(vms_string("datastring")));
    params->sq= 1;
  }

  /* Misc. switches */
  if (vms_present("inside")) {
    params->inner_surface= 1;
  }
  if (vms_present("boundingbox")) {
    params->bbox= 1;
  }
  
  /* If only one iteration is to be done but a rotation is requested,
   * rotate the single model forever.
   */
  if ((params->iterations == 1) && params->rotq) 
    params->rotate_single_model= 1;

  /* output file handling- must arbitrate with renderer */
  if (vms_present("output")) {
    string= mylower(vms_string("output"));
    if (params->rq) {
      if (!strcmp(params->renderer,"p3d")) { /* P3D renderer */
	strcpy(params->driver,string);
      }
      else if (!strcmp(params->renderer,"painter")) { /* Painter renderer */
	strcpy(params->datastr,string);
      }
    }
    else {
      strcpy(params->renderer,"p3d");
      strcpy(params->driver,string);
      strcpy(params->datastr,"junk");
    }
  }
}

#else /* ifdef VMS */

static void usage()
     /* This is the Unix usage message.  It exits rather then returning. */
{
  fprintf(stderr, "\nusage: marchp3d -x xdim -y ydim -z zdim\n\
                -v value [max step] [-n iterations]\n\
                [-c c_scheme clr_data min max ] [-i] [-b]\n\
                [-l x|y|z|-x|-y|-z] [-t x_com y_com z_com angle]\n\
                [-r renderer] [-d device] [-s datastr] array_data\n\n\
		xdim - x dimension of array\n\
		ydim - y dimension of array\n\
		zdim - z dimension of array\n\
		value - value at which isosurface is desired\n\
                max, step - controls for value crawl\n\
                iterations - the number of data sets to be input (default 1)\n\
		c_scheme - color table scheme (if not a valid color scheme\n\
                           number, the program will attempt to use a file\n\
                           with the given name as a color table)\n\
		clr_data - file which contains color scalars to fill array\n\
			of size xdim * ydim * zdim\n\
		min max - scalars to correspond to lowest and highest color\n\
                -i - specifies that inside surface should be drawn\n\
                -b - specifies that a bounding box should be drawn\n\
                -l - axis to look down on (defaults to z)\n\
                     (examples: '-l x', '-l -x', '-l -z'...)\n\
                x_com - x-component of rotation vector\n\
                y_com - y-component of rotation vector\n\
                z_com - z-component of rotation vector\n\
                angle - number of degrees to rotate per frame\n\
                renderer - specifies the renderer to use (defaults to p3d)\n\
                device - specifes which device to use (defaults to '-')\n\
                datastr - specifies data to the renderer (defaults to '-')\n\
		array_data - data file which contains values to fill array\n\
			of size xdim * ydim * zdim\n");
  
  exit(1);
}

static void parse_params(argc, argv, params)
     int argc;
     char *argv[];
     Params *params;
     /* This is the Unix parameter parsing mechanism */
{
  char c;
  int neg; /*For negative axis detection*/
  int rox=0, roy=0, roz=0; /*For consistancy checks reading rotation vector*/
  
  if (argc < 9) usage();
  
  while ((c=getopt(argc, argv, "x:y:z:v:n:r:d:s:l:tibc")) != ((char)-1))
    switch (c) {
    case 'x':
      params->xq = sscanf(optarg, "%d", &(params->nx));
      break;
    case 'y':
      params->yq = sscanf(optarg, "%d", &(params->ny));
      break;
    case 'z':
      params->zq = sscanf(optarg, "%d", &(params->nz));
      break;
    case 'v':
      params->vq = sscanf(optarg, "%f", &(params->min));
      if ((argv[optind][0] == '.') || ((argv[optind][0] >= '0') &&
				       (argv[optind][0] <= '9'))) {
	if (!(params->gq = ((sscanf(argv[optind++], "%f", &(params->max)) +
		      sscanf(argv[optind++], "%f", &(params->step))) == 2)))
	  usage();
      }
      else
	params->max=params->min;
      break;
    case 'n':
      params->iq = sscanf(optarg, "%d", &(params->iterations));
      params->forever = (params->iq && (! params->iterations));
      break;
    case 'r':
      params->rq = sscanf(optarg, "%s", params->renderer);
      break;
    case 'd':
      params->dq = sscanf(optarg, "%s", params->driver);
      break;
    case 's':
      params->sq = sscanf(optarg, "%s", params->datastr);
      break;
    case 'i':
      params->inner_surface++;
      break;
    case 'b':
      params->bbox++;
      break;
    case 'l':
      lookfrom.z = 0.0;
      if (*optarg == '-') {
	neg = -1;
	optarg++;
      }
      else
	neg=1;
      switch (*optarg) {
      case 'x':
	lookfrom.x = 2.5*neg;
	break;
      case 'y':
	/*Hack, hack, hack... we need to change the up vector
	  of the camera if the user does this... let's just stuff
	  it into the x-direction...*/
	up.x = 1.0;
	up.y = 0.0;
	lookfrom.y = 2.5*neg;
	break;
      case 'z':
	lookfrom.z = 2.5*neg;
	break;
      }
      break;
    case 't':
      if (optind == argc)
	usage();
      if (rox = sscanf(argv[optind++], "%f", &axis.x))
	roy = sscanf(argv[optind++], "%f", &axis.y);
      else
	usage();
      if (rox && roy)
	roz = sscanf(argv[optind++], "%f", &axis.z);
      else
	usage();
      if ((! sscanf(argv[optind++], "%f", &(params->angle))) || (! rox) ||
	  (! roy) || (! roz))
	usage();
      else
	params->rotq = 1;
      break;
    case 'c':
      /*Attempt to read an integer into c_scheme*/
      if (optind == argc)
	usage();
      if ((! sscanf(argv[optind], "%d", &(params->c_scheme))) ||
	  (params->c_scheme < 0) || (params->c_scheme > 4)) {
	params->clr_table = fsetup (argv[optind], "r");
	params->c_tableq = 1;
      }
      else
	params->c_schemeq = 1;
      
      optind++;
      if (strlen(argv[optind]) == 1 && *(argv[optind]) == '-')
	params->clr_data = stdin;
      else
	params->clr_data = fsetup (argv[optind], "r");
      optind++;
      params->c_dataq = 1;
      if (params->loq = sscanf(argv[optind++], "%f", &(params->loscale)))
	params->hiq = sscanf(argv[optind++], "%f", &(params->hiscale));
      if ((! params->loq) || (! params->hiq))
	usage();
      break;
    case '?':
      usage();
      break;
    default:
      ger_fatal("%s: illegal option %c\n", argv[0], c);
    }
  
  /* If only one iteration is to be done but a rotation is requested,
   * rotate the single model forever.
   */
  if ((params->iterations == 1) && params->rotq) 
    params->rotate_single_model= 1;

  /* makes sure that all required data has been inputted */
  if (params->xq != 1 || params->yq != 1 || params->zq != 1 || params->vq != 1)
    ger_fatal("%s: Must specify -x, -y, -z, and -v parameters", argv[0]);

  /* sets up movie_data to point to last argument on command line or
     to stdin if last argument is just a dash */
  
  if (argv[optind]) /*I don't like to pass a - ;)*/
    if (strlen(argv[optind]) == 1 && *(argv[optind]) == '-')
      params->arr_data= stdin;
    else
      params->arr_data= fsetup(argv[optind], "r");
  else
    params->arr_data = stdin;

}

#endif /* ifdef VMS */
    
main(argc, argv)
int argc; 
char *argv[];
{
  Color *make_c_table();	/* makes a color table holding color data */
  FILE *arr_data;	  /* points to array data file */
  float *my_arr; /* will point to array made by reading in arr_data */
  float *my_clr; /* will point to array made by reading in clr_data */
  float *my_arr_copy; /* for use in reading in data files */
  float *my_clr_copy; /* for use in reading in data files */
  float rotmul; /*For remembering how far we've come*/
  int count;	/* for use in reading in data files */
  float val;
  P_Point corner1, corner2;
  corner1.x = corner1.y = corner2.z = -.5;
  corner2.x = corner2.y = corner1.z = .5;

  ger_init(argv[0]);
  
  ger_debug("marchp3d:");
  
  parse_params(argc, argv, &params);
  
  /* Memory allocation for the my_arr */
  if ( !(my_arr= (float *)malloc(
	       params.nx * params.ny * params.nz * sizeof(float)) ) )
    ger_fatal ("unable to allocate %d bytes to fill array",
	           params.nx * params.ny * params.nz * sizeof(float) );
  
  /* Memory allocation for the my_clr if necessary */
  if (params.c_dataq == 1) {
    if ( !(my_clr= (float *)malloc(
            params.nx * params.ny * params.nz * sizeof(float)) ) )
      ger_fatal ("unable to allocate %d bytes to fill color array",
	      params.nx * params.ny * params.nz * sizeof(float) );
  }
  
  if (params.c_tableq)
    this_color_table= make_c_table(params.clr_table);
  
  /*Okay... initialize the renderer.*/
  dp_init_ren(argv[0], params.renderer, params.driver, params.datastr);
  
  dp_open("best-lights");
  dp_light(&lightloc, &lightcolor);
  dp_ambient(&amblightcolor);
  dp_close();
  
  dp_camera("best-camera", &lookfrom, &lookat, &up, FOVEA, HITHER, YON);
  
  if (params.c_tableq)
    dp_set_cmap(params.loscale, params.hiscale, colorfun);
  
  if (params.c_schemeq)
    dp_std_cmap(params.loscale, params.hiscale, params.c_scheme);
  
  rotmul = 0.0;

  while ( params.forever || (params.iterations--)) {
    /* filling my_arr */
    my_arr_copy = my_arr;
    count = 0;
    while (! feof (params.arr_data) 
	   && count < params.nx * params.ny * params.nz) {
      fscanf(params.arr_data, "%f", my_arr_copy);
      my_arr_copy++;
      count++; 
    }
    if (count < params.nx * params.ny * params.nz)
      ger_error (
		 "array data does not fill expected array size.\n\
May cause errors.");

    /* filling my_clr if necessary */
    if (params.c_dataq == 1) {
      my_clr_copy = my_clr;
      count = 0;
      while (! feof (params.clr_data) 
	     && count < params.nx * params.ny * params.nz) {
	fscanf(params.clr_data, "%f", my_clr_copy);
	my_clr_copy++;
	count++; 
      }
      if (count < params.nx * params.ny * params.nz)
	ger_error ("color array data does not fill expected array size.\n\
May cause errors.");
    }
    
    for ( val=params.min; 
	 (((params.step >= 0.0)&&(val<=params.max) ||
	   ((params.step < 0.0)&&(val>=params.max)))); 
	 val+=params.step) {
      dp_open("best-model");

      if (params.bbox)
	dp_boundbox(&corner1, &corner2);
      
      if (params.loq && params.hiq && params.c_dataq 
	  && (params.c_tableq || params.c_schemeq))
	dp_isosurface( P3D_CVNVTX, (float *)my_arr,
		      (float *)my_clr, params.nx, params.ny, params.nz, 
		      val, &corner1,
		      &corner2, params.inner_surface);
      else
	dp_isosurface( P3D_CNVTX, (float *)my_arr, (float *)my_arr,
		      params.nx, params.ny, params.nz, val, 
		      &corner1, &corner2,
		      params.inner_surface);
      
      dp_translate(-.0, 0, -.0);
      dp_close();
      /* This inner loop repeats if a rotation was requested, but
       * only one iteration is occurring.  It allows rotations of
       * a single model.
       */
      if (params.rotq) { /*Include the rotation angle... (sigh)*/
	do {
	  dp_open("rot-gob");
	  dp_child("best-model");
	  dp_rotate(&axis, (params.angle)*rotmul);
	  rotmul=rotmul+1.0;
	  dp_close();
	  dp_snap("rot-gob", "best-lights", "best-camera");
	  dp_free("rot-gob");
	} while (params.rotate_single_model);
      }
      else /*Don't worry about creating a rotgob*/
	dp_snap("best-model", "best-lights", "best-camera");
      dp_free("best-model");
    }
  }
  
  dp_shutdown();
}


/* Reads in a table of color values */
Color *make_c_table (c_file)
FILE *c_file;            
{
    int ind;	/* array index */
    int c;		/* counts succesful conversions of sscanf */
    float r, g, b, a;
    char s[CMAXINP];	/* holds a line of input */
    Color *my_color_table;

      ger_debug("make_c_table:");
    
    /* Memory allocation for the color table */
    if ( !(my_color_table= 
	   (Color *)malloc(CTABSIZ * sizeof(Color)) ) )
	ger_fatal (
		   "make_c_table: unable to allocate %d bytes",
		   CTABSIZ * sizeof(Color) );
    
    loindex= CTABSIZ;
    hiindex= 0;
    
    while (fgets(s, CMAXINP, c_file)) {
	c = sscanf (s, "%d %f %f %f %f", &ind, &r, &g, &b, &a);
	my_color_table[ind].r= r;
	my_color_table[ind].g= g;
	my_color_table[ind].b= b;
	if (c == 5)		/* checking that there is an alpha */
	    my_color_table[ind].a= a;
	else my_color_table[ind].a= 1.0;
	if (ind < loindex)
	    loindex= ind;
	if (ind > hiindex)
	    hiindex= ind;
    }
    return(my_color_table);
}	


static int interpolate_val(scalar)
float scalar;
{
  /* Remember that scalar has been mapped to 0.0 to 1.0 inclusive */
  return floor( (float)loindex + (hiindex-loindex)*scalar + 0.5 );
}


static void colorfun (thisvalue, r, g, b, a)
float *thisvalue;
float *r, *g, *b, *a;

{
    int ind;
    
    ind= interpolate_val(*thisvalue);
    *r = this_color_table[ind].r;
    *g = this_color_table[ind].g;	       
    *b = this_color_table[ind].b;	       
    *a = this_color_table[ind].a;
}

static FILE *fsetup(fname, fmode)
char fname[]; 
char fmode[];
/* fsetup safely opens a file and returns its pointer. */
{
    FILE *fp;
    
    if (fmode[0] == 'w' )
	{
	    if (strcmp(fname,"-") == 0) fp = stdout;
	    else fp= fopen(fname, fmode);
	    if ( fp == NULL )
		ger_fatal("Error opening %s", fname);  
	}
    else if ( (fp = fopen(fname, fmode)) == NULL )               
	ger_fatal("Error opening %s", fname);
    return (fp);
}
