/*---------------------------------------------------------------------------*/
/*                                                                           */
/* File:     TEST_PDIM.C                                                     */
/*                                                                           */
/* Purpose:  The purpose of this program is to test the pdim routines.       */
/*                                                                           */
/* Contains: main                                                            */
/*           Init                                                            */
/*           Print                                                           */
/*           Read                                                            */
/*           Write                                                           */
/*           Free                                                            */
/*           Append                                                          */
/*           Window                                                          */
/*           Scale                                                           */
/*           Rotate                                                          */
/*           Translate                                                       */
/*           Map                                                             */
/*           Show                                                            */
/*           Copy                                                            */
/*                                                                           */
/* Author:   John Gauch                                                      */
/*                                                                           */
/* Date:     July 21, 1986                                                   */
/*                                                                           */
/*---------------------------------------------------------------------------*/

/* Include files */
#include <stdio.h>
#include <math.h>
#include "/usr/image/lib/image.h"

/* Global variables */
PDIMREC pdim;
char *malloc();

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This program is used to test the PDIM routines.  It uses a      */
/*	     crude menu based interface.                                     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
main()
   {
   /* Local Variables */
   int Test = -1;

   /* Loop until user quits */
   while (Test != 0)
      {
      /* Print menu */
      printf("\nPDIM Test Program\n");
      printf("-----------------\n\n");
      printf(" 0)  Quit\n");
      printf(" 1)  Init  PDIM string\n");
      printf(" 2)  Print PDIM string\n");
      printf(" 3)  Read\n");
      printf(" 4)  Write\n");
      printf(" 5)  Free\n");
      printf(" 6)  Append\n");
      printf(" 7)  Window\n");
      printf(" 8)  Scale\n");
      printf(" 9)  Rotate\n");
      printf("10)  Translate\n");
      printf("11)  Map\n");
      printf("12)  Show\n");
      printf("13)  Copy\n");

      /* Read user selection */
      printf("\nEnter test number:");
      scanf("%d", &Test);
      printf("\n");

      /* Handle the test cases */
      switch(Test){
         case 0: break;
         case 1: Init(); break;
         case 2: Print(); break;
         case 3: Read(); break;
         case 4: Write(); break;
         case 5: Free(); break;
         case 6: Append(); break;
         case 7: Window(); break;
         case 8: Scale(); break;
         case 9: Rotate(); break;
         case 10: Translate(); break;
         case 11: Map(); break;
         case 12: Show(); break;
         case 13: Copy(); break;
         default:  printf("Invalid test number\n"); break;
         }
      }
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine initializes the pdim string.                       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Init()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];
   char pdimstr[2000];

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);

   /* Open image files */
   if ((image1 = imopen(name1,UPDATE)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}

   /* Read pdim info field from user */
   printf("Enter pdim string:");
   scanf("%s", pdimstr);

   /* Save string in image */
   if (imputinfo(image1, "pdim", pdimstr) == INVALID){
      fprintf(stderr,"Can not put info field\n");
      return(1);}
   if (imputinfo(image1, "tdim", pdimstr) == INVALID){
      fprintf(stderr,"Can not put info field\n");
      return(1);}
   
   /* Close images */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine prints the pdim string.                            */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Print()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];
   char *pdimstr;

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);

   /* Open image files */
   if ((image1 = imopen(name1,READ)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}

   /* Read pdim info field of specified image */
   pdimstr = (char *) imgetinfo(image1,"pdim");
   if (pdimstr == NULL){ 
      fprintf(stderr,"Can not get info field\n");
      return(1);}

   /* Print pdim string */
   printf("pdim: %s\n", pdimstr);

   /* Close images */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads a pdim field from an image.                  */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Read()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);

   /* Open image files */
   if ((image1 = imopen(name1,READ)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}

   /* Read pdim info field of specified image */
   if (pdim_read(image1,&pdim) == INVALID) {
      fprintf(stderr, "Read error: %s\n", _imerrbuf);
      return(1);}

   /* Close images */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes a pdim field to an image.                   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Write()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);

   /* Open image files */
   if ((image1 = imopen(name1,UPDATE)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}

   /* Write pdim info field of specified image */
   if (pdim_write(image1,&pdim) == INVALID) {
      fprintf(stderr, "Write error: %s\n", _imerrbuf);
      return(1);}

   /* Close images */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine frees a pdim record.                               */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Free()
   {

   /* Clear old pdim record */
   if (pdim_free(&pdim) == INVALID) {
      fprintf(stderr, "Free error: %s\n", _imerrbuf);
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine appends 2 pdim records.                            */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Append()
   {
   IMAGE *image1;
   IMAGE *image2;
   char name1[80];
   char name2[80];
   PDIMREC pdim1;
   PDIMREC pdim2;

   /* Get image file names */
   printf("The second image is appended to the end of the first.\n");
   printf("Enter first image name:");
   scanf("%s", name1);
   printf("Enter second image name:");
   scanf("%s", name2);

   /* Open image files */
   if ((image1 = imopen(name1,UPDATE)) == NULL){
      fprintf(stderr,"Can not open first image\n"); 
      return(1);}
   if ((image2 = imopen(name2,READ)) == NULL){
      fprintf(stderr,"Can not open second image\n"); 
      return(1);}

   /* Read the pdim fields */
   if (pdim_read(image1, &pdim1) == INVALID){
      fprintf(stderr,"Can not get first pdim field\n");
      return(1); }
   if (pdim_read(image2, &pdim2) == INVALID){
      fprintf(stderr,"Can not get second pdim field\n");
      return(1); }

   /* Append pdim info fields */
   if (pdim_append(&pdim1, &pdim2) == INVALID) {
      fprintf(stderr,"Append error: %s\n", _imerrbuf);
      return(1); }

   /* Write pdim info field of specified image */
   if (pdim_write(image1,&pdim1) == INVALID) {
      fprintf(stderr, "Write error: %s\n", _imerrbuf);
      return(1);}

   /* Close image files */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}
   if (imclose(image2) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine windows a pdim record.                             */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Window()
   {
   char dimension[10];
   int low, high;

   /* Get parameters */
   printf("Enter dimension:");
   scanf("%s", dimension);

   printf("Enter low:");
   scanf("%d", &low);

   printf("Enter high:");
   scanf("%d", &high);

   /* Do windowing */
   if (pdim_window(&pdim, dimension[0], low, high) == INVALID) {
      fprintf(stderr,"Window error: %s\n", _imerrbuf);
      return(1); }

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine scales a pdim record.                              */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Scale()
   {
   char field[10];
   char dimension[10];
   float scale;
   int low, high;

   /* Get parameters */
   printf("Enter field:");
   scanf("%s", field);

   printf("Enter dimension:");
   scanf("%s", dimension);

   printf("Enter scale:");
   scanf("%f", &scale);

   printf("Enter low:");
   scanf("%d", &low);

   printf("Enter high:");
   scanf("%d", &high);

   /* Do scaling */
   if (pdim_scale(&pdim, field[0], dimension[0], scale, low, high) == INVALID) {
      fprintf(stderr,"Scale error: %s\n", _imerrbuf);
      return(1); }

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine rotates a pdim record.                             */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Rotate()
   {
   char field[10];
   char dimension[10];
   float angle;
   int low, high;

   /* Get parameters */
   printf("Enter field:");
   scanf("%s", field);

   printf("Enter dimension:");
   scanf("%s", dimension);

   printf("Enter angle:");
   scanf("%f", &angle);

   printf("Enter low:");
   scanf("%d", &low);

   printf("Enter high:");
   scanf("%d", &high);

   /* Do rotation */
   if (pdim_rotate(&pdim, field[0], dimension[0], angle, low, high) == INVALID){
      fprintf(stderr,"Rotate error: %s\n", _imerrbuf);
      return(1); }

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine translates a pdim record.                          */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Translate()
   {
   char field[10];
   char dimension[10];
   float dist;
   int low, high;

   /* Get parameters */
   printf("Enter field:");
   scanf("%s", field);

   printf("Enter dimension:");
   scanf("%s", dimension);

   printf("Enter distance:");
   scanf("%f", &dist);

   printf("Enter low:");
   scanf("%d", &low);

   printf("Enter high:");
   scanf("%d", &high);

   /* Do translation */
   if (pdim_translate(&pdim, field[0], dimension[0], dist, low, high) 
      == INVALID) {
      fprintf(stderr,"Translate error: %s\n", _imerrbuf);
      return(1); }

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine uses pdim info to interpret x,y,z coordinates.     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Map()
   {
   /* Local variables */
   char field[10];
   int u,v,w;
   float x,y,z,t;

   /* Get parameters */
   printf("Enter field:");
   scanf("%s", field);

   /* Loop getting pixels and converting to patient coordinates */
   while (TRUE)
      {
      /* Initialize input array */
      printf("Enter pixel coordinate (u,v,w):");
      scanf("%d %d %d", &u, &v, &w);
      if (u == -1) break;

      /* Convert to patient coordinates */
      if (pdim_map(&pdim, field[0], u,v,w, &x,&y,&z,&t) == INVALID) {
         fprintf(stderr,"Map error: %s\n", _imerrbuf);
         return(1); }

      /* Printing patient coordinates */
      printf("Patient coordinate (x,y,z,t): %f %f %f %f\n", x,y,z,t);
      }

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  It is often hard to visualize the relationships between         */
/*	     the slices of a 3D image, especially when the "pdim" field      */
/*	     specifying this relationship is complex.  To solve this         */
/*           problem, we display the slices in scale and in their proper     */
/*	     relative positions on the Vector General.                       */
/*                                                                           */
/*           This routine reads the "pdim" field of an image and generates   */
/*           four vectors representing the maximum extent of each of the     */
/*           slices in the image.  These can be displayed (along with the    */
/*           X,Y and Z axis) on the Vector General by "vgvu xxx.vg".         */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Show()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];
   char name2[80];
   char field[10];
   FILE *Fd;
   int pixformat, dimc, *dimv;
   int  slice, i, j;
   int u[4], v[4];
   float x[4], y[4], z[4], time;

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);
   printf("Enter vector file name:");
   scanf("%s", name2);
   printf("Enter field:");
   scanf("%s", field);

   /* Open image files */
   if ((image1 = imopen(name1,READ)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}

   /* Open data file */
   Fd = fopen(name2, "w");
   if (Fd == NULL){
      fprintf(stderr,"Could not open vector file\n");
      return(1);}
   
   /* Initialize image parameters */
   if (imdim(image1, &pixformat, &dimc) == INVALID){
      fprintf(stderr,"Can not dimension\n"); 
      return(1);}
   if (dimc!=3) {
      fprintf(stderr,"Image not 3D\n"); 
      return(1);}
   dimv = (int *)malloc((unsigned)sizeof(int)*dimc);
   if (dimv==NULL){ 
      fprintf(stderr,"Allocation error\n");
      return(1); }
   if (imbounds(image1,dimv) == INVALID){
      fprintf(stderr,"Can not get bound\n");
      return(1); }

   /* Read the pdim field for the image */
   if (pdim_read(image1, &pdim) == INVALID){
      fprintf(stderr,"Can not get pdim field\n");
      return(1); }

   /* Set extents in the X and Y dimensions */
   u[0] = 0;           /* xmin */
   v[0] = 0;           /* ymin */
   u[1] = 0;           /* xmin */
   v[1] = dimv[1];     /* ymax */
   u[2] = dimv[2];     /* xmax */
   v[2] = dimv[1];     /* ymax */
   u[3] = dimv[2];     /* xmax */
   v[3] = 0;           /* ymin */

   /* Loop through each slice in the Z dimension */
   for(slice=0; slice<dimv[0]; slice++)
      {
      /* Convert 4 pixels to patient coordinates */
      for (i=0; i<4; i++) {
         if (pdim_map(&pdim, field[0], u[i],v[i],slice, 
            &x[i],&y[i],&z[i],&time) == INVALID) {
            fprintf(stderr,"Map error: %s\n", _imerrbuf);
            return(1); }
         }
         
      /* Print the world coordinates in VGVU format */
      for (i=0; i<4; i++) {
         j = (i+1) % 4;
         fprintf(Fd, "%d %d %d %d %d %d\n", 
            (int)(x[i]), (int)(y[i]), (int)(z[i]), 
            (int)(x[j]), (int)(y[j]), (int)(z[j]));
         }
      }

   /* Close image files */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}
   fclose(Fd);

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine copys all info fields from one image to another.   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Copy()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];
   IMAGE *image2;
   char name2[80];

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);
   printf("Enter destination image name:");
   scanf("%s", name2);

   /* Open image files */
   if ((image1 = imopen(name1,READ)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}
   if ((image2 = imopen(name2,UPDATE)) == NULL){
      fprintf(stderr,"Can not open destination image\n"); 
      return(1);}

   /* Copy info fields to specified image */
   if (imcopyinfo(image1,image2) == INVALID) {
      fprintf(stderr, "Copy error: %s\n", _imerrbuf);
      return(1);}

   /* Close images */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}
   if (imclose(image2) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }
