/*
  File: wff2rgb.c
  Authors: Yun Wang,
           K.R. Sloan
  Last Modified: 21 March 1991
  Purpose: convert a .wff file to Img RGB format

  NOTE: The Img RGB file has four files:
        <name>.a:  contains attributes like width and height
        <name>.r:  contains  RED component
        <name>.g:  contains  GREEN component
        <name>.b:  contains  BLUE component

*/

#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <sys/types.h>
#include "wff.h"

unsigned char *malloc();

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is:\n\t%s [-h][-v][-f FileName]\n",RoutineName);
 } 

static void FatalError(s)
 char *s;
 { 
  fprintf(stderr,"%s:FatalError(%s)\n",RoutineName,s);
  exit(1);
 }

static void Pass(inFD, outFD)
 FILE *inFD;
 FILE *outFD[];
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right, BitsPerBand;
  int KnownColorSystem, BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  int x,y,b,width,height,length,i,c;
  unsigned char *Red, *Green, *Blue;
  unsigned short *WFFScanLine;
  int WFFScanLineLength;

  FrameBuffer = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer)) 
   {
    fprintf("%s: OpenFB failed\n", RoutineName);
    exit(-1);
   }
  
  if (FAILURE == ReadImage(inFD, FrameBuffer)) 
   {
    fprintf(stderr,"%s: ReadImage failed\n", RoutineName);
    (void)CloseFB(&FrameBuffer);
    exit(-1);
   }

  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right)) 
   { 
    fprintf(stderr,"%s: GetBounds failed\n", RoutineName);
     (void)CloseFB(&FrameBuffer); exit(-1);
   }

  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   { 
    fprintf(stderr,"%s: GetColorsystem failed\n", RoutineName);
    (void)CloseFB(&FrameBuffer); exit(-1);
   }

  if (0 == strcmp(WhatBands,"RGB")) BandsPerPixel = 3; 
  else 
   {
    fprintf(stderr,"%s: Sorry, we only do RGB format\n",RoutineName);
    (void)CloseFB(&FrameBuffer); exit(-1);
   }

  width = Right - Left + 1;
  height = Top - Bottom + 1;
 
  if (VERBOSE)
   {
    fprintf(outFD[0],"%4d %4d \n",width,height);
    fprintf(stderr,"%4d %4d \n",width,height);
   }
 
  if (VERBOSE)
   fprintf(stderr,"%s: BandsPerPixel = %d; BitsPerBand = %d; \n",
     RoutineName, BandsPerPixel, BitsPerBand);


  WFFScanLineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanLine = (unsigned short *) malloc(WFFScanLineLength);
  if ((unsigned short *)0 == WFFScanLine)
   {
    fprintf (stderr,"%s: no memory for WFFScanLine\n", RoutineName);
    (void)CloseFB(&FrameBuffer); exit(-1);
   }

  length = width*height;

  Red = (unsigned char *)calloc(sizeof(unsigned char),length);
  if (!Red) FatalError("no memory for Red data");

  Green = (unsigned char *)calloc(sizeof(unsigned char),length);
  if (!Green) FatalError("no memory for Green data");

  Blue = (unsigned char *)calloc(sizeof(unsigned char),length);
  if (!Blue) FatalError("no memory for Blue data");

  if (VERBOSE) fprintf(stderr,"%s: Started to write the image\n", RoutineName);

  i=0;
  for (y=Top;y>=Bottom;y--)
   {
    if (FAILURE == GetBlock(FrameBuffer,y,Left,y,Right, WFFScanLine))
     { 
      fprintf("%s: GetBlock Failed",RoutineName);
      free(Red); free(Green); free(Blue);
      free(WFFScanLine);(void)CloseFB(&FrameBuffer);exit(-1);
     }
    for(x=Left;x<=Right;x++)
      {
        Red[i]   = (unsigned char) WFFScanLine[0+x*BandsPerPixel];
        Green[i] = (unsigned char) WFFScanLine[1+x*BandsPerPixel];
        Blue[i]  = (unsigned char) WFFScanLine[2+x*BandsPerPixel];
        i++;
       }
   }
   if (VERBOSE)
    {
     fprintf(stderr,"length=%d\n",length);
     fprintf(stderr,"i=%d\n",i);
    }  

   fwrite(Red,sizeof(unsigned char),length,outFD[1]);
   if (ferror(outFD[1]))
    fprintf(stderr,"%s: Error writing Red File \n", RoutineName);
   if (VERBOSE)
    fprintf(stderr,"%s: finished writing Red File\n", RoutineName);
   fflush(outFD[1]);
   free(Red);
 
   fwrite(Green,sizeof(unsigned char),length,outFD[2]);
   if (ferror(outFD[2]))
    fprintf(stderr,"%s: Error writing Green File \n", RoutineName);
   if (VERBOSE)
    fprintf(stderr,"%s: finished writing Green File\n", RoutineName);
   fflush(outFD[2]);
   free(Green);
  
   fwrite(Blue,sizeof(unsigned char),length,outFD[3]);
   if (ferror(outFD[3]))
    fprintf(stderr,"%s: Error writing Blue File  \n", RoutineName);
   if (VERBOSE)
    fprintf(stderr,"%s: finished writing Blue File\n", RoutineName);
   fflush(outFD[3]);
   free(Blue);

   free(WFFScanLine);
   (void)CloseFB(&FrameBuffer);  
 }

int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int b;
  char *FileName[4],*BaseName;
  FILE *fdOut[4];

  BaseName = (char *)0;
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'f': BaseName = argv[ArgsParsed++];
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': { usage(); exit(-1); }
     }
   }
 
  if ((char *)0 == BaseName) { usage(); exit(-1); }

  for (b=0;b<4;b++)
   {
   FileName[b]=(char *)malloc(sizeof(char)*20);
   strcpy(FileName[b],BaseName);
   }
  strcat(FileName[0],".a");
  strcat(FileName[1],".r"); 
  strcat(FileName[2],".g");
  strcat(FileName[3],".b");

  if (VERBOSE)
   {
    fprintf(stderr,"FileName[0] %s\n",FileName[0]);
    fprintf(stderr,"FileName[1] %s\n",FileName[1]);
    fprintf(stderr,"FileName[2] %s\n",FileName[2]);
    fprintf(stderr,"FileName[3] %s\n",FileName[3]);
   }

  for (b=0;b<4;b++)
    if (!(fdOut[b] = fopen(FileName[b],"w"))) FatalError("fopen failed");

  Pass(stdin, fdOut);

  exit(0); 
 }
