/*
  File: WSINC.c 
  Authors: K.R. Sloan
  Last Modified: 28 June 1989
  Purpose: Convolve with a windowed sinc, parameterized by:
             z = distance to first zero crossing       (default 2.0 pixels)
             w = number of zero crossings on each side (default 4)
             g = gain          (default 1.0)
             d = dc            (default 0.0)  
           Gain is the area under EACH of the two 1D filters.
           DC is the amount added to EACH of the two 1D filters.

           An important special case is when the images are offset
           so that middle-gray is considered to be the "zero" value.

           In this case, when we apply a gain factor g, we should
           automatically apply a DC correction of (g-1.0)/2.0.

           To do this, we offer a composite parameter:
             G = gain          (default 1.0)
                 dc = (gain-1.0)/2.0

           Yet another special case is (very) scattered data points
           in a field of gray.  Here, we want to set the gain (and DC)
           indirectly by setting the peak value of the kernel.  So...another
           parameter (-P). 

             P = peak [gain and dc derived indirectly]

 */

#include <stdio.h>
#include <math.h>
#include "wff.h"

#define ROUND(x) ((int)(0.5+(x)))

#define SwitchChar '-'
#define MaxFilterWidth 1000

double atof();


static int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr, "Usage is\n\t%s [-h][-v][-z z][-n n][-g gain][-d DC][-G gain][-P p]\n",
              RoutineName);
 }

int main(argc, argv)
 int argc;
 char *argv[]; 
 {
  int ArgsParsed = 0;
  double z = 2.0;
  int w = 4;
  double gain = 1.0;
  double DC = 0.0;
  double Peak = 0.0;
  int n, HalfN, k, k0, v;
  int Kernel[MaxFilterWidth];
  double c, u;
  int Sum;
  double Correction;
  int Correction2;

  RoutineName = argv[ArgsParsed++];

  while (argc > ArgsParsed)
   {
    if (SwitchChar == argv[ArgsParsed][0])
     switch (argv[ArgsParsed++][1])
      {
       case 'z': z     = atof(argv[ArgsParsed++]); break;
       case 'w': w     = atoi(argv[ArgsParsed++]); break;
       case 'g': gain  = atof(argv[ArgsParsed++]); break;
       case 'd': DC    = atof(argv[ArgsParsed++]); break;
       case 'G': gain  = atof(argv[ArgsParsed++]);
                 DC    = 0.5*(1.0-gain);
                 break;
       case 'P': Peak  = atof(argv[ArgsParsed++]); break;
       case 'v': VERBOSE = 1; break;
       case 'h':
       default:  usage(); exit(-1);  
      }
    else  { usage(); exit (-1); }	
   }

  HalfN = ROUND(z*w);
  n = 2+HalfN+HalfN;
  if (n>MaxFilterWidth)
    {
     n = MaxFilterWidth;
     HalfN = (n-2)/2;      /* belt */
     n = 2+HalfN+HalfN;    /* suspenders */
    } 

  if (0.0 != Peak)
   {
    gain = Peak*z;
    DC   = 0.5*(1.0-gain);
    c    = 2048*Peak;
   }
  else c = 2048.0*gain/z;

  Kernel[0] = ROUND(2048.0*DC);
  Kernel[1+HalfN] = c;
  for(k=1;k<=HalfN;k++)
   {
    u = 3.1415926*(double)k/z;
    v = c*sin(u)*sin(u/w)*w/(u*u);
    Kernel[1+HalfN+k] = v;
    Kernel[1+HalfN-k] = v;
   }

  /* correct errors due to truncating the kernel */
   
  Sum = 0;  for(k=1;k<n;k++) Sum += Kernel[k];
  Correction = ROUND(2048.0*gain)/(double)Sum;
  if (VERBOSE)
   fprintf(stderr,
          "z = %f; w = %d; gain = %f; DC = %f\nSum = %d; Correction = %f\n",
           z, w, gain, DC, Sum, Correction);

  /* correct round-off errors */

  for (k=1;k<n;k++) Kernel[k] = (int)((double)Kernel[k]*Correction);
  Sum = 0;  for(k=1;k<n;k++) Sum += Kernel[k];
  Correction2 = ROUND(2048.0*gain)-Sum;
  Kernel[1+HalfN] += Correction2;

  if (VERBOSE)
   {
    fprintf(stderr,"Secondary Correction = %f\n",Correction2/2048.0);
   }

  /* look for zero tails - these are common with wide filters */
  for(k0=0;(k0<HalfN) && (0 == Kernel[1+k0]);k0++) ;

  if (0<k0)
   {
    HalfN = HalfN-k0;
    n = 2+HalfN+HalfN;
    for(k=1;k<n;k++) Kernel[k] = Kernel[k+k0];
   }

  if (VERBOSE)
   {
    fprintf(stderr,"DC  : %15.12f\n",(double)Kernel[0]/2048.0);
    for(k=1;k<n;k++)
     fprintf(stderr,"%4d: %15.12f\n", k-HalfN-1,(double)Kernel[k]/2048.0);
   } 

  (void)wffConvHV(stdin,stdout,n,n,Kernel,Kernel);

  exit (0);
 }
