/*
 File: Delaunay.c
 Authors: K.R. Sloan
 Last Modified: 28 August 1992
 Purpose: triangulates a set of points

          Options:
            -A AnimationFile    the name of an ANIMATION file, 
                              a sgp meta-file.  See DCEL.c to decipher the
                              format.  "-" indicates stdout. 

 The intention is that this file will be read by a display process,
 which will show triangles being created and destroyed.

 NOTE: this version reads the first two numbers on each line,
       and discards the remainder of the line.  This allows us 
       to look at point sets which have information associated
       with each 2D point (like, say, a height)


 */
#include <stdio.h>
#include <math.h>
#include <DCEL.h>
#include <GrahamHull.h>
#include <Triangulate.h>

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-h][-v][-z][-A AnimationFileName]\n",
               RoutineName);
 }

static int VERBOSE = 0;
static int DISCARDZ = -1;
static int LastPoint;
static double wL,wB,wR,wT, Cx, Cy, R;

 /*
   Read the data file and initialize
  */
static int ReadPoints(s)
 FILE *s;
 {
  int i;

  for(i=0;;i++)
   {
    if (   feof(s)
        || (2 != fscanf(s," %lf%*[ ,]%lf", &Points[i].x, &Points[i].y)) )
     {
      /*
        some programs have the convention that the last two points are
        (L,B) and (R,T) - to establish a particular window.  Look for this
        case, and fix it.
       */
      if ((wR == Points[i-1].x) && (wT == Points[i-1].y)) i--;
      if ((wL == Points[i-1].x) && (wB == Points[i-1].y)) i--;
      if ((wR == Points[i-1].x) && (wT == Points[i-1].y)) i--;
      if ((wL == Points[i-1].x) && (wB == Points[i-1].y)) i--;
       
      Cx = 0.5*(wL+wR); Cy = 0.5*(wB+wT);
      if ((wR-wL)>(wT-wB)) R = 0.5*(wR-wL); else R = 0.5*(wT-wB);
      wL = Cx - R;  wB = Cy - R;
      wR = Cx + R;  wT = Cy + R;

      return(i);
     }
    if (DISCARDZ) fscanf(s,"%*[^\n]");
    if ((0 == i) || (wL > Points[i].x)) wL = Points[i].x;
    if ((0 == i) || (wB > Points[i].y)) wB = Points[i].y;
    if ((0 == i) || (wR < Points[i].x)) wR = Points[i].x;
    if ((0 == i) || (wT < Points[i].y)) wT = Points[i].y;
    Points[i].Edges = (EdgePointer)0;
   }
 }

 /*
   Write out the triangulation.  The set of triangles is Pointed at by
   the GLOBAL variable Triangles.

   ASSUMPTION: Triangles is NOT nil.
  */
static void TriangleDump()
 {
  TrianglePointer T;
  int count;

  count = 0;      
  T = Triangles;
  for (;;)
   {
    count = count + 1;
    T = T->Next;
    if (T == Triangles) break;
   }
  fprintf(stdout,"%5d\n", count);
  T = Triangles;
  for (;;)
   {
    fprintf(stdout,"%5d %5d %5d\n", T->P1, T->P2, T->P3);
    T = T->Next;
    if (T == Triangles) break;
   }  
 } /* TriangleDump  */

 /*
   Write the output file.  Copy the input, and tack on the triangles.
  */
static void OutputFile(s)
 FILE *s;
 {
  int i;

  fprintf(s,"%5d\n",LastPoint);
  for (i=0;i<LastPoint;i++)
   {
    fprintf(s,"%15.8e %15.8e\n", Points[i].x, Points[i].y);
   }
  if (Triangles != (TrianglePointer)0) TriangleDump();
 } /* OutputFile */

 /*
  At last...the main line (are we in Philadelphia, yet?)
  */

int main (argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed=0;
  char *AFileName = (char *)0;

  ANIMATING = false;
  Triangles = (TrianglePointer)0;
  Suspects = (EdgePointer)0;
  Hull = (PointSetPointer)0;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     switch (argv[ArgsParsed++][1])
      {
       case 'A': 
        if (ArgsParsed >= argc) {usage(); exit(-1);}
        AFileName = argv[ArgsParsed++];
        Animate(AFileName);
        break;
       case 'z': DISCARDZ = -1; break;
       case 'v': VERBOSE  = -1; break;
       default:
       case 'h': usage(); exit(-1);
      }
   }

  LastPoint = ReadPoints(stdin);
  if (VERBOSE) fprintf(stderr,"%s: Read %d points\n",RoutineName,LastPoint);

  if (ANIMATING)
   {
    double Wider;

    Wider = 1.01 * R;
    fprintf(AnimationFile,"%f %f %f %f sgpSetWindow\n",
             Cx-Wider,Cy-Wider,Cx+Wider,Cy+Wider);
   }

  Triangulate(LastPoint);
  if (VERBOSE) fprintf(stderr,"%s: Triangulation done\n",RoutineName);

  if (!ANIMATING || (stdout != AnimationFile)) OutputFile(stdout);

  if (ANIMATING) fprintf(AnimationFile,"sgpQuit\n");

  exit(0);
 }
