/*
   File: framebuffer.c
   Authors: Tony DeRose,
            K.R. Sloan
   Last Modified: 30 August 1990
   Purpose: simple framebuffer abstraction, using X11
            this is a binary display, with the origin at the lower left 

            rectangles are specified as: Left, Top, Right, Bottom

            It is assumed (BUT NOT CHECKED) that:
                     Left   <= Right
                     Bottom <= Top

            This version adds mouse tracking and button pushes.

 */

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include "framebuffer.h"

XWMHints xwmh =
 {
  (InputHint|StateHint),    /* flags */
  False,                    /* input */
  NormalState,              /* initial state */
  0,                        /* icon pixmap */
  0,                        /* icon window */
  0, 0,                     /* icon location */
  0,                        /* icon mask  */
  0                         /* Window group */      
 };

/* X-Framebuffer definitions */
#define fb_NAME "FrameBuffer"
#define fb_FONT "6x13"

/* Variables local to the library */
int WindowOpened = FALSE;         /* TRUE if the window is opened */
static Display *dpy ;             /* The display to post the window to */
static Window win;                /* Window descriptor of framebuffer */
static GC gc;                     /* GC to draw with */        
static XFontStruct *fontstruct;   /* font descriptor */
static XEvent event;              /* Event received */
static XSizeHints xsh;            /* Size hints for window manager */
static char *geomspec;            /* Window geometry string */
static XSetWindowAttributes xswa; /* Temporary Set Window Attribute */
static XWindowAttributes xwa;     /* the returned truth */ 
static Cursor NoButtons, Buttons; /* cursors, eh? */
static unsigned long fg, bg, bd;  /* pixel values */

/* Exported variables */
/*
  fb_error_message points at an explanation of your error
 */
char *fb_error_message = NULL;
  
/*
  Create the frame buffer window, display it on the screen,
  and return the range of its device coordinates.
*/
int fb_open(Left, Bottom, Right, Top)
 int *Left, *Bottom, *Right, *Top;
 {
  unsigned long bw;           /* border width */ 
  unsigned long pad;          /* padding for text */
  XGCValues gcv;              /* Struct for creating GC */
  
  /*
    Open the display, using the DISPLAY environment variable
   */
  if (WindowOpened) 
   {
    fb_error_message = "fb_open: attempt to open the already opened framebuffer\n";
    return(FB_ERROR);
   }
  dpy = XOpenDisplay((char *)0);
  if ((Display *)0 == dpy)
   {
    fb_error_message = "fb_open: couldn't open the display";
    return(FB_ERROR);
   }



  /*
     Load a font
   */
/*
  DEBUGGING 

  fontstruct = XLoadQueryFont(dpy, fb_FONT); 
  if ((XFontStruct *)0 == fontstruct)
   {
    fb_error_message = "fb_open: couldn't load a font";
    return(FB_ERROR);
   }
 */



  /*
    Select colors
   */
  bd = BlackPixel(dpy,DefaultScreen(dpy));
  bg = BlackPixel(dpy,DefaultScreen(dpy));
  fg = WhitePixel(dpy,DefaultScreen(dpy));

  /*
    set border width and pad
   */
  pad = 1; bw = 1;
  
  /*
    provide hints for initial position and size
   */

  xsh.flags = (PPosition | PSize | PMinSize | PMaxSize | PResizeInc);
  xsh.height = 240;
  xsh.width  = 320;
  xsh.x = (DisplayWidth (dpy, DefaultScreen(dpy)) - xsh.width ) / 2;
  xsh.y = (DisplayHeight(dpy, DefaultScreen(dpy)) - xsh.height) / 2;
  xsh.min_width = 320;  xsh.min_height = 240;
  xsh.max_width = 1280;  xsh.max_height = 920;
  xsh.width_inc = 16;    xsh.height_inc = 12;
  
  /*
    set the border, background, foreground pixels
   */
  xswa.background_pixel = bg;
  xswa.border_pixel = bd;

  /*
    create two cursors
   */
  NoButtons = XCreateFontCursor(dpy,XC_crosshair);
  Buttons = XCreateFontCursor(dpy,XC_circle);

  xswa.cursor = NoButtons;

  /*
    make sure we have a colormap
    set bit gravity to minimize Expose events
   */
  xswa.colormap = DefaultColormap(dpy, DefaultScreen(dpy));
  xswa.bit_gravity = CenterGravity;


  /*
    Create the window
   */
  win = XCreateWindow( dpy, DefaultRootWindow(dpy),
            xsh.x, xsh.y, xsh.width, xsh.height, bw,
            CopyFromParent, InputOutput, CopyFromParent,
     (CWBackPixel | CWBorderPixel | CWColormap | CWBitGravity | CWCursor),
	    &xswa);
  /*
    Set standard properties for the window managers
   */
  XSetStandardProperties(dpy, win, fb_NAME, fb_NAME, None, "", 0, &xsh);
  XSetWMHints(dpy, win, &xwmh);

  /*
    create Graphics Context
   */
/*  gcv.font = fontstruct->fid; */
  gcv.foreground = fg;
  gcv.background = bg;
  gc = XCreateGC(dpy, win, (/* GCFont | */ GCForeground | GCBackground), &gcv);
 
  /*
    allow Exposure events    
   */

  XSelectInput(dpy, win, ExposureMask);
 
  /*
    map window to make it visible
   */

  XMapWindow(dpy, win);

  /*
    wait for an exposure event
   */
  while(1)
   {
    XNextEvent(dpy, &event);
    if (  (Expose == event.type) && (0 == event.xexpose.count) )
     {
      while (XCheckTypedEvent(dpy, Expose, &event)); /* get them all */
      break;
     }
   }
 
  /*
    refuse all events
   */
  XSelectInput(dpy,win,NoEventMask);
  
  /*
    Find out what size the window ended up being set to 
   */
  if (0 == XGetWindowAttributes(dpy,win,&xwa))
   {
    fb_error_message = "Couldn't query the frame buffer window.";
    return FB_ERROR;
   }

  /*
    Set the output parameters - notice that we are reporting the
    coordinate system we are exporting - NOT the coordinate system
    provided by X11.  
   */
  *Left = 0;
  *Bottom = 0;
  *Right = xwa.width  - 1;
  *Top   = xwa.height - 1;



  /*
    Everything went well...
   */
  WindowOpened = TRUE;
  return FB_SUCCESS;
 }


/*
  Destroy the frame buffer window.
 */
int fb_close()
 {
  if (!WindowOpened) 
   {
    fb_error_message = 
        "fb_close: attempt to close already closed framebuffer.";
    return FB_ERROR;
   }

  XDestroyWindow(dpy,win);
  WindowOpened = FALSE;
  XSync(dpy,1);
  return FB_SUCCESS;
 }

/*
   Write a pixel in the given color.
 */
int fb_writePixel(x, y, color)
 int x, y;
 FB_COLOR color;
 {
  if (!WindowOpened) 
   {
    fb_error_message = "fb_writePixel: window not opened.";
    return FB_ERROR;
   }
  switch (color) 
   {
    case FB_WHITE:
     XSetForeground(dpy, gc, WhitePixel(dpy,DefaultScreen(dpy)));
     break;
    case FB_BLACK:
     XSetForeground(dpy, gc, BlackPixel(dpy,DefaultScreen(dpy)));
     break;
    default:
     fb_error_message = "fb_writePixel: unknown color.";
     return FB_ERROR;
   }
  XDrawPoint(dpy, win, gc,
              x, xwa.height-1-y); /* reverse y coord */
  return FB_SUCCESS;
 }

/*
   paint a rectangle in the given color
 */
int fb_writeRectangle(L, B, R, T, color)
 int L, T, R, B, color;
 {
  if (!WindowOpened) 
   {
    fb_error_message = "fb_writeRectangle: window not opened.";
    return FB_ERROR;
   }
  switch (color) 
   {
    case FB_WHITE:
     XSetForeground(dpy, gc, WhitePixel(dpy,DefaultScreen(dpy)));
     break;
    case FB_BLACK:
     XSetForeground(dpy, gc, BlackPixel(dpy,DefaultScreen(dpy)));
    break;
   }
  XFillRectangle(dpy, win, gc, 
                  L, xwa.height-1-T,  /* reverse y-coordinate */
                  R-L+1, T-B+1);
  fb_flush();
  return FB_SUCCESS;
 }

/*
  Draw a line in the given color.
 */
int fb_drawLine(x1, y1, x2, y2, color)
 int x1, y1, x2, y2;
 FB_COLOR color;
 {
  if (!WindowOpened) 
   {
    fb_error_message = "fb_drawLine: window not opened.";
    return FB_ERROR;
   }

  switch (color) 
   {
    case FB_WHITE:
     XSetForeground(dpy, gc, WhitePixel(dpy,DefaultScreen(dpy)));
     break;
    case FB_BLACK:
     XSetForeground(dpy, gc, BlackPixel(dpy,DefaultScreen(dpy)));
    break;
   }
  /* reverse y coordinate */ 
  XDrawLine(dpy, win, gc,
              x1, xwa.height-1-y1,   /* reverse y-coordinate */
              x2, xwa.height-1-y2);  /* reverse y-coordinate */
  fb_flush();
 }

/*
  Draw some text at the given position.
 */
int fb_text( x, y, str, color)
 int x, y;
 char *str;
 FB_COLOR color;
 {
  if (!WindowOpened) 
   {
    fb_error_message = "fb_text: window not opened.";
    return FB_ERROR;
   }

  switch (color) 
   {
    case FB_WHITE:
     XSetForeground(dpy, gc, WhitePixel(dpy,DefaultScreen(dpy)));
     break;
    case FB_BLACK:
     XSetForeground(dpy, gc, BlackPixel(dpy,DefaultScreen(dpy)));
    break;
   }
  /* reverse y-coordinate */
  XDrawString(dpy, win, gc, 
                x, xwa.height-1-y, /* reverse y-coordinate */
               str, strlen(str));
  fb_flush();
  return FB_SUCCESS;
 }

/*
 fb_flush() flushes all buffers
 */
int fb_flush()
 {
  XFlush(dpy);    
  return(FB_SUCCESS);
 }

/*
 fb_pick( x, y, buttons ) returns the location of the mouse, and
 the status of the buttons
 */
int fb_pick( x, y, buttons )
 int *x, *y;
 int *buttons;
 {
  int MouseX, MouseY, MouseButtons;
  int GotEvent;
  
  /*
    allow Motion and Button
   */

  XSelectInput(dpy, win,
               PointerMotionMask | ButtonMotionMask 
             | ButtonPressMask | ButtonReleaseMask);
 
  /*
    wait for an event
   */
  do
   {
    XNextEvent(dpy, &event);
    switch (event.type)
     {
      case ButtonPress:
        MouseButtons = 1;
        MouseX = event.xbutton.x;
        MouseY = event.xbutton.y;
        break;
      case ButtonRelease:
        MouseButtons = 0;
        MouseX = event.xbutton.x;
        MouseY = event.xbutton.y;
        break;
      case MotionNotify:
        MouseButtons = event.xmotion.state & (  Button1Mask
                                              | Button2Mask
                                              | Button3Mask
                                              | Button4Mask
                                              | Button5Mask);
        MouseX = event.xmotion.x;
        MouseY = event.xmotion.y;
        break;
      default: break;
     }
   } while (XPending(dpy));
  /*
    refuse all events
   */
  XSelectInput(dpy,win,NoEventMask);

  *x = MouseX; 
  *y = xwa.height-1-MouseY;
  *buttons = (MouseButtons != 0);

  if (*buttons) XDefineCursor(dpy,win,Buttons);
  else          XDefineCursor(dpy,win,NoButtons);
  XSync(dpy,1);

  return(0); /* success */
 }
 
