/*
 * Copyright (c) 1991, 1992 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
char copyright[] =
    "@(#) Copyright (c) 1991, 1992 The Regents of the University of California.\nAll rights reserved.\n";
static  char rcsid[] =
    "@(#)$Header: histtrim.c,v 1.16 92/01/26 13:32:33 leres Exp $ (LBL)";
#endif

/*
 * histtrim - trim the history file
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#ifndef SYSV
#include <strings.h>
#else
#include <string.h>
#include <unistd.h>
#define bcopy(s,d,l) memmove((d),(s),(l))
#define index strchr
#define rindex strrchr
#endif
#include <ctype.h>
#include <errno.h>

#include "dexpire.h"
#include "hash.h"
#include "util.h"

/* Private data */
static char *dfeedback = DFEEDBACK;
static char *history = "history";
static char *nhistory = "history.n";
static char *ehistory = "history.e";
static char *expires = EXPIRES;
static char *nexpires = NEXPIRES;
static char *spool_dir = SPOOL_DIR;

static time_t currenttime;	/* current timestamp */
static time_t mintime = 0;	/* keep all history entries newer than this */

/* Public data */
char *prog;

int debug = 0;			/* debugging modes */
int verbose = 0;		/* verbose information to stdout */
int doexpires = 0;		/* update the explicit expires file */

/* Statistics */
int bad = 0;
int newnogroup = 0;		/* number of newly created "no groups" */
int oldnogroup = 0;		/* number of "no groups" omitted */
int nogroup = 0;		/* number of "no groups" in new history */
int nofeedback = 0;		/* number not in the feedback file */
int neverexpired = 0;
int emitted = 0;
int numread = 0;		/* number of old history lines read */

/* External data */

extern char *optarg;
extern int optind, opterr;

extern int errno;

main(argc, argv)
	int argc;
	char **argv;
{
	register int op;
	register char *cp;
	register FILE *fin, *fout, *fexp, *eout;
	int status;
	char *usage = "usage: %s [-duv] [-m days]\n";

	if (cp = rindex(argv[0], '/'))
		prog = cp + 1;
	else
		prog = argv[0];

	/* Process arguments */
	while ((op = getopt(argc, argv, "duvm:")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'm':
			mintime = atoi(optarg);
			if (mintime <= 0 || mintime > 365) {
				(void) fprintf(stderr,
				    "%s: \"%s\" invalid argument to -m\n",
				    prog, optarg);
				exit(1);
				/* NOTREACHED */
			}
			/* Convert days to seconds */
			mintime *= 24 * 60 * 60;
			break;

		case 'u':
			++doexpires;
			break;

		case 'v':
			++verbose;
			break;

		default:
			(void) fprintf(stderr, usage, prog);
			exit(1);
			/* NOTREACHED */
		}

	if (optind != argc) {
		(void) fprintf(stderr, usage, prog);
		exit(1);
		/* NOTREACHED */
	}

	/* Fetch current time (used in various calculations) */
	currenttime = time(0);

	/* Convert from delta to timestamp */
	mintime = currenttime - mintime;

	/* Report various times */
	if (verbose) {
		(void) printf("%s: Current time: %s\n",
		    prog, fmtdate(currenttime));
		if (mintime != currenttime)
			(void) printf("%s: Minimum time: %s\n",
			    prog, fmtdate(mintime));
		else
			(void) printf("%s: No minimum time\n", prog);
		(void) fflush(stdout);
	}

	/* Load in the feedback file */
	if ((fin = fopen(dfeedback, "r")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(dfeedback);
		exit(1);
		/* NOTREACHED */
	}
	if (!readfeedback(fin, hashadd))
		exit(1);
	(void)fclose(fin);

	/* Dump hash table if debugging */
	if (debug > 2)
		hashdump();

	/* Open the current history file */
	if ((fin = fopen(history, "r")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(history);
		exit(1);
		/* NOTREACHED */
	}

	/* Create the new history file */
	if ((fout = fopen(nhistory, "w")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(nhistory);
		exit(1);
		/* NOTREACHED */
	}

	/* Create the error history file */
#ifdef FUNET_EDEBUG
	if ((eout = fopen(ehistory, "w")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(nhistory);
		exit(1);
		/* NOTREACHED */
	}
#endif

	/* If requested, create the explicit expires file */
	fexp = 0;
	if (doexpires && (fexp = fopen(nexpires, "w")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(nexpires);
		/* Non-fatal error */
	}

	/* Process the history file */
	status = histtrim(fin, fout, fexp, eout);

	/* Check for errors and close the files */
	if (ferror(fin)) {
		(void) fprintf(stderr, "%s: ferror(): ", prog);
		perror(history);
		status |= 1;
	}
	if (ferror(fout)) {
		(void) fprintf(stderr, "%s: ferror(): ", prog);
		perror(nhistory);
		status |= 1;
	}
	if (fexp && ferror(fexp)) {
		(void) fprintf(stderr, "%s: ferror(): ", prog);
		perror(nexpires);
		/* Non-fatal error */
		(void)fclose(fexp);
		fexp = 0;
	}
	(void)fclose(fin);
	(void)fclose(fout);
#ifdef FUNET_EDEBUG
	(void)fclose(eout);
#endif
	if (fexp)
		(void)fclose(fexp);

	/* Rename the explicit expires file */
	if (fexp && rename(nexpires, expires) < 0 &&
	    unlink(expires) < 0 &&
	    rename(nexpires, expires) < 0) {
		(void)fprintf(stderr, "%s: rename(): %s -> ", prog, nexpires);
		perror(expires);
		/* Non-fatal error */
	}

	if (verbose) {
		(void) fflush(stderr);
		(void) printf("%s: %d bad\n", prog, bad);
		(void) printf("%s: %d newly created \"no groups\"\n",
		    prog, newnogroup);
		(void) printf("%s: %d old \"no groups\"\n", prog, oldnogroup);
		(void) printf("%s: %d \"no groups\" omitted\n", prog, nogroup);
		(void) printf("%s: %d not in the nofeedback file\n",
		    prog, nofeedback);
		(void) printf("%s: %d neverexpired\n", prog, neverexpired);
		(void) printf("%s: %d numread\n", prog, numread);
		(void) printf("%s: %d emitted\n", prog, emitted);
		(void) printf("%s: Took %s\n",
		    prog, fmtdelta(time(0) - currenttime));
	}

	exit(status);
}

histtrim(fin, fout, fexp, eout)
	FILE *fin, *fout, *fexp, *eout;
{
	/* XXX want this to be a register */
	char *cp, *cp2;
	register int h;
	register int numkept;
	time_t t, ht;
	char line[10240];
	register int n;
	register int hasexpires;
	int keep;

	n = 1;

	while (fgets(line, sizeof(line), fin)) {
		/* Statistics */
		++numread;

		/* Flag the fact that we haven't counted these yet */
		numkept = -1;

		/* Assume no explicit expires date */
		hasexpires = 0;

		/* Step over message id */
		cp = index(line, '\t');
		if (cp == 0) {
			(void) fprintf(stderr, "%s: parse error #1 line %d\n",
			    prog, n);
			++bad;
#ifdef FUNET_EDEBUG
			fputs("Error #1: ", eout);
			fputs(line, eout);
#endif
			goto emit;
		}
		++cp;

		/* Arrival timestamp */
		t = atol(cp);

		/* Optionally look for explicit expires */
		if (fexp) {
			cp = index(cp, '~');
			if (cp == 0) {
				(void) fprintf(stderr,
				    "%s: parse error #2 line %d\n", prog, n);
				++bad;
#ifdef FUNET_EDEBUG
				fputs("Error #2: ", eout);
				fputs(line, eout);
#endif
				goto emit;
			}

			/* Step over '~' */
			++cp;

			/* Look for explicit expires time and date */
			hasexpires = (*cp != '-');
		}

		/* Find newsgroups */
		cp = index(cp, '\t');

		/* Check for newsgroups */
		if (cp == 0) {
			/* No newsgroups */
			if (t < mintime) {
				/* History entry is too old; suppress */
				++oldnogroup;
				continue;
			}
			/* Keep history entry */
			++nogroup;
			goto emit;
		}

		/* Point to start of newsgroups */
		++cp;

		/* Save pointer to start of first newsgroup */
		cp2 = cp;

		/* Loop through newsgroup(s) */
		numkept = 0;
		for (;;) {
			keep = 0;
			/* Hash groupname and get group time */
			h = hashgroup(&cp2);
			ht = hashfind(h, cp);

			if (ht < 0) {
				/* Not in feedback file */
				++nofeedback;

				/* Keep history entry if article exists */
				if (artexists(cp))
					++keep;
			} else if (ht == 0) {
				/* "Never expired" newsgroup */
				++neverexpired;

				/* Keep history entry if article exists */
				if (artexists(cp))
					++keep;
			} else if (t >= ht) {
				/* Keep if not too old */
				++keep;
			}

			/* Update number of groups kept */
			numkept += keep;

			if (*cp2=='\0') {
				goto mta_failback;
			}
			/* Step over article number */
			if (*cp2 != '/') {
				(void) fprintf(stderr,
				    "%s: parse error #3 on line %d\n",
				    prog, n);
				++bad;
#ifdef FUNET_EDEBUG
				fputs("Error #3: ", eout);
				fputs(line, eout);
#endif
				goto emit;
			}
			++cp2;
			while (isdigit(*cp2))
				++cp2;

			if (*cp2 == '\n') {
				/* End of line; all done */
				if (!keep) {
					--cp;
					if (!isspace(*cp))
						++cp;
					*cp++ = '\n';
					*cp = '\0';
				}
				break;
			} else if (*cp2 == ' ') {
				/* Step over whitespace */
				++cp2;
				if (!keep) {
					/* Remove newsgroup */
					bcopy(cp2, cp, strlen(cp2) + 1);
					cp2 = cp;
				} else
					cp = cp2;
			} else {
				(void) fprintf(stderr,
				    "%s: parse error #4 on line %d\n",
				    prog, n);
#ifdef FUNET_EDEBUG
				fputs("Error #4: ", eout);
				fputs(line, eout);
#endif
				goto emit;
			}
		}

mta_failback:
		if (numkept == 0) {
			/* History entry now has "no groups" */
			++newnogroup;

			/* Suppress if too old */
			if (t < mintime) {
				++oldnogroup;
				continue;
			}
			++nogroup;
		}
emit:
		/* Output this (possibly modified) history line */
		fputs(line, fout);
		++n;

		/* Only add to expires file if there are newsgroups */
		if (hasexpires && numkept != 0) {
			/* Look for at least one newsgroup if necessary */
			if (numkept < 0 &&
			    (cp = index(line, '\t')) && index(cp, '\t') == 0)
				numkept = 1;
			if (numkept > 0)
				fputs(line, fexp);
		}
	}
	emitted = n - 1;
	return(0);
}

/* Check if an article actually exists */
artexists(art)
	register char *art;
{
	register int i;
	register char *path;


	path = artpath(spool_dir, art);
	i = access(path, F_OK);
	if (debug > 1)
		(void) fprintf(stderr, "%s: checking (%s): %s\n",
		    prog, i < 0 ? "fails" : "succeeds", path);
	if (i < 0)
		return(0);
	return(1);
}

/* Read and parse the feedback file */
readfeedback(f, fn)
	register FILE *f;
	int (*fn)();
{
	register int n;
	register char *cp, *cp2;
	char line[1024];
	time_t t;

	n = 0;
	while (fgets(line, sizeof(line), f)) {
		++n;
		cp = line;
		/* String trailing newline */
		cp2 = cp + strlen(cp) - 1;
		if (cp2 >= cp && *cp2 == '\n')
			*cp2++ = '\0';
		t = atoi(cp);
		if ((cp = index(cp, '\t')) == 0) {
			(void) fprintf(stderr,
			    "%s: feedback file syntax error #1 line %d\n",
			    prog, n);
			return(0);
		}
		++cp;
		/* Convert '/'s to '.'s a la the history file */
		for (cp2 = cp; *cp2 != '\0'; ++cp2)
			if (*cp2 == '/')
				*cp2 = '.';
		(void)(*fn)(cp, t);
	}

	/* Paranoid safety check */
	if (ferror(f)) {
		(void) fprintf(stderr, "%s: ", prog);
		perror("readfeedback()");
		return(0);
	}

	return(1);
}
