/*
 * Copyright (c) 1991, 1992 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
char copyright[] =
    "@(#) Copyright (c) 1991, 1992 The Regents of the University of California.\nAll rights reserved.\n";
static  char rcsid[] =
    "@(#)$Header: dexpire.c,v 1.74 92/01/26 13:31:56 leres Exp $ (LBL)";
#endif

/*
 * dexpire - dynamic expire for netnews
 *
 * Most (if not all) other expire programs force the news administrator
 * to predict how many days worth of news articles they have room for.
 * This approach is fundamentally incorrect.
 *
 * A better way is to specify the relative importance or priority
 * of newsgroups and have the expire program delete just enough
 * articles in the desired proportion to free the required number
 * of disk blocks. This is how dexpire works.
 *
 * Dexpire breaks all newsgroups into "classes." Each class has a
 * priority associated with it. High priority groups are kept for
 * a longer period than low priority groups.
 *
 * The "standard" class is the highest priority class. Articles in
 * newsgroups in the standard class are kept the longest. The standard
 * class is also used to determine how long lower priority classes
 * are kept.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef SYSV
#include <strings.h>
#else
#define bzero(s,l) memset((s), 0, (l))
#include <string.h>
#define rindex strrchr
#endif
#include <ctype.h>

#include "dexpire.h"
#include "disk.h"
#include "file.h"
#include "util.h"
#include "patchlevel.h"

/* Class node */
struct class {
	long pri;		/* priority of this class */
	long deleted;		/* number of articles deleted */
	long freed_blk;		/* number of 512 bytes blocks freed */
	time_t minimum;		/* minimum time to keep articles */
	time_t to;		/* timestamp of oldest article in class */
	struct group *group;	/* group list */
	struct class *next;	/* forward link */
};

/* Expire rule list */
struct rule {
	char *ngn;		/* newsgroup name, e.g. alt/slack */
	char mod;		/* moderation character */
	char chud[3];
	struct class *class;	/* pointer back to class */
	struct rule *next;	/* forward link */
};

/* Newsgroup node */
struct group {
	char *ngn;		/* newsgroup name, e.g. alt/slack */
	long no;		/* number of oldest article */
	long nn;		/* number of newest article */
	time_t to;		/* timestamp of oldest article */
	struct group *next;	/* forward link */
};

/* Converts 512 byte blocks to Kbytes */
#define BLK2KB(b) ((b) / 2)

#ifdef SYSV
#define B2BLK(b) (((b) + 511) / 512)
#endif

/* Private data */
static char *active_file = ACTIVE_FILE;
static char *spool_dir = SPOOL_DIR;
static char *dexplist = DEXPLIST;
static char *dfeedback = DFEEDBACK;
static char *ndfeedback = NDFEEDBACK;

static long free_kb;		/* current number of free Kbytes in spool_dir */
static long used_kb;		/* current number of used Kbytes in spool_dir */
static long inodes;		/* current number of used inodes in spool_dir */
static long want_kb = 4000;	/* desired number of free Kbytes in spool_dir */
static long need_kb;		/* needed number of free Kbytes in spool_dir */
static long freed_blk = 0;	/* number of 512 bytes blocks freed */

static time_t standardtime;	/* "standard" timestamp */
static time_t currenttime;	/* current timestamp */
static time_t deltatime = 0;	/* delta of current and standard timestamps */
static time_t dtime;		/* delta added to standardtime in loop */
static time_t ioldesttime;	/* initial oldest timestamp */

static int deleted = 0;		/* number of articles deleted */
static double density;		/* Kbytes/inode for spool dir */
static togo;			/* iterations before dtime calculation */

static struct class *classlist = 0;
static struct class *standardclass = 0;
static struct rule *rulelist = 0;

static FILE *feed;		/* feedback file descriptor */

static int newgroup(), nextrule();
static int expireone(), dexpire();
static void findstandard(), newdtime(), dump();

/* Public data */
char *prog;

int nflag = 0;			/* non-destructive mode */
int feedback = 0;		/* update the feedback file */
int debug = 0;			/* debugging modes */
int verbose = 0;		/* verbose information to stdout */

/* External data */

extern char *optarg;
extern int optind, opterr;

extern char *version;

main(argc, argv)
	int argc;
	char **argv;
{
	register int op, status;
	register char *cp;
	register FILE *f;
	register struct class *cl;
	register struct group *gl;
	register int i;
	time_t d, oldesttime;
	double r;
	char *usage =
    "usage: %s [-dnuv] [-a active] [-c dexplist] [-s spool_dir] [-f Kbytes]\n";

	if (cp = rindex(argv[0], '/'))
		prog = cp + 1;
	else
		prog = argv[0];

	/* Process arguments */
	while ((op = getopt(argc, argv, "dnuva:c:d:f:s:")) != EOF)
		switch (op) {

		case 'a':
			active_file = optarg;
			break;

		case 'c':
			dexplist = optarg;
			break;

		case 'd':
			++debug;
			break;

		case 'f':
			want_kb = atoi(optarg);
			if (want_kb <= 0) {
				(void)fprintf(stderr,
				    "%s: \"%s\" invalid argument to -f\n",
				    prog, optarg);
				exit(1);
				/* NOTREACHED */
			}
			/* Trailing 'm' means Mbytes */
			for (cp = optarg; isdigit(*cp); ++cp)
				;
			if (*cp == 'm' || *cp == 'M')
				want_kb *= 1024;
			break;

		case 'n':
			++nflag;
			break;

		case 's':
			spool_dir = optarg;
			break;

		case 'u':
			++feedback;
			break;

		case 'v':
			++verbose;
			break;

		default:
			(void)fprintf(stderr, usage, prog);
			exit(1);
			/* NOTREACHED */
		}

	if (optind != argc) {
		(void) fprintf(stderr, usage, prog);
		exit(1);
		/* NOTREACHED */
	}

	if (verbose>2) { /* mta */
		(void)fprintf(stderr, "%s: Verbose level 3\n", prog);
	}
	/* Fetch current time which is used in various calculations */
	currenttime = time(0);
	if (verbose) {
		(void)printf("%s: Version %s, patchlevel %d\n",
		    prog, version, PATCHLEVEL);
		(void)printf("%s: Current time: %s\n",
		    prog, fmtdate(currenttime));
		(void)printf("%s: Want %ld Kbyte%s free\n",
		    prog, want_kb, PLURAL(want_kb));
		(void)printf("%s: Spool directory is %s\n", prog, spool_dir);
	}

	/* Update various disk parameters */
	if (disk_usage(spool_dir, &used_kb, &free_kb, &inodes) < 0) {
		(void)fflush(stdout);
		(void)fprintf(stderr, "%s: disk_usage(): ", prog);
		perror(spool_dir);
		exit(1);
		/* NOTREACHED */
	}

	/* Calculate how much space we need */
	need_kb = want_kb - free_kb;
	if (verbose)
		(void)printf("%s: Have %ld Kbyte%s free\n",
		    prog, free_kb, PLURAL(free_kb));

	/* Bail if we already have enough free space */
	if (need_kb <= 0) {
		(void)printf("%s: Nothing to do!\n", prog);
		exit(0);
		/* NOTREACHED */
	}
	if (verbose)
		(void)printf("%s: Need to free %ld Kbyte%s\n",
		    prog, need_kb, PLURAL(need_kb));

	/* Complain if asked to free too much */
	r = ((double)want_kb) / (((double) free_kb) + ((double)used_kb));
	if (r >= MAX_FREE && !nflag) {
		(void)fflush(stdout);
		(void)fprintf(stderr,
		    "%s: Cannot free %d%% of the filesystem (%d%% max)",
		    prog, (int)(100.0 * r), (int)(100.0 * MAX_FREE));
		(void)fprintf(stderr, "; use -n if testing\n");
		exit(1);
		/* NOTREACHED */
	}

	/* Calculate file system density (in Kbytes/inode) */
	if (verbose)
		(void)printf("%s: Spool directory ", prog);
	if (inodes < 0) {
		density = DEFAULT_DENSITY;
		if (verbose)
			(void)printf("inode usage unknown, assuming ");
	} else {
		density = ((double)used_kb) / ((double)inodes);
		if (verbose)
			(void)printf("averages ");
	}
	if (verbose) {
		(void)printf("%.1f Kbytes/inode\n", density);
		(void)fflush(stdout);
	}

	/* Read in the dexplist config file */
	if ((f = fopen(dexplist, "r")) == 0) {
		(void)fprintf(stderr, "%s: fopen(): ", prog);
		perror(dexplist);
		exit(1);
		/* NOTREACHED */
	}
	if (!readexplist(f, nextrule)) {
		exit(1);
		/* NOTREACHED */
	}
	(void)fclose(f);

	/* Create the new feedback file, if necessary */
	if (feedback && !nflag) {
		if (verbose)
			(void)printf("%s: Creating feedback file\n", prog);
		if ((feed = fopen(ndfeedback, "w")) == 0) {
			(void)fprintf(stderr, "%s: fopen(): ", prog);
			perror(ndfeedback);
			status |= 1;
			feedback = 0;
		}
	}

	/* Read in the active file */
	if ((f = fopen(active_file, "r")) == 0) {
		(void)fprintf(stderr, "%s: fopen(): ", prog);
		perror(active_file);
		exit(1);
		/* NOTREACHED */
	}
	if (!readactive(f, newgroup)) {
		exit(1);
		/* NOTREACHED */
	}
	(void)fclose(f);

	/* Dump out data structures if debugging */
	if (debug)
		dump();

	/* Determine the "standard" time */
	findstandard();

	/* Go delete some articles */
	status = dexpire();

	/* Update the feedback file, if necessary */
	if (feedback) {
		if (nflag) {
			(void)fprintf(stderr,
			    "%s: Skipping feedback file update\n", prog);
		} else {
			if (verbose)
				(void)printf("%s: Updating feedback file\n",
				    prog);
			for (cl = classlist; cl; cl = cl->next)
				for (gl = cl->group; gl; gl = gl->next)
					(void)fprintf(feed, "%d\t%s\n",
					    gl->to, gl->ngn);
			(void)fclose(feed);
			if (rename(ndfeedback, dfeedback) < 0 &&
			    unlink(dfeedback) < 0 &&
			    rename(ndfeedback, dfeedback) < 0) {
				(void)fprintf(stderr, "%s: rename(): %s -> ",
				    prog, ndfeedback);
				perror(dfeedback);
				status |= 1;
			}
		}
	}

	if (verbose) {
		/* Report final class times */
		for (cl = classlist; cl; cl = cl->next)
			(void)printf("%s: Final %3d%%: %s (%s)\n",
			    prog, (100 * cl->pri) / standardclass->pri,
			    fmtdate(cl->to), fmtdelta(currenttime - cl->to));

		/* Report per-class deletion statistics */
		for (cl = classlist; cl; cl = cl->next) {
			(void)printf("%s: Class %3d%%",
			    prog, (100 * cl->pri) / standardclass->pri);
			if (cl->minimum) {
				i = cl->minimum / (24 * 60 * 60);
				(void)printf(" (%d day%s)", i, PLURAL(i));
			}
			(void)printf(": %seleted %d article%s (%ld Kbyte%s)\n",
			    nflag ? "Would have d" : "D",
			    cl->deleted, PLURAL(cl->deleted),
			    BLK2KB(cl->freed_blk),
			    PLURAL(BLK2KB(cl->freed_blk)));
		}

		(void)printf(
		    "%s: %seleted %d article%s total (%ld Kbyte%s), took %s\n",
		    prog, nflag ? "Would have d" : "D",
		    deleted, PLURAL(deleted),
		    BLK2KB(freed_blk), PLURAL(BLK2KB(freed_blk)),
		    fmtdelta(time(0) - currenttime));

		/* Determine new oldest article time */
		oldesttime = currenttime;
		for (cl = classlist; cl; cl = cl->next)
			for (gl = cl->group; gl; gl = gl->next)
				if (gl->to < oldesttime)
					oldesttime = gl->to;

		d = currenttime - ioldesttime;
		(void)printf("%s: Old oldest %s (%s)\n",
		    prog, fmtdate(ioldesttime), fmtdelta(d));
		d = currenttime - oldesttime;
		(void)printf("%s: New oldest %s (%s)\n",
		    prog, fmtdate(oldesttime), fmtdelta(d));

		d = ioldesttime - oldesttime;
		cp = "gained";
		if (d < 0) {
			d = -d;
			cp = "lost";
		}
		(void)printf("%s: Article delta %s %s\n",
		    prog, cp, fmtdelta(d));
		(void)fflush(stdout);
	}

	exit(status);
	/* NOTREACHED */
}

/* Handle the next rule */
static int
nextrule(group, mod, pri, days)
	char *group, mod;
	int pri, days;
{
	register struct class *cl, *cl2;
	register struct rule *rp;
	register time_t minimum;
	static struct rule *lastrule = 0;

	if (pri < 0) {
		(void)fprintf(stderr,
		    "%s: priority for group %s is negative (%d)",
		    prog, group, pri);
		exit(1);
		/* NOTREACHED */
	}
	if (days < 0) {
		(void)fprintf(stderr,
		    "%s: minimum days for group %s is negative (%d)",
		    prog, group, days);
		exit(1);
		/* NOTREACHED */
	}
	minimum = days * 24 * 60 * 60;

	/* Save new rule */
	rp = (struct rule *)mymalloc(sizeof(*rp), "rule");
	bzero((char *)rp, sizeof(*rp));
	if (rulelist == 0)
		rulelist = rp;
	else
		lastrule->next = rp;
	lastrule = rp;
	rp->ngn = savestr(group);
	rp->mod = mod;

	/* Don't create a new class for priority 0 */
	if (pri == 0)
		return(1);

	/* Look for existing class with the same priority and minimum */
	cl2 = 0;
	for (cl = classlist; cl; cl = cl->next) {
		if (cl->pri == pri && cl->minimum == minimum)
			break;
		if (cl->pri < pri || (cl->pri == pri && cl->minimum < minimum))
			cl2 = cl;
	}

	/* New class; insert in ascending order */
	if (cl == 0) {
		cl = (struct class *)mymalloc(sizeof(*cl), "class");
		bzero((char *)cl, sizeof(*cl));
		if (cl2) {
			cl->next = cl2->next;
			cl2->next = cl;
		} else {
			cl->next = classlist;
			classlist = cl;
		}
		cl->pri = pri;
		cl->minimum = minimum;
	}

	/* Find the "standard" class */
	for (cl2 = classlist; cl2; cl2 = cl2->next)
		standardclass = cl2;

	/* Backpointer from new rule to class */
	rp->class = cl;

	return(1);
}

/* Handle the next newsgroup from the active file */
static int
newgroup(group, nn, no, mod)
	char *group;
	int nn, no;
	char mod;
{
	register struct group *gl;
	register struct class *cl;
	register struct rule *rp;
	register char *cp, *cp2;

	/* Locate the class for this newsgroup */
	for (rp = rulelist; rp; rp = rp->next) {
		/* Check moderated flag */
		if (rp->mod != mod &&
		    rp->mod != 'x' &&
		    (rp->mod != 'u' || mod != 'y'))
			continue;

		/* Check for a group match */
		cp = group;
		cp2 = rp->ngn;
		while (*cp != '\0' && *cp == *cp2) {
			++cp;
			++cp2;
		}
		if (*cp2 == '\0' && (*cp == '\0' || *cp == '/'))
			break;

		/* "all" always matches */
		if (strcmp(rp->ngn, "all") == 0)
			break;
	}

	/* Gotta have a rule */
	if (rp == 0) {
		(void)fprintf(stderr,
		    "%s: Warning: no default rule, discarding %s\n",
		    prog, group);
		return(1);
	}

	/* Ingore groups in the 0 priority class */
	if (rp->class == 0) {
		if (verbose)
			(void)printf("%s: Skipping %s\n", prog, group);
		if (!nflag && feedback)
			(void)fprintf(feed, "0\t%s\n", group);
		return(1);
	}

	/* Build a new group entry */
	gl = (struct group *)mymalloc(sizeof(*gl), "group");
	bzero((char *)gl, sizeof(*gl));

	gl->ngn = savestr(group);
	gl->no = no;
	gl->nn = nn;
	cl = rp->class;
	gl->next = cl->group;
	cl->group = gl;

	return(1);
}

/* Fetch initial timestamp info */
static void
findstandard()
{
	register struct class *cl;
	register struct group *gl;
	register time_t start, t, d;
	register long n;
	register int numstats;
	register char *art;
	char file[512];
	struct stat sbuf;

	/* Get initial timestamp in case we need it later */
	start = time(0);

	/* Determine timestamp of oldest article in each class */
	numstats = 0;
	ioldesttime = currenttime;
	for (cl = classlist; cl; cl = cl->next) {
		/* Safe initial value */
		cl->to = currenttime;

		for (gl = cl->group; gl; gl = gl->next) {
			/*
			 * Change our working directory. Although this
			 * seems inefficient, it's about as expensive
			 * as checking to see if the newsgroup
			 * directory exists and can greatly reduce
			 * overhead if the third field of the active
			 * file isn't up to date.
			 */
			(void)sprintf(file, "%s/%s", spool_dir, gl->ngn);
			if (chdir(file) < 0) {
				gl->to = currenttime;
				continue;
			}

			t = 0;
			art = "?";
			for (n = gl->no; n <= gl->nn; ++n) {
				/* Construct article filename */
				art = long2str(n);

				/* Attempt to get timestamp */
				++numstats;
				if (stat(art, &sbuf) >= 0 &&
				    (t = sbuf.st_mtime) != 0)
					break;
			}
			if (t == 0) {
				gl->to = currenttime;
				continue;
			}

			/* Complain if from the first year of Unix history */
			if (t > 0 && t < 365 * 24 * 60 * 60)
				(void)fprintf(stderr,
				    "%s: Warning: %s/%s is old: %s\n",
				    prog, gl->ngn, art, fmtdate(t));

			gl->no = n;
			gl->to = t;
			if (t < cl->to) {
				cl->to = t;
				if (t < ioldesttime)
					ioldesttime = t;
			}
		}
	}

	/* Determine standard time */
	standardtime = currenttime;
	for (cl = classlist; cl; cl = cl->next) {
		if (verbose)
			(void)printf("%s: Oldest %3d%%: ",
			    prog, (100 * cl->pri) / standardclass->pri);

		/* Safety check */
		if (cl->to == 0) {
			if (verbose)
				(void)printf("has zero time!\n");
			continue;
		}

		/* Determine class delta time */
		d = currenttime - cl->to;
		if (verbose)
			(void)printf("%s (delta %s)\n",
			    fmtdate(cl->to), fmtdelta(d));

		/* Increase class delta by class ratio */
#ifdef ORIGINAL_DEXPIRE
		d = (d * standardclass->pri) / cl->pri;
#else
#if 0
		d = (d * standardclass->pri) / (standardclass->pri-cl->pri+1);
#endif
		d += (d * cl->pri ) / standardclass->pri;
#endif

		/* Convert to absolute time */
		t = currenttime - d;
		if (verbose > 1)
			(void)printf("%s: Expanded     %s (delta %s)\n",
			    prog, fmtdate(t), fmtdelta(d));

		if (t < standardtime)
			standardtime = t;
	}

	/* Calculate delta time */
	deltatime = currenttime - standardtime;

	if (verbose) {
		(void)printf("%s: Initial Standard: %s (%s)\n",
		    prog, fmtdate(standardtime), fmtdelta(deltatime));
		(void)printf("%s: (Took %s to check %d file%s)\n",
		    prog, fmtdelta(time(0) - start),
		    numstats, PLURAL(numstats));
		(void)fflush(stdout);
	}
}

/* Calculates new dtime and togo */
static void
newdtime()
{
	register long n, need;
	double r;

	need = need_kb - BLK2KB(freed_blk);
	if (verbose > 1)
		(void)printf("%s: newdtime(): Need to free %ld Kbyte%s\n",
		    prog, need, PLURAL(need));

	/* Calculate new togo based on file system density */
	togo = (int)(need / density * 0.25);
	if (verbose > 1)
		(void)printf("%s: New togo is ", prog);
	if (togo < MIN_TOGO) {
		if (verbose > 1)
			(void)printf("too small (%d), using ", togo);
		togo = MIN_TOGO;
	} else if (togo > MAX_TOGO) {
		if (verbose > 1)
			(void)printf("too big (%d), using ", togo);
		togo = MAX_TOGO;
	}
	if (verbose > 1)
		(void)printf("%d article%s\n", togo, PLURAL(togo));

	/* Calculate new dtime based on file system usage */
	n = used_kb;
	if (n == 0)
		n = 1;
	r = ((double)need) / ((double)n);
	if (verbose > 1)
		(void)printf("%s: Need to used ratio is %.1e\n", prog, r);
	dtime = (int)(deltatime * r * 0.25);
	if (verbose > 1)
		(void)printf("%s: New dtime is ", prog);
	if (dtime < MIN_DTIME) {
		if (verbose > 1)
			(void)printf("too short (%s), using ", fmtdelta(dtime));
		dtime = MIN_DTIME;
	} else if (dtime > MAX_DTIME) {
		if (verbose > 1)
			(void)printf("too long (%s), using ", fmtdelta(dtime));
		dtime = MAX_DTIME;
	}
	if (verbose > 1) {
		(void)printf("%s\n", fmtdelta(dtime));
		(void)fflush(stdout);
	}
}

/* Make a pass over one newsgroup. Returns true if enough space was freed */
static int
expireone(gl, cl, cutoff)
	register struct group *gl;
	register struct class *cl;
	register time_t cutoff;
{
	register long n;
	register int i, status;
	register int olddeleted;
	register time_t t;
	register char *art;
	char file[512];
	struct stat sbuf;

	/* Make sure there's something to do */
	if (gl->no > gl->nn) {
		if (verbose > 2)
			(void)printf("%s: No articles in %s\n", prog, gl->ngn);
		return(0);
	}

	/* Cached article time might tell us there's nothing to do */
	/* (This test fails if time isn't cached) */
	if (gl->to >= cutoff) {
		if (verbose > 2)
			(void)printf("%s: Cached time skip of %s\n",
			    prog, gl->ngn);
		return(0);
	}

	/* Change working directory to speed up directory lookups */
	(void)sprintf(file, "%s/%s", spool_dir, gl->ngn);
	if (chdir(file) < 0) {
		/* Skip next time and save the unsuccessful chdir() */
		gl->no = gl->nn + 1;
		gl->to = currenttime;
		if (verbose > 1)
			(void)printf("%s: No directory %s\n", prog, gl->ngn);
		return(0);
	}

	/* Nuke 'em, Dano! */
	if (verbose > 1)
		(void)printf("%s: Running %s\n", prog, gl->ngn);

	status = 0;
	t = 0;
	olddeleted = deleted;
	for (n = gl->no; n <= gl->nn; ++n) {
		/* Construct article filename */
		art = long2str(n);

		/* Skip if no file */
		t = 0;
		if (stat(art, &sbuf) < 0)
			continue;

		/* Complain if from the first year of Unix history */
		t = sbuf.st_mtime;
		if (t < 365 * 24 * 60 * 60)
			(void)fprintf(stderr,
			    "%s: Warning: %s/%s is old: %s\n",
			    prog, gl->ngn, art, fmtdate(t));

		/* If not old enough, we're done with this group */
		if (t >= cutoff)
			break;

		if (verbose>2) { /* mta */
			(void)fprintf(stderr,
			    "%s: Unlink %s/%s\n", prog, file, art);
		}

		if (!nflag) {
			/* Only chalk up the article if it was the last link */
			if (unlink(art) >= 0 && sbuf.st_nlink <= 1) {
				++deleted;
#ifndef SYSV
				freed_blk += sbuf.st_blocks;
#else
				freed_blk += B2BLK(sbuf.st_size);
#endif
				++cl->deleted;
#ifndef SYSV
				cl->freed_blk += sbuf.st_blocks;
#else
				cl->freed_blk += B2BLK(sbuf.st_size);
#endif
			}
		} else {
			/* Pretend to delete the article */
			++deleted;
			freed_blk += sbuf.st_blocks;
			++cl->deleted;
			cl->freed_blk += sbuf.st_blocks;
		}

		if (BLK2KB(freed_blk) >= need_kb) {
			if (verbose)
				(void)printf("%s: We're done!\n", prog);
			status = 1;
			break;
		}

		/* Occasionally recalculate dtime */
		if (--togo < 0)
			newdtime();
	}

	if (verbose > 1) {
		i = n - gl->no + 1;
		(void)printf("%s: wanted to delete %d article%s\n",
		    prog, i, PLURAL(i));
		i = deleted - olddeleted;
		if (i)
			(void)printf("%s: deleted %d article%s\n",
			    prog, i, PLURAL(i));
		(void)fflush(stdout);
	}

	/* Update oldest article and its timestamp */
	gl->no = n;
	if (t != 0)
		gl->to = t;
	else
		gl->to = currenttime;

	return(status);
}

/* Article deletion routine; returns desired exit() status */
static int
dexpire()
{
	register struct class *cl;
	register struct group *gl;
	register time_t cutoff, d;
	register int i, pass, olddeleted;
	register long n, oldfreed_blk;
	int done;

	/* Calculate initial dtime */
	newdtime();

	/*
	 * Basic rules:
	 *
	 *   - We hit the lowest priority classes first and work our
	 *     way up to the highest priority classes
	 *
	 *   - If a pass through all classes and newsgroups doesn't
	 *     yield enough space, we lower the "standard" by some
	 *     amount of time and start over.
	 *
	 *   - We don't test the free space as often when we have a
	 *     long ways to go.
	 *
	 *   - We lower the "standard" time at the start of the loop
	 *     since findstandard() picks a value that won't delete
	 *     any files.
	 */
	pass = 0;
	done = 0;
	for (;;) {
		/* Internal book keeping */
		olddeleted = deleted;
		oldfreed_blk = freed_blk;
		++pass;

		/* Increase standard time */
		standardtime += dtime;
		deltatime = currenttime - standardtime;
		if (deltatime <= 0) {
			(void)fprintf(stderr,
			    "%s: Standard time is in the future!\n", prog);
				return(1);
		}
		if (verbose > 1) {
			(void)printf("%s: Delta time: %s\n",
			    prog, fmtdelta(deltatime));
			(void)fflush(stdout);
		}

		/* Proceed from low class to high class */
		for (cl = classlist; cl; cl = cl->next) {
			/* Calculate cutoff for this class */
#if BROKEN_DELTA_ON_32_BIT_MACHINES
                        d = (deltatime * cl->pri) / standardclass->pri;
#else
                        d = (deltatime / standardclass->pri) * cl->pri;
#endif
			cutoff = currenttime - d;

			/* Skip entire class if we're reached the minimum */
			if (cl->minimum >= d) {
				if (verbose > 1) {
					(void)printf(
					    "%s: Skip   %3d%%: %s (%s)\n",
					    prog, (100 * cl->pri) /
					    standardclass->pri,
					    fmtdate(currenttime - cl->minimum),
					    fmtdelta(cl->minimum));
					(void)fflush(stdout);
				}
				continue;
			}

			/* Update class time */
			cl->to = cutoff;

			if (verbose > 1) {
				(void)printf("%s: Cutoff %3d%%: %s (%s)\n",
				    prog, (100 * cl->pri) / standardclass->pri,
				    fmtdate(cutoff), fmtdelta(d));
				(void)fflush(stdout);
			}

			/* Run through the groups */
			for (gl = cl->group; gl; gl = gl->next)
				if (expireone(gl, cl, cutoff)) {
					++done;
					goto bail;
				}
		}
bail:
		i = deleted - olddeleted;
		if ((verbose && i > 0) || verbose > 1) {
			n = BLK2KB(freed_blk - oldfreed_blk);
			(void)printf(
		    "%s: Pass %d %sdeleted %d article%s (%ld Kbyte%s)\n",
			    prog, pass, nflag ? "would have " : "",
			    i, PLURAL(i), n, PLURAL(n));
			(void)fflush(stdout);
		}
		if (done)
			return(0);
	}
}

/* Dump out internal data structures */
static void
dump()
{
	register struct class *cl;
	register struct rule *rp;
	register struct group *gl;

	(void)printf("------\nDiagnostic dump:\n");
	(void)printf("\nClasses:\n");
	for (cl = classlist; cl; cl = cl->next) {
		(void)printf("%3d:", cl->pri);
		if (cl->minimum > 0)
			(void)printf(" minimum %s", fmtdelta(cl->minimum));
		if (cl == standardclass) {
			if (cl->minimum > 0)
				putchar(',');
			(void)printf(" standard");
		}
		putchar('\n');
	}

	printf("\nRules:\n");
	for (rp = rulelist; rp; rp = rp->next) {
		(void)printf("%-32s %c\t",
		    rp->ngn, rp->mod);
		if (rp->class == 0)
			(void)printf("<never>");
		else {
			(void)printf("%d", rp->class->pri);
			if (rp->class->minimum)
				(void)printf("\t%s",
				    fmtdelta(rp->class->minimum));
		}
		putchar('\n');
	}

	(void)printf("\nGroups:\n");
	for (cl = classlist; cl; cl = cl->next) {
		(void)printf("%3d:", cl->pri);
		if (cl->group == 0) {
			putchar('\n');
			continue;
		}
		for (gl = cl->group; gl; gl = gl->next) {
			(void)printf("\t%-32s [%ld, %ld]",
			    gl->ngn, gl->no, gl->nn);
			if (gl == cl->group && cl->minimum > 0)
				(void)printf(" (%s)", fmtdelta(cl->minimum));
			putchar('\n');
		}
	}
	(void)printf("------\n");
	(void)fflush(stdout);
}
