static char rcsid[] = "urld.c,v 1.13 1996/01/04 04:14:44 duane Exp";
/*
 *  urld.c - Daemon that serves URL using an FTP connection cache.
 *
 *  David Merkel, University of Colorado - Boulder, July 1994
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include <stdio.h>
#include <sys/types.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include "url.h"
#include "ccache.h"
#include "urld.h"
#include "util.h"

/* Local functions */
static void ReportError();
static void ShutDown();
static void disconnect();

/* Local variables */
/* client conn queue and srvr socket global so signal handler can clean up */
static Queue *sockQueue = NULL;
static int servSocket;

static void usage()
{
	fprintf(stderr, "Usage: urld [-qMax client q connect] [-fMax ftp cache connect] [-tFtp cache timeout (in secs)]\n");
	exit(1);
}



/*
 * main() - starts up urld, TERM signal sent to process will cleanly shut down
 */
int main(argc, argv)
     int argc;
     char *argv[];
{
	int newSocket, lastSocket;
	int ftpConn, ftpTimeout, cQConn, i, errflg = 0;
	fd_set sockDetect;
	QNode *newQNode = NULL, *tempQNode = NULL, *delQNode;
	Boolean forceDQ;

	if (argc > 4)
		usage();

	/* set defaults and check for flags for user supplied
	 * parameters
	 */
	ftpConn = DEFAULT_MAX_CONN;
	ftpTimeout = DEFAULT_TIMEOUT;
	cQConn = DEFAULT_MAX_CONNECT;

	for (i = 1; (i < argc) && (!errflg); i++) {
		if (argv[i][0] != '-')
			errflg = 1;
		else
			switch (argv[i][1]) {
			case 'q':
				cQConn = atoi(&argv[i][2]);
				break;
			case 'f':
				ftpConn = atoi(&argv[i][2]);
				break;
			case 't':
				ftpTimeout = atoi(&argv[i][2]);
				break;
			default:
				errflg++;
				break;
			}
	}

	if (errflg)
		usage();

#ifdef DEBUG
	Log("urld v1.0\n");
#endif

#ifdef DAEMON
	disconnect();
#endif
	/* set signal, INT & TERM shuts down urld */
	signal(SIGINT, ShutDown);
	signal(SIGTERM, ShutDown);

	/* init liburl, logging, and FTP connection cache */
	init_log(stderr, stderr);
	init_url();
	url_initCache(ftpConn, ftpTimeout);

	/* Init daemon socket */
	servSocket = InitDSocket(SERVER_PORT);

	if (servSocket < 0) {
		ReportError(TRUE);
		exit(1);
	}
	/* Init server socket queue */
	sockQueue = InitQueue(cQConn);
	if (!sockQueue) {
		ReportError(FALSE);
		exit(1);
	}
	/* ready for connection accepts, go to loop */
	lastSocket = servSocket;
	while (1) {
		FD_ZERO(&sockDetect);
		FD_SET(servSocket, &sockDetect);
		tempQNode = sockQueue->head;

		/* set to listen on all open socks in q */
		while (tempQNode != NULL) {
			FD_SET(tempQNode->theSocket, &sockDetect);
			tempQNode = tempQNode->next;
		}

		/* wait for client requests and connections */
		newSocket = WaitForRequest(servSocket, &sockDetect, lastSocket);
		if (newSocket > lastSocket)
			lastSocket = newSocket;

		/* error while waiting for service requests and
		 * new client connections
		 */
		if ((newSocket < 0) && (GetURLDErr() != SERVICE_REQUEST)) {
			ReportError(TRUE);
			ShutDown(1);	/* END_OF_PROGRAM */
		}
		/* add newSocket to queue */
		if (GetURLDErr() == NO_ERROR) {
			if (fcntl(newSocket, O_NONBLOCK) < 0) {
#if DEBUG > 3
				log_errno("fcntl(O_NONBLOCK)");
#endif
				ShutDown(1);	/* END_OF_PROGRAM */
			}
			newQNode = InitQNode(newSocket);
			if (!newQNode) {
				ReportError(FALSE);
				ShutDown(1);	/* END_OF_PROGRAM */
			}
			tempQNode = Enqueue(sockQueue, newQNode);
			if ((!tempQNode) && (GetQErr() == DEQ_ERROR)) {
				ReportError(FALSE);
				ShutDown(1);	/* END_OF_PROGRAM */
			}
			/* notify client of dequeue disconnect */
			if (tempQNode) {

				write(tempQNode->theSocket, DISCNCT_STR, MSG_LEN);
				close(tempQNode->theSocket);
				free(tempQNode);
#ifdef	REPORT_OVERFLOW_DQ
				ReportError(FALSE);
#endif
			}
			FD_SET(newSocket, &sockDetect);
		}
		tempQNode = sockQueue->head;

		/* service all clients in queue */
		while (tempQNode) {
			forceDQ = FALSE;
			if (FD_ISSET(tempQNode->theSocket, &sockDetect)) {
#if DEBUG > 2
				printf("servicing client on socket: %d\n", tempQNode->theSocket);
#endif
				if (ServiceClient(tempQNode->theSocket) < 0) {
					if ((GetURLDErr() == NO_REQ_PRESENT) ||
					    (GetURLDErr() == CLIENT_DISCONNECT)) {
						delQNode = tempQNode;
						tempQNode = delQNode->next;
						close(delQNode->theSocket);
						(void) Dequeue(sockQueue,
						    delQNode);
						forceDQ = TRUE;
					}
					ReportError(TRUE);
				}
#if DEBUG > 2
				printf("client service complete\n\n");
#endif
			}
			if (!forceDQ)
				tempQNode = tempQNode->next;
		}
	}
	exit(0);
}


/*
 * ReportError() - reports error to STDOUT
 */
static void ReportError(whichError)
     Boolean whichError;
{
	/* a urld error occurred, report it */
	if (whichError) {
		switch (GetURLDErr()) {
		case URLD_GET_SOCK_ERR:
			log_errno("urld: socket");
			break;
		case BIND_SOCK_ERR:
			log_errno("urld: bind ");
			break;
		case URLD_GET_HOSTNAME_ERR:
			log_errno("urld: gethostname");
			break;
		case URLD_GET_HOSTBYNAME_ERR:
			log_errno("urld: gethostbyname");
			break;
		case URLD_GET_SOCKNAME_ERR:
			log_errno("urld: getsockname");
			break;
		case SELECT_ERR:
			log_errno("urld: select");
			break;
		case ACCEPT_ERR:
			log_errno("urld: accept");
			break;
		case URLD_LISTEN_ERR:
			log_errno("urld: listen");
			break;
		default:
			break;
		}
	}
	/* a q error occurred, report it */
	else {
		switch (GetQErr()) {
		case DEQ_ERROR:
			log_errno("urld: error on client dequeue");
			break;
		case DEQ_ON_ENQ:
			log_errno("urld: exceed maxConnect, dequeue one client");
			break;
		case CANT_MALLOC_Q:
			log_errno("urld: cannot malloc queue");
			break;
		case CANT_MALLOC_QNODE:
			log_errno("urld: cannot malloc queue node");
			break;
		default:
			break;
		}
	}
}


/*
 * ShutDown() - shuts down connection queue on urld termination
 * caused by SIGTERM signal, or clean up if urld main encounters error
 */
static void ShutDown(n)
     int n;
{
	QNode *tempQNode;

#if DEBUG > 3
	Log("Shutting down urld...\n");
#endif
	/* do clean up */
	url_shutdowncache();
	tempQNode = sockQueue->head;
	while (tempQNode) {
		write(tempQNode->theSocket, DISCNCT_STR, MSG_LEN);
		close(tempQNode->theSocket);
		tempQNode = tempQNode->next;
	}
	DestroyQueue(sockQueue);
	close(servSocket);
	exit(n);
}

/* 
 *  disconnect() - Disconnect the process from the controlling terminal.
 *  Adapted from Harvest's floodd daemon in the replicator.
 */
static void disconnect()
{
	int pid, fd;

#ifdef SIGTTOU
	signal(SIGTTOU, SIG_IGN);
#endif
#ifdef SIGTTIN
	signal(SIGTTIN, SIG_IGN);
#endif
#ifdef SIGTSTP
	signal(SIGTSTP, SIG_IGN);
#endif

	if ((pid = fork()) < 0) {
		log_errno("fork");
		return;
	}
	if (pid) {		/* parent */
		exit(0);	/* quietly exit to let child do the work */
	}
#ifdef HAVE_GETDTABLESIZE
	/* Close all file descriptors */
	for (fd = 0; fd < getdtablesize(); fd++)
		(void) close(fd);
#endif

	if (setpgrp(0, getpid()) < 0) {
		log_errno("setpgrp");
		exit(1);
	}
	/* Redirect the stdin, stdout, and stderr to /dev/null */
	if ((fd = open("/dev/null", O_RDWR)) < 0) {
		exit(1);
	}
	(void) dup2(fd, 0);
	(void) dup2(fd, 1);
	(void) dup2(fd, 2);
}
