static char rcsid[] = "oid.c,v 1.21 1996/01/05 20:28:56 duane Exp";
/*
 *  oid.c - Object Identifier code for the Essence system
 *
 *  Darren Hardy, hardy@cs.colorado.edu, February 1994
 *
 *  DEBUG: section  67, level 1         Gatherer essence object-ID code
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "util.h"
#include "essence.h"

/*
 *  generate_oid() - Generates an OID structure based on the given data.
 */
struct OID *generate_oid(url, gid, object)
     char *url;
     struct GID *gid;
     DataObject *object;
{
	time_t mtime, ttl, refresh;

	Debug(67, 1, ("generate_oid: Generating OID for %s\n", url));

	ttl = default_ttl;
	refresh = default_refresh;

	mtime = object->url->lmt;
	Debug(67, 1, ("generate_oid: mtime=%ld\n", mtime));

	return (new_oid(url, gid, mtime, ttl, refresh));
}

struct OID *new_oid(url, gid, timestamp, ttl, refresh)
     char *url;
     struct GID *gid;
     time_t timestamp, ttl, refresh;
{
	static struct OID *oid = NULL;
	Debug(67, 1, ("new_oid: New OID for %s\n", url));

	oid = xmalloc(sizeof(struct OID));
	oid->url = strdup(url);
	oid->gid.name = strdup(gid->name);
	oid->gid.host = strdup(gid->host);
	oid->gid.version = strdup(gid->version);
	oid->timestamp = timestamp;
	oid->ttl = ttl;
	oid->refresh = refresh;
	return (oid);
}

/*
 *  free_oid() - Cleans up an OID.
 */
void free_oid(oid)
     struct OID *oid;
{
	if (oid == NULL)
		return;
	if (oid->url)
		xfree(oid->url);
	if (oid->gid.name)
		xfree(oid->gid.name);
	if (oid->gid.host)
		xfree(oid->gid.host);
	if (oid->gid.version)
		xfree(oid->gid.version);
	xfree(oid);
}

/*
 *  extract_oid() - Extract an OID structure from a Template.  
 *  Returns NULL on error.
 */
struct OID *extract_oid(template)
     Template *template;
{
	char *url = NULL;
	struct GID gid;
	time_t timestamp = 0, ttl = 0, refresh = 0;
	AVList *walker = NULL;
	static struct OID *oid = NULL;

	/* Extract fields from the template */
	for (walker = template->list; walker; walker = walker->next) {
		if (!strcmp(walker->data->attribute, T_URL))
			url = strdup(walker->data->value);
		else if (!strcasecmp(walker->data->attribute, T_GNAME))
			gid.name = strdup(walker->data->value);
		else if (!strcasecmp(walker->data->attribute, T_GHOST))
			gid.host = strdup(walker->data->value);
		else if (!strcasecmp(walker->data->attribute, T_GVERSION))
			gid.version = strdup(walker->data->value);
		else if (!strcasecmp(walker->data->attribute, T_TTL))
			ttl = (time_t) atoi(walker->data->value);
		else if (!strcasecmp(walker->data->attribute, T_REFRESH))
			refresh = (time_t) atoi(walker->data->value);
		else if (!strcasecmp(walker->data->attribute, T_TIMESTAMP))
			timestamp = (time_t) atoi(walker->data->value);
	}

	/* Did we get all of the values needed? */
	if (!url || !gid.host || !gid.name || !gid.version)
		return (NULL);

	/* Create an OID, then free tmp space */
	oid = new_oid(url, &gid, timestamp, ttl, refresh);
	xfree(url);
	xfree(gid.name);
	xfree(gid.host);
	xfree(gid.version);

	return (oid);
}

/*
 *  create_template_with_oid() - Creats a new Template structure and
 *  integrates the given OID with it.
 */
Template *create_template_with_oid(type, url, oid)
     char *type;
     char *url;
     struct OID *oid;
{
	static Template *t = NULL;
	char buf[BUFSIZ];

	t = create_template(type, url);

	/* Integrate OID with the template */
	sprintf(buf, "%d", (int) time(NULL));
	t->list = create_AVList(T_UPDATE, buf, strlen(buf));

	if (do_minimalbooks)
		return (t);

	if (oid->timestamp > 0) {
		sprintf(buf, "%d", (int) oid->timestamp);
		add_AVList(t->list, T_TIMESTAMP, buf, strlen(buf));
	}
	sprintf(buf, "%u", (unsigned int) oid->ttl);
	add_AVList(t->list, T_TTL, buf, strlen(buf));

	sprintf(buf, "%d", (unsigned int) oid->refresh);
	add_AVList(t->list, T_REFRESH, buf, strlen(buf));

	add_AVList(t->list, T_GNAME, oid->gid.name, strlen(oid->gid.name));
	add_AVList(t->list, T_GHOST, oid->gid.host, strlen(oid->gid.host));
	add_AVList(t->list, T_GVERSION, oid->gid.version,
	    strlen(oid->gid.version));
	return (t);
}
