static char rcsid[] = "log.c,v 1.49 1996/01/17 10:07:39 duane Exp";
/*
 *   log.c -- Collects statistics information for the broker and logs it.
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include "broker.h"
#include "log.h"

/*
 *  USE_FAST_LOGGING - will omit fflush() calls if defined.
 */
#ifndef USE_FAST_LOGGING
#undef USE_FAST_LOGGING
#endif

/* Global variables */
extern time_t Cur_Time;

/* Local variables */
static FILE *log_file = NULL;
static unsigned int log_master;

int Log_Terse = 0;

/* Initialize logging facility */
int LOG_Init()
{
	char *filename;

	if (log_file) {
		Log("Log file already initialized.\n");
		return ERROR;
	}
	filename = UTIL_make_admin_filename("LOG");
	if ((log_file = fopen(filename, "a+")) == NULL) {
		log_errno(filename);
		xfree(filename);
		errorlog("Cannot initialize log file.\n");
		return ERROR;
	}
	xfree(filename);
	(void)chmod(filename, 0640);	/* not readable by public */

	(void)UTIL_Get_Time();
	fprintf(log_file, "BEGIN-SESSION\t%d::\n", Cur_Time);
	(void)LOG_flush_log();	/* ensure that this is printed */

	return SUCCESS;
}

/* Log Something */
int LOG_statlog(va_alist)
va_dcl
{
	va_list args;
	time_t curtime;
	unsigned int type;
	char *outstr = NULL;
	char *statname = NULL;
	char *url = NULL;
	char *gn = NULL;
	char *gh = NULL;
	char *gv = NULL;
	int gid;
	fd_t FD;

	if (!log_file)
		return ERROR;

	va_start(args);
	type = va_arg(args, unsigned int);
	if (LOG_do_logging(type) == TRUE) {
		statname = va_arg(args, char *);
		curtime = UTIL_Get_Time();
		fprintf(log_file, "%s\t%d::\t", statname, curtime);
		switch (type) {
		case L_UPDATE:
		case L_DELETE:
		case L_REFRESH:
		case L_CLEAN:
		case L_QUERY_R:
		case L_BULK_R:
			url = va_arg(args, char *);
			FD = va_arg(args, fd_t);
			gid = va_arg(args, int);
			if (Log_Terse) {
				fprintf(log_file, "%d\n", FD);
			} else {
				fprintf(log_file, "%d\t%d\t%s\n", FD, gid, url);
			}
			break;
		case L_COLTNG:
			vfprintf(log_file, "%s:%d\twhen:%d\n", args);
			break;
		case L_QUERY:
		case L_BULK:
		case L_ADMINC:
		case L_CONNECT:
		case L_ADMINC_R:
		case L_QUERYERR:
		case L_BULKERR:
			outstr = va_arg(args, char *);
			fprintf(log_file, "%s\n", outstr);
			break;
		case L_CLEANS:
		case L_COMPRESS:
			fprintf(log_file, "\n");
			break;
		case L_GID:
			gid = va_arg(args, int);
			gn = va_arg(args, char *);
			gh = va_arg(args, char *);
			gv = va_arg(args, char *);
			fprintf(log_file, "%d\t%s\t%s\t%s\n",gid,gn,gh,gv);
			break;
		default:
			errorlog("LOGGER: Serious internal error!\n");
			break;
		}
#ifndef USE_FAST_LOGGING
		LOG_flush_log();
#endif
	}
	va_end(args);
	return SUCCESS;
}

/* turn on logging for a type */
int LOG_turn_on(type)
unsigned int type;
{
	log_master = log_master | type;
	return SUCCESS;
}

int LOG_turn_off(type)
unsigned int type;
{
	if (LOG_do_logging(type) == TRUE)
		log_master = log_master ^ type;

	return SUCCESS;
}

/* log nothing */
int LOG_reset_master()
{
	log_master = 00;
	return SUCCESS;
}

/* flush log file */
int LOG_flush_log()
{
	fflush(log_file);
	return SUCCESS;
}

int LOG_close_log()
{
	(void)UTIL_Get_Time();
	fprintf(log_file, "END-SESSION\t%d::\n", Cur_Time);
	(void)LOG_flush_log();

	if (fclose(log_file) != 0) {
		log_file = NULL;
		return ERROR;
	}
	log_file = NULL;

	return SUCCESS;
}

unsigned int LOG_match_name(name)
char *name;
{
	if (strcasecmp(name, L_UPDATE_S) == 0)
		return L_UPDATE;
	if (strcasecmp(name, L_DELETE_S) == 0)
		return L_DELETE;
	if (strcasecmp(name, L_REFRESH_S) == 0)
		return L_REFRESH;
	if (strcasecmp(name, L_QUERY_S) == 0)
		return L_QUERY;
	if (strcasecmp(name, L_QUERY_RS) == 0)
		return L_QUERY_R;
	if (strcasecmp(name, L_CLEAN_S) == 0)
		return L_CLEAN;
	if (strcasecmp(name, L_COLTNG_S) == 0)
		return L_COLTNG;
	if (strcasecmp(name, L_ADMINC_S) == 0)
		return L_ADMINC;
	if (strcasecmp(name, L_ADMINC_RS) == 0)
		return L_ADMINC_R;
	if (strcasecmp(name, L_BULK_S) == 0)
		return L_BULK;
	if (strcasecmp(name, L_BULK_RS) == 0)
		return L_BULK_R;
	if (strcasecmp(name, L_CLEANS_S) == 0)
		return L_CLEANS;
	if (strcasecmp(name, L_COMPRESS_S) == 0)
		return L_COMPRESS;
	if (strcasecmp(name, L_CONNECT_S) == 0)
		return L_CONNECT;
	if (strcasecmp(name, L_GID_S) == 0)
		return L_GID;

	return (00);
}

/* 
 *  LOG_rotate - Rotates the log file.  On return, the logging begins again.
 */
int LOG_rotate()
{
	char *f1 = NULL;
	char *f2 = NULL;
	char *timebuf = NULL;

	(void)UTIL_Get_Time();
	fprintf(log_file, "ROTATE\t%d::\n", Cur_Time);
	if (LOG_close_log() != SUCCESS)
		return ERROR;

#ifdef HAVE_STRFTIME
	timebuf = xmalloc(BUFSIZ);
	(void)strftime(timebuf, BUFSIZ, "LOG.%y%m%d", localtime(&Cur_Time));
	f2 = UTIL_make_admin_filename(timebuf);
#else
	f2 = UTIL_make_admin_filename("LOG.old");
#endif
	f1 = UTIL_make_admin_filename("LOG");
	if ((access(f2, F_OK) == 0) ||
	    (rename(f1, f2) < 0)) {
		errorlog("LOG_rotate: Cannot rotate log: %s -> %s\n", f1, f2);
	}
	xfree(f1);
	xfree(f2);
	if (timebuf)
		xfree(timebuf);
	return LOG_Init();
}

FILE *LOG_fptr()
{
	return (log_file);
}

