/* $Id: client.c,v 1.7 1995/05/31 20:34:11 dante Exp $ */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <fcntl.h>
#include <syslog.h>
#include <command.h>
#if defined (HAVE_ARPA_INET_H)
#include <arpa/inet.h>
#endif
#include <netdb.h>
#include <dynamic_string.h>
#include <floodd.h>
#include <time.h>
#include <io-handler.h>

#define HTTP 1

char *end_of_line_marker = "\r\n";
char *end_of_data = "\n.\n";
char *prompt = ">";

int client_count = 0;		/* Number of active client sessions */
int cmd_execute_line (void *line, int fd, void *data);
int client_initialized = 0;

typedef struct
{
  char *buffer;
  int count;			/* used for bytes read or bytes written */
  int max;			/* used for max buffer size, or just size of
				 * buffer is this is for output
				 */
} Buffer;

Buffer *
buffer_new ()
{
  Buffer *buffer;

  buffer = xmalloc (sizeof (Buffer));
  buffer->count = 0;
  buffer->max = 16;
  
  buffer->buffer = xmalloc (buffer->max);
  *buffer->buffer = '\0';

  return (buffer);
}

Buffer *
buffer_new_full (void *buf, int buf_length)
{
  Buffer *buffer;
  buffer = xmalloc (sizeof (Buffer));
  buffer->buffer = buf;
  buffer->count = 0;
  buffer->max = buf_length;

  return (buffer);
}

void
buffer_free (Buffer *buffer)
{
  if (buffer->buffer)
    xfree (buffer->buffer);
  xfree (buffer);
}

void
buffer_insert_char (Buffer *buffer, char c)
{  
  if (buffer->count >= (buffer->max - 1))
    {
      if (buffer->max == 0)
	buffer->max = 16;
      else
	buffer->max += 16;

      buffer->buffer = 
	xrealloc (buffer->buffer, buffer->max);
    }
  buffer->buffer[buffer->count] = c;
  buffer->count++;
  buffer->buffer[buffer->count] = '\0';
}

typedef struct 
{
  void *from;
  int from_length;
  int bytes_requested;
  SiteID *send_to_siteid;
  Buffer *buffer;
  int content_length;		/* content length in HTTP stuff */
  int content_read;		/* how much of the content length has been
				 * read
				 */
} ClientState;

ClientState *
client_state_new (struct sockaddr_in *from, int from_length)
{
  ClientState *state;
  state = xmalloc (sizeof(ClientState));
  state->from = xmalloc (from_length);
  memcpy (state->from, from, from_length);

  state->bytes_requested = -1;
  state->send_to_siteid = NULL;
  state->buffer = buffer_new ();
  state->content_length = 0;
  state->content_read = 0;
  client_count++;
  return (state);
}

void
client_state_free (ClientState *state)
{
  xfree (state->from);
  buffer_free (state->buffer);
  if (state->send_to_siteid != (SiteID *)-1)
    xfree (state->send_to_siteid);
  xfree (state);
  client_count--;
}

int 
client_connection_write (int fd, Buffer *state)
{
  int n;
  n = write (fd, 
	     &state->buffer[state->count], 
	     state->max - state->count);

  if (n == -1)
    return (-1);
  state->count += n;

  /* If we are funished return the number of bytes written */
  if (state->count == state->max)
    return (state->count);

  /* We are not done yet so return a 0 */
  return (0);
}

/* output a preallocate buffer.  The buffer must be freeable with
 * free ().
 */

void
client_output_buffer (int fd, void *buffer, int length)
{
  IO_output (fd, 
	     client_connection_write,		/* write function */
	     buffer_free,			/* Cleanup function */
	     buffer_new_full (buffer, length));	/* State */
}

/* A simple utility funciton.
 * Output a string that needs to be allocated.
 */
 
void
client_output_string (int fd, char *string)
{
  client_output_buffer (fd, strdup (string), strlen (string));
}


void
client_cmd_groupmembership (char **args, int fd, ClientState *foo)
{
  Group *group;
  string_declare (output);
  int i;
 
  /* We don't free the output string here since it gets freed when
   * after is has been sent.
   */
 
  string_init (output, 64);

  if (args[1] == NULL)
    {
      string_append_string (output, "ERROR: Usage: ");
      string_append_string (output, args[0]);
      string_append_string (output, " group-name\n");
      client_output_buffer (fd, output, output_length);
    }
 
  group = group_by_name (args[1]);
  string_append_string (output, "(:group");
  for (i = 0; group->sites != NULL && group->sites[i] != NULL; i++)
    {
      string_append_string (output, "\n");
      string_append_string_field (output,
                                  "  (",
                                  ":site-name", sites[i]->name,
                                  "");
 
      string_append_string_field (output,
                                  "\n   ",
                                  ":hostname", sites[i]->host,
                                  "");
      string_append_int_field (output,
                               "\n   ",
                               ":client-port", sites[i]->client_port,
                               "");
 
      string_append_int_field (output,
                               "\n   ",
                               ":data-port", sites[i]->data_port,
                               ")");    }
  string_append_string (output, "\n");
  client_output_buffer (fd, output, output_length);
}

void
client_cmd_groups (char **args, int fd)
{
  int i;
  string_declare (result);

  string_init (result, 32);

  string_append_string (result, "(:group-list");
  for (i = 0; groups != NULL && groups[i] != NULL; i++)
    {
      string_append_string (result, " ");
      string_append_string (result, groups[i]->name);
    }
  string_append_string (result, ")\n");

  client_output_buffer (fd, result, result_length);
}

void
client_cmd_group_parameters (char **args, int fd)
{
  Group *group;
  string_declare (output);

  string_init (output, 128);

  if (args[1] == NULL)
    {
      string_append_string (output, "ERROR: Usage: ");
      string_append_string (output, args[0]);
      string_append_string (output, group->name);
      client_output_buffer (fd, output, output_length);
      return;

    }

  group = group_by_name (args[1]);
  if (group == NULL)
    {
      string_append_string (output, "Default group: `");
      group = groups[0];
    }

  string_append_string (output, "(:group-parameters \n");
 
  string_append_string_field (output, "\t", ":group-name",
                              group->name, "\n");
  if (group->master_site != NULL)
    string_append_string_field (output, "\t", ":master-site",
                                group->master_site->name, "\n");
  string_append_int_field (output, "\t", ":ping-period",
                           group->ping_period, "\n");
  string_append_int_field (output, "\t", ":bandwidth-period",
                           group->bandwidth_period, "\n");
  string_append_int_field (output, "\t", ":update-period",
                           group->update_period, "\n");
  string_append_int_field (output, "\t", ":estimates-period",
                           group->estimates_period, "");
  string_append_string (output, ")\n");
 
  client_output_buffer (fd, output, output_length);
}

#ifdef STATISTICS
/* start adding by erhyuan */
void
client_cmd_reset_counters (char **args, int fd)
{
	initialize_counters();
}
/* End   adding by erhyuan */
#endif /* STATISTICS */


void
client_cmd_statistics (char **args, int fd, ClientState *foo)
{
  int i, j;
  extern Queue *export_queue;
  extern int log_size_in_bytes;
  extern int log_size_in_messages;
  string_declare (output); 

  string_init (output, 128);

  string_append_string (output, "(:statistics \n");

  string_append_string_field (output, " ", ":site-name", whoami->name, "\n");

  string_append_string (output, " (:site-id ");
  string_append_siteid (output, &whoami->id);
  string_append_string (output, ")");

  string_append_int_field (output, "\n ", 
			   ":clients-count", client_count,
			   "");

  string_append_int_field (output, "\n ",
			   ":log-size", log_size_in_bytes,
			   "");

  string_append_int_field (output, "\n ", 
			   ":log-length", log_size_in_messages, 
			   "");

  string_append_int_field (output, "\n ",
			   ":export-queue-length", export_queue->size,
			   "");

  string_append_int_field (output, "\n ",
			   ":holding-queue", whoami->datablocks_to_send->size,
			   "");

  string_append_int_field (output, "\n ",
			   ":data-storage-free", datablock_space (0),
			   "");

  string_append_int_field (output, "\n ",
			   ":data-storage-max", datablock_space (-1),
			   "");
#ifdef STATISTICS
/* Start of adding by erhyuan */

  string_append_count_field (output, "\n ",
			   ":block-send-out", count_sendout_block( 0 ),
			   count_sendout_byte( 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-received", count_received_block( 0 ),
			   count_received_byte( 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-duplicate", count_duplicate_block( 0 ),
			   count_duplicate_byte( 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-purged-bandwidth", count_purge_block( 0, 0 ),
			   count_purge_byte( 0, 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-purged-topology", count_purge_block( 1, 0 ),
			   count_purge_byte( 1, 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-purged-estimates", count_purge_block( 2, 0 ),
			   count_purge_byte( 2, 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-purged-data", count_purge_block( 3, 0 ),
			   count_purge_byte( 3, 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-purged-fedex", count_purge_block( 4, 0 ),
			   count_purge_byte( 4, 0 ),
			   "");

  string_append_count_field (output, "\n ",
			   ":block-purged-join", count_purge_block( 5, 0 ),
			   count_purge_byte( 5, 0 ),
			   "");

  string_append_int_field (output, "\n ",
			   ":export-forked", count_export ( 0 ),
			   "");

/* End   of adding by erhyuan */
#endif /* STATISTICS */

  string_append_string (output, "\n (:neighbor-list");
  /* Go through groups and get neighbor information  */

  for (j = 0; groups[j] != NULL; j++)
    {
      for (i = 0; groups[j]->neighbors != NULL && groups[j]->neighbors[i] != NULL; i++)
	{
	  SiteID *siteid = groups[j]->neighbors[i];
	  SiteInfo *site = site_by_id (siteid);

	  if (site && site != whoami)
	    {
	      Statistics *stat;
	      int bandwidth;
	      int rtt;

	      string_append_string (output, "\n  (:neighbor\n");
	      string_append_string_field (output, "   ", ":site-name",
			  	site->name, "\n");

	      string_append_string (output, "   (:site-id ");
	      string_append_siteid (output, &site->id);
	      string_append_string (output, ")\n");
	            
	      stat = 
		LIST_FIND (whoami->estimates, &site->id, statistics_by_id);

	      if (stat == NULL)
		{
		  bandwidth = -1;
		  rtt = -1;
		}
	      else
		{
		  bandwidth = stat->bandwidth;
		  rtt = stat->round_trip_time;
		}
	      string_append_int_field (output, "   ", ":bandwidth",
				       bandwidth,
				       "\n");

	      string_append_int_field (output, "   ", ":round-trip-time",
				       rtt,
				       "\n");

	      string_append_int_field (output, "   ", ":send-queue-length",
				       site->datablocks_to_send->size,
				       "\n");
	      string_append_string_field (output, "   ", ":status", 
				    (site->status > 0) ? "up" : "down", "");

	      string_append_string (output, ")");
	    }
	}
    }

  string_append_string (output, "))\n");
  client_output_buffer (fd, output, output_length);
}


/* return true if buffer ends in a blank.
 * i.e. \n\n, or \r\n\r\n 
 */
buffer_blank_line (Buffer *buffer)
{
  int end = buffer->count - 1;

  if (end < 1)
    return (0);

  if (buffer->buffer[end] == '\n' && buffer->buffer[end - 1] == '\n')
    return (1);
  
  if (end < 3)
    return (0);

  if (buffer->buffer[end - 0] == '\n' && buffer->buffer[end - 1] == '\r' &&
      buffer->buffer[end - 2] == '\n' && buffer->buffer[end - 3] == '\r')
    return (1);

  return (0);
}
/* Reader function for HTTP.  After we grab the command lines, 
 * we need to grab the mime headers
 */

/* Read the mime headers */
client_cmd_http_read (int fd, ClientState *state)
{
  int n;
  char c;

  while ((n = read (fd, &c, 1)) > 0)
    {
      buffer_insert_char (state->buffer, c);

      /* Check to see if we are done with the mime headers
       * and need to read the content data
       */
      if (state->content_length == 0 && buffer_blank_line (state->buffer))
	{
	  char *content_length;

	  /* Search for "Content-length to see if we need to snarf anything */
	  content_length = strstr (state->buffer->buffer, "Content-length");

	  /* If we don't find a content length header, assume we are done
	   * I don't know if this is to spec.
	   */

	  if (content_length == NULL)
	    goto done;
	  
	  /* find the number */
	  while (*content_length && !isspace (*content_length))
	    content_length++;
	  while (*content_length && isspace (*content_length))
	    content_length++;

	  state->content_length = atoi (content_length);
	  continue;
	}
      if (state->content_length > 0)
	{
	  if (state->content_length <= state->content_read)
	    goto done;
	  state->content_read++;
	}
    }

  return (0);
 done:

  if (debug) 
    printf ("http-request:: `%s'\n", state->buffer->buffer);

  exec_http_command (fd, state->buffer->buffer);
  /* We can terminate too quickly here since we want our
   * output to get written out before we kill the io handler.
   * (the other option would be to return -1)
   */
  IO_close (fd);
  state->buffer->count = 0;
  state->buffer->buffer[state->buffer->count] = '\0';

  return (1);
}


void
client_cmd_http (char **args, int fd, ClientState *state)
{
  int i;
  char a_name[100];
  int found = 0;

  if (args[1] == NULL)
    {
      IO_close (fd);
      return;
    }

  if (debug && args[1])
    fprintf(stderr, "user typed: %s\n", args[1]); /* PBD */

  IO_set_reader (fd, (IOFunction *)client_cmd_http_read);
  return;
}


int 
num_estimates_sites()
{
  int no;
  Statistics  **stats;

  stats = (Statistics **) whoami->estimates->list;
  for (no = 0; stats && stats[no]; no++);
  return (no);
}

void
client_cmd_estimates (char **args, int fd)
{
  int i;
  Statistics  **stats;

  string_declare (output); 

  string_init (output, 128);

  string_append_string (output, "(:site-estimates ");
  string_append_siteid (output, &whoami->id);
  
  stats = (Statistics **) whoami->estimates->list;
  for (i = 0; stats && stats[i]; i++)
    {
      string_append_string (output, "\n (:site-id");
      string_append_siteid (output, &(stats[i]->id));
      string_append_string (output, ") ");
      string_append_int_field (output, "   ", ":round-trip-time",
			       stats[i]->round_trip_time,
			       " ");

      string_append_int_field (output, "   ", ":bandwidth",
			       stats[i]->bandwidth,
			       ")");
    }
  string_append_string (output, ")\n");
  client_output_buffer (fd, output, output_length);
}

void
client_cmd_set (char **args, int fd)
{
  parse_config_string (args[1], fd);
  client_output_string (fd, "\n");
}

/* Receive a client data block 
 * Wait to receive a `\n.\n'
 */
void
client_cmd_textual_data (char **args, int fd, struct sockaddr_in *address)
{
  int client_cmd_text_read (int fd, ClientState *state);

  IO_set_reader (fd, (IOFunction *)client_cmd_text_read);
}

void
client_cmd_binary_data (char **args, int fd, struct sockaddr_in *address)
{
  ClientState *state;
  SiteID *siteid = NULL;
  int send_to_neighbor = 0;
  int bytes;
  int client_cmd_binary_read (int fd, ClientState *state);
  
  if (args[1] == NULL)
    {
      client_output_string (fd, "ERROR: Usage: ");
      client_output_string (fd, args[0]);
      client_output_string (fd, "number-of-bytes [ siteid | \"neighbor\"]\n");
      return;
    }

  bytes = atoi(args[1]);
  if (bytes < 0 || bytes > 2 * 1024 * 1024)
    {
      client_output_string (fd, "ERROR: Datablock too large.\n");
      return;
    }


  if (args[2] != NULL)
    {
      if (strcmp (args[2], "neighbor") == 0)
	siteid = (SiteID *) -1;
      else
	{
	  siteid = (SiteID *) scan_siteid (args[2], NULL);
	  if (siteid == NULL)
	    {
	      client_output_string (fd, "ERROR: Invalid siteid\n");
	      return;
	    }
	}
    }

  state = IO_read_state (fd);
  state->bytes_requested = bytes;
  state->send_to_siteid = siteid;
  IO_set_reader (fd, (IOFunction *)client_cmd_binary_read);
}

void
client_cmd_quit (char **args, int fd, void *foo)
{
  IO_close (fd);
}

#ifdef EXIT_CMD
void
client_cmd_exit (char **args, int fd, void *foo)
{
  exit (1);
}
#endif

void
client_cmd_force_topology (char **args, int fd, ClientState *state)
{
  Group *group;
  string_declare (output);

  string_init (output, 128);

  if (args[1] == NULL)
    {
      string_append_string (output, "ERROR: Usage: ");
      string_append_string (output, args[0]);
      string_append_string (output, group->name);
      client_output_buffer (fd, output, output_length);
      return;

    }

  group = group_by_name (args[1]);
  if (group == NULL)
    {
      string_append_string (output, "ERROR: Invalid group name `");
      string_append_string (output, args[1]);
      string_append_string (output, "'\n");
      client_output_buffer (fd, output, output_length);
      return;
    }
  
  reschedule_topology (group);
}

#define after_decimal_point(f) (int) (abs((f - (int) f) * 100))

void
client_cmd_sites (char **args, int fd, void *foo)
{
  int i;
  string_declare (output);
  string_declare (address);

  string_init (output, 64);
  string_init (address, 17);

  string_append_string (output, "(:site-list\n");

  for (i = 0; sites[i]; i++)
    {
      if (i != 0)
	string_append_string (output, "\n");
      
      string_append_string_field (output, 
				  "  (", 
				  ":site-name", sites[i]->name, 
				  "");

      string_append_string (output, "\n   (:site-id ");
      string_append_siteid (output, &sites[i]->id);
      string_append_string (output, ")");

      string_reset (address);
      string_append_address (address, sites[i]->id.addr.bytes);

      string_append_string_field (output, 
				  "\n   ", 
				  ":inet-address", address, 
				  "");

      string_append_string_field (output, 
				  "\n   ", 
				  ":hostname", sites[i]->host, 
				  "");

      string_append_int_field (output, 
			       "\n   ",
			       ":client-port", sites[i]->client_port, 
			       "");

      string_append_int_field (output, 
			       "\n   ", 
			       ":data-port", sites[i]->data_port, 
			       "");
      
      string_append_float_field (output, 
				 "\n   ",
				 ":longitude",  sites[i]->longitude,
				 "");

      string_append_float_field (output, 
				 "\n   ",
				 ":lattitude",  sites[i]->lattitude,
				 ")");
    }
  string_append_string (output, ")\n");
  client_output_buffer (fd, output, output_length);
}


/* print out local parameters */
void
client_cmd_local_parameters (char **args, int fd, void *foo)
{
  string_declare (output);

  string_init (output, 128);

  string_append_string (output, "(:local");

  string_append_int_field (output, "\n ",
			   ":data-storage-max", datablock_space (-1),
			   "");

  string_append_int_field (output, "\n ",
			   ":site-purge-period", 
			   site_purge_period,
			   "");

  string_append_int_field (output, "\n ",
			   ":log-purge-period", 
			   log_purge_period,
			   "");

  string_append_int_field (output, "\n ",
			   ":bandwidth-size", 
			   bandwidth_size,
			   "");


  string_append_string_field (output, "\n ",
			      ":export-command", 
			      (export_command) ? export_command : "not-set",
			      "");
  string_append_string (output, ")\n");


  client_output_buffer (fd, output, output_length);

}

/* Display a topology for a given group */
void
client_cmd_topology (char **args, int fd, void *foo)
{
  Group *group;
  string_declare (output);


  if (args[1] == NULL)
    {
      string_init (output, 64);
      string_append_string (output,"Topology for default group ");
      group = groups[0];
      string_append_string (output, group->name);
    } else  {
       group = group_by_name (args[1]);
    }
  if (group == NULL)
    {
      string_init (output, 64);
      string_append_string (output, "ERROR: No group `");
      string_append_string (output, args[1]);
      string_append_string (output, "'\n");
      client_output_buffer (fd, output, output_length);
      return;
    }
  
  if (group->topology != NULL)
    client_output_string (fd, group->topology);
  client_output_string (fd, "\n");
}

/* return the nearest floodd to the requesting client.
 * for right now, just return this one.
 */

void
client_cmd_nearest (char **args, int fd, void *foo)
{
  string_declare (output);
  string_init (output, 16);

  string_append_siteid (output, &whoami->id);
  string_append_string (output, "\n");

  client_output_buffer (fd, output, output_length);
}

#ifdef INSIDE_CONTROL
char *inside_switches[] = {
  "data_receive",
  "data_connect",
  "*"
};
#define sw_DataReceive 0
#define sw_DataConnect 1

int sw_data_receive = 0;
int sw_data_connect = 0;

int
check_switch( char *name )
{
  int i;
  for( i = 0; *inside_switches[i] != '*'; ++i ) {
    if( strcmp( name, inside_switches[i] ) == 0 ) return(i);
  }
  return( -1 );
}

void 
client_cmd_inside_control (char **args, int fd)
{
  string_declare (output);   
  int control;
  int on = -1;
  int off = -1;

  string_init (output, 64);

  if (args[1] == NULL || (control=check_switch( args[1] )) == -1 ||
      args[2] == NULL ||
      ( (on=strcmp(args[2],"on")) != 0 && (off=strcmp(args[2],"off")) != 0 )
     ) 
    {
      /* string_declare (output); */
      string_append_string (output, "ERROR: Usage: ");
      string_append_string (output, args[0]);
      string_append_string (output, " <switch> on|off \n");
      client_output_buffer (fd, output, output_length);
      return;
    }
  switch (control) {
    case sw_DataConnect:
      string_append_string (output, "Blocking DataConnect ");
      if (on == 0) {
	sw_data_connect = 1;
        string_append_string (output, "on");
      } 
      else {
	sw_data_connect = 0;
        string_append_string (output, "off");
      }
      break;
    case sw_DataReceive:
      string_append_string (output, "Blocking DataReceive ");
      if (on == 0) {
	sw_data_receive = 1;
        string_append_string (output, "on");
      } 
      else {
	sw_data_receive = 0;
        string_append_string (output, "off");
      }
      break;
  }
  string_append_string (output, " \n");
  client_output_buffer (fd, output, output_length);

}
#endif /* INSIDE_CONTROL */

client_initialize ()
{
  cmd_add_command ("data", client_cmd_textual_data,
		   "Enter textual data to be flooded.  End with a `<CR>.'", NULL);
  cmd_add_command ("binary", client_cmd_binary_data,
		   "Enter binary data.  The second argument give the number of bytes to read", NULL);
  cmd_add_command ("estimates", client_cmd_estimates,
                   "Return estimates for RTT and Bandwidth to other sites", NULL);
  cmd_add_command ("force-topology", client_cmd_force_topology,
                   "Run a topology calculation for the group", NULL);
  cmd_add_command ("group", client_cmd_groupmembership, 
		   "Return the membership of the given group", NULL);
  cmd_add_command ("groups", client_cmd_groups, 
		   "Return the list of groups", NULL);
  cmd_add_command ("group-parameters", client_cmd_group_parameters,
		   "Return paramters for floodd", NULL);
  cmd_add_command ("local-parameters", client_cmd_local_parameters,
		   "Return local site parameters", NULL);
  cmd_add_command ("nearest", client_cmd_nearest,
		   "Return the nearest known floodd daemon", NULL);
  cmd_add_command ("sites", client_cmd_sites,
		   "Return the list sites.", NULL);
  cmd_add_command ("statistics", client_cmd_statistics,
		   "Return some statistics on the floodd", NULL);
  cmd_add_command ("topology", client_cmd_topology,
		   "Return the topology for the named group", NULL);
#ifdef STATISTICS
  cmd_add_command ("reset", client_cmd_reset_counters,
		   "Reset counters for statistics", NULL);
#endif /* STATISTICS */
#ifdef INSIDE_CONTROL
  cmd_add_command ("control", client_cmd_inside_control,
		   "Control the inside switches to on/off", NULL);
#endif /* INSIDE_CONTROL */
  cmd_add_command ("quit", client_cmd_quit,
		   "Close connection", NULL);
#ifdef EXIT_CMD
  cmd_add_command ("exit", client_cmd_exit,
		   "Terminate floodd", NULL);
#endif
  cmd_add_command ("set", client_cmd_set, "Set parameters - the syntax is the \n\tsame as the Floodd configuration file.", NULL);

#ifdef HTTP
  cmd_add_command ("GET", client_cmd_http, "HTTP get interface", NULL);
  cmd_add_command ("POST", client_cmd_http, "HTTP post interface", NULL);
#endif /* HTTP */

  client_initialized = 1;
}


int
client_connection_read (int fd, ClientState *state)
{
  int n;
  char c;
  int read_something;
  extern void *IO_get_reader ();
  if (client_initialized == 0)
    client_initialize ();

  read_something = 0;

  while ((n = read (fd, &c, 1)) > 0)
    {
      read_something = 1;
      buffer_insert_char (state->buffer, c);

      /* If we have and end of line, then execute the command*/
      if (c != '\0' && strchr (end_of_line_marker, c))
	{
	  /* if only exeute if the line is non-null (i.e. not just a eol) */
	  if (state->buffer->count > 1) {
	    cmd_execute_line (state->buffer->buffer, fd, state);
	  }
	  
	  /* Only output a prompt if  the current read function is
	   * this one
	   */
	  if ((IO_get_reader (fd) == client_connection_read) && c == '\n')
	    client_output_string (fd, prompt);
	  
	  if (IO_get_reader (fd) != client_cmd_http_read)
	      {
		state->buffer->count = 0;
		state->buffer->buffer[state->buffer->count] = '\0';
	      }
	  return (0);
	}
    }
  
  /* if we didn't actually read any characters, something is wrong */
  if (!read_something)
    return (-1);

  /* If an the error returned was EWOULDBLOCK, then we just return
   * a zero.
   */
  if (errno == EWOULDBLOCK)
    return (0);

  if (n == -1)
    {
      if (debug)
	syslog (LOG_DEBUG, "Closing client session\n");
      return (-1);
    }
  return (n);
}
/* Read data to be flooded from a client.
 * This command replaces the existing client_connection_read command
 * while it reads data to be flooded.  Once it has received the data
 * it needs to reinstall the orignal handler client_connection_read().
 */
int
client_cmd_text_read (int fd, ClientState *state)
{
  DataBlock *datablock;
  char c;
  int done = 0;
  int n;
  int length;
  int read_something;

  read_something = 0;

  while ((n = read (fd, &c, 1)) == 1)
    {
      read_something = 1;

      buffer_insert_char (state->buffer, c);
      length = strlen (end_of_data);
      if (state->buffer->count > 2)
	if (strncmp (&state->buffer->buffer[state->buffer->count - length],
		     end_of_data, length) == 0)
	done = 1;

      if (done)
	{
	  /* Create a datablock and flood it */
	  datablock = 
	    datablock_new (TYPE_DATA,
			   whoami->id,
			   state->buffer->buffer, state->buffer->count);

#ifdef NEVER
	  state->buffer->buffer = NULL;
	  state->buffer->count = 0;
	  state->buffer->max = 0;
#endif
	  /* Log it before we do the receive so we don't receive it 
	   * via flooding. 
	   */
	  log_message (&datablock->header.id);
	  dispatch_datablock (datablock);
	  datablock_free (datablock);

	  /* Reinstall the data client_connection_read ()*/
	  IO_set_reader (fd, (IOFunction *) client_connection_read);
	  client_output_string (fd, prompt);	
	  return (0);
	}
    }
  
  /* If we didn't read anyting this function sould not have been called,
   * hence an error must have occured.
   */
  if (!read_something)
    return (-1);

  if (n == -1 && errno != EWOULDBLOCK)
    return (-1);

  return (0);
}

/* Read binary date until we get state->bytes_requested data */
int
client_cmd_binary_read (int fd, ClientState *state)
{
  DataBlock *datablock;
  int n;
  int read_something;

  read_something = 0;

  if (!datablock_space (state->bytes_requested))
    purge_datablocks (0);

  if (!datablock_space (state->bytes_requested))
    goto return_eof;

  if (state->bytes_requested > state->buffer->max)
    {
      state->buffer->max = state->bytes_requested;

      state->buffer->buffer =
	xrealloc (state->buffer->buffer, state->buffer->max);
    }
				     
  while ((n = 
	  read (fd, &state->buffer->buffer[state->buffer->count], 
		state->bytes_requested - state->buffer->count)) > 0)
    {
      read_something = 1;

      state->buffer->count += n;

      if (state->buffer->count == state->bytes_requested)
	{
	  /* Create a datablock and flood it - we cheat and use the
	   * memory allocated to the buffer to avoid copying it.
	   */
	  if (state->send_to_siteid == NULL)
	    datablock = 
	      datablock_new (TYPE_DATA, whoami->id, NULL, 0);
	  else
	    datablock = 
	      datablock_new (TYPE_FEDEX, whoami->id, NULL, 0);
	  
	  datablock_set_data (datablock, 
			      state->buffer->buffer,
			      state->buffer->count);
	  
	  state->buffer->buffer = NULL;
	  state->buffer->count = 0;
	  state->buffer->max = 0;

	  if (!datablock_space (0))
	    {
	      datablock_free (datablock);
	      goto return_eof;
	    }

	  /* Check to see if this this is point-to-point, or a regular
	   * floodd block.
	   */
	  if (datablock->header.type == TYPE_DATA)
	    {
	      /* Log it before we do the receive so we don't receive it 
	       * via flooding. 
	       */
	      log_message (&datablock->header.id);
	      dispatch_datablock (datablock);
	    }
	  else
	    {
	      if (send_to_siteid (state->send_to_siteid, datablock) != 0)
		{
		  datablock_free (datablock);
		  goto return_eof;
		}
	    }
	  datablock_free (datablock);
	  state->bytes_requested = -1;
	  
	  /* Reinstall the data client_connection_read ()*/
	  IO_set_reader (fd, (IOFunction *) client_connection_read);
	  client_output_string (fd, prompt);

	  return (0);
	}
    }
  
  /* If we didn't read anyting this function sould not have been called,
   * hence an error must have occured.
   */
  if (!read_something)
    goto return_eof;

  if (n == -1 && errno != EWOULDBLOCK)
    goto return_eof;
  return (0);

 return_eof:
  return (-1);
}
