/* WIDE AREA INFORMATION SERVER SOFTWARE:
   No guarantees or restrictions.  See the readme file for the full standard
   disclaimer.

   Brewster@think.com
*/

/* this file defines a set of helper functions
 * for indexing common types of files.
 * -brewster 7/90
 */

/* I encourage adding customizations.
 * (too bad they all have to be hard coded, but
 *  C did not have convenient dynamic linking facilities)
 *
 * Add three functions to this file:
 * boolean foo_separator_function(char *line){}
 * void foo_header_function(char *line){}
 * long foo_date_function(char *line){}
 * void foo_finish_header_function(char *header){}
 *
 * then add the prototypes to ircfiles.h
 * then add the functions to the big case statement in irbuild.c
 *
 *
 * to do:
 *   filter for digests
 *
 */


/* Change log:
 * 8/90 brewster added the library customizations
 * 6/91 and before - added a bunch of other filters - JG
 * $Log:	ircfiles.c,v $
 * Revision 1.23  92/02/20  09:50:14  jonathan
 * Added bibtex and nhyp filters from S.P.vandeBurgt@research.ptt.nl.
 * 
 * Revision 1.22  92/02/12  13:11:25  jonathan
 * Changed library catalog functions for new format (from fad).
 * 
 * 
 *
 */

#include <string.h>
#include <ctype.h>
#include "cutil.h"
#include "ircfiles.h"

#define MAX_HEADER_LEN 100

#define MAX_AUTHOR_LEN 25
#define MAX_DATE_LEN 4

static char* trim_trailing_newline _AP((char* string));

static char*
trim_trailing_newline(string)
char* string;
{
  if(string)
    if(strlen(string) > 0)
      if(string[strlen(string) -1] == '\n')
	string[strlen(string) -1] = '\0';
  return(string);
}

/* =================================
 * ===  Groliers Customizations  ===
 * =================================
 */

boolean groliers_separator_function(line)
char *line;
{
  if((strlen(line) > strlen("ARTICLE")) &&
     substrcmp(line, "ARTICLE")){
    /* printf("hit %s\n", line); */
    return(true);
  }
  else{
    return(false);
  }
}

char groliers_header[MAX_HEADER_LEN + 1];

void groliers_header_function(line)
char *line;
{
  if(groliers_separator_function(line)){
    s_strncpy(groliers_header, line + strlen("ARTICLE") + 2, MAX_HEADER_LEN);
  }
}

void groliers_finish_header_function(header)
char *header;
{
  if(strlen(groliers_header) == 0){
    s_strncpy(header, "Unknown Title", MAX_HEADER_LEN);
  }
  else{
    s_strncpy(header, groliers_header, MAX_HEADER_LEN);
  }
  groliers_header[0] = '\0';
}


/* ==============================
 * ===  RMail Customizations  ===
 * ==============================
 */

/* this is just a preliminary version. A good version would
 * produce a headline like gnu emacs RMAIL
 */


boolean mail_separator_function(line)
char *line;
{
  /* this should really look for a "<cr><cr>From " rather than "<cr>From " */
  if((strlen(line) > strlen("From ")) &&
     substrcmp(line, "From ")){
    return(true);
  }
  else{
    return(false);
  }
}

boolean rmail_separator_function(line)
char *line;
{
  if(0 == strcmp(line, "\n")){
    return(true);
  }
  else{
    return(false);
  }
}

/* This one is portable, but might get the wrong answer.
   I'm open to better code.  - Jonny G
*/

static char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

long my_getdate(line)
char *line;
{
  char date[255], *temp;
  int day, month, year;
  char cmonth[25], dow[5], tod[10];

  strcpy(date, line);

  temp = date;

  while(!isdigit(*temp)) temp++;

  sscanf(temp, "%d %s %d", &day, cmonth, &year);

  for(month = 0; months[month] != NULL; month++)
    if(!strcmp(cmonth, months[month])) break;

  if (year > 99) year = year % 100;

  if(day > 0 && 
     month < 12 &&
     year > 0) {
    return (10000 * year + 100 * (month+1) + day);
  }

  month= -1; day= -1; year= -1;

  sscanf(temp, "%d/%d/%d", &month, &day, &year);

  if (year > 99) year = year % 100;

  if(day > 0 && 
     month < 12 &&
     year > 0) {
    return (10000 * year + 100 * (month+1) + day);
  }

  month= -1; day= -1; year= -1;

  sscanf(temp, "%d/%d/%d", &year, &month, &day);

  if (year > 99) year = year % 100;

  if(day > 0 && 
     month < 12 &&
     year > 0) {
    return (10000 * year + 100 * (month+1) + day);
  }

  temp = date;

  sscanf(temp, "%s %s %d %s %d", dow, cmonth, &day, tod, &year);

  for(month = 0; months[month] != NULL; month++)
    if(!strcmp(cmonth, months[month])) break;

  if (year > 99) year = year % 100;

  if(day > 0 && 
     month < 12 &&
     year > 0) {
    return (10000 * year + 100 * (month+1) + day);
  }

  return 0;
}

long mail_date_function(line)
char *line;
{
  if((strlen(line) > strlen("Date: ")) &&
     substrcmp(line, "Date: ")){
    return(my_getdate(line+6));
  }
  else if((strlen(line) > strlen("From ")) &&
	  substrcmp(line, "From ")){
    char *p;
    p = (char*)index(line+5, ' ');
    if(p != NULL)
      return(my_getdate(p+1));
  }
  else return -1;
}

  

char mail_subject[MAX_HEADER_LEN + 1];
char mail_from[MAX_HEADER_LEN + 1];

void mail_header_function(line)
char *line;
{
  if((strlen(line) > strlen("Subject: ")) &&
     substrcmp(line, "Subject: ") &&
     (strlen(mail_subject) == 0)){
    strcpy(mail_subject, "Re: ");
    s_strncat(mail_subject, line + strlen("Subject: "), MAX_HEADER_LEN, MAX_HEADER_LEN);
    trim_trailing_newline(mail_subject);
  }
  else if((strlen(line) > strlen("From: ")) &&
     substrcmp(line, "From: ") &&
     (strlen(mail_from) == 0)){
    /* this should find the <foo@bar> field in the from list */
    s_strncpy(mail_from, line + strlen("From: "), MAX_HEADER_LEN);
    trim_trailing_newline(mail_from);
  }
  
}

void mail_finish_header_function(header)
char *header;
{
  if(strlen(mail_subject) != 0 &&
     strlen(mail_from) != 0){
    /* trim the from line if needed */
    if(strlen(mail_from) > 10){
      mail_from[10] = '\0';
    }
    s_strncpy(header, mail_from, MAX_HEADER_LEN);
    s_strncat(header, " ", MAX_HEADER_LEN, MAX_HEADER_LEN);
    s_strncat(header, mail_subject, MAX_HEADER_LEN, MAX_HEADER_LEN);
    /* printf("%s\n", header); */
  }
  else if(strlen(mail_subject) != 0){
    s_strncpy(header, mail_subject, MAX_HEADER_LEN);
  }
  else if(strlen(mail_from) != 0){
    s_strncpy(header, mail_from, MAX_HEADER_LEN);
  }
  else{
    strcpy(header, "Unknown Subject");
  }
  mail_from[0] = '\0';
  mail_subject[0] = '\0';
}




boolean mail_or_rmail_separator(line)
char *line;
{
  static boolean blank_line = false;

  if((strlen(line) > strlen("From ")) &&
     substrcmp(line, "From ") &&
     blank_line == true){
    blank_line = false;
    return(true);
  }
  
  if(substrcmp(line, "")){
    blank_line = true;
    return(true);
  }    
  
  if(!strcmp(line, "\n")){
      blank_line = true;
    }
    else{
      blank_line = false;
    }

  return(false);
}


/* ========================================
 * ===  Mail Digest Customizations     ====
 * ========================================
 */

boolean mail_digest_separator_function(line)
char *line;
{
  if((strlen(line) > strlen("-----------------------------")) &&
     substrcmp(line, "------------------------------")){
    return(true);
  }
  else{
    return(false);
  }
}


/* ========================================
 * ===  Library Catalog Customizations  ===
 * ========================================
 */

#define TITLE_MARKER "Title...... "
#define FIRST_LINE_MARKER "Call No...."

/* just use the title */

boolean catalog_separator_function(line)
char *line;
{
  if (strstr(line, FIRST_LINE_MARKER)) {
    return(true);
  }
  else{
    return(false);
  }
}

char catalog_header[MAX_HEADER_LEN + 1];

void catalog_header_function(line)
char *line;
{
  char * title_start;
  if (title_start = strstr(line, TITLE_MARKER))
    {
      strncpy(catalog_header, title_start + strlen(TITLE_MARKER), MAX_HEADER_LEN);
    }
}

void catalog_finish_header_function(header)
char *header;
{
  if(strlen(catalog_header) == 0){
    strcpy(header, "Unknown Title");
  }
  else{
    s_strncpy(header, catalog_header, MAX_HEADER_LEN);
  }
  catalog_header[0] = '\0';
}



/* ============================
 * ===  Bio Customizations  ===
 * ============================
 */

/* customizations for a DB of genetic abstracts */

boolean hit_header = false;

boolean bio_separator_function(line)
char *line;
{
  if((strlen(line) > strlen(">>>")) &&
     substrcmp(line, ">>>")){
    return(true);
  }
  else{
    return(false);
  }
}

char bio_header[MAX_HEADER_LEN + 1];

void bio_header_function(line)
char *line;

{
  if(hit_header			/* we just hit a seperator previous to this */
     && (!bio_separator_function(line)) /* we are not on the separator now */
     && strlen(bio_header) == 0){ /* and we have not saved the headline yet */
    strcpy(bio_header, line);
    waislog(WLOG_MEDIUM, WLOG_INDEX, "storing line: %s", bio_header);
    hit_header = false;
  }
}

void bio_finish_header_function(header)
char *header;

{
  hit_header = true; /* turn on the flag */
  if(strlen(bio_header) == 0){
    strcpy(header, "Unknown Title");
  }
  else{
    strcpy(header, bio_header);
  }
  bio_header[0] = '\0';
}

/* =================================
 * ===  CMApp   Customizations  ===
 * =================================
 */

boolean cmapp_separator_function(line)
char *line;
{
  if((strlen(line) > strlen("@A")) &&
     substrcmp(line, "@A")){
    /* printf("hit %s\n", line); */
    return(true);
  }
  else{
    return(false);
  }
}

char cmapp_header[MAX_HEADER_LEN + 1];

void cmapp_header_function(line)
char *line;
{
  if((strlen(line) > strlen("APPLICATION:")) &&
     substrcmp(line, "APPLICATION:")){
    /* printf("hit %s\n", line); */
    s_strncpy(cmapp_header, line + strlen("APPLICATION:"), MAX_HEADER_LEN);
  }
}

void cmapp_finish_header_function(header)
char *header;
{
  if(strlen(cmapp_header) == 0){
    s_strncpy(header, "Unknown Title", MAX_HEADER_LEN);
  }
  else{
    s_strncpy(header, cmapp_header, MAX_HEADER_LEN);
  }
  cmapp_header[0] = '\0';
}

/* =================================
 * ===  Jargon   Customizations  ===
 * =================================
 *
 * GW - updated for Jargon File 2.9.8
 */

/*

Format of an entry:

[blank line]
:Title of This entry: first line of text of this entry
   second line of text of this entry
   third line of text of this entry
[blank line]

Any line which starts with a colon is considered to be the beginning
of an entry.

-GW

*/

static int jargon_seen_entry = 0;

boolean jargon_separator_function(line)
register char *line;
{
  if(!jargon_seen_entry && line[0] == ':')
    jargon_seen_entry = 1;
  return line[0] == ':';
}

char jargon_header[MAX_HEADER_LEN + 1];

void jargon_header_function(line)
char *line;
{
  if(line[0] != ':')
    return;

  strncpy(jargon_header,line+1,MAX_HEADER_LEN);
  jargon_header[MAX_HEADER_LEN] = '\0';

  if(NULL != (line = strchr(jargon_header,':'))){
    if(line[1] == ':')
      line++;
    line++;
    line[0] = '\0';
  }
}	

void jargon_finish_header_function(header)
char *header;
{
  if(jargon_seen_entry) {
    strncpy(header, jargon_header, MAX_HEADER_LEN);
  }
  jargon_header[0] = '\0';
}


/* =================================
 * ===  Internet Resource Guide  ===
 * =================================
 */


char irg_header[MAX_HEADER_LEN + 1];
boolean irg_header_set = FALSE;

boolean irg_separator_function(line)
char *line;
{
  if(line[0] == 12){  /* control L */
    irg_header_set = FALSE;
    return(true);
  }
  else
    return(false);
}

void irg_header_function(line)
char *line;
{
  if((irg_header_set == FALSE) &&
     (line[0] == 32 )){ /* space */
    s_strncpy(irg_header, line + strspn(line, " "), MAX_HEADER_LEN);
    irg_header_set = TRUE;
  }
  
}

void irg_finish_header_function(header)
char *header;
{
  if(strlen(irg_header) == 0){
    s_strncpy(header, "Unknown Title", MAX_HEADER_LEN);
  }
  else{
    s_strncpy(header, irg_header, MAX_HEADER_LEN);
  }
  irg_header[0] = '\0';
  irg_header_set = FALSE;
}

/* ========================
 * ===  Dash Separator  ===
 * ========================
 */


/*
 * dash-seperate entries
 * used in Introduction to Algorithms bug.list, suggestions, etc.
 * --------------------... at least 20 dashes
 * header
 * item
 *  ..
 * --------------------... at least 20 dashes
 */

boolean dash_separator_function(line)
char *line;
{
  if((strlen(line) > 20) && substrcmp(line,"--------------------")){
    /* printf("hit %s\n", line); */
    return(true);
  }
  else{
    return(false);
  }
}

char dash_header[MAX_HEADER_LEN + 1];

void dash_header_function(line)
char *line;
{
  if(!dash_separator_function(line) &&
     (strlen(dash_header) < (MAX_HEADER_LEN - 1))){
    s_strncat(dash_header, line, 
	      MAX_HEADER_LEN, MAX_HEADER_LEN);
    trim_trailing_newline(dash_header);
    strncat(dash_header, "  ", MAX_HEADER_LEN);
  }
}

void dash_finish_header_function(header)
char *header;
{
  if (strlen(dash_header) == 0) {
    strcpy(header, "No Title");
  }
  else {
    s_strncpy(header, dash_header, MAX_HEADER_LEN);
  }
  dash_header[0] = '\0';
}


/* ============================
 * ===  one_line Separator  ===
 * ============================
 */

/* this is where each line is a document (good for databases) */

boolean one_line_hit_header = false;

boolean one_line_separator_function(line)
char *line;
{
    return(true);
}

char one_line_header[MAX_HEADER_LEN + 1];

void one_line_header_function(line)
char *line;
{
    s_strncpy(one_line_header, line, MAX_HEADER_LEN);
}

void one_line_finish_header_function(header)
char *header;
{
  if (strlen(one_line_header) == 0) {
    strcpy(header, "No Title");
  }
  else {
    s_strncpy(header, one_line_header, MAX_HEADER_LEN);
  }
  one_line_header[0] = '\0';
}

/* =============================
 * ===  Paragraph Separator  ===
 * =============================
 */

/* paragraph files - seperated by a blank line.  Next line is the header */

char para_header[MAX_HEADER_LEN +1];
static boolean para_start = true;

boolean para_separator_function(line)
char *line;
{
  if (para_start == true) {
    para_start = false;
    return true;
  }
  if (strlen(line) < 2)
    para_start = true;
  return false;
}

void para_header_function(line)
char *line;
{
  if (para_header[0] == 0)
    s_strncpy(para_header, line, MAX_HEADER_LEN);
}

void para_finish_header_function(header)
char *header;
{
  if (strlen(para_header) == 0) {
    strcpy(header, "No Title");
  }
  else {
    s_strncpy(header, para_header, MAX_HEADER_LEN);
  }
  para_header[0] = 0;
}  

/* ==========================
 * ===  Seeker Separator  ===
 * ==========================
 */

boolean seeker_separator_function(line)
char *line;
{
  return(dash_separator_function(line));
}

char seeker_header[MAX_HEADER_LEN + 1];
boolean in_headline = FALSE;

void seeker_header_function(line)
char *line;
{
  if(strlen(line) > strlen("Headline:") &&
     substrcmp(line, "Headline:")){
    in_headline = TRUE;
    seeker_header[0] = '\0';
    /* printf("hit headline!\n"); */
  }
  else if(in_headline == TRUE &&
	  (strlen(seeker_header) < (MAX_HEADER_LEN - 1))){
    s_strncat(seeker_header, line, 
	      MAX_HEADER_LEN, MAX_HEADER_LEN);
    trim_trailing_newline(seeker_header);
  }
}

void seeker_finish_header_function(header)
char *header;
{
  if (strlen(seeker_header) == 0) {
    strcpy(header, "No Title");
  }
  else {
    s_strncpy(header, seeker_header, MAX_HEADER_LEN);
  }
  seeker_header[0] = '\0';
  in_headline = TRUE;
}  

/* ==========================
 * ===  RLIN Separator  ===
 * ==========================
 */

boolean rlin_separator_function(line)
char *line;
{
  return(dash_separator_function(line));
}

char rlin_header[MAX_HEADER_LEN + 1];
boolean rlin_in_headline = FALSE;

void rlin_header_function(line)
char *line;
{
  if(rlin_separator_function(line)){
    rlin_in_headline = TRUE;
    rlin_header[0] = '\0';
    /* printf("hit headline!\n"); */
  }
  else if(rlin_in_headline == TRUE &&
	  (strlen(rlin_header) < (MAX_HEADER_LEN - 1))){
    s_strncat(rlin_header, line, 
	      MAX_HEADER_LEN, MAX_HEADER_LEN);
    trim_trailing_newline(rlin_header);
  }
}

void rlin_finish_header_function(header)
char *header;
{
  if (strlen(rlin_header) == 0) {
    strcpy(header, "No Title");
  }
  else {
    s_strncpy(header, rlin_header, MAX_HEADER_LEN);
  }
  rlin_header[0] = '\0';
  in_headline = TRUE;
}  

/* ========================================
 * ===  MH_BBoard  Customizations     ====
 * ========================================
 */

/* gcardwel@uci.edu
MH bboards use a series of control A's to do a blank line.. yuk!
*/

boolean mh_bboard_separator_function(line)
char *line;
{
  static boolean blank_line = false;

  if((strlen(line) > strlen("BBoard-ID: ")) &&
     substrcmp(line, "BBoard-ID: ") &&
     blank_line == true){
    blank_line = false;
    return(true);
  }
  
  if(!strcmp(line, "\001\001\001\001\n")){
    blank_line = true;
  }
  else{
    blank_line = false;
  }
  return (false);
}

/*
 * Customization for files saved from within the 'rn' newsreader.
 *
 * These can either be in 'mail' format, or they can be in a similar
 * format which starts each article with the pseudo-header
 * 'Article: 42 of comp.sys.foobar'.  Other than that, we treat this
 * just like 'mail'.
 *
 * wollman@uvm.edu, Sun Sep  8 20:12:21 EDT 1991
 */
boolean rn_separator_function(line)
char *line;
{
  if(!strncmp(line,"From ",5) ||
     !strncmp(line,"Article ",7) ||
     !strncmp(line,"Article: ",9))
    return true;
  return false;
}

/*
 * Customizations for GNU Emacs Info files
 *
 * When indexing info files, the user must index the files with real text
 * in them, rather than the file with the tag and indirect tables; otherwise
 * you'll end up with lots of garbage in your index.
 *
 * G. Wollman
 */

static int done_headline = 0;

boolean emacs_info_separator_function(line) /* hate K&R-style definitions */
char *line;
{
  if(line[0] == (char)31) {
    done_headline = 0;
    return true;
  }
  return false;
}

static char emacs_info_headline[MAX_HEADER_LEN+1];

void emacs_info_header_function(line)
register char *line;
{
  int i;

  if(done_headline)
    return;

  if(strncmp(line,"File: ",6))
    return;

  done_headline = 1;
  line += 6;			/* skip over "File: " */

  i = 1;
  emacs_info_headline[0] = '(';
  while(*line && *line != ',' && (i < MAX_HEADER_LEN-1))
    emacs_info_headline[i++] = *line++;

  emacs_info_headline[i++] = ')';

  line += 9;			/* skip over ", Node: " */

  /* copy the name of the info node into the headline */
  while(*line && (i < MAX_HEADER_LEN) && (*line != ','))
    emacs_info_headline[i++] = *line++;

  emacs_info_headline[i++] = '\0';
}

void emacs_info_finish_header_function(header)
char *header;
{
  strcpy(header,emacs_info_headline);
}

/* ========================================
 * ===    Medline  Customizations      ====
 * ========================================
 */

/* 
	Francois Schiettecatte
	with help from:
	Tom Emmel
	Karen Phipps
*/

char medline_header[MAX_HEADER_LEN +1];
char medline_title[MAX_HEADER_LEN + 1];
char medline_date[MAX_HEADER_LEN + 1];
char medline_author[MAX_HEADER_LEN + 1];

static boolean medline_start = true;


boolean medline_separator_function(line)
char *line;
{
  if (medline_start == true) {
    medline_start = false;
    return true;
  }
  if (strlen(line) < 2)
    medline_start = true;
  return false;
}


void medline_header_function(line)
char *line;
{
 char *ptr;

 if((strlen(line) > strlen("TI ")) && 
     (substrcmp(line, "TI "))){    
    strncpy(medline_title, line + strlen("TI "), MAX_HEADER_LEN);
  }
 
  if((strlen(line) > strlen("SO ")) &&
     (substrcmp(line, "SO "))){
   ptr = strchr(line,'1');
   strncpy(medline_date, ptr, MAX_DATE_LEN);
  }

  if((strlen(line) > strlen("AU ")) &&
     (substrcmp(line, "AU "))){
    ptr = strtok(line + strlen("AU "),".,");
    strcpy(medline_author,ptr);
    strncat(medline_author, "  ", MAX_AUTHOR_LEN); 
  } 
}

void medline_finish_header_function(header)
char *header;
{
  if(strlen(medline_author) > 0 ){
   strncat(medline_header,medline_author, MAX_HEADER_LEN);
  }
  
  if(strlen(medline_date) > 0 ){
    strncat(medline_header,"(", MAX_HEADER_LEN);
    strncat(medline_header,medline_date, MAX_HEADER_LEN);
    strncat(medline_header,") ", MAX_HEADER_LEN);
  }

  if(strlen(medline_title) > 0 ){
    strncat(medline_header,medline_title, MAX_HEADER_LEN);
  }
  
  if(strlen(medline_header) == 0){
    strcpy(header, "No Title");
  }
  else{
    strncpy(header, medline_header, MAX_HEADER_LEN);
  }

  medline_header[0] = '\0';
  medline_title[0] = '\0';
  medline_date[0] = '\0';
  medline_author[0] = '\0';
}




/* ========================================
 * ===    Refer  Customizations      ====
 * ========================================
 */


/* 
	Francois Schiettecatte
	with help from:
	Tom Emmel
	Karen Phipps
*/

char refer_header[MAX_HEADER_LEN +1];
char refer_title[MAX_HEADER_LEN + 1];
char refer_date[MAX_HEADER_LEN + 1];
char refer_author[MAX_HEADER_LEN + 1];

static boolean refer_start = true;


boolean refer_separator_function(line)
char *line;
{
  if (refer_start == true) {
    refer_start = false;
    return true;
  }
  if (strlen(line) < 2)
    refer_start = true;
  return false;
}


void refer_header_function(line)
char *line;
{
  if((strlen(line) > strlen("%T ")) && 
     (substrcmp(line, "%T "))){    
    strncpy(refer_title, line + strlen("%T "), MAX_HEADER_LEN);
  }
  else if((strlen(line) > strlen("%B ")) && 
     (substrcmp(line, "%B ")) && (strlen(refer_title) == 0)){    
    strncpy(refer_title, line + strlen("%B "), MAX_HEADER_LEN);
  }
  
  if((strlen(line) > strlen("%D ")) &&
     (substrcmp(line, "%D "))){
    strncpy(refer_date, line + strlen("%D "), MAX_DATE_LEN);
  }

  if((strlen(line) > strlen("%A ")) &&
     (substrcmp(line, "%A ")) && (strlen(refer_author) == 0)){
    strncpy(refer_author, line + strlen("%A "), MAX_AUTHOR_LEN);
    strncat(refer_author, " ", MAX_AUTHOR_LEN);
  }
  else if((strlen(line) > strlen("%E ")) &&
     (substrcmp(line, "%E ")) && (strlen(refer_author) == 0)){
    strncpy(refer_author, line + strlen("%E "), MAX_AUTHOR_LEN);
    strncat(refer_author, " ", MAX_AUTHOR_LEN);
  }
}

void refer_finish_header_function(header)
char *header;
{
  if(strlen(refer_author) > 0 ){
    strncat(refer_header,refer_author, MAX_HEADER_LEN);
  }
  
  if(strlen(refer_date) > 0 ){
    strncat(refer_header,"(", MAX_HEADER_LEN);
    strncat(refer_header,refer_date, MAX_HEADER_LEN);
    strncat(refer_header,") ", MAX_HEADER_LEN);
  }

  if(strlen(refer_title) > 0 ){
    strncat(refer_header,refer_title, MAX_HEADER_LEN);
  }
  
  if(strlen(refer_header) == 0){
    strncpy(header, "No Title", MAX_HEADER_LEN);
  }
  else{
    strncpy(header, refer_header, MAX_HEADER_LEN);
  }

  refer_header[0] = '\0';
  refer_author[0] = '\0';
  refer_date[0] = '\0';
  refer_title[0] = '\0';
}




/* ===========================================
 * ===    First Line  Customizations      ====
 * ===========================================
 */

/* this means the first line of the file is the headline.
   useful for the lyrics server */

/* paragraph files - seperated by a blank line.  Next line is the header */

char first_line_header[MAX_HEADER_LEN +1];

boolean first_line_separator_function(line)
char *line;
{
  return false;
}

void first_line_header_function(line)
char *line;
{
  if (first_line_header[0] == '\0')
    s_strncpy(first_line_header, line, MAX_HEADER_LEN);
}

void first_line_finish_header_function(header)
char *header;
{
  if (strlen(first_line_header) == 0) {
    strcpy(header, "No Title");
  }
  else {
    s_strncpy(header, first_line_header, MAX_HEADER_LEN);
  }
  first_line_header[0] = 0;
}  

/* =========================
 * ===  BIBTEX Separator ===
 * =========================
 * S.P.vandeBurgt@research.ptt.nl (Stan)
 *
 * BibTeX entries
 *
 * @......{
 * ......
 * title = header
 * .......}
 *
 */

static char bibtex_header[MAX_HEADER_LEN + 1];

boolean bibtex_separator_function(line)
char *line;
{
    char *p = line;

    while (isspace(*p)) p++; /* skip space */
    return(*p == '@');
}

void bibtex_header_function(line)
char *line;
{
    char *p = line;

    p = strstr(line, "title");
    if (p == NULL) p = strstr(line, "Title");
    if (p == NULL) p = strstr(line, "TITLE");
    if (p != NULL)
    {
      p += 5;

      while (isspace(*p)) p++; /* skip space */
      if (*p == '=')          /* should be an '=' now */
      {
          p++;
          /* skip bibtex char's */
          while (isspace(*p) || *p == '"' || *p == '{') p++;
          strncpy(bibtex_header, p, MAX_HEADER_LEN);
          for (p = bibtex_header; *p != '\0'; p++)
          {
              /* replace bibtex char's */
              if (*p == '\n' || *p == '"' || *p == '}' || *p == '{')
              {
                  *p = ' ';
              }
          }
      }
    }
}

void bibtex_finish_header_function(header)
char *header;
{
    if (bibtex_header[0] == '\0')
    {
      strcpy(header, "Unknown Title");
    }
    else{
      strncpy(header, bibtex_header, MAX_HEADER_LEN);
    }
    bibtex_header[0] = '\0';
}


/* =========================
 * ===  NHYP Separator ===
 * =========================
 * S.P.vandeBurgt@research.ptt.nl (Stan)
 * Nhyp entries
 *
 * ?:? header
 * ......
 * ......
 *
 */

static char nhyp_header[MAX_HEADER_LEN + 1];

boolean nhyp_separator_function(line)
char *line;
{
    return(strstr(line, "?:?") != NULL);
}

void nhyp_header_function(line)
char *line;
{
    char *p = line;

    p = strstr(line, "?:?");
    if (p != NULL)
    {
      p += 3;
      while (isspace(*p)) p++; /* skip space */
      strncpy(nhyp_header, p, MAX_HEADER_LEN);
      trim_trailing_newline(nhyp_header);
    }
}

void nhyp_finish_header_function(header)
char *header;
{
    if (nhyp_header[0] == '\0')
    {
      strcpy(header, "Unknown Title");
    }
    else{
      strncpy(header, nhyp_header, MAX_HEADER_LEN);
    }
    nhyp_header[0] = '\0';
}
