;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Copyright (c) 1993 by William M. Perry (wmperry@indiana.edu)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions for compatibility with XMosaic
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(require 'w3)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Support for global history file
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun w3-write-global-history (&optional fname)
  "Write the global history file into w3-global-history-file"
  (interactive)
  (if (not fname) (setq fname w3-global-history-file))
  (if (not (file-exists-p w3-global-history-file))
      (progn
	(message "Creating history file %s." w3-global-history-file)
	(set-buffer (get-buffer-create " *W3HIST*"))
	(erase-buffer)
	(insert "ncsa-mosaic-history-format-1\nGlobal\n"))
    (progn
      (set-buffer (get-buffer-create " *W3HIST*"))
      (erase-buffer)
      (insert-file-contents w3-global-history-file)))
  (let ((tmp w3-history-list)
	url)
    (while tmp
      (setq url (car (car tmp)))
      (goto-char (point-min))
      (if (not (re-search-forward (regexp-quote url) nil t))
	  (progn
	    (goto-char (point-max))
	    (insert (format "%s %s\n" url (current-time-string)))))
      (setq tmp (cdr tmp))))
  (write-file w3-global-history-file)
  (kill-buffer (current-buffer)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Hotlist Handling Code
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun w3-remove-from-hotlist ()
  "Deletes a document from your hotlist file"
  (interactive)
  (if (not w3-hotlist) (message "No hotlist in memory!")
    (if (not (file-exists-p w3-hotlist-file))
	(message "Hotlist file %s does not exist." w3-hotlist-file)
      (let* ((title (car (assoc (completing-read "Delete Document: "
						 w3-hotlist nil t)
				w3-hotlist)))
	     (buffer (get-buffer-create " *HOTW3*")))
	(set-buffer buffer)
	(erase-buffer)
	(insert-file-contents w3-hotlist-file)
	(if (re-search-forward (regexp-quote title) nil t)
	    (progn
	      (previous-line 1)
	      (beginning-of-line)
	      (kill-line 2)
	      (write-file w3-hotlist-file)
	      (setq w3-hotlist (w3-delete-from-alist title w3-hotlist))
	      (kill-buffer (current-buffer)))
	  (message "%s was not found in %s" title w3-hotlist-file))))))

(defun w3-parse-hotlist (&optional fname)
  "Read in the hotlist specified by FNAME"
  (if (not fname) (setq fname w3-hotlist-file))
  (setq w3-hotlist nil)
  (if (not (file-exists-p fname))
      (message "%s does not exist!" fname)
    (let* ((buffer (get-buffer-create " *HOTW3*"))
	   cur-link
	   cur-alias)
      (set-buffer buffer)
      (erase-buffer)
      (insert-file-contents fname)
      (w3-replace-regexp "^$\n" "")
      (goto-line 3)
      (while (not (equal (point) (point-max)))
	(re-search-forward "^[^ ]*" nil t)
	(setq cur-link (buffer-substring (match-beginning 0) (match-end 0)))
	(setq cur-alias (buffer-substring (progn
					    (forward-line 1)
					    (beginning-of-line)
					    (point))
					  (progn
					    (end-of-line)
					    (point))))
	(if (not (equal cur-alias ""))
	    (setq w3-hotlist (cons (list cur-alias cur-link) w3-hotlist))))
      (kill-buffer buffer))))

(defun w3-use-hotlist ()
  "Possibly go to a link in the hotlist"
  (interactive)
  (if (not w3-hotlist) (message "No hotlist in memory!")
    (let* ((url (car (cdr (assoc
			   (completing-read "Goto Document: " w3-hotlist nil t)
			   w3-hotlist)))))
      (w3-fetch url))))

(defun w3-add-document-to-hotlist (&optional gopher-link)
  "Add this documents url to the hotlist"
  (interactive)
  (let* ((buffer (get-buffer-create " *HOTW3*"))
	 (title (if gopher-link (gopher-object-descr gopher-link)
		  (buffer-name)))
	 (url (cond
	       (gopher-link (format "gopher://%s:%d/%c%s"
				    (gopher-object-host gopher-link)
				    (gopher-object-port gopher-link)
				    (gopher-object-type gopher-link)
				    (gopher-object-selector gopher-link)))
	       ((equal w3-current-type "ftp")
		(concat "ftp://" w3-current-server
			(if (= ?/ (string-to-char w3-current-file))
			    "" "/") w3-current-file))
	       ((equal w3-current-type "gopher")
		(format "gopher://%s:%s/%s" w3-current-server
			w3-current-port w3-current-file))
	       ((equal w3-current-type "http")
		(concat "http://" w3-current-server ":"
			w3-current-port "/" w3-current-file))
	       ((equal w3-current-type nil)
		(concat "file:" w3-current-file)))))
    (set-buffer buffer)
    (erase-buffer)
    (setq w3-hotlist (cons (list title url) w3-hotlist))
    (if (not (file-exists-p w3-hotlist-file))
	(progn
	  (message "Creating hotlist file %s" w3-hotlist-file)
	  (insert "ncsa-xmosaic-hotlist-format-1\nDefault\n\n")
	  (backward-char 1))
      (progn
	(insert-file-contents w3-hotlist-file)
	(goto-char (point-max))
	(backward-char 1)))
    (insert "\n" url " " (current-time-string) "\n" title)
    (write-file w3-hotlist-file)
    (kill-buffer (current-buffer))))

(defun w3-add-gopher-hotlist-entry ()
  "Add this gopher link to the hotlist file"
  (interactive)
  (w3-add-document-to-hotlist gopher-obj))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Annotation server handling
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun w3-fetch-annotations ()
  "Fetch the annotations for the current document"
  (if (not w3-group-annotation-server)
      (error "No group annotation server defined!")
    (let ((cmd (format "ANN_GET /url=\"%s\";=\n" (w3-view-url t)))
	  (proc nil)
	  (tmp 0)
	  (tmp2 0))
      (save-excursion
	(set-buffer (get-buffer-create " *W3*"))
	(goto-char (point-max))
	(insert "\n"))
      (setq proc (w3-open-stream "*anno*" (get-buffer-create " *W3*")
				 w3-group-annotation-server
				 w3-group-annotation-port))
      (message "Fetching annotations...")
      (if (processp proc)
	  (progn
	    (process-send-string proc cmd)
	    (while (memq (process-status proc) '(run open))
	      (if (= 0 (setq tmp2 (% (1+ tmp2) 200)))
		  (message "Fetching annotations..%s" (make-string
						       (setq tmp (% (1+ tmp) 50))
						       ?.)))
	      (accept-process-output)))
	(message proc))
      
      (condition-case ()
	  (delete-process proc);; make sure its dead
	(error nil))
      (w3-replace-regexp (regexp-quote (substring cmd 0 -1)) "")
      (w3-replace-regexp
       "Process .*anno.* \\(exit\\|kill\\|finish\\).*" ""))))

(defun w3-is-annotation ()
  "Is this a group annotation?"
  (and (equal w3-current-server w3-group-annotation-server)
       (= (string-to-int w3-current-port) w3-group-annotation-port)))

(defun w3-delete-group-annotation ()
  "Delete this group annotation"
  (interactive)
  (if (not w3-group-annotation-server)
      (error "No group annotation server defined!")
    (if (w3-is-annotation)
	(let ((cmd (format "ANN_DELETE /url=\"%s\";=\n" (w3-view-url t)))
	      (proc nil)
	      (parseit nil)
	      (buff (current-buffer)))
	  (save-excursion
	    (set-buffer (get-buffer-create " *W3*"))
	    (erase-buffer)
	    (setq proc (w3-open-stream "*anno*" (get-buffer-create " *W3*")
				       w3-group-annotation-server
				       w3-group-annotation-port))
	    (if (processp proc)
		(process-send-string proc cmd))
	    (while (memq (process-status proc) '(run open))
	      (accept-process-output))
	    (condition-case ()
		(delete-process proc);; make sure its dead
	      (error nil))
	    (w3-replace-regexp (regexp-quote cmd) "")
	    (w3-replace-regexp "Process .*anno.* exit.*" "")
	    (goto-char (point-min))
	    (setq w3-current-type nil
		  w3-current-file "historylist"
		  w3-current-last-buffer buff)
	    (if (and (re-search-forward "[^ \t\n]+" nil t)
		     (not
		      (progn
			(goto-char (point-min))
			(re-search-forward "success!" nil t))))
		(setq parseit t)
	      (setq parseit nil)))
	  (if parseit
	      (progn
		(setq w3-continuation (append w3-default-continuation
					      '(w3-prepare-buffer)))
		(w3-use-continuation))
	    (message "Deleted...")))
      (message "This is not an annotation."))))

(defun w3-add-group-annotation ()
  "Add an annotation to the current url"
  (interactive)
  (let ((url (w3-view-url t))
	(buf (get-buffer-create "*Annotation*")))
    (if w3-mutable-windows (pop-to-buffer buf) (switch-to-buffer buf))
    (set-buffer buf)
    (erase-buffer)
    (insert "</PRE>\n\n")
    (html-mode)
    (setq w3-current-annotation url)
    (define-key html-mode-map "\C-c\C-c" 'w3-do-group-annotation)
    (message "Hit C-cC-c to send this annotation.")))	

(defun w3-do-group-annotation ()
  "Finish adding an annotation"
  (interactive)
  (let* ((bufsize (buffer-size))
	 (bufstr  (buffer-string))
	 (cmd (format "ANN_%s /url=\"%s\";title=\"%s\";user=\"%s\";date=%s;length=%d;=%s\n"
		      (if w3-editing-annotation "CHANGE" "SET")
		      w3-current-annotation
		      (read-string "Title: "
				   (format "Annotation by %s (%s@%s)"
					   (user-full-name)
					   (user-real-login-name)
					   (system-name)))
		      (format "%s@%s" (user-real-login-name)
			      (system-name))
		      (current-time-string)
		      bufsize bufstr))
	 (proc nil)
	 (parseit nil))
    (save-excursion
      (not-modified)
      (kill-buffer (current-buffer))
      (set-buffer (get-buffer-create " *W3*"))
      (erase-buffer)
      (setq proc (w3-open-stream "*anno*" (get-buffer-create " *W3*")
				 w3-group-annotation-server
				 w3-group-annotation-port))
      (process-send-string proc cmd)
      (while (memq (process-status proc) '(run open))
	(accept-process-output))
      (condition-case ()
	  (delete-process proc);; make sure its dead
	(error nil))
      (w3-replace-regexp (regexp-quote cmd) "")
      (w3-replace-regexp "Process .*anno.* exit.*" "")
      (goto-char (point-min))
      (setq w3-current-type nil
	    w3-current-file "historylist")
      (if (re-search-forward "[^ \t\n]+" nil t)
	  (setq parseit t)
	(setq parseit nil)))
    (if parseit
	(progn
	  (setq w3-continuation (append w3-default-continuation
					'(w3-prepare-buffer)))
	  (w3-use-continuation))
      (message "Annotation delivered..."))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Private annotation support
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun w3-parse-personal-annotations ()
  "Read in personal annotation file"
  (if (and 
       (file-exists-p (format "%s/LOG" w3-personal-annotation-directory))
       (file-readable-p (format "%s/LOG" w3-personal-annotation-directory)))
      (save-excursion
	(setq w3-personal-annotations nil);; nuke the old list
	(let ((start nil)
	      (end nil)
	      (txt nil)
	      (url nil)
	      (num nil))
	  (set-buffer (get-buffer-create " *panno*"))
	  (erase-buffer)
	  (insert-file-contents
	   (format "%s/LOG" w3-personal-annotation-directory))
	  (goto-char (point-min))
	  (kill-line 2);; nuke the header lines
	  (while (/= (point-max) (point))
	    (setq start (point)
		  end (prog2 (end-of-line) (point) (forward-char 1))
		  txt (buffer-substring start end)
		  url (substring txt 0 (string-match " " txt))
		  num (substring txt (1+ (string-match " " txt)) nil)
		  w3-personal-annotations 
		  (cons
		   (list url (list num
				   (w3-grok-annotation-format num)))
		   w3-personal-annotations)))
	  (kill-buffer " *panno*")))))


(defun w3-grok-annotation-format (anno)
  "Grab the title from an annotation"
  (save-excursion
    (set-buffer (get-buffer-create " *annotmp*"))
    (erase-buffer)
    (if (file-exists-p (format "%s/PAN-%s.html"
			       w3-personal-annotation-directory anno))
	(insert-file-contents (format "%s/PAN-%s.html"
				      w3-personal-annotation-directory anno)))
    (goto-char (point-min))
    (prog1
	(if (re-search-forward "<title>\\(.*\\)</title>" nil t)
	    (buffer-substring (match-beginning 1) (match-end 1))
	  "No title")
      (kill-buffer " *annotmp*"))))    

(defun w3-fetch-personal-annotations ()
  "Grab any personal annotations for the current url"
  (let ((url  (w3-view-url t))
	(anno w3-personal-annotations)
	(annolist nil))
    (if (assoc url anno)
	(while anno
	  (if (equal (car (car anno)) url)
	      (setq annolist
		    (cons
		     (format "<A HREF=\"file:/%s/PAN-%s.html\">%s</A>"
			     w3-personal-annotation-directory
			     (car (car (cdr (car w3-personal-annotations))))
			     (car (cdr (car (cdr (car w3-personal-annotations))))))
		     annolist)))
	  (setq anno (cdr anno))))
    annolist))

(defun w3-is-personal-annotation (url)
  "Is URL a personal annotation?"
  (if (string-match "file:/.*/PAN-.*\\.html" url) t nil))

(defun w3-delete-personal-annotation ()
  "Delete a personal annotation."
  (interactive)
  (if (w3-is-personal-annotation (w3-view-url t))
      (let ((num nil)
	    (annotated-url nil)
	    (anno w3-personal-annotations))
	(string-match "file:/.*/PAN-\\(.*\\)\\.html" (w3-view-url t))
	(setq num (substring (w3-view-url t) (match-beginning 1)
			     (match-end 1)))
	(while anno
	  (if (equal num (car (car (cdr (car anno)))))
	      (setq annotated-url (car (car anno))))
	  (setq anno (cdr anno)))
	(if annotated-url
	    (save-excursion
	      (set-buffer (get-buffer-create " *annotmp*"))
	      (erase-buffer)
	      (insert-file-contents (format "%s/LOG"
					    w3-personal-annotation-directory))
	      (delete-matching-lines (format "%s %s" annotated-url num))
	      (let ((make-backup-files nil)
		    (version-control nil)
		    (require-final-newline t))
		(write-region (point-min) (point-max)
			      (format "%s/LOG"
				      w3-personal-annotation-directory)))
	      (kill-buffer " *annotmp*")
	      (setq anno w3-personal-annotations
		    w3-personal-annotations nil)
	      (while anno
		(if (not (string= num (car (car (cdr (car anno))))))
		    (setq w3-personal-annotations
			  (cons anno w3-personal-annotations)))
		(setq anno (cdr anno)))
	      (delete-file (format "%s/PAN-%s.html"
				   w3-personal-annotation-directory num)))
	  (message "Couldn't find url that this is annotating!")))
    (message "This isn't a personal annotation.")))  

(defun w3-add-personal-annotation ()
  "Add an annotation to this document."
  (interactive)
  (let ((url (w3-view-url t))
	(buf (get-buffer-create "*Personal Annotation*")))
    (set-buffer buf)
    (if w3-mutable-windows (pop-to-buffer buf) (switch-to-buffer buf))
    (erase-buffer)
    (insert "</PRE>\n\n")
    (html-mode)
    (setq w3-current-annotation url)
    (define-key html-mode-map "\C-c\C-c" 'w3-do-personal-annotation)
    (message "Hit C-cC-c to send this annotation.")))

(defun w3-do-personal-annotation ()
  "Finish doing a personal annotation."
  (interactive)
  (if (or (not w3-personal-annotation-directory)
	  (not (file-exists-p w3-personal-annotation-directory))
	  (not (file-directory-p w3-personal-annotation-directory)))
      (error "No personal annotation directory!")
    (let ((url w3-current-annotation)
	  (txt (buffer-string))
	  (title (read-string "Title: "
			      (format "Annotation by %s on %s"
				      (user-real-login-name)
				      (current-time-string))))
	  (fname nil)
	  (num nil))
      (save-excursion
	(not-modified)
	(kill-buffer (current-buffer))
	(set-buffer (get-buffer-create " *annotmp*"))
	(erase-buffer)
	(if (file-exists-p 
	     (format "%s/LOG" w3-personal-annotation-directory))
	    (insert-file-contents
	     (format "%s/LOG" w3-personal-annotation-directory))
	  (progn
	    (goto-char (point-min))
	    (insert "ncsa-mosaic-personal-annotation-log-format-1\n")
	    (insert "Personal\n")))
	(goto-char (1- (point-max)))
	(beginning-of-line)
	(if (re-search-forward ".* \\([0-9]+\\)" nil t)
	    (setq num (buffer-substring (match-beginning 1) (match-end 1)))
	  (setq num "1"))
	(setq fname (format "%s/PAN-%s.html"
			    w3-personal-annotation-directory num))
	(goto-char (point-max))
	(insert (format "%s %s\n" url num))
	(let ((make-backup-files nil)
	      (version-control nil)
	      (require-final-newline t))
	  (write-region (point-min) (point-max)
			(format "%s/LOG" w3-personal-annotation-directory))
	  (erase-buffer)
	  (insert (format "%s\n<title>%s</title>\n<h1>%s</h1>\n"
			  w3-annotation-marker
			  title title))
	  (insert
	   (format "<address>%s (%s@%s)</address>\n<address>%s</address>\n"
		   (user-full-name)
		   (user-real-login-name)
		   (system-name)
		   (current-time-string)))
	  (insert "______________________________________\n<pre>" txt)
	  (write-region (point-min) (point-max)
			(format "%s/PAN-%s.html"
				w3-personal-annotation-directory num)))
	(setq w3-personal-annotations
	      (cons (list url (list num title)) w3-personal-annotations))))))

(provide 'w3-mosaic)
