;;;; rcs.el deals with rcs'd files in a natural way
;; Copyright (C) 1989 Chris Siebenmann under the terms of the FSF GNU
;; General Public License, version 1
;; modified and added to by
;; Jatinder Singh (jat@water.ca.gov)
;; Ralph Finch (rfinch@water.ca.gov)

;; $Header: /site/lib/emacs/RCS/rcs.el,v 1.0.4.125 1992/07/13 14:57:19 rfinch Exp $

(require 'emerge)
(provide 'rcs)

;;;;			Some Documentation
;; This file provides a convenient interface to rcs that allows you
;; to easily check out, check in, and log changes to files that you
;; edit, as well as automatically check out a file when you try and
;; edit it, and record on the mode line who has locked the file.

;; Please report any bugs and improvements to rfinch@water.ca.gov
;; This version of rcs.el works only with version 5.5 or later of RCS.

;; Suggested usage: enable strict locking, and make all source files
;; non-writable. Then whenever you try to edit a file that's
;; read-only, you know you have to check it out; if the file hasn't
;; already been placed under RCS, you will be prompted for an initial
;; description. Conversly, a modifiable file is one you have locked,
;; and can thus change freely.

;; If you're su'd to root, files you check out will have the locker
;; recorded as the real you, not root. i.e. No branches can be created
;; for root.

;; Rcs itself is free, and can be obtained via ftp from
;; arthur.cs.purdue.edu from pub/RCS, or from a number of uucp
;; archive sites. You will also need a version of GNU diff, also
;; widely available from ftp and uucp archive sites.

;; To use, bind the available functions to convenient keys. A good
;; set of functions to start by binding are rcs-ci-buffer and
;; rcs-co-file; we use C-c C-s and C-c C-f. Note that rcs-co-file will
;; automatically create an RCS directory and check in the current
;; buffer if the file is not currently under RCS.


;;; Functions available:
;; rcs-co-file	        	Check out a file. If the
;;				file isn't already RCS'd, it will be
;;				checked in automatically.
;; rcs-ci-{file,buffer}		Check in a file or buffer. The file
;;				will be left checked out unlocked.
;; rcs-diff-{file,buffer}	rcsdiff the current version of the
;;				file against the most recent RCS
;;				version. Useful for seeing what
;;				changes you just made.
;; rcs-show-log-file		Show the RCS change log for a given
;;				file or buffer.
;; rcs-revert-{file,buffer}	Revert a file or buffer back to the
;;				last checked in version. An easy way
;;				to blow away changes you've decided
;;				you don't want, or back out of
;;				checking out locked a file you decide
;;				not to change.
;; rcs-refresh-buffer		Reload the current buffer from its file,
;;				usually because someone else has just
;;				unlocked it.
;; rcs-make-user-{branch,variation} Make a user branch attached to the
;;                              main stem, or a user variation attached
;;                              to the user's branch.

;; The following function is intended to be placed on a hook,
;; instead of being called directly.

;; rcs-hack-modeline	-	Put this on your find-file-hooks hook;
;;				when run, it will put a legend up
;;				about who has locked the file being
;;				edited in that buffer.
;;				Caution -- this can be slow.
;; Eg, to put on a hook, add the following lisp code to your .emacs.
;; (setq find-file-hooks (list 'rcs-hack-modeline))

;;; Hooks available:
;; rcs-hook	-	run at the end of this file (ie after all
;;			the functions have been loaded).
;; rcs-new-dir-hook	run when a new RCS directory has been created.
;; rcs-new-file-hook	run when a file is checked in for the first time.
;; (see rcs-ci-file for more details on the latter two)
;; All hooks are run with run-hooks.

;; Terminology used in the routines.
;; Suppose that one has the following symbolic names in an RCS file:
;; main: 1
;; user1: 1.0.1
;; user1_var2: 1.0.1.2.2
;; user1_var1: 1.0.1.2.1

;; 'symbol' is the RCS symbol bound to an RCS main stem or branch.  Here,
;; the symbol is almost always an alphanumeric string, either assigned
;; by the program or selected by the user. Above, the symbols are 'main',
;; 'user1', 'user1_var1', and 'user1_var2'.  We never bind symbols to
;; entire revision numbers, only to branches.

;; 'Revision' is the entire revision number, branch+tip.  It always has
;; an odd number of dots.  No entire revision numbers are shown above,
;; they appear in the full rlog output.

;; 'Item' is the revision number minus the tip.  In the example the
;; items are 1, 1.0.1, 1.0.1.2.2, and 1.0.1.2.1.  Items always
;; have an even number of dots.

;; 'Tip' is the last number in the entire revision number.  There is not
;; enough information in the example to tell what the tip of main is,
;; but we can tell that at least at one time the tip of user1 was '2'
;; (user1 may have a later tip now).  Tips have no dots.

;; 'main stem' is the main trunk of the RCS file.  Here, its symbol is
;; 'main', and its branch number is '1'.

;; 'user branches' (or branches) come off the main stem, possibly at
;; different tips of the main stem.  Their branch number at creation time
;; is derived as follows: main.maintip.user# where user# is the highest
;; existing user number, plus 1.  Their symbol is the user login name.

;; 'user variations' (or variations) come off user branches, just like
;; user branches come off the main stem.  They are numbered:
;; userbranch.userbranchtip.var#, where var# is the highest existing
;; variation number for the user, + 1.  Their symbol is assigned by the
;; user.

;; Thus, one can form a matrix showing all possibilities for an example:

;;	type		symbol		item		tip	revision

;;	main		"main"		1		1	1.1
;;	branch		"user1"		1.1.3		5	1.1.3.5
;;	variation	"user1_var1"	1.1.3.5.1	2	1.1.3.5.1.2



;;; User-changeable variables.

(defvar rcs-use-user-branch t
  "*If non-nil, implements user-branch concept.
Each user will be provided with one branch in each checked in file.")

(defvar rcs-check-outdated t
  "*If non-nil, checks for outdated version on checkout.")

(defvar rcs-check-symbols ""
  "*If string, check against these symbols for outdated version,
in addtion to default symbols.")

(defvar rcs-show-freeze-symbols nil
  "*If non-nil, show RCS symbols which are for a freeze in the
 Variation Menu")

(defvar rcs-initial-access nil
  "*If non-nil, access list to assign to an initial checkin.
rcs.el does not check who is on the access list.  However, an RCS
error will occur when trying to checkin a file that the user is not
the owner of and is not on the access list.")

(defvar rcs-use-login-name t
  "*If non-nil, use the name you're currently logged in under, instead
of the real name.  For example, if t, and you su to root and use emacs
to check in a file, the RCS log will have 'root'.")

(defvar rcs-make-backup-files nil
  "*If non-nil, backups of checked-in files are made according to
the make-backup-files variable.  Otherwise, prevents backups being made.")

(defvar rcs-edit-mode "text-mode"
  "*If non-nil, the log buffer will be placed in this mode.
Otherwise, the log buffer will be in default-major-mode.")

(defvar rcs-keep-log nil
  "*If non-nil, uses old log for check in of later revisions.
Otherwise gives user blank log message for each checkin.")

(defvar rcs-clean-before-copy nil
  "*If non-nil, attempts to clean up file before copying to new version.")

(defvar rcs-edit-before-copy nil
  "*If non-nil, allow user to edit file before copying to new version.")

(defvar rcs-log-before-copy nil
  "*If non-nil, allow user to enter log message before copying to new version.")

(defvar rcs-diff-options nil
  "*If a string, use these options for diff operations.")

(defvar rcs-gnu-diff-options nil
  "*If a string, the options to use for a GNU diff in rcsdiff.")

(defvar rcs-list-all-lockers t
  "*If non-nil, list all the lockers on the mode line, else just list
the user.")

;; Some switches that user can set for *VARIATION LIST* buffer.

(defvar rcs-use-previous-symbols t
  "*For multiple file functions.  If t, functions skip over a file
 if symbols don't exist; if nil, prompt user for new symbol.")

(defvar rcs-list-all-symbols nil
  "*If non-nil, all symbols associated with a file will be listed.
This listing appears in the *VARIATION BUFFER* buffer.  Otherwise,
only the symbols created by the user associated with the file are
displayed.  Set to t, if the only revision that exists in a file is the
stem.")

;;; Global variables.

(defvar rcs-symbol ""
  "The name of the symbol that user wants to work with.
It is selected from the *VARIATION LIST* buffer, but stored as a
global variable.")

(defvar rcs-item-number ""
  "The RCS number of the symbol that user has selected from *VARIATION LIST*")

(defvar rcs-revision ""
  "An RCS revision number.")

(defvar rcs-checkin-locked nil
  "If non-nil symbol should be checked in as locked.")

(defvar rcs-filename ""
  "Current rcs file.")

(defvar rcs-buf-ancestor ""
  "Ancestor used by emerge.")

(defvar rcs-quit nil
  "If a quit is signaled in variation buffer.")

(defvar rcs-command nil
  "RCS command to do on the files in rcs-do-many.")

(defvar rcs-save-for-many nil
  "T when the Variation menu commands should save their command name
and args used.")

(defvar rcs-items '()
  "List of items to use with rcs-command.")

(defvar rcs-tmp-files '()
  "Alist of temporary file names:
( ("realname" "tmpname1")...("realname" "tmpnameN"))")

(defvar rcs-win-conf nil
  "Window configuration to return to.")

(defvar rcs-use-prev-log nil
  "If non-nil, rcs-ci-file will not prompt for new log.")

(defvar rcs-use-1st-log nil
  "If non-nil, use 1st log for all logs for polyfile checkin.")

(defvar Variation-return-func nil
  "Function to return to from 2-part variation buffer command.")

(defvar Variation-return-args '()
  "Args (as list) to use in return function.")

;;; Hooks
(defvar rcs-hook nil
  "*Hooks run at the end of this file.")
(defvar rcs-new-dir-hook nil
  "*Hooks run when a new ./RCS directory is created.")
(defvar rcs-new-file-hook nil
  "*Hooks run when a file has been checked in for the first time.")

;;; Variables the user won't normally want to change
(defvar rcs-log-buffer "*rcs log*")

(defvar rcs-temp-buffer "*rcs temp*")

(defvar rcs-error-buffer "*rcs error*") ; M. Gregory  23-Aug-91 11:09

(defvar rcs-executable-path nil
  "If non-nil, the default path to find an RCS command on.")

(defvar rcs-exec-path nil
  "Path rcs searches to find executables.
Built from rcs-executable-path and exec-path.")

;; set this to a small shell that starts fast.
(defvar rcs-shell-path "/bin/sh"
  "If non-nil, the file name to load inferior shells for RCS commands from.
If nil, shell-file-name's value is used instead.")

(defvar rcs-initial-branch "1"
  "Branch number to assign an initial checkin.")

(defvar rcs-initial-rev "0"
  "Revision number to assign an initial checkin.")

(defvar rcs-main-symbolic "main"
  "Symbolic name used for main stem.")

(defvar rcs-varbuf "*Variation List*"
  "The buffer that contains all file symbols that can be accessed.")


;;; utility functions.

(defun rcs-interactive-init ()
  "Initialization of rcs."
  (setq rcs-user-name
	(if rcs-use-login-name
	    (user-login-name)
	  (user-real-login-name))
	))

(defun rcs-buf-kill-and-reload (fn)
  "Given FILE, load the current version of that file into a buffer.
If the file was in a buffer already, the buffer will be refreshed to
contain the latest version of the file. If the buffer has been
modified, the new changes will NOT be saved.  Attempts to
keep the point and mark the same if the buffer was around before:
they are put at the same line and column as they were before this call"
  (if (setq buf (get-file-buffer fn))
      (if (string= fn (buffer-file-name buf))
	  (progn
	    (set-buffer buf)
	    (let ((curp-col (current-column))
		  (curp-line
		   (save-excursion
		     (beginning-of-line) ; get past bug in count-lines
		     (+ 1 (count-lines (point-min) (point))) )))
	      (if (mark)
		  (goto-char (mark)) )
	      (let ((curm-col (current-column))
		    (curm-line
		     (save-excursion
		       (beginning-of-line) ; get past bug in count-lines
		       (+ 1 (count-lines (point-min) (point))) )))
		(set-buffer-modified-p nil)
		(kill-buffer buf)
		(rcs-find-file fn)
		(goto-line curm-line)
		(move-to-column curm-col)
		(set-mark (point))
		(goto-line curp-line)
		(move-to-column curp-col) )))
	(rcs-find-file fn))
    (rcs-find-file fn)))

(defun rcs-make-rcs-name (fn)
  "Make the name for an RCS file from the normal file name."
  (if (string-match ",v$" fn)
      fn
    (let ((rcsfn nil))
      (if (or
	   (file-exists-p (setq rcsfn (concat
				       (file-name-directory fn)
				       "RCS/" (file-name-nondirectory fn)
				       ",v")))
	   (file-exists-p (setq rcsfn (concat fn ",v"))))
	  rcsfn
	nil))))

(defun rcs-make-normal-name (fn)
  "Make a normal file name from an RCS file name."
  (let ((case-fold-search nil))
    (if (string-match ",v$" fn)
	(setq fn (substring fn 0 (match-beginning 0))))
    (if (string-match "RCS/" fn)
	(setq fn (concat
		  (substring fn 0 (match-beginning 0))
		  (substring fn (match-end 0)))))
    fn))

(defun rcs-is-rcs-file-p (fn)
  "Return t if FILE is an RCS file."
  (string= (substring fn -2) ",v"))

(defun rcs-has-rcs-file-p (fn)
  "Return t if FILE has an RCS file."
  (if (or
       (string-match ",v$" fn)
       (rcs-make-rcs-name fn))
      t
    nil))

(defun rcs-list-files (&optional directory-name recurse inquire list-to-date)
  
  "Returns a list of RCS files (with working file name) in optional
DIRECTORY (current if nil).  If RECURSE non-nil, descend recursively
into subdirectories; if string, use it as include-only regexp for
directories.  If INQUIRE non-nil, ask before adding a new directory's
files to list.  Final arg LIST is list of files so far."
  
  (if (not (and directory-name
		(file-directory-p directory-name)))
      (setq directory-name "./"))
  (setq directory-name (file-name-as-directory directory-name))
  (let ((list-dirs '())
	(list-this-dir
	 (mapcar 'rcs-make-normal-name
		 (if (file-directory-p
		      (setq rcs-directory (concat directory-name "RCS")))
		     (directory-files rcs-directory t ",v$")
		   (directory-files directory-name t ",v$")))))
    (if (and list-this-dir
	     (or (not inquire)
		 (y-or-n-p (concat "Add files from " directory-name " "))))
	(setq list-to-date (append list-to-date list-this-dir)))
    (if (and recurse
	     (setq list-dirs (rcs-directory-dirs
			      directory-name
			      (if (stringp recurse)
				  recurse))))
	(while list-dirs
	  (setq list-to-date
		(append (rcs-list-files (car list-dirs) recurse inquire list-to-date)))
	  (setq list-dirs (cdr list-dirs))))
    list-to-date))

(defun rcs-directory-dirs (&optional directory-name only-regexp)
  "Return list of non-RCS directories in optional DIRECTORY.
Include only directory names matching optional REGEXP."
  (if (not (and directory-name
		(file-directory-p directory-name)))
      (setq directory-name "./"))
  (let ((list-files (directory-files directory-name t (or
						       only-regexp
						       "^[^\\.]")))
	(list-dirs '()))
    (while list-files
      (if (and
	   (not (string-match "^RCS"
			      (file-name-nondirectory
			       (car list-files))))
	   (file-directory-p (car list-files)))
	  (setq list-dirs (append list-dirs (list (car list-files)))))
      (setq list-files (cdr list-files)))
    list-dirs))

(defun rcs-make-revision (fn object)
  "Using FILE, make a revision from OBJECT."
  (let ((type (car (rcs-what-is-object fn object))))
    (cond
     ((string= "r" type)
      object)
     ((string= "i" type)
      (concat object "." (rcs-get-last-tip fn object)))
     ((string= "s" type)
      (concat (rcs-get-item-from-symbolic fn object) "."
	      (rcs-get-last-tip fn object))))))

(defun rcs-make-symbol (fn object)
  "Using FILE, make a symbol from OBJECT."
  (let* ((type (car (rcs-what-is-object fn object)))
	 (symbol (cond
		  ((string= "r" type)
		   (car (rcs-rassoc
			 (substring object 0 -2)
			 (rcs-get-file-symbols-items fn rcs-list-all-symbols))))
		  ((string= "i" type)
		   (car (rcs-rassoc object (rcs-get-file-symbols-items fn rcs-list-all-symbols))))
		  ((string= "s" type)
		   object))))
    symbol))

(defun rcs-make-item (fn object)
  "Using FILE, make an item from OBJECT."
  (let ((type (car (rcs-what-is-object fn object))))
    (cond
     ((string= "r" type)
      (substring object 0 -2))
     ((string= "i" type)
      object)
     ((string= "s" type)
      (let ((all-symbols (rcs-get-info-from-rlog fn "symbolic names")))
	(if (not (string-match
		  (concat "^.*" object ":[ \t]*\\([^;\n]+\\)")
		  all-symbols))
	    nil
	  (substring all-symbols (match-beginning 1) (match-end 1))))))))

;; find where a command is.
(defun rcs-find-exec-command (cmd paths)
  "Return the full path to CMD from PATHS or just CMD if not found."
  (cond ((not paths) cmd)
	(t (if (file-exists-p (concat (car paths) "/" cmd))
	       (concat (car paths) "/" cmd)
	     (rcs-find-exec-command cmd (cdr paths))))))

;; get a log entry into the log buffer
(defun rcs-get-log (banner buf)
  "Prompting with BANNER, get a RCS log entry into the given BUFFER."
  (save-excursion
    (save-window-excursion
      (switch-to-buffer-other-window buf)
      (if (not rcs-keep-log)
	  (erase-buffer))
      (if rcs-edit-mode
	  (funcall rcs-edit-mode))
      (message
       (substitute-command-keys
	(concat banner " entry; \\[exit-recursive-edit] to end, \\[abort-recursive-edit] to abort.")))
      (recursive-edit)
      (message "Finished entry")))
  (bury-buffer buf))

(defun rcs-string-to-list (string-of-lists)
  "Given a STRING-OF-LISTS, (characters separated by whitespace),
returns a list with the string elements."
  (let ((i 0)
	(is 0)
	(list-all nil))
    (while (string-match
	    "[ \t\n]*\\([^ \t\n:]+\\)[ \t\n]*"
	    string-of-lists
	    is)
      (setq list-all (cons (substring string-of-lists
				      (match-beginning 1)
				      (match-end 1))
			   list-all)
	    is (match-end 1)
	    i (1+ i)))
    (reverse list-all)))

(defun rcs-count-regexps-in-string (regexp string)
  "Return how many times REGEXP appears in STRING."
  (let ((count 0)
	(last-match 0))
    (while (string-match (concat "\\(" regexp "\\)") string last-match)
      (setq count (+ 1 count)
	    last-match (match-end 0)))
    count))

(defun rcs-what-is-object (fn object)
  "For FILE, return string list of what OBJECT is: if symbol, 's'.
Else, first element will be 'i' or 'r' (item or revision).  Second
element will be 'm', 'b', or 'v' (main, branch, or variation).
If the object is not in the file, return nil."
  (if (not (stringp object))
      nil
    (save-excursion
      (set-buffer (rcs-show-log-file fn))
      (goto-char (point-min))
      (if (not (re-search-forward object nil t))
	  nil
	(if (string-match "^[0-9.]+$" object)
	    ;; all numbers
	    (let ((ndots (rcs-count-regexps-in-string "\\." object))
		  (list '()))
	      (cond
	       ((= 0 ndots)
		(cons "i" "m"))
	       ((= 1 ndots)
		(cons "r" "m"))
	       ((= 2 ndots)
		(cons "i" "b"))
	       ((= 3 ndots)
		(cons "r" "b"))
	       ((= 4 ndots)
		(cons "i" "v"))
	       ((= 5 ndots)
		(cons "r" "v"))))
	  ;; probably a symbol
	  (let ((all-symbols (rcs-get-info-from-rlog fn "symbolic names")))
	    (if (string-match
		 (concat "^.*" object ":[ \t]*\\([^;\n]+\\)")
		 all-symbols)
		(list "s")
	      nil)))))))

(defun rcs-call-process (command options fn &optional not-put-trailing-fn)
  "Do COMMANDS with OPTIONS on FILE.  Optional arg indicates to not put on
a trailing filename to the command and options.
cd to the correct directory, and then execute the command."
;;; check buffer for error??
  (let ((buftmp (get-buffer-create rcs-temp-buffer)))
    (save-excursion
      (set-buffer buftmp)
      (erase-buffer))
    (call-process rcs-shell-path nil buftmp nil "-c"
		  (concat "cd " (expand-file-name
				 (or (file-name-directory fn) ".")) " && exec "
				 (rcs-find-exec-command command rcs-exec-path)
				 " " options " "
				 (if (not not-put-trailing-fn)
				     (file-name-nondirectory fn)))
		  )
    (bury-buffer buftmp)
    t))

(defun rcs-item-is-locked-p (fn item)
  "Returns t if in FILE, ITEM is locked, else nil.  If item is nil,
check for any locks."
  (if (not (rcs-has-rcs-file-p fn))
      nil
    (if (not item)
	(setq item "[0-9.]+")
      (setq item (rcs-make-item fn item))) ; ensure that ITEM really is
    (string-match (concat "[a-zA-Z0-9]*: " item ".[0-9]+$")
		  (rcs-get-info-from-rlog fn "locks"))))

(defun rcs-make-tmp-fn (fn)
  "Returns a temporary filename for FILENAME, if such exists, or nil
if it doesn't."
  ;; use alist to track real names and temp names
  (if (not (and fn
		(setq fn (expand-file-name fn))
		(file-exists-p fn)))
      nil
    (or
     (cdr (assoc fn rcs-tmp-files))	; tmp file already created for fn
     (let ((tmp-name (concat "/tmp/#rcs-" (file-name-nondirectory fn) "-0"))
	   (i 0))
       (while (or (file-exists-p tmp-name) (rcs-rassoc tmp-name rcs-tmp-files))
	 (setq i (1+ i)
	       tmp-name (concat
			 (substring tmp-name
				    0
				    (- (string-match "[0-9]*$" tmp-name) 1))
			 "-" i)))
       (setq rcs-tmp-files (append rcs-tmp-files (list (cons fn tmp-name))))
       tmp-name))))

(defun rcs-rassoc (value alist)
  "Reverse assoc: given ELT in ALIST, return the element whose cdr is ELT.
Uses equal."
  (while (and (not (equal value (cdr (car alist))))
	      (car alist))
    (setq alist (cdr alist)))
  (car alist))

(defun rcs-assoc-buf-file (buffer fn)
  "Associate file in BUFFER to FILENAME.
Used to update buffer if filename changes; the buffer did not
come from the file.  If no filename, unassociate the buffer.
Adapted from set-visited-file-name.  Yes, it's a kluge."
  (save-excursion
    (set-buffer buffer)
    (if fn
	(setq fn
	      (if (string-equal fn "")
		  nil
		(expand-file-name fn))))
    ;; generate a dummy filename so that file-newer-than-file-p will work
    (let ((fndummy (rcs-make-tmp-fn fn)))
      (setq buffer-file-name fndummy)
      (setq buffer-backed-up t)
      (set-buffer-modified-p nil)
      (clear-visited-file-modtime)
      (if fndummy
	  (progn
	    (write-region 1 (if (>= (point-max) 2)
				2
			      1)
			  fndummy nil "")
	    (message "")))
      (kill-local-variable 'write-file-hooks)
      (kill-local-variable 'revert-buffer-function))))

(defun rcs-get-command (fn)
  "Get an rcs command to operate on a list of files by using the
variation menu buffer.  FILE is used as an example."
  (setq rcs-save-for-many t)
  (rcs-variation-menu-many fn)
  (setq rcs-save-for-many nil))

(defun rcs-get-file-list ()
  "Get a list of files for an RCS command."
  (message "Please enter type of file list (t*ag or d*ir): ")
  (let ((type (char-to-string (downcase (read-char)))))
    (cond
     ((string= "t" type)
      (tag-table-files))
     ((string= "d" type)
      (rcs-list-files
       (expand-file-name
	(read-file-name "Please enter directory: " nil nil t))))
     )))

(defun rcs-do-many ()
  "Do an rcs command on many files."
  (interactive)
  (rcs-interactive-init)
  (let ((rcs-file-list (rcs-get-file-list))
	(initial t)
	(old-rcs-keep-log rcs-keep-log)
	(old-rcs-use-prev-log rcs-use-prev-log))
    (if (not rcs-file-list)
	(message "No RCS files to work on.")
      (rcs-get-command (car rcs-file-list))
      (while rcs-file-list
	(let ((file-name (car rcs-file-list)))
	  (rcs-find-file file-name)
	  (cond ((eq rcs-command 'rcs-ci-file)
		 (if initial
		     (if rcs-use-1st-log
			 (progn
			   (setq rcs-keep-log t)
			   (setq rcs-use-prev-log nil)))
		   (if (setq rcs-use-prev-log rcs-use-1st-log)
		       (setq rcs-keep-log t)))))
	  (if rcs-items
	      (apply rcs-command file-name rcs-items)
	    (funcall rcs-command file-name))
	  (setq rcs-file-list (cdr rcs-file-list))
	  (setq initial nil)))
      (setq rcs-use-prev-log old-rcs-use-prev-log)
      (setq rcs-keep-log old-rcs-keep-log)
      (message "Done."))))

;;;; mainline functions.
;; Most functions have two forms; one which operates on a file, and one
;; which operates on the current buffer. The latter are obviously made out
;; the former

;;; checkout functions
;; edit an RCS file by checking it out locked.

(defun rcs-co-file (fn &optional get-main)
  "Checkout the RCS FILE.  With optional arg, use main stem regardless
of user branches; else if user has variations call variation menu."
  (interactive "FRCS Find File: \nP")
  (if (not (rcs-has-rcs-file-p fn))
      (rcs-find-file fn)
    (rcs-interactive-init)
    (if get-main
	(rcs-checkout-file fn rcs-main-symbolic)
      (let* ((variation-list (rcs-get-file-symbols-items fn rcs-list-all-symbols))
	     (number-of-variations (length variation-list)))
	(if (> number-of-variations 1) ; more than one item exists...
	    (Variation-menu fn variation-list)
	  (rcs-checkout-file fn (car (car variation-list))))))))

(defun rcs-checkout-file (fn &optional object)
  "Check out and visit a FILE, using optional OBJECT."
  (catch 'co-finish
    (setq fn (rcs-make-normal-name fn))
    (if (and (file-exists-p fn) (file-writable-p fn))
	(progn
	  (if (rcs-item-is-locked-p fn (or
					object
					(and rcs-use-user-branch
					     (rcs-make-user-branch fn))
					(rcs-get-info-from-rlog fn "head")))
	      (progn
		(rcs-find-file fn)
		(throw 'co-finish nil))
	    (if (yes-or-no-p (concat "File "
				     (file-name-nondirectory fn )
				     " is writeable; overwrite it ? "))
		(call-process "rm" nil nil nil "-f" (expand-file-name fn))
	      (message (concat "Not checking-out file "
			       (file-name-nondirectory fn)
			       ", but finding it..."))
	      (sit-for 3)
	      (rcs-find-file fn)
	      (message "")
	      (throw 'co-finish nil)))))
    
    ;; Check out the item in the file
    (cond
     (object
      ;; check for outdated version if not main stem
      ;; compare branch (or variation) with main stem (and branch) (and others)
      (if (and rcs-check-outdated
	       (not (string= (rcs-make-symbol fn object)
			     rcs-main-symbolic)))
	  (if (rcs-update-obj1-wrt-obj2 fn nil object)
	      (message "Checking out file %s using branch %s..."
		       fn (rcs-make-symbol fn object))
	    (if (not (yes-or-no-p "Continue with checkout? "))
		(progn
		  (message "Checkout aborted.")
		  (throw 'co-finish nil)))))
      (rcs-checkout fn (concat "-l" object)))
     
     ;; get the user's branch, or make them one if rcs-use-user-branch
     (rcs-use-user-branch
      (let ((branch (rcs-make-user-branch fn)))
	;; check for outdated version
	(if rcs-check-outdated
	    (if (rcs-update-obj1-wrt-obj2 fn)
		(message "Checking out file... %s with user branch %s"
			 fn rcs-user-name)
	      (if (not (yes-or-no-p "Continue with checkout? "))
		  (progn
		    (message "Checkout aborted.")
		    (throw 'co-finish nil)))))
	(rcs-checkout fn (concat "-l" branch))))
     ;; No branch or variation ?... Must want "main".
     (t
      (message "Checking out file %s" fn)
      (rcs-checkout fn "-l "))))
  t)

(defun rcs-checkout (fn options)
  "Perform RCS file check out on FILE with OPTIONS."
  (rcs-call-process "co" options fn)
  (rcs-refresh-log fn)
  (rcs-buf-kill-and-reload fn)
  (if (get-buffer rcs-varbuf)
      (Variation-menu-refresh))
  (message "Checkout done"))

(defun rcs-co-copy (fn object buf)
  "In FILE, checkout an OBJECT into BUF."
  (call-process rcs-shell-path nil buf nil "-c"
		(concat
		 "cd " (expand-file-name
			(or (file-name-directory fn) "."))
		 " && exec "
		 (rcs-find-exec-command "co" rcs-exec-path)
		 " -q -r" object " -p "
		 (file-name-nondirectory fn)))
  ;; remove ld.so messages, what a kluge
  (save-excursion
    (goto-char (point-min))
    (replace-regexp "ld\\.so: warning:.*\n" ""))
  t)


;;; checkin functions

;; Check back in a file
(defun rcs-ci-buffer (&optional locked)
  "Do RCS checkin on the current buffer; optional prefix arg means
remained locked.  If more than one locked revision in the file exists
then the user will be prompted by the *VARIATION BUFFER* for symbol to
check in, otherwise no *VARIATION BUFFER* will appear."
  
  (interactive "P")
  (rcs-interactive-init)
  (setq fn (buffer-file-name))
  (setq rcs-checkin-locked locked)
  (let ((number-of-locks 0)
	(variation-list (rcs-get-file-symbols-items fn rcs-list-all-symbols t))
	(locked-variations-list '()))
    (while variation-list
      (if (string-match "*" (car (car variation-list)))
	  (setq number-of-locks (1+ number-of-locks)
		locked-variations-list
		(append locked-variations-list
			(list (car variation-list)))))
      (setq variation-list (cdr variation-list)))
    
    ;; If more than one item is locked, determine which one needs
    ;; to be checked-in.
    (if (> number-of-locks 1)
	(progn
	  (message "More than one item is locked...")
	  (sit-for 3)
	  (Variation-menu fn locked-variations-list))
      ;; Only one variation or the branch.
      (setq checkin-this-item (cdr (car locked-variations-list)))
      (rcs-ci-file fn locked checkin-this-item))))

(defun rcs-ci-file (fn &optional locked object additional-log no-user-log)
  "Use RCS to check back in FILE; leave locked with optional PREFIX
argument.  The working file will still exist. If the file is
unchanged, it is simply left locked or unlocked.  If the file is in a
buffer and has been modified, it will be saved first.  Check in
specific OBJECT if not nil.  If ADDITIONAL-LOG is a string, add to the
end of user's log message, which they are prompted for if NO-USER-LOG
is not nil. Return nil if an error detected in the ci command.

The hook rcs-new-dir-hook is run after a new RCS directory is created;
the hook rcs-new-file-hook is run after a file is checked in the first
time. When this happens, 'fn' is the file being checked in, and 'dir'
is the just-created directory."
  (interactive "fFile to check in: \nP")
  (rcs-interactive-init)
  (setq fn (rcs-make-normal-name fn))
  (let ((initial nil))
    (catch 'ci-error
      (if (not (file-exists-p fn))
	  (progn
	    (message "File %s doesn't exist to check in." fn)
	    (sit-for 3)
	    (throw 'ci-error nil)))
      (if (and (rcs-has-rcs-file-p fn) (not (rcs-item-is-locked-p fn object)))
	  (progn
	    (message "File is not locked.")
	    (throw 'ci-error t)))
      (if (get-buffer rcs-error-buffer)	;   M. Gregory  23-Aug-91 09:36
	  (kill-buffer rcs-error-buffer) ) ; rcs-error-buffer could still be
					; lying around from a previous error.
					; Kill it so that it does not appear
					; if this ci works.
      (if (and (get-file-buffer fn)
	       (buffer-modified-p (get-file-buffer fn)))
	  (progn
	    (if (not rcs-make-backup-files)
		(progn
		  (make-local-variable 'make-backup-files)
		  (setq make-backup-files nil)))
	    (if (setq checkin-buffer (get-file-buffer fn))
		(progn
		  (set-buffer checkin-buffer)
		  (save-buffer)))
	    ))
      (let ((log (get-buffer-create rcs-log-buffer))
	    (buf (get-file-buffer fn))
	    (buftmp (get-buffer-create rcs-temp-buffer))
	    (dir (expand-file-name (concat (file-name-directory fn) "RCS")))
	    (inital nil)
	    (end-of-log nil)
	    (end-of-add-log nil))
	(bury-buffer buftmp)
	(if (rcs-has-rcs-file-p fn)
	    (progn
	      (if (and (not rcs-use-prev-log)
		       (not no-user-log))
		  (rcs-get-log "Log" log))
	      (setq initial nil))
	  ;; initial checkin
	  (if (not (file-directory-p dir))
	      (progn
		(message "Creating RCS directory %s..." dir)
		(call-process "mkdir" nil nil nil dir)
		(if (not (file-directory-p dir))
		    (progn
		      (message "Cannot create RCS directory")
		      (throw 'ci-error nil)))
		(run-hooks 'rcs-new-dir-hooks)))
	  (if (not rcs-use-prev-log)
	      (rcs-get-log "General descriptive text" log))
	  (setq initial t))
      
	(save-excursion
	  (set-buffer buftmp)
	  (erase-buffer)
	  (save-excursion
	    (set-buffer log)
	    (if (stringp additional-log)
		(progn
		  (goto-char (setq end-of-log (point-max)))
		  (insert (concat
			   (if (not (char-equal (preceding-char)
						(string-to-char "\n")))
			       "\n")
			   additional-log))
		  (setq end-of-add-log (point))))
	    (message "Checking in %s..." fn)
	    (call-process-region
	     1 (point-max) rcs-shell-path nil buftmp t "-c"
	     (concat "cd " (expand-file-name (or (file-name-directory fn) "."))
		     " && exec "
		     (rcs-find-exec-command "ci" rcs-exec-path)
		     " -w" rcs-user-name " "
		     (if locked "-l" "-u")
		     (if (and initial rcs-initial-branch rcs-initial-rev)
			 (concat rcs-initial-branch "." rcs-initial-rev " ")
		       ;; more than one item is locked (variation(s) and/or branch).
		       (if object
			   (concat object " ")
			 " "))
		     (file-name-nondirectory fn)))
	  
	    (if (not rcs-keep-log)
		(kill-buffer log)
	      (if (stringp additional-log)
		  (delete-region end-of-log end-of-add-log))))
	
	  ;; check if no change in versions
	  (goto-char (point-min))
	  (if (re-search-forward "unchanged" nil t)
	      (progn
		(message (concat "RCS file " fn " unchanged; left "
				 (if locked "locked." "unlocked.")))
		(sit-for 3)
		(if (not locked)
		    (progn
		      ;; unlock the file instead
		      (erase-buffer)
		      (call-process
		       rcs-shell-path nil buftmp t "-c"
		       (concat "cd "
			       (expand-file-name
				(or (file-name-directory fn) "."))
			       " && exec "
			       (rcs-find-exec-command "co" rcs-exec-path)
			       " -f -u"
			       ;; more than one item is locked
			       ;; (variation(s) and/or branch).
			       object " "
			       (file-name-nondirectory fn))))))))
      
	;; check for error in checkin
	(if (get-buffer rcs-temp-buffer)
	    (set-buffer rcs-temp-buffer))
	(goto-char (point-min))
	(if (re-search-forward "error:" nil t)
	    (progn
	      (message "Cannot check in RCS file %s due to error..." fn)
	      (sit-for 3)
	      (if (get-buffer rcs-temp-buffer)
		  (progn
		    (rename-buffer rcs-error-buffer)
		    (switch-to-buffer-other-window
		     (get-buffer rcs-error-buffer)))
		(throw 'ci-error nil))))
      
	(if initial
	    (progn
	      (run-hooks 'rcs-new-file-hook)
	      (if rcs-initial-access
		  (let ((buftmp (get-buffer-create rcs-temp-buffer)))
		    (save-excursion
		      (set-buffer buftmp)
		      (erase-buffer)
		      (call-process
		       rcs-shell-path nil buftmp t "-c"
		       (concat "cd " (expand-file-name
				      (or (file-name-directory fn) "."))
			       " && exec "
			       (rcs-find-exec-command "rcs" rcs-exec-path)
			       " -a" rcs-initial-access " "
			       (file-name-nondirectory fn)))
		      (goto-char (point-min))
		      (if (re-search-forward "error:" nil t)
			  (progn
			    (message "Cannot check in RCS file.")
			    (throw 'ci-error nil))
			(kill-buffer buftmp)))))
	      ;; Make user a "main" if none exists
	      (if (not (rcs-get-item-from-symbolic fn rcs-main-symbolic))
		  (rcs-call-process "rcs"
				    (concat "-u -n" rcs-main-symbolic ":"
					    rcs-initial-branch)
				    fn))
	      (if rcs-use-user-branch
		  ;; get the user's branch, or make them one
		  (if (setq branch (rcs-make-user-branch fn))
		      (message
		       (concat "Using your branch " branch
			       " instead of main trunk "
			       (rcs-get-info-from-rlog fn "head") "."))
		    (message "Unable to open a branch for %s." rcs-user-name)
		    (throw 'ci-error nil)))))
	(if (and (not locked)
		 (setq buffer-to-kill (get-file-buffer fn)))
	    (kill-buffer buffer-to-kill)
	  (rcs-buf-kill-and-reload fn))
	(message "Checkin done")
	t))
    (if (get-buffer rcs-varbuf)
	(Variation-menu-refresh))))

(defun rcs-ci-checked-out (&optional directory-name locked)
  "Perform rcs checkin of checked-out files in DIRECTORY-NAME.
Current directory is the default.  Leave locked with optional PREFIX."
  (interactive "D(Default Directory) \nP")
  (rcs-interactive-init)
  (if (not (and directory-name
		(file-directory-p directory-name)))
      (setq directory-name "./"))
  (setq directory-name (file-name-as-directory directory-name))
  (if (y-or-n-p "Do you want to use the same log message for all files ? ")
      (setq use-1st-log t)
    (setq use-1st-log nil))
  (message "")
  (let ((next-file-list (rcs-list-locked directory-name))
	(old-rcs-keep-log rcs-keep-log)
	(old-rcs-use-prev-log rcs-use-prev-log))
    
    (setq initial t)
    (setq rcs-symbol nil)
    (while (setq fn (car next-file-list))
      (setq locked-list '())
      (rcs-find-file fn)
      (setq variation-list (rcs-get-file-symbols-items fn rcs-list-all-symbols t))
      (while variation-list
	(if (string-match "*" (car (car variation-list)))
	    (setq locked-list
		  (append locked-list (list (car variation-list)))))
	(setq variation-list (cdr variation-list)))
      
      (if initial
	  (if use-1st-log
	      (progn
		(setq rcs-keep-log t)
		(setq rcs-use-prev-log nil)))
	(if (setq rcs-use-prev-log use-1st-log)
	    (setq rcs-keep-log t)))
      
      (cond
       ( (> (length locked-list) 1)
	 (message "More than one item locked in file %s..." fn)
	 (sit-for 3)
	 (setq item-list '())
	 (cond (initial
		(Variation-menu fn locked-list)
		(setq initial nil))
	       
	       (rcs-use-previous-symbols
		(if (setq item-list (or (assoc (concat "*"  rcs-symbol)
					       locked-list)
					(assoc rcs-symbol locked-list)))
		    (rcs-ci-file fn locked (cdr item-list))))
	       
	       ( (not rcs-use-previous-symbols)
		 (if (not (setq item-list (or (assoc (concat "*"  rcs-symbol)
						     locked-list)
					      (assoc rcs-symbol locked-list))))
		     (Variation-menu fn locked-list)
		   (rcs-ci-file fn locked (cdr item-list))))))
       
       ( (= (length locked-list) 1)
	 (if initial
	     (progn
	       (setq rcs-symbol (substring (car (car locked-list)) 1))
	       (setq initial nil)))
	 (rcs-ci-file fn locked (cdr (car locked-list))))
       ;; Hopefully this will never occur
       ( t
	 (message "Nothing checked-out in file %s." fn)
	 (if (setq buffer-to-kill (get-file-buffer fn))
	     (kill-buffer buffer-to-kill))
	 (sit-for 3)))
      (setq next-file-list (cdr next-file-list)))
    (setq rcs-use-prev-log old-rcs-use-prev-log)
    (setq rcs-keep-log old-rcs-keep-log)
    (if (get-buffer rcs-varbuf)
	(kill-buffer rcs-varbuf))
    (if (get-buffer (other-buffer rcs-log-buffer))
	(switch-to-buffer (other-buffer rcs-log-buffer)))
    (message "All files processed.")))

(defun rcs-diff-file (fn &optional diff-revs symbol)
  "Run an rcsdiff on FILE and display the differences in another
buffer.  If optional DIFF-REVS a string, use those for rcsdiff
revisions; else, if not checked out, diff two previous revs, else
working file and last rev, using optional SYMBOL, or user branch."
  (interactive "FFile to Diff: \n")
  (if (not (rcs-has-rcs-file-p fn))
      (message "No RCS file exists for %s" fn)
    (rcs-interactive-init)
    (let* ((buf (get-buffer-create (concat "# rcs diff : " fn " #")))
	   (symbol (or
		    symbol
		    (if rcs-use-user-branch
			rcs-user-name
		      rcs-main-symbolic)))
	   (top-revision nil)
	   (top-revision-int (rcs-get-last-tip fn symbol))
	   (all-diff-options (concat
			      " "
			      (if (stringp rcs-diff-options)
				  rcs-diff-options)
			      " "
			      (if (stringp rcs-gnu-diff-options)
				  rcs-gnu-diff-options)
			      " ")))
      (if top-revision-int
	  (setq top-revision (string-to-int top-revision-int))
	;; couldn't get a tip from "symbol"!
	(if rcs-use-user-branch
	    (setq symbol rcs-main-symbolic
		  top-revision (string-to-int (rcs-get-last-tip fn symbol)))
	  (message "Unable to decipher log file!")
	  (beep)
	  (sit-for 3)
	  (setq diff-revs nil
		all-diff-options nil)))
      (if (stringp diff-revs)
	  (setq all-diff-options (concat all-diff-options diff-revs " "))
	;; if checked-out, diff working file and symbol, else on two prev revs
	(if (rcs-item-is-locked-p fn symbol)
	    (setq all-diff-options (concat all-diff-options "-r" symbol " "))
	  (if (> top-revision (string-to-int rcs-initial-rev))
	      (setq all-diff-options (concat all-diff-options
					     "-r" symbol "." (1- top-revision)
					     " -r" symbol))
	    (setq all-diff-options nil)
	    (message "Revision %s.%s is the initial revision.  Can not diff."
		     symbol rcs-initial-rev)
	    (sit-for 3))))
      (if all-diff-options
	  (progn
	    (save-excursion
	      (set-buffer buf)
	      (erase-buffer))
	    (if (interactive-p)
		(message "Diffing %s..." fn))
	    (call-process rcs-shell-path nil buf nil "-c"
			  (concat
			   "cd "
			   (expand-file-name (or (file-name-directory fn) "."))
			   " && exec "
			   (rcs-find-exec-command "rcsdiff" rcs-exec-path)
			   all-diff-options " "
			   (file-name-nondirectory fn)))
	    (switch-to-buffer-other-window buf)
	    (goto-char (point-min))
	    (set-buffer-modified-p nil)
	    (if (interactive-p)
		(message "Done")))))))

(defun rcs-revert-file (fn &optional revert-locked object)
  "Unlock and revert FILE to its last RCS'd version.
Handy when you locked a file that you later decided not to change.
If the file was in a buffer, reload the buffer with the reverted
version; otherwise, the file is not loaded.
If a prefix argument is given, the file is left locked; otherwise, it
is left unlocked. Optional OBJECT reverts that specific item."
  (interactive "fFile to revert: \nP")
  (catch 'no-revert
    (rcs-interactive-init)
    (let ((inbuf (get-file-buffer fn))
	  (buf (get-buffer-create rcs-temp-buffer)))
      (bury-buffer buf)
      (if (not (rcs-has-rcs-file-p fn))
	  (message "No RCS file exists for %s" fn)
	(if (and (file-newer-than-file-p fn (rcs-make-rcs-name fn))
		 (not (y-or-n-p
		       "Revert-file: any changes made will be lost.  Proceed? ")))
	    (message "File not reverted.")
	  (if (not object)
	      ;; Might have a branch or a variation locked.
	      (let ((variation-list (rcs-get-file-symbols-items fn rcs-list-all-symbols t))
		    (locked-list '()))
		(while variation-list
		  (if (string-match "*" (car (car variation-list)))
		      (setq locked-list
			    (append locked-list (list (car variation-list)))))
		  (setq variation-list (cdr variation-list)))
		(if (> (length locked-list) 1)
		    (progn
		      (Variation-menu rcs-filename locked-list)
		      (throw 'no-revert nil))
		  (setq object (cdr (car locked-list))))))
	  
	  ;; Otherwise revert to main
	  (if (not object)
	      (setq object (rcs-get-info-from-rlog fn "head")))
	  
	  (message
	   "Reverting RCS file %s using branch %s ..." fn
	   (rcs-make-symbol fn object))
	  (call-process
	   rcs-shell-path nil buf nil "-c"
	   (concat
	    "cd " (expand-file-name (or (file-name-directory fn) "."))
	    " && rcs -u" object " "
	    (file-name-nondirectory fn)
	    " && rm -f " (file-name-nondirectory fn)
	    " && exec " (rcs-find-exec-command "co" rcs-exec-path)
	    (if revert-locked " -l" " -u")
	    object " "
	    (file-name-nondirectory fn)))
	  (if (not revert-locked)
	      (setq buffer-read-only t))
	  (if (get-file-buffer fn)
	      (rcs-buf-kill-and-reload fn))
	  (message "Done"))))))

;; revert the current buffer.
(defun rcs-revert-buffer (&optional revert-locked)
  "Unlock and revert the current buffer to its last RCS'd version.
Does not save any changes.  If a prefix argument is given, the file is
left locked; otherwise, it is left unlocked."
  (interactive "P")
  (rcs-interactive-init)
  (if (not (setq rcs-filename (buffer-file-name)))
      (message "No file associated with the current buffer.")
    (if (or (not (buffer-modified-p))
	    (y-or-n-p
	     "Revert-buffer: any changes made will be lost.  Proceed? "))
	(progn
	  (set-buffer-modified-p nil)
	  (rcs-revert-file rcs-filename revert-locked) )
      (message "Buffer not reverted."))))

(defun rcs-delete-object (fn object &optional entire)
  "Delete from FILE the OBJECT; with optional ENTIRE, delete the object's
entire range and symbol."
  (if (string= rcs-initial-rev
	 (rcs-get-last-tip fn object))
      (setq entire t))
  (if (or (not entire)
	  (and entire
	       (yes-or-no-p (format "This will delete the ENTIRE branch of %s: proceed? " object))))
      (progn
	(let ((buf (get-buffer-create rcs-temp-buffer)))
	  (save-excursion
	    (set-buffer rcs-temp-buffer)
	    (erase-buffer)
	    (message "Removing last item from symbol %s in RCS file %s..."
		     (rcs-make-symbol fn object) fn)
	    (call-process rcs-shell-path nil buf nil "-c"
			  (concat
			 "cd " (expand-file-name (or (file-name-directory fn) "."))
			 " && rcs -o"
			 (if entire ":") object
			 (if entire (concat "." (rcs-get-last-tip fn object)))
			 " " (file-name-nondirectory fn)))
	    (rcs-refresh-log fn)
	    (goto-char (point-min))
	    (if (re-search-forward "error:" nil t)
		(progn
		  (message "Could not outdate the item.")
		  (beep)
		  (sit-for 5))
	      (if entire (rcs-remove-symbol fn object)))
	    (bury-buffer buf))
	  (message "Done")))))

(defun rcs-remove-symbol (fn symbol)
  "Remove from FILE the RCS SYMBOL."
  (setq symbol (rcs-make-symbol fn symbol))
  (rcs-call-process "rcs" (concat "-n" symbol) fn)
  (rcs-refresh-log fn))

(defun rcs-emerge-objA-objB (fn symbol-A symbol-B)
  "Perform emerge of FILE items A and B."
  (catch 'rcs-emerge-items
    (setq rcs-revision (rcs-make-item fn symbol-B))
    (message "Getting item-A: %s..." symbol-A)
    (setq rcs-buf-A (get-buffer-create
		     (concat "A: "
			     (file-name-nondirectory fn)
			     " (" symbol-A ")")))
    (save-excursion
      (set-buffer rcs-buf-A)
      (erase-buffer))
    (if (not (rcs-co-copy fn symbol-A rcs-buf-A))
	(progn
	  (message "Unable to get item-A...emerge aborted.")
	  (sit-for 3)
	  (throw 'rcs-emerge-items nil)))

    (message "Getting item-B_%s..." symbol-B)
    (setq rcs-buf-B (get-buffer-create
		       (concat "B: "
			       (file-name-nondirectory fn)
			       " (" symbol-B ")")))
    (save-excursion
      (set-buffer rcs-buf-B)
      (erase-buffer))
    (if (not (rcs-co-copy fn symbol-B rcs-buf-B))
	(progn
	  (message "Unable to get item-B...emerge aborted.")
	  (sit-for 3)
	  (throw 'rcs-emerge-items-error nil)))
    (setq rcs-win-conf (current-window-configuration))
    (setq rcs-filename fn)
    (emerge-buffers rcs-buf-A rcs-buf-B nil 'rcs-emerge-quit1)
    (recursive-edit)))

(defun rcs-emerge-with-ancestor-objA-objB (fn symbol-A symbol-B symbol-ancestor)
  "Perform emerge of FILE items A and B with ANCESTOR."
  (catch 'rcs-emerge-items-ancestor
       (message "Getting item-A: %s..." symbol-A)
      (setq rcs-buf-A (get-buffer-create
		       (concat "A: " (file-name-nondirectory fn)
			       " (" symbol-A ")")))
      (save-excursion
	(set-buffer rcs-buf-A)
	(erase-buffer))
      (if (not (rcs-co-copy fn symbol-A rcs-buf-A))
	  (progn
	    (message "Unable to get item-A...emerge aborted.")
	    (sit-for 3)
	    (throw 'rcs-emerge-items-ancestor-error nil)))
      
      (message "Getting item-B: %s..." symbol-B)
      (setq rcs-buf-B (get-buffer-create
		       (concat "B: "
			       (file-name-nondirectory fn)
			       " (" symbol-B ")")))
      (save-excursion
	(set-buffer rcs-buf-B)
	(erase-buffer))
      (if (not (rcs-co-copy fn symbol-B rcs-buf-B))
	  (progn
	    (message "Unable to get item-B...emerge aborted.")
	    (sit-for 3)
	    (throw 'rcs-emerge-items-ancestor-error nil)))
      
      (message "Getting ancestor: %s..." symbol-ancestor)
      (setq rcs-buf-ancestor (get-buffer-create
			      (concat "Ancestor: "
				      (file-name-nondirectory fn)
				      " (" symbol-ancestor ")")))
      (save-excursion
	(set-buffer rcs-buf-ancestor)
	(erase-buffer))
      (if (not (rcs-co-copy fn symbol-ancestor rcs-buf-ancestor))
	  (progn
	    (message "Unable to get ancestor...emerge aborted.")
	    (sit-for 3)
	    (throw 'rcs-emerge-items-ancestor-error nil)))
      
      (setq rcs-win-conf (current-window-configuration))
      (setq rcs-filename fn)
      (emerge-buffers-with-ancestor
       rcs-buf-A rcs-buf-B rcs-buf-ancestor nil 'rcs-emerge-quit1)
      (recursive-edit)))

;;; show logfiles
;; refresh the logfile of filename
(defun rcs-refresh-log (fn &optional show-log)
  "Refresh the RCS log for FILE.  Optional prefix arg means show the log."
  (interactive "FFile to refresh log of: \nP")
  (rcs-show-log-file fn show-log t))

;; show the rcs logfile for a given file.
(defun rcs-show-log-file (fn &optional show-log refresh-log)
  "Show the RCS log for FILE.  Return the buffer name if not interactive.
If interactive, or optional show-log non-nil, switch to log buffer.
Optional refresh-log, if not nil, forces new log."
  (interactive "FFile to show log of: ")
  (catch 'rcs-show-log-end
    ;; expand fn and use working file name so it's consistent
    ;; across invocations
    (setq fn (rcs-make-normal-name (expand-file-name fn)))
    (if (not (rcs-has-rcs-file-p fn))
	(if (interactive-p)
	    (message "No RCS file exists for %s" fn)
	  (throw 'rcs-show-log-end nil)))
    
    (let ((buf (get-buffer-create (concat "# rcs log : " fn " #")))
	  (rcsfn (rcs-make-rcs-name fn)))
      (if (and (not refresh-log)
	       (buffer-file-name buf)
	       (file-newer-than-file-p (buffer-file-name buf) rcsfn))
	  (if (or show-log (interactive-p))
	      (progn
		(switch-to-buffer-other-window buf)
		(goto-char (point-min))
		(select-window (previous-window))
		(message "Done")
		(throw 'rcs-show-log-end t))
	    (throw 'rcs-show-log-end buf)))
      (save-excursion
	(set-buffer buf)
	(setq buffer-read-only nil)
	(rcs-assoc-buf-file buf nil)
	(erase-buffer))
      (if (interactive-p) (message "Getting log for %s..." fn))
      (call-process
       rcs-shell-path nil buf nil "-c"
       (concat
	"cd " (expand-file-name (or (file-name-directory fn) ".")) " && exec "
	(rcs-find-exec-command "rlog" rcs-exec-path) " "
	(file-name-nondirectory fn)))
      (rcs-assoc-buf-file buf rcsfn)
      (save-excursion
	(set-buffer buf)
	(setq buffer-read-only t))
      ;; if no symbol defined for main, do it (initial checkin not done with
      ;; this package)
      (if (not (rcs-get-item-from-symbolic fn rcs-main-symbolic))
	  (rcs-call-process "rcs"
			    (concat "-u -n"  rcs-main-symbolic
				    ":"
				    (rcs-make-item
				     fn (rcs-get-info-from-rlog fn "head")))
			    fn))
      (if (or show-log (interactive-p))
	  (progn
	    (switch-to-buffer-other-window buf)
	    (goto-char (point-min))
	    (select-window (previous-window))
	    (message "Done.")))
      (if (not (interactive-p))
	  (throw 'rcs-show-log-end buf)))))

(defun rcs-refresh-buffer ()
  "Reload the current buffer."
  (interactive)
  (if (buffer-file-name)
      (rcs-buf-kill-and-reload (buffer-file-name))
    (message "No file associated with current buffer")))

(defun rcs-list-of-lockers (fn)
  "Returns a list (of strings) of the people who have FILE locked.
Otherwise, nil if there are no lockers or the file is not under RCS."
  (if (not (rcs-has-rcs-file-p fn))
      nil
    (save-excursion
      (set-buffer (rcs-show-log-file fn))
      (goto-char (point-min))
      ;; find the "locks: ..." line
      (if (not (re-search-forward "^locks:[ \t]*\\(\\(strict\\)?[\n;]\\)?"
				  nil t))
	  nil
	(rcs-match-lockers-forward nil)))))

(defun rcs-match-lockers-forward (list-to-date)
  "Adjunct to rcs-list-of-lockers; recursively builds the list of lockers."
  ;; look for " <user>: <revision level>"
  (let ((pat "[ \t]*\\([^\n \t:;]*\\): [0-9][.0-9]*[\n;]"))
    (if (not (looking-at pat))
 	list-to-date	;; match failed, return what we have
      ;; skip to the end of the match, pull out the <user> portion, and recurse.
      (re-search-forward pat nil t)
      (rcs-match-lockers-forward (cons (buffer-substring (match-beginning 1)
							 (match-end 1))
				       list-to-date)))))

(defun rcs-item1-isolderthan-item2-p (fn item1 item2)
  "For RCS FILE, compare ITEM1 with ITEM2; return t if item1 is older
than item2."
  (setq item1 (rcs-make-item fn item1)
	item2 (rcs-make-item fn item2))
  (let ((date-pat "^date:[ \t]*\\([0-9:/ ]+\\);")
	(rev-pat1 (concat "^revision[ \t]*" item1))
	(rev-pat2 (concat "^revision[ \t]*" item2)))
    (save-excursion
      (set-buffer (rcs-show-log-file fn))
      (goto-char (point-min))
      ;; look for " revision: <revision level>" for first item, get date
      (re-search-forward rev-pat1 nil t)
      ;; limit search to this revision of item
      (save-excursion
	(re-search-forward "^-\\|=+$")
	(setq revision-limit (point)))
      (re-search-forward date-pat revision-limit t)
      (setq date1 (buffer-substring (match-beginning 1) (match-end 1)))
      ;; get 2nd item and compare dates
      (goto-char (point-min))
      (re-search-forward rev-pat2 nil t)
      (re-search-forward date-pat nil t)
      (setq date2 (buffer-substring (match-beginning 1) (match-end 1)))
      (if (string< date1 date2)
	  t
	nil))))

(defun rcs-item1-iscopyof-item2-p (fn item1 item2)
  "For RCS FILE, compare ITEM1 with ITEM2; return t if item1
is a copy of item2."
  (catch 'rcs-iscopy
    (if (equal item1 item2)
	(throw 'rcs-iscopy t))
    (setq item1 (rcs-make-item fn item1)
	  item2 (rcs-make-item fn item2))
    (let* ((rev2 (rcs-make-revision fn item2))
	   (rev-pat1 (concat "^revision " item1))
	   (rev-pat2 (concat "^revision " item2))
	   (revision-limit nil))
      ;; first check rcsdiff on both items
      (save-excursion
	(save-window-excursion
	(rcs-diff-file fn (concat "-r" item1 " -r" item2))
	(goto-char (point-min))
	(re-search-forward "^diff *" nil t)
	(end-of-line)
	(if (= (1+ (point)) (point-max))
	    (throw 'rcs-iscopy t))))
      (save-excursion
	(set-buffer (rcs-show-log-file fn))
	(goto-char (point-min))
	;; look for " revision: <revision level>" for first item
	(re-search-forward rev-pat1 nil t)
	;; limit search to this revision of item
	(save-excursion
	  (re-search-forward "^-\\|=+$")
	  (setq revision-limit (point)))
	;; search for 'Copied from <item2>' line; means item1 a copy
	(or (re-search-forward (concat "^Copied from " rev2 " ")
			       revision-limit t)
	    ;; 'Start of branch/variation', with rev starting from item2, ok also
	    (and (> (length item1) (length item2))
		 (string= (substring item1 0 -2) rev2)
		 (re-search-forward "Start of \\(branch\\|variation\\)"
				    revision-limit t)))
	))))

(defun rcs-item1-wascopyof-item2-p (fn item1 item2)
  "For RCS FILE, compare ITEM1 with ITEM2 and return t if item1 was
at any time copied from item2."
  (setq item1 (rcs-make-item fn item1)
	item2 (rcs-make-item fn item2))
  (let* ((rev2 (rcs-make-revision fn item2))
	 (rev-pat1 (concat "^revision " item1))
	 (revision-limit nil))
    (or
     (and (> (length item1) (length item2)) ; item1 possible branch of item2
	  (equal (rcs-get-last-tip fn item2)
		 (substring (and
			     (string-match "\\([0-9]+\\)\\.[0-9]+$" item1)
			     item1)
			    (match-beginning 1)
			    (match-end 1))))
     ;; item1 not a direct branch of item2
     (save-excursion
       (set-buffer (rcs-show-log-file fn))
       (goto-char (point-min))
       ;; look for " revision: <revision level>" for item1
       (re-search-forward rev-pat1 nil t)
       ;; limit search to item1 and all its revisions
       (save-excursion
	 (re-search-forward "^\\(Copied from\\|Start of \\(branch\\|variation\\)\\|Initial revision\\)")
	 (forward-line)
	 (setq revision-limit (point)))
       ;; if item2 possible branch of item1, search for "Copied from <item2>"
       ;; line; else search for "Copied from <rev2>" line (means item1 a
       ;; copy of latest revision of item2).
       (if (> (length item2) (length item1))
	   (re-search-forward (concat "^Copied from " item2)
			      revision-limit t)
	 (re-search-forward (concat "^Copied from " rev2 " ")
			    revision-limit t))))))

(defun rcs-item1-isderivedfrom-item2-p (fn item1 item2)
  "For FILE, compare ITEM1 with ITEM2; return t if item1
is derived from item2, i.e, item1 and item2 were copies, and
item1 was modified."
  (if (string= (rcs-get-last-tip fn item1) rcs-initial-rev) ; item1 not modified
      nil
    (setq item1 (rcs-make-item fn item1) ; just in case it's not really an item
	  item2 (rcs-make-item fn item2))
    (and (or (rcs-item1-wascopyof-item2-p fn item1 item2)
	     (rcs-item1-wascopyof-item2-p fn item2 item1))
	 (not (rcs-item1-isolderthan-item2-p fn item1 item2)))))

(defun rcs-list-member (element list)
  "How many times ELEMENT appears in LIST."
  (setq is-a-member 0)
  (if (car-safe list)
      (if (string= element (car list))
	  (progn
	    (rcs-list-member element (setq list (cdr list)))
	    (setq is-a-member (+ is-a-member 1)))
	(rcs-list-member element (setq list (cdr list))))))



(defun rcs-find-file (fn)
  "rcs alternative to using find-files-hook.  Bind this to C-x C-f;
Allows tag files and others to avoid running rlog."
  (interactive "FFind file: ")
  (switch-to-buffer (find-file-noselect fn))
  (rcs-hack-modeline))

;;; show who has a file locked in the modeline
;; Give us some vaguely useful information in the modeline.
(defun rcs-hack-modeline ()
  "Modify the current modeline to tell whether the file is under RCS.
Gives who the lockers are. Can be called interactively."
  (interactive)
  (if (and buffer-file-name
	   (rcs-has-rcs-file-p buffer-file-name))
      (progn
	;; since this is a buffer-local thing, we don't want everyone to
	;; share this nice label...
	(make-local-variable 'global-mode-string)
	(setq global-mode-string '( "" display-time-string))
	(let ((locker-list (rcs-list-of-lockers buffer-file-name))
	      (l-string ""))
	  (if (not locker-list)
	      (setq global-mode-string (append global-mode-string
					       '(" [unlocked]")))
	    ;; Code for multiple locks
	    (while  (car-safe locker-list)
	      (if (or rcs-list-all-lockers
		      (string= (car locker-list) rcs-user-name))
		  (progn
		    (setq l-string (concat l-string  " " (car locker-list)))
		    (if ( > (setq locks (rcs-list-member (car locker-list) locker-list)) 1)
			(progn
			  (setq locker-list
				(let ( (new-locker-list '())
				       (index -1))
				  (while (< (setq index (+ index 1))
					    (length locker-list))
				    (if (not (string= (car locker-list)
						      (nth index locker-list)))
					(setq new-locker-list
					      (nconc new-locker-list
						     (list (nth index locker-list))))))
				  new-locker-list))
			  (setq l-string (concat l-string "(" locks ")"))))))
		      (setq locker-list (cdr-safe locker-list)))
	    (setq global-mode-string
		  (append global-mode-string
			  (list (concat
				 (if (string= l-string "")
				     " [locked"
				   " [locked by")
				 l-string "]")))))))))

(defun rcs-list-locked (&optional directory-name)
  "Generate a list of RCS files locked by user.
The default is the current directory, else in optional DIRECTORY-NAME."
  (if (not (and directory-name
		(file-directory-p directory-name)))
      (setq directory-name "./"))
  (setq directory-name (expand-file-name
			(file-name-as-directory directory-name)))
  (setq rcs-lock-buffer "*rcs locked files*")
  (if (not (get-buffer rcs-lock-buffer))
      (get-buffer-create rcs-lock-buffer))
  (set-buffer rcs-lock-buffer)
  (erase-buffer)
  (setq locked-files-list '(nil))
  (call-process rcs-shell-path nil rcs-lock-buffer nil "-c"
		(concat "cd " directory-name "; "
			(rcs-find-exec-command "rlog" rcs-exec-path)
			" -R -L -l" rcs-user-name " " directory-name
			(if (file-exists-p (concat directory-name "RCS"))
			    "RCS/*"
			  "*,v ")))
  (goto-char (point-min))
  (while (re-search-forward "^.+$" (point-max) t)
    (if (and (setq locked-file (buffer-substring (match-beginning 0) (match-end 0)))
	     (not (string-match "^rlog error:" locked-file)))
	(setq locked-files-list
	      (append (list (rcs-make-normal-name locked-file))
		      locked-files-list))))
  (cdr (reverse locked-files-list)))

;; user branch functions

(defun rcs-get-info-from-rlog (fn name)
  "Return the info in FILE associated with the NAME.
Return nil if file not an RCS file, or if no such name."
  (if (not (rcs-has-rcs-file-p fn))
      nil
    (save-excursion
      (set-buffer (rcs-show-log-file fn))
      (goto-char (point-min))
      ;; find the "<name>: ..." line
      (if (re-search-forward (concat "^" name  ":[ \t\n]")
			     (point-max) t)
	  (progn
	    (setq start-pos (match-end 0))
	    (if (re-search-forward (concat "^[a-z]")
				   (point-max) t)
		(buffer-substring start-pos (- (match-beginning 0) 1))))))))


(defun rcs-get-item-from-symbolic (fn &optional symbol)
  "Return an item number in FILE associated with the optional
SYMBOL; use rcs-user-name if none given.
Return nil if file not an RCS file, or if no such symbolic name."
  (if (not symbol)
      (setq symbol rcs-user-name))
  (rcs-make-item fn symbol))

(defun rcs-get-symbol-from-item (fn item)
  "Return a symbol in FILE associated with the ITEM.
Return nil if file not an RCS file, or if no such item."
  (car (rcs-rassoc item (rcs-get-file-symbols-items fn t))))

(defun rcs-get-last-tip (fn item)
  "Returns as a string the latest revision number for file FN of ITEM.
E.g. (rcs-get-last-tip test 1.0) -> 8 assuming 1.0.8 is the highest
revision.  If no branches or revisions then 0 is returned."
  (setq item (rcs-make-item fn item))	; ensure that ITEM really is
  (if (not item)
      nil
    (save-excursion
      (set-buffer (rcs-show-log-file fn))
      (goto-char (point-min))
      ;; find the "revision ..." line
      (if (re-search-forward
	   (concat "^revision " item "." "\\([0-9]+\\)")
	   (point-max) t)
	  (buffer-substring (match-beginning 1) (match-end 1))
	"0"))))

(defun rcs-get-tip (string)
  "Get the tip from STRING (i.e. return the last string of digits
in STRING)."
  (and (string-match "[0-9]+$" string)
       (substring string (match-beginning 0))))

(defun rcs-get-rev-date (fn revision)
  "Returns the change date in FILE of REVISION, or nil if
no such revision."
  (save-excursion
    (set-buffer (rcs-show-log-file fn))
    (goto-char (point-min))
    ;; find the "revision ..." line
    (if (re-search-forward
	 (concat "^revision " revision ".*\ndate: +\\([^;]+\\)")
	 (point-max) t)
	(buffer-substring (match-beginning 1) (match-end 1))
      nil)))


(defun rcs-make-user-branch (fn &optional)
  "Make a user branch in the given FILE.
A user branch is a branch created from the top level of the main
trunk or default branch, with the user's login name as the symbolic
name of the branch.
Return nil if it can't create the branch, otherwise return the branch
revision number."
  (interactive "FPlease enter the filename: ")
  (catch 'make-branch-end
    (if (or (not rcs-use-user-branch)
	    (not (rcs-has-rcs-file-p fn)))
	(throw 'make-branch-end nil))
    
    (let* ((user-branch (rcs-get-item-from-symbolic fn))
	   (head-rev (rcs-get-info-from-rlog fn "head")))
      ;; check if user already has a branch; if so, return branch number
      (if user-branch
	  (throw 'make-branch-end user-branch))
      ;; No existing branch for user.
      ;; Create the branch from the head rev level. The user's branch number
      ;; will be created from the number of branches, plus 1.
      (setq user-branch (concat head-rev "."
				(1+ (rcs-get-number-branches fn head-rev))
				"." rcs-initial-rev))
      ;; now create the branch by checking out the file locked, and
      ;; checking it back in with the branch number; then fix the
      ;; branch name to the user's name.
      (and (file-exists-p fn)
	   (file-writable-p fn)
	   (not (yes-or-no-p
		 (concat "File " fn " is writable; overwrite it ? ")))
	   (throw 'make-branch-end nil))
      (rcs-call-process "co" (concat "-f -l" rcs-main-symbolic) fn)
      (rcs-call-process "ci" (concat "-f -u" user-branch
				     " -m'Start of branch for "
				     rcs-user-name "' ") fn)
      (rcs-call-process "rcs" (concat "-u" head-rev) fn)
      (rcs-call-process "rcs" (concat "-n" rcs-user-name ":"
				      (substring user-branch 0 -2)) fn)
      user-branch)))

(defun rcs-update-obj1-wrt-obj2 (fn &optional use-emerge object1 object2
				    make-branch no-query)
  "Update the object in the given FILE.  Optional ARG means call
emerge instead of normal copy.  If optional OBJ1 nil, use user branch.
If optional OBJ2 nil, use main stem (if obj1 is user branch) or user
branch (if obj1 is variation).  Optional 4th arg MAKE-BRANCH if
non-nil means make the user branch if none exists and that's what obj1
is.  Optional 5th arg NO-QUERY if non-nil means don't query user to perform
update.  Return nil if still outdated."
  (interactive "FPlease enter the filename: \nP")
  (rcs-interactive-init)
  (catch 'update-item-end
    (if (or (not (rcs-has-rcs-file-p fn))
	    (not rcs-use-user-branch))
	(throw 'update-item-end nil))
    
    ;; check if user already has a branch
    (if (not object1)
	(if rcs-use-user-branch
	    (progn
	      (setq object1 rcs-user-name)
	      (if (not (rcs-get-item-from-symbolic fn))
		  (if make-branch
		      (progn
			(message "No user branch exists, creating one...")
			(rcs-make-user-branch fn)
			(throw 'update-item-end t))
		    (message "Branch %s does not exist." rcs-user-name)
		    (sit-for 3)
		    (throw 'update-item-end nil))))
	  (message "Cannot use branches.")
	  (sit-for 3)
	  (throw 'update-item-end nil))
      (setq object1 (rcs-make-symbol fn object1)))
    
    (if (not object2)
	(setq object2 (if (string= object1 rcs-user-name)
			  rcs-main-symbolic
			rcs-user-name))
      (if (rcs-make-symbol fn object2)
	  (setq object2 (rcs-make-symbol fn object2))))
	
    ;; check that both objects exist in the file
    (if (not (rcs-what-is-object fn object1))
	(progn
	  (message "Symbol %s does not exist in file %s" object1 fn)
	  (sit-for 3)
	  (throw 'update-item-end t)))
    (if (not (rcs-what-is-object fn object2))
	(progn
	  (message "Symbol %s does not exist in file %s" object2 fn)
	  (sit-for 3)
	  (throw 'update-item-end t)))
      
    (if (or
	 (rcs-item1-iscopyof-item2-p fn object1 object2)
	 (rcs-item1-iscopyof-item2-p fn object2 object1)
	 (rcs-item1-isolderthan-item2-p fn object2 object1)
	 (and
	  (rcs-item1-isderivedfrom-item2-p fn object1 object2)
	  (rcs-item1-isolderthan-item2-p fn object2 object1)))
	(throw 'update-item-end t)
      
      ;; branch is outdated
      (if (rcs-item-is-locked-p fn object1)
	  (progn
	    (message "Item %s is checked-out! Cannot update!" object1)
	    (beep)
	    (sit-for 7)
	    nil)
	(if (string= (rcs-get-last-tip fn object1) rcs-initial-rev)
	    (message "Item %s is out-of-date w.r.t. %s, but not modified." object1 object2)
	  (message "Item %s is out-of-date w.r.t. %s, and there are changes!"
		   object1 object2))
	(beep)
	(sit-for 5)
	(if (or
	     no-query
	     (yes-or-no-p (format "Update %s from %s " object1 object2)))
	    (if use-emerge
		(rcs-emerge-objA-objB fn object1 object2)
	      (rcs-copy-objects fn object2 object1 (not rcs-log-before-copy)))
	  (message "Item %s remains out-of-date." object1)
	  (beep)
	  (sit-for 3)
	  nil)))))

(defun rcs-copy-objects (fn &optional object-from object-to no-user-log)
  "In FILE, copy from OBJECT-FROM into OBJECT-TO.
If object-from not given, assume main stem.  If object-to not given,
use user branch item. Optional NO-USER-LOG means don't let the user enter
a log message."
  (interactive "FRCS file to update: ")
  (rcs-interactive-init)
  (catch 'rcs-copy-end
    (if (not object-from)
	(setq object-from rcs-main-symbolic))
    (if (not object-to)
	(setq object-to rcs-user-name))
    (let ((item-from (rcs-make-item fn object-from))
	  (item-to (rcs-make-item fn object-to))
	  (symbol-from (rcs-make-symbol fn object-from))
	  (symbol-to (rcs-make-symbol fn object-to)))
      
      ;; check that both objects exist in the file
      (if (not symbol-from)
	  (progn
	    (message "%s does not exist in file %s" object-from fn)
	    (sit-for 3)
	    (throw 'rcs-copy-end t)))
      (if (not symbol-to)
	  (progn
	    (message "%s does not exist in file %s" object-to fn)
	    (sit-for 3)
	    (throw 'rcs-copy-end t)))
      
      (message "Copying %s into %s..." symbol-from symbol-to)
      (if rcs-clean-before-copy
	  (progn
	    (setq rcs-buf-edit (get-buffer-create
				(concat (file-name-nondirectory fn)
					" (" symbol-from ")")))
	    (save-excursion
	      (set-buffer rcs-buf-edit)
	      (erase-buffer)
	      (rcs-co-copy fn symbol-from rcs-buf-edit)
	      ;; if there are log messages from a copy *from* the object-to,
	      ;; get rid of them
	      (goto-char (point-min))
	      (if (re-search-forward "\\$Log:" nil t)
		  (progn
		    (forward-line 1)
		    (let ((start (point)))
		      (if (re-search-forward
			   (concat "^\\(Copied from "
				   (rcs-make-revision fn item-to)
				   " \\)\\|"
				   "\\(Initial revision\\)\\|"
				   "\\(Start of branch\\)")
			   nil t)
			  (progn
			    (forward-line 2)
			    (delete-region start (point)))))))
	      (rcs-call-process "chmod" "u+w" fn)
	      (write-file fn)
	      (rcs-call-process "chmod" "u-w" fn))
	    (kill-buffer rcs-buf-edit))
	(rcs-call-process "co" (concat "-f -r"  symbol-from) fn))
      (if rcs-edit-before-copy
	  (progn
	    (rcs-call-process "chmod" "u+w" fn)
	    (rcs-find-file fn)
	    (message
	     (substitute-command-keys
	      (concat "Last edit before copy; \\[exit-recursive-edit] to end, \\[abort-recursive-edit] to abort.")))
	    (recursive-edit)))
      (rcs-call-process "rcs" (concat "-l" symbol-to) fn)
      (rcs-ci-file fn nil symbol-to
		   (concat "Copied from " (rcs-make-revision fn item-from)
			   " by " rcs-user-name)
		   no-user-log)
      (kill-buffer (get-file-buffer fn))
      (message "Copying complete.")
      t)))

(defun rcs-get-file-symbols-items (fn list-all-symbols &optional lock-info)
  "Function that gets symbolic information from the 'rlog' of FILE.
The information is returned as an alist with the first element as the
symbol of the item and the second element as the rcs item number
associated with that symbol.  Both elements are strings.
LIST-ALL-SYMBOLS, if t, means list all the symbols in the rlog,
otherwise just the user's. Optional LOCK-INFO means put an * in front
of the symbol if it's locked."
  (let ((items (rcs-get-info-from-rlog fn "symbolic names")))
    (if (null items)
	nil
      (if (stringp items)
	  (let ((items-list (rcs-string-to-list (concat " " items)))
		(return-list '())
		(rcs-name "")
		(rcs-number "")
		(locked-items (rcs-get-info-from-rlog fn "locks")))
	    (while (car items-list)
	      (setq rcs-name (car items-list)
		    items-list (cdr items-list)
		    rcs-number (car items-list)
		    items-list (cdr items-list)
		    name nil)
	      ;; Make sure that:
	      ;; if list-all-symbols, all items always appear
	      ;; if not list-all-symbols:
	      ;;   if rcs-use-user-branch, only user items appear
	      ;;   if not rcs-use-user-branch, only main stem appears
	      (if list-all-symbols
		  (setq name rcs-name)
		;; not list all items
		(if rcs-use-user-branch
		    (if (string-match rcs-user-name rcs-name)
			(if (not (string= rcs-user-name rcs-name))
			    (setq name (substring rcs-name (1+ (string-match "_" rcs-name))))
			  (setq name rcs-name)))
		  ;; not user branches
		  (if (string= rcs-main-symbolic rcs-name)
		      (setq name rcs-main-symbolic))))
	      
	      (if name
		  (progn
		    (setq rcs-name name)
		    ;; Indicate locked items by placing a "*"  in front of them.
		    (if (and lock-info
			     (string-match (concat  ".+: "  rcs-number  "\.[0-9]+$")
					   locked-items))
			(setq rcs-name (concat "*" rcs-name)))
		    (setq return-list (append return-list
					      (list (cons rcs-name rcs-number)))))))
	    return-list)))))

(defun rcs-get-number-branches (fn revision)
  "Returns from FILE the number of branches (or variations) for REVISION."
  (setq revision (rcs-make-revision fn revision))
  (let ((symbol-list (rcs-get-file-symbols-items fn t))
	(count 0))
    (while symbol-list
      (if (string-match (concat revision "\\.[0-9]+$") (cdr (car symbol-list)))
	  (setq count (+ 1 count)))
      (setq symbol-list (cdr symbol-list)))
    count))


(defun rcs-check-for-outdated (fn &optional symbol-list1 symbol-list2)
  "Checks for outdated items in FILE; if optional SYMBOL-LIST1 nil,
use rcs user name; if optional SYMBOL-LIST2 nil, use main stem.  Both
items may contain multiple symbolic names, separated by spaces.
Return nil if all symbol-list1's are newer than all symbol-list2's;
else return list of outdated combos."
  (interactive "FEnter RCS file name: ")
  (if (not (rcs-has-rcs-file-p fn))
      (progn
	(message "File %s does not have an RCS file." fn)
	nil)
    
    (rcs-interactive-init)
    
    (if (not symbol-list2)
	(setq symbol-list2 (rcs-get-item-from-symbolic fn rcs-main-symbolic)))
    (if (not symbol-list1)
	(setq symbol-list1 (rcs-get-item-from-symbolic fn rcs-user-name)))
    
    (let ((return-list nil)
	  (symbol-list1 (string-to-list symbol-list1))
	  (symbol-list2 (string-to-list symbol-list2)))
      
      (while symbol-list1
	(let ((symbol1 (car symbol-list1))
	      (symbol-list2-temp symbol-list2))
	  (setq symbol-list1 (cdr symbol-list1))
	  
	  ;; if the symbols are actually item numbers, don't itemize them
	  (if (not (string-match "^[0-9.]+$" symbol1))
	      (setq symbol1 (rcs-get-item-from-symbolic fn symbol1)))
	  
	  (while symbol-list2-temp
	    (let ((symbol2 (car symbol-list2-temp)))
	      (setq symbol-list2-temp (cdr symbol-list2-temp))
	      (if (not (string-match "^[0-9.]+$" symbol2))
		  (setq symbol2 (rcs-get-item-from-symbolic fn symbol2)))
	      
	      (if (and
		   (not (rcs-item1-iscopyof-item2-p fn symbol1 symbol2))
		   (not (rcs-item1-iscopyof-item2-p fn symbol2 symbol1))
		   (not (and
			 (rcs-item1-isderivedfrom-item2-p fn symbol1 symbol2)
			 (rcs-item1-isolderthan-item2-p fn symbol2 symbol1))))
		  (progn
		    (setq return-list (nconc return-list
					     (list (list symbol1 symbol2))))
		    (if (interactive-p)
			(progn
			  (message "Item %s out-of-date wrt item %s" symbol1 symbol2)
			  (ding)
			  (sit-for 7)))
		    ))))))
      return-list)))


;;; functions to make variations

(defun rcs-valid-name (symbol-name)
  "Check that NAME is valid for an RCS symbol."
  (if (or (not symbol-name)
	  (string= symbol-name ""))
      nil
    (if (setq invalid-pos (string-match "[$,.:;@]" symbol-name))
	(progn
	  (message (concat "Illegal character: "
			   (substring symbol-name invalid-pos
				      (+ invalid-pos 1))
			   " in name: " symbol-name "."))
	  (sit-for 3)
	  nil)
      t)))
	    
(defun rcs-get-variation-name (fn &optional variation-name)
  "Check in FILE for valid and new VARIATION-NAME.
If not valid get a valid one."
  (while (or (not variation-name)
	     (not (rcs-valid-name variation-name))
	     (assoc (concat rcs-user-name "_" variation-name)
		    (rcs-get-file-symbols-items fn t)))
    (setq variation-name (read-string "Enter new variation name > "))
    (if (assoc (concat rcs-user-name "_" variation-name)
	       (rcs-get-file-symbols-items fn t))
	(progn
	  (message "Variation name %s already used" variation-name)
	  (beep)
	  (sit-for 3))))
  variation-name)

(defun rcs-make-user-variation (fn variation-name)
  "Make a user variation in the given FILE with given NAME.
A user variation is a variation created from the user's branch if it
exists, else from the main trunk.  The symbolic variation name is
created by concatenating the user's name, '_', and the entered
variation name. Nil is returned, if it can't create the variation;
otherwise return the full main.tip.user.tip.variation number."
  (interactive "FFile to create variation in: \nsName of variation: \n")
  (setq fn (rcs-make-normal-name fn))
  (catch 'no-variation-created
    (rcs-interactive-init)
    (let ((variation-name (rcs-get-variation-name fn variation-name))
	  (parent-item "")
	  (var-item ""))
      (if (not (rcs-has-rcs-file-p fn))
	  (progn
	    (message "No rcs file for %s...make variation aborted." fn)
	    (sit-for 3)
	    (throw 'no-variation-created nil)))
      (if (file-writable-p fn)
	  (progn
	    (message "File %s is writable...make variation aborted." fn)
	    (throw 'no-variation-created nil)))
      ;; check that user has branch
      (if (setq parent-item (rcs-get-item-from-symbolic fn))
	  (progn
	    (message "Using branch %s to create variation." parent-item)
	    (sit-for 3))
	;; else must be attached to main
	(progn
	  (setq parent-item (rcs-get-item-from-symbolic fn rcs-main-symbolic))
	  (message "Using main stem %s to create variation." parent-item)
	  (sit-for 3)))
      ;; Create the variation by using the latest revision of the parent item.
      (setq parent-item (concat parent-item "." (rcs-get-last-tip fn parent-item)))
      (setq var-item (concat parent-item "."
			     (1+ (rcs-get-number-branches fn parent-item))
			     "." rcs-initial-rev))
      ;; now create the variation by checking out the file locked, and
      ;; checking it back in with the variation number; then fix the
      ;; symbolic name
      (message "Making variation %s with RCS number %s..."
	       variation-name var-item )
	    (rcs-call-process "co"  (concat "-l" parent-item) fn)
	    (rcs-call-process "ci"
			      (concat "-u -f"  var-item
				      " -m'Start of variation "
				      variation-name " for "
				      rcs-user-name
				      "' ")
			      fn)
	    (rcs-call-process "rcs" (concat "-u" parent-item) fn)
	    (rcs-call-process
	     "rcs" (concat "-n" rcs-user-name "_" variation-name ":"
			   (substring var-item 0 -2)) fn)
	    (message "Make variation completed.")
	    (sit-for 3)
	    var-item)))

(defun rcs-get-freeze-name (fn &optional freeze-name)
  "Check in FILE for valid and new FREEZE-NAME.
If not valid get a valid one."
  (while (or (not freeze-name)
	     (not (rcs-valid-name freeze-name))
	     (assoc freeze-name (rcs-get-file-symbols-items fn t)))
    (setq freeze-name (read-string "Enter new freeze name > "))
    (if (assoc freeze-name (rcs-get-file-symbols-items fn t))
	(progn
	  (message "Freeze name %s already used" freeze-name)
	  (beep)
	  (sit-for 3))))
  freeze-name)

(defun rcs-make-freeze (fn symbol freeze-name)
  "Freeze in the given FILE with the tip of SYMBOL with given FREEZE-NAME.
That is, create a new symbol (freeze-name) in the file which holds the
full revision number of the latest revision of symbol."
  (setq fn (rcs-make-normal-name fn))
  (catch 'no-freeze-created
    (let ((revision (rcs-make-revision fn symbol)))
      ;; check that the symbol exists
      (if (not revision)
	  (progn
	    (message "Symbol %s does not exist in file %s, cannot freeze."
		     symbol fn)
	    (sit-for 2)
	    (throw 'no-freeze-created nil)))
      ;; check that freeze name doesn't exist
      (if (assoc freeze-name (rcs-get-file-symbols-items fn t))
	  (progn
	    (message "Freeze name %s already used in file %s, freeze aborted."
		     freeze-name fn)
	    (sit-for 2)
	    (throw 'no-freeze-created nil)))
      ;; now create the freeze by setting the freeze name to the revision
      (message "Making freeze %s at RCS number %s..."
	       freeze-name revision)
      (sit-for 2)
      (rcs-call-process
       "rcs" (concat "-n" freeze-name ":" revision) fn)
      (message "Make freeze completed.")
      revision
      )))


;;;; startup actions

(rcs-interactive-init)

;; Expand exec-path to include rcs-executable-path if necessary
(if rcs-executable-path
    (setq rcs-exec-path (cons rcs-executable-path exec-path))
  (setq rcs-exec-path exec-path))
;; set rcs-shell-path if necessary
(if (not rcs-shell-path)
    (setq rcs-shell-path shell-file-name))

;; Run any startup hooks necessary.
(run-hooks 'rcs-hook)


;; Variation menu main function and support functions.

;; The main (first level) Variation map and the secondary map (for
;; completing 2-part commands.

(defvar Variation-main-menu-mode-map nil "")
(defvar Variation-2nd-menu-mode-map nil "")

(if (not Variation-main-menu-mode-map)
    (progn
      (setq Variation-main-menu-mode-map (make-keymap))
      (suppress-keymap Variation-main-menu-mode-map t)
      (define-key Variation-main-menu-mode-map "a" 'rcs-emerge-select-objects1)
      (define-key Variation-main-menu-mode-map "w" nil)
      (define-key Variation-main-menu-mode-map "wa" 'rcs-emerge-select-objects-with-ancestor1)
      (define-key Variation-main-menu-mode-map "i" 'Variation-menu-check-in)
      (define-key Variation-main-menu-mode-map "o" 'Variation-menu-check-out)
      (define-key Variation-main-menu-mode-map "r" 'Variation-menu-remove)
      (define-key Variation-main-menu-mode-map "k" 'Variation-menu-delete)
      (define-key Variation-main-menu-mode-map "v" 'Variation-menu-revert)
      (define-key Variation-main-menu-mode-map "u" 'Variation-menu-update1)
      (define-key Variation-main-menu-mode-map "m" 'Variation-menu-make-variation)
      (define-key Variation-main-menu-mode-map "c" 'Variation-menu-copy1)
;;      (define-key Variation-main-menu-mode-map "#" 'Variation-menu-renumber)
      (define-key Variation-main-menu-mode-map "d" 'Variation-menu-diff)
      (define-key Variation-main-menu-mode-map "D" 'Variation-menu-Ddiff1)
      (define-key Variation-main-menu-mode-map "f" 'Variation-menu-new-file)
      (define-key Variation-main-menu-mode-map "l" 'Variation-menu-show-log)
      (define-key Variation-main-menu-mode-map "t" 'Variation-menu-toggle)
      (define-key Variation-main-menu-mode-map "g" 'Variation-menu-refresh)
      (define-key Variation-main-menu-mode-map "q" 'Variation-menu-kill)
      (define-key Variation-main-menu-mode-map "z" 'Variation-menu-freeze)
      (define-key Variation-main-menu-mode-map " " 'next-line)
      (define-key Variation-main-menu-mode-map "\177" 'previous-line)
      (define-key Variation-main-menu-mode-map "?" 'describe-mode)))

(if (not Variation-2nd-menu-mode-map)
    (progn
      (setq Variation-2nd-menu-mode-map (make-keymap))
      (suppress-keymap Variation-2nd-menu-mode-map t)
      (define-key Variation-2nd-menu-mode-map "x" 'Variation-2nd-complete)
      (define-key Variation-2nd-menu-mode-map "b" 'Variation-2nd-complete)
      (define-key Variation-2nd-menu-mode-map "n" 'Variation-2nd-complete)
      (define-key Variation-2nd-menu-mode-map "t" 'Variation-menu-toggle)
      (define-key Variation-2nd-menu-mode-map "g" 'Variation-menu-refresh)
      (define-key Variation-2nd-menu-mode-map "q" 'Variation-menu-kill)
      (define-key Variation-2nd-menu-mode-map " " 'next-line)
      (define-key Variation-2nd-menu-mode-map "\177" 'previous-line)
      (define-key Variation-2nd-menu-mode-map "?" 'describe-mode)))

(put 'Variation-main-menu-mode 'mode-class 'special)
(put 'Variation-2nd-menu-mode 'mode-class 'special)

(defun Variation-main-menu-mode ()
  "Major mode for editing and merging a list of RCS items.
Each line describes one of the RCS item in a file.
Letters do not insert themselves; instead, they are commands.

VARIATION MENU COMMANDS:
t -- Toggle variable that adds all user symbols in the variation menu.
     With prefix arg, toggle freeze symbols.
g -- Refresh Variation menu.
q -- Kill Variation menu.
? -- This HELP MENU.
, -- Next file (if in file loop).
SPC -- Move down a line.
Delete -- Back up a line.

BASIC CHECK IN AND CHECK OUT OF RCS ITEMS:
o -- Check out the item that is on the same line as the cursor.
i -- Check in the item that is on the same line as the cursor.
NOTE: * before an item indicates that item is locked.

AUXILIARY RCS COMMANDS:
m -- Make a variation.  Note: Check in all versions of the file before
     attempting to make a variation, otherwise it will abort.
c -- Copy rcs item to another rcs item.
d -- (1) If item is checked out (i.e. locked), then diff the working
         file against the last checked in version.
     (2) If item is not checked out, then diff the two most recent
         versions.
D -- User is prompted for both items that will be used to find differences.
f -- Change the filename in the variation buffer menu.
l -- Show RCS log file.
r -- Remove the item's symbolic name but do not delete ('outdate') it
     from the RCS file.
k -- Delete the tip (latest revision) of the item from the RCS file
     ('outdate' it) and remove its symbol, if the tip is the only revision
     remaining.
v -- Revert the buffer of the checked out file, i.e. nuke all changes
     made to the file and call up the most recent checked in version locked.
u -- Update first item with respect to another item (copy item2 to item1
     if item1 out-of-date wrt item2).  If given prefix arg (C-u), use emerge
     instead of copy.  Intended for multiple files.
z -- Freeze an rcs item.  Create a symbol with a freeze name set to the
     latest revision of the rcs item.  Intended for multiple files.

MERGING TWO ITEMS IN A COMMON FILE:
a -- Select item a for merge.
wa -- Select item a for merge with an ancestor item.

Precisely,\\{Variation-main-menu-mode-map}"
  
  (kill-all-local-variables)
  (use-local-map Variation-main-menu-mode-map)
  (setq truncate-lines t)
  (setq buffer-read-only t)
  (setq major-mode 'Variation-main-menu-mode)
  (setq mode-name "Variation Menu")
  (set-buffer-modified-p (buffer-modified-p))
  (run-hooks 'Variation-main-menu-mode-hook)
  (message " ? for help. * = locked item."))

(defun Variation-2nd-menu-mode ()
  "Secondary mode for editing and merging a list of RCS items.
This mode completes 2-part commands from the main menu.

VARIATION MENU COMMANDS:
t -- Toggle variable that adds all file items in the variation menu.
     With prefix arg, toggle freeze symbols.
g -- Refresh Variation menu.
q -- Kill Variation menu.
? -- This HELP MENU.
, -- Next file (if in file loop).
SPC -- Move down a line.
Delete -- Back up a line.

x -- Select item to copy into, or to diff against.
b -- Select item b for merge with item a.
n -- Select ancestor for merge.

Precisely,\\{Variation-2nd-menu-mode-map}"
  
  (kill-all-local-variables)
  (use-local-map Variation-2nd-menu-mode-map)
  (setq major-mode 'Variation-2nd-menu-mode)
  (setq mode-name "Variation Completion Menu")
  (set-buffer-modified-p (buffer-modified-p)))


;; utility *VARIATION LIST* functions

(defun rcs-eoln ()
  "Give position of current end-of-line."
  (save-excursion
    (end-of-line)
    (point)))

;; mainline *VARIATION LIST* functions

(defun rcs-interactive-variation-menu (fn)
  "Call the variation menu using FILE."
  (interactive "FRCS File to view in variations buffer: \n")
  (if (not (rcs-has-rcs-file-p fn))
      (message "File %s doesn't exist or it is not an RCS file." fn)
    (rcs-interactive-init)
    (setq rcs-filename fn)
    (let ( (return-list (rcs-get-file-symbols-items fn rcs-list-all-symbols t)))
      (if (> (length return-list) 0)
	  (Variation-menu fn return-list)
	(message "No Variations for file %s." fn)))))

(defun Variation-menu (fn variation-list &optional 2nd-menu)
  "Make a menu of Variations so you can save, delete or select them.
Type ? after invocation to get help on commands available.
Type q to make the Variation menu go away."
  (setq rcs-filename fn)
  (let ((col1 0)
	(col2 0)
	(col3 0)
	(rev "")
	(freeze-sym nil)
	(position nil))
    (if (not (get-buffer rcs-varbuf))
	(get-buffer-create rcs-varbuf)
      (set-buffer rcs-varbuf)
      (setq position (point)))
    (switch-to-buffer rcs-varbuf)
    (setq buffer-read-only nil)
    (erase-buffer)
    (forward-line 2)
    (insert "\n\tFile: " fn "\n")
    (insert "\n Symbol\t\t\t")
    (setq col1 (current-column))
    (insert "Item\t\t")
    (setq col2 (current-column))
    (insert "Tip\t")
    (setq col3 (current-column))
    (insert "Date       Time\n")
    (insert " ---------\t\t-------\t\t---\t---------- --------\n")
    
    ;; add items to variations buffer
    (while variation-list
      (if (or
	   (not (setq freeze-sym
		      (string= "r" (car (rcs-what-is-object
					 rcs-filename
					 (cdr (car variation-list)))))))
	   rcs-show-freeze-symbols)
	  (progn
	    (setq rev (concat (cdr (car variation-list)) "."
			      (rcs-get-last-tip rcs-filename
						(cdr (car variation-list)))))
	    (insert " " (car (car variation-list)) (make-string 70 ? ))
	    (move-to-column col1)
	    (insert (cdr (car variation-list)))
	    (move-to-column col2)
	    (insert (if freeze-sym " "
		      (rcs-get-last-tip rcs-filename
					(cdr (car variation-list)))))
	    (move-to-column col3)
	    (insert (or (rcs-get-rev-date rcs-filename rev) " ") "\n")))
      (setq variation-list (cdr variation-list)))
    (fixup-whitespace)
    (if position
	(goto-char position)
      (goto-char (point-min))
      (forward-line 5))
    (if 2nd-menu
	(Variation-2nd-menu-mode)
      (if (string-match "^Variation" (symbol-name major-mode))
	  (funcall major-mode)
	(Variation-main-menu-mode)))))

(defun rcs-variation-menu-many (fn &optional message)
  "Produce Variation buffer on FILE, displaying optional MESSAGE."
  (save-excursion
    (save-window-excursion
      (let ((variation-list (rcs-get-file-symbols-items fn rcs-list-all-symbols t)))
	(if (null variation-list)
	    (progn			; must be an initial checkin
	      (message "No RCS file, will perform initial checkin.")
	      (beep)
	      (sit-for 5)
	      (if (y-or-n-p "Do you want to use the same log message for all files ? ")
		  (setq rcs-use-1st-log t)
		(setq rcs-use-1st-log nil))
	      (message "")
	      (setq rcs-command 'rcs-ci-file
		    rcs-items (list nil)))
	  (Variation-menu fn variation-list)
	  (define-key (current-local-map) "," 'Variation-menu-next-file)
	  (if message (message message))
	  (recursive-edit)
	  (define-key (current-local-map) "," nil))))))

(defun Variation-menu-symbol-selected ()
  ;; return the RCS symbol on the selected line
  (let ((item (Variation-menu-item-selected)))
    (and item
	 (rcs-get-symbol-from-item rcs-filename item))))
	
(defun Variation-menu-item-selected ()
  "Return item RCS number described by this line of Variation menu."
  (save-excursion
    (beginning-of-line)
    (if (re-search-forward "[ \t]\\([0-9.]+\\)[ \t]" (rcs-eoln) t)
	(setq rcs-item-number
	      (buffer-substring (match-beginning 1) (match-end 1)))
      (message "No item on this line.")
      nil)))

(defun Variation-menu-check-in (arg)
  (interactive "P")
  (let ((symbol (Variation-menu-symbol-selected)))
    (and
     symbol
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-ci-file rcs-filename (or arg rcs-checkin-locked) symbol)
	   (switch-to-buffer rcs-varbuf)
	   (Variation-menu-refresh))
       (if (y-or-n-p "Do you want to use the same log message for all files ? ")
	   (setq rcs-use-1st-log t)
	 (setq rcs-use-1st-log nil))
       (message "")
       (setq rcs-command 'rcs-ci-file
	     rcs-items (list (or arg rcs-checkin-locked) symbol))
       (Variation-menu-next-file)))))

(defun Variation-menu-check-out ()
  (interactive)
  (let ((symbol (Variation-menu-symbol-selected)))
    (and
     symbol
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-checkout-file rcs-filename symbol)
	   (Variation-menu-refresh)
	   (switch-to-buffer (get-file-buffer rcs-filename)))
       (setq rcs-command 'rcs-checkout-file
	     rcs-items (list symbol))
       (Variation-menu-next-file)))))

(defun Variation-menu-remove ()
  ;; Remove an RCS symbol
  (interactive)
  (let ((symbol (Variation-menu-symbol-selected)))
    (and
     symbol
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-remove-symbol rcs-filename symbol)
	   (Variation-menu-refresh)
	   (message " %s removed from variation list." symbol))
       (setq rcs-command 'rcs-remove-symbol
	     rcs-items (list symbol))
       (Variation-menu-next-file)))))

(defun Variation-menu-delete (entire)
  (interactive "P")
  (let ((symbol (Variation-menu-symbol-selected)))
    (and
     symbol
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-delete-object rcs-filename symbol entire)
	   (switch-to-buffer rcs-varbuf)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-delete-object
	     rcs-items (list symbol entire))
       (Variation-menu-next-file)))))

(defun Variation-menu-revert (revert-locked)
  (interactive "P")
  (let ((item (Variation-menu-symbol-selected)))
    (and
     item
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-revert-file rcs-filename revert-locked item)
	   (switch-to-buffer rcs-varbuf)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-revert-file
	     rcs-items (list revert-locked item))
       (Variation-menu-next-file)))))

(defun Variation-menu-update1 (use-emerge)
  "Update first symbol wrt to second symbol."
  (interactive "P")
  (message "Please choose second item...(x)")
  (setq Variation-return-func 'Variation-menu-update2
	Variation-return-args (list use-emerge (Variation-menu-symbol-selected)))
  (Variation-2nd-menu-mode))

(defun Variation-menu-update2 (use-emerge symbol1)
  (let ((symbol2 (Variation-menu-symbol-selected)))
    (and
     symbol2
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-update-obj1-wrt-obj2 rcs-filename use-emerge symbol1 symbol2)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-update-obj1-wrt-obj2
	     rcs-items (list use-emerge symbol1 symbol2 nil t))
       (Variation-menu-next-file)))))

(defun Variation-menu-make-variation ()
  (interactive)
  (let ((variation-name (rcs-get-variation-name rcs-filename)))
    (and
     (rcs-valid-name variation-name)
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-make-user-variation rcs-filename variation-name)
	   (switch-to-buffer rcs-varbuf)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-make-user-variation
	     rcs-items (list variation-name))
       (Variation-menu-next-file)))))

(defun Variation-menu-freeze ()
  (interactive)
  (let ((symbol (Variation-menu-item-selected))
	(freeze-name (rcs-get-freeze-name rcs-filename)))
    (and
     symbol
     (rcs-valid-name freeze-name)
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-make-freeze rcs-filename symbol freeze-name)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-make-freeze
	     rcs-items (list symbol freeze-name))
       (Variation-menu-next-file)))))

(defun Variation-menu-copy1 ()
  (interactive)
  (message "Please choose item to copy %s into. Hit (x) when complete"
	   (Variation-menu-symbol-selected))
  (setq Variation-return-func 'Variation-menu-copy2
	Variation-return-args (list (Variation-menu-symbol-selected)))
  (Variation-2nd-menu-mode))

(defun Variation-menu-copy2 (symbol-from)
  (let ((symbol-to (Variation-menu-symbol-selected)))
    (and
     symbol-to
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-copy-objects rcs-filename symbol-from symbol-to
			     (not rcs-log-before-copy))
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-copy-objects
	     rcs-items (list symbol-from symbol-to (not rcs-log-before-copy)))
       (Variation-menu-next-file)))))

(defun Variation-menu-renumber ()
  ;; renumber an item to a higher number
  (let ((symbol (Variation-menu-symbol-selected))
	(rcs-num-list '()))
    (and
     symbol
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-diff-file rcs-filename nil symbol)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-renumber-item
	     rcs-items (list symbol))
       (Variation-menu-next-file)))))

(defun Variation-menu-diff ()
  (interactive)
  (let ((symbol (Variation-menu-symbol-selected)))
    (and
     symbol
     (if (not rcs-save-for-many)
	 (progn
	   (rcs-diff-file rcs-filename nil symbol)
	   (Variation-menu-refresh))
       (setq rcs-command 'rcs-diff-file
	     rcs-items (list nil symbol))
       (Variation-menu-next-file)))))

(defun Variation-menu-Ddiff1 ()
  (interactive)
  (message "Please choose item to diff %s against. Hit (x) when complete."
	   (Variation-menu-symbol-selected))
  (setq Variation-return-func 'Variation-menu-Ddiff2
	Variation-return-args (list (Variation-menu-symbol-selected)))
  (Variation-2nd-menu-mode))

(defun Variation-menu-Ddiff2 (symbol2)
  (let ((symbol1 (Variation-menu-symbol-selected))
	(diff-options nil))
    (if (> (length (rcs-get-file-symbols-items rcs-filename rcs-list-all-symbols))
	   1)
	(and
	 symbol1
	 (progn
	   (setq diff-options (concat "-r" symbol1 " -r" symbol2))
	   (if (not rcs-save-for-many)
	       (progn
		 (rcs-diff-file rcs-filename diff-options)
		 (Variation-menu-refresh))
	     (setq rcs-command 'rcs-diff-file
		   rcs-items (list diff-options))
	     (Variation-menu-next-file))))
      (message (concat "File " rcs-filename " has only one item..."
		       " Use (d) to diff revisions of the same item")))))

(defun Variation-menu-new-file (fn)
  (interactive "FEnter new file for Variation menu: ")
  (rcs-interactive-variation-menu fn))

(defun Variation-menu-show-log ()
  (interactive)
  (if (not rcs-save-for-many)
      (progn
	(rcs-show-log-file rcs-filename t)
	(Variation-menu-refresh))
    (setq rcs-command 'rcs-show-log-file
	  rcs-items (list t))
    (Variation-menu-next-file)))

(defun Variation-menu-toggle (arg)
  (interactive "P")
  (if arg
      (if rcs-show-freeze-symbols
	  (setq rcs-show-freeze-symbols nil)
	(setq rcs-show-freeze-symbols t))
    (if rcs-list-all-symbols
	(setq rcs-list-all-symbols nil)
      (setq rcs-list-all-symbols t)))
  (Variation-menu-refresh))

(defun Variation-menu-refresh ()
  (interactive)
  (save-excursion
    (Variation-menu rcs-filename (rcs-get-file-symbols-items rcs-filename rcs-list-all-symbols t))))

(defun Variation-menu-next-file ()
  (interactive)
  (if (and rcs-save-for-many (interactive-p))
      (setq rcs-command 'rcs-variation-menu-many
	    rcs-items
	    (list (concat "Hit \""
			  (substitute-command-keys "\\[Variation-menu-next-file]")
			  "\" to move on to next Variation Buffer menu; \""
			  (substitute-command-keys "\\[Variation-menu-kill]")
			  "\" to quit."))))
  (if (>= (recursion-depth) 1)
      (exit-recursive-edit)))

(defun Variation-menu-kill ()
  ;; Quit of a the main or 2nd menu:  if main, leave completely,
  ;; else just return to main menu.
  (interactive)
  (if (eq 'Variation-main-menu-mode major-mode)
      (progn
	(kill-buffer rcs-varbuf)
	(top-level)
	(message ""))
    (Variation-main-menu-mode)
    (Variation-menu-refresh))
  (setq rcs-quit t))

(defun Variation-2nd-complete ()
  "Finish a 2-part command and return to Variation-main-menu-mode."
  (interactive)
  (Variation-main-menu-mode)
  (apply Variation-return-func Variation-return-args))


;;; Interface functions between variations menu and emerge.

(defun rcs-emerge-select-objects1 ()
  "User selects items from menu, then call emerge-buffers to do merge."
  (interactive)
  (message "Please choose item B...(b)")
  (setq Variation-return-func 'rcs-emerge-select-objects2
	Variation-return-args (list (Variation-menu-symbol-selected)))
  (Variation-2nd-menu-mode))

(defun rcs-emerge-select-objects2 (symbol-A)
  (interactive)
  (let ((symbol-B (Variation-menu-symbol-selected)))
    (and
     symbol-B
     (if (not rcs-save-for-many)
	 (rcs-emerge-objA-objB rcs-filename symbol-A symbol-B)
       (setq rcs-command 'rcs-emerge-objA-objB
	     rcs-items (list symbol-A symbol-B))
       (Variation-menu-next-file)))))

(defun rcs-emerge-select-objects-with-ancestor1 ()
  "User selects items from menu, then call emerge-buffers to do merge
with ancestor."
  (interactive)
  (message "Please choose item B...(b)")
  (setq Variation-return-func 'rcs-emerge-select-objects-with-ancestor2
	Variation-return-args (list (Variation-menu-symbol-selected)))
  (Variation-2nd-menu-mode))

(defun rcs-emerge-select-objects-with-ancestor2 (symbol-A)
  (interactive)
  (let ((symbol-B (Variation-menu-symbol-selected)))
    (message "Please choose the ancestor...(n)")
    (setq Variation-return-func 'rcs-emerge-select-objects-with-ancestor3
	  Variation-return-args (list symbol-A (Variation-menu-symbol-selected)))
    (Variation-2nd-menu-mode)))

(defun rcs-emerge-select-objects-with-ancestor3 (symbol-A symbol-B)
  (interactive)
  (let ((symbol-ancestor (Variation-menu-symbol-selected)))
    (and
     symbol-ancestor
     (if (not rcs-save-for-many)
	 (rcs-emerge-with-ancestor-objA-objB rcs-filename symbol-A symbol-B symbol-ancestor)
       (setq rcs-command 'rcs-emerge-with-ancestor-objA-objB
	     rcs-items (list symbol-A symbol-B symbol-ancestor))
       (Variation-menu-next-file)))))

(defun rcs-emerge-quit1 ()
  "Checkin files and cleanup after emerge."
  (let ((merge-buffer (get-buffer (buffer-name))) ; merge buffer name
	(merge-fn (emerge-make-temp-file "-rcs-"))) ; save the merge buffer
    (write-file merge-fn)
    (if emerge-prefix-argument
	(progn
	  (message "Abort merge; merge buffer saved in file %s." merge-fn)
	  (rcs-kill-emerge-bufs merge-buffer)
	  (sit-for 5)
	  (if (>= (recursion-depth) 1)
	      (exit-recursive-edit)))
      (if (not (yes-or-no-p
		(concat "Do you want to checkin the merges to file "
			rcs-filename "? ")))
	  (progn
	    (message "Merge buffer saved in file %s." merge-fn)
	    (sit-for 5)
	    (if (>= (recursion-depth) 1)
		(exit-recursive-edit)))
	(message "Use (x) to indicate RCS item to save merge.")
	(switch-to-buffer rcs-varbuf)
	(Variation-menu-refresh)
	(setq Variation-return-func 'rcs-emerge-quit2
	      Variation-return-args (list merge-buffer))
	(Variation-2nd-menu-mode)))))

(defun rcs-emerge-quit2 (merge-buffer)
  (let ((symbol (Variation-menu-symbol-selected)))
    (catch 'rcs-emerge-quit-error
      (if rcs-quit
	  (progn
	    (setq rcs-quit nil)
	    (message "Save merge aborted.")
	    (sit-for 3)
	    (throw 'rcs-emerge-quit-error nil)))
      (rcs-call-process "co" (concat "-f -l"  symbol) rcs-filename)
      (message "Saving merges into item %s file %s..." symbol rcs-filename)
      (rcs-call-process
       "cp"
       (concat (buffer-file-name merge-buffer) " "
	       (file-name-directory rcs-filename)
	       (file-name-nondirectory rcs-filename))
       (file-name-nondirectory rcs-filename) t)
      (rcs-ci-file rcs-filename nil symbol
		   (concat "Merged from " (rcs-make-revision fn rcs-revision)
			   " by " rcs-user-name) (not rcs-log-before-copy))
      (message "Merge back done")
      (rcs-kill-emerge-bufs merge-buffer))))

(defun rcs-kill-emerge-bufs (merge-buffer)
  ;; kill merge buffers
  (if merge-buffer
      (kill-buffer merge-buffer))
  (if (get-buffer rcs-buf-A)
      (kill-buffer rcs-buf-A))
  (if (get-buffer rcs-buf-B)
      (kill-buffer rcs-buf-B))
  (if (get-buffer rcs-buf-ancestor)
      (kill-buffer rcs-buf-ancestor))
  (if (get-buffer "*emerge-diff*")
      (kill-buffer "*emerge-diff*"))
  (set-window-configuration rcs-win-conf)
  (if (>= (recursion-depth) 1)
      (exit-recursive-edit)))

