/*****************************************************************************

NAME
   inews - insert a news article

SYNOPSIS
   main(argc, argv)		-- main sequence of inews/rnews
   int argc; char **argv;

   catch_t xxit(status)		-- exit the program, cleaning up all locks
   int status;

DESCRIPTION
   This program functions as a front end to rnews. It generates Path, From,
Sender and Organization lines based on the identity of the calling user.
For detailed rules on the transformations used, see newpost.c.

   The message-parsing code tries to cope with the Received, >From, and From 
headers generated by many UNIX mailers. The former two kinds it discards; the
latter two kinds it will accept in lieu of an RFC-1036 Path: header if none
is supplied. If several UNIX-style Froms occur in the header, only the address
associated with the last one will be kept. See also the comments within
accept() on mailfrom[] handling.

   The intent of these features is to allow moderators, under most
circumstances, to simply feed the entire text of mailed submissions to inews
and have it do the right thing.

NOTE
   Compiling with DEBUG on enables a -D option that sends the message rnews
would otherwise see to stdout. Piping 'inews -D | rnews -D' will test rnews
as you'd expect.

SEE ALSO
   newpost.c -- user-interface-independent functions for new postings

FILES
   ADM/log		-- event log file
   ADM/errlog		-- error event log file
   ADM/active		-- active-groups file (through active.c)
   /tmp/newsraw??????	-- used to store incoming news
   /tmp/newsin??????	-- used to store incoming news after header munging

AUTHORS
   Eric S. Raymond
   This software is Copyright (C) 1989 by Eric S. Raymond for the sole purpose
of protecting free redistribution; see the LICENSE file for details.

*****************************************************************************/
/*LINTLIBRARY*/
#include "news.h"
#include "vio.h"
#include "header.h"
#include "libpost.h"
#include "procopts.h"
#include "active.h"
#include "dballoc.h"

/* miscellaneous local defines */
#define OPTFORM \
    "usage: inews -t title [ -n newsgroups ] [ -e expdate ] [ -f sender]\n"

/* possible modes for inews program */
#define UNCOMMITTED 0x00    /* Don't know what we're doing yet */
#define INEWS	    0x01    /* Accept input from stdin to post */
#define RNEWS	    0x02    /* Accept stuff from the net to post locally */
#define	CANCEL	    0x04    /* Cancel an article */
#define	MAKENG	    0x08    /* Create a new newsgroup */

char	*Progname = "inews";	/* so xerror identifies failing program */

/* things only inews.c needs to see */
private int	mode = UNCOMMITTED;	/* mode of news program */
private int	hflag;			/* if TRUE, accept hdrs from stdin */
private int	Mflag;			/* fake back-paths to originators */
private int	vflag;			/* show returned Article ID */
private char	inbuf[LBUFLEN];		/* common input buffer */
private char	titlebuf[BUFLEN];	/* title of the article */
private char	grouplist[BUFLEN];	/* list of groups to post to */
private char	distbuf[SBUFLEN];	/* distributions to send to */
private char	expbuf[SBUFLEN];	/* expiration date buffer */
private char	followbuf[SBUFLEN];	/* followup ID buffer */
private char	filename[BUFLEN];	/* general-use file name */
private char	forgedname[NAMELEN];	/* a user specified -f option. */
private char	orgbuf[BUFLEN];		/* organization name buffer */
private char	appbuf[BUFLEN];		/* moderator-approval buffer */
private char	not_here[BUFLEN];	/* do not send to this list */
private char	*artfile;		/* copy of batched input section */
private char	*rawfile;		/* the raw input file */

private option_t postopts[] =
{ /*
opt  filchar flag   mode    buf	                 meaning   */
't', ' ',    NONE,  INEWS, INEWS, STRING, titlebuf,	/* Subject: */
'n', NGDELIM,NONE,  INEWS, INEWS, STRING, grouplist,	/* To: */
'd', '\0',   NONE,  INEWS, INEWS, STRING, distbuf,	/* Distribution: */
'e', ' ',    NONE,  INEWS, INEWS, STRING, expbuf,	/* expire date */
'p', '\0',   NONE,  RNEWS, RNEWS, STRING, filename,	/* run rnews mode */
'f', '\0',   NONE,  INEWS, INEWS, STRING, forgedname,   /* sender name */
'F', ' ',    NONE,  INEWS, INEWS, STRING, followbuf,	/* Followup: */
'c', '\0',   NONE,  CANCEL,CANCEL,STRING, filename,     /* cancel mode */
#ifndef FLEXGROUPS
'C', ' ',    NONE,  MAKENG,MAKENG,STRING, grouplist,	/* create grp */
#endif /* FLEXGROUPS */
'h', '\0',   &hflag,INEWS, INEWS, OPTION, (char *)NULL, /* accept headers */
'o', '\0',   NONE,  INEWS, INEWS, STRING, orgbuf,	/* Organization: */
'a', '\0',   NONE,  INEWS, INEWS, STRING, appbuf,	/* Approved: */
'x', '\0',   NONE,  INEWS, INEWS, STRING, not_here,	/* don't send to */
'v', '\0',   &vflag,INEWS, INEWS, OPTION, (char *)NULL,	/* print returned ID */
'M', '\0',   &Mflag,INEWS, INEWS, OPTION, (char *)NULL,	/* moderator option */
#ifdef DEBUG
'D', '\0',   &debug,INEWS, INEWS, NUMBER, (char *)NULL,	/* debugging option */
#endif /* DEBUG */
'\0','\0',   0,	    0,	   0,    0,      (char *)NULL
};

/*
 * Save partial news.
 */
#define PARTIAL	    "dead.article"	/* place to save partial news */

/*
 * Trap interrupts.
 */
catch_t ionsig(n)
int n;
{
    static int numsigs = 0;
    if (++numsigs > 100)
    {
	(void) fprintf(stderr, "inews ran away looping on signal %d\n", n);
	xxit(1);
    }
#if !defined(BSD4_2) && !defined(SYSV3)
    /*
     * Older UNIX systems reset caught signals to SIG_DFL.
     * This bad design requires that the trap be set again here.
     * Unfortunately, if the signal recurs before the trap is set,
     * the program will die.
     */
    (void) signal(n, ionsig);
#endif /* !defined(BSD4_2) && !defined(SYSV3) */
    sigcaught = n;
}

/* ARGSUSED */
private void newssave(text)
char *text;	/* file to save from if pointer is null */
{
    register FILE *tofd, *fromfd;
    char sfname[BUFLEN];
    register int c;
    time_t now;

    if ((fromfd = fopen(text, "r")) == (FILE *)NULL)
	return;
    (void) setgid(gid);
    (void) setuid(uid);

    (void) sprintf(sfname, "%s/%s", userhome, PARTIAL);
    if ((tofd = fopen(sfname, "a")) == (FILE *)NULL)
    {
	(void) fprintf(stderr, "Cannot save partial news in %s\n", sfname);
	xxit(1);
    }
    /* Unisoft 5.1 doesn't seek to EOF on "a" */
    (void) fseek(tofd, (off_t)0, SEEK_END);
    (void) time(&now);
    (void) fprintf(tofd, "----- News saved at %s\n", arpadate(&now));

    while ((c = getc(fromfd)) != EOF)
	(void) putc(c, tofd);
    (void) fclose(fromfd);
    (void) fclose(tofd);
    (void) printf("News saved in %s\n", sfname);
    xxit(0);
}

private char *gettext(tmptext)
/* accept text from stdin, put it in a tempfile, return the file name */
char	*tmptext;	    /* name of the file to put the text in */
{
    register int    c;
    register char   *cp;
    FILE	    *tfp;
    int		    consec_newlines = 0, lcount = 0;
    int		    tty = isatty(fileno(stdin));
    char	    rawname[BUFLEN];

    (void) strcpy(rawname, tmptext);
    (void) mktemp(rawname);
    tfp = xfopen(rawname, "w");

    sigcaught = FALSE;	/* becomes true if a signal has been caught */

    /* now grab text lines until we hit a message ending */
    while (!sigcaught && fgets(inbuf, BUFLEN, stdin) != (char *)NULL)
    {
	/* break on ancient-binmail-style end of input if input is from tty */
	if (strcmp(inbuf, ".\n") == 0)
	    if (tty)
		break;
	    else
	    {
		(void) fprintf(stderr,
		    "line %d: some news versions choke on \".\\n\"\n",
		    lcount);
		xxit(1);
	    }

	/* break on an MMDF-style message delimiter */
	if (!tty && strncmp(inbuf, "\1\1\1\1", 4) == 0)
	    break;

	if (inbuf[0] == '\n') {	/* 1 empty line, to go */
	    lcount++, consec_newlines++;	/* count it, in case */
	    continue;		/* but don't write it*/
	}

	/* foo! a non-empty line. write out all saved lines. */
	while (consec_newlines > 0) {
	    (void) putc('\n', tfp);
	    consec_newlines--;
	}

	/* we got a real line, send to the article file */
	for (cp = inbuf; c = *cp; cp++)
	{
	    if (isascii(c) && (isprint(c) || isspace(c) || c=='\b'))
		(void) putc(c, tfp);
	    else
	    {
		(void) fprintf(stderr,
		    "line %d: illegal character (octal %o)\n", lcount, c);
		xxit(1);
	    }

	    if (c == '\n')
		lcount++;
	}
    }

    /* now deal with any signals that may have come in */
    if (sigcaught)
    {
	if (tty)
	    (void) fprintf(stderr, "Interrupt\n");
	if (tty && lcount)
	    newssave(rawname);
	if (!tty)
	    (void) fprintf(stderr,"Blown away by interrupt %d\n",sigcaught);

	(void) unlink(tmptext);
	xxit(1);
    }
    else if (tty)
	(void) printf("EOT\n");
    (void) fflush(stdout);
    (void) fclose(tfp);

    return(savestr(rawname));
}

private int accept(fname)
/*
 * accept a message from the stdin, load header with its header info,  and end
 * up with header and text in a given temp file
 */
char	*fname;	    /* name of the file to put the text in */
{
    FILE *article;
    char mailfrom[LBUFLEN];
    long lastline = 0L;

    /* snarf text into a tempfile connected to stdin */
    rawfile = gettext("/tmp/inewsrawXXXXXX");
    (void) freopen(rawfile, "r", stdin);

    /* if the -h flag was given, accept headers from the input */
    if (hflag)
    {
	/* there may be leading headers from a mailer */
	while (fgets(bfr, LBUFLEN, stdin) != (char *)NULL)
	{
	    header.h_intnumlines--;
	    if (!strncmp(bfr, "Received: ", 13))
		continue;
	    else if (bfr[0] == ' ' && bfr[0] == '\t')
		continue;
	    else if (!strncmp(bfr, "From ", 5) || !strncmp(bfr, ">From ", 6))
	    {
		char *ep, *lp, *fb = strchr(bfr, ' ');

		mailfrom[0] = '\0';
		lp = bfr + strlen(bfr) - 1;
	        *lp-- = '\0';	/* trash the trailing \n */

		/* we need to look at the last token */
		while (lp > bfr && *lp != ' ')
		    lp--;

		/* pick up the 'remote' site if there is one */
		if (lp > bfr && lp[-1] == 'm')	/* preceding 'from'? */
		{
		    (void) strcat(mailfrom, lp + 1);
		    (void) strcat(mailfrom, "!");
		}

		/* concatenate the sender address in the first token */
		while (isspace(*fb))
		    fb++;
		if (ep = strchr(fb, ' '))
		    *ep = '\0';
		(void) strcat(mailfrom, fb);
	    }
	    else
	    {
		/* we hit a real header, back up a line and quit */
		(void) fseek(stdin, (off_t)lastline, SEEK_SET);
		break;
	    }

	    lastline = ftell(stdin);
	}

	/* Now allow the user to supply some headers. */
	(void) hread(&header, 0L, stdin);
	lcase(header.h_newsgroups);

	/* force the From: line into canonical form */
	if (hlnblank(header.h_from))
	{
	    fixaddress(header.h_from, bfr);
	    (void) hlcpy(header.h_from, bfr);
	}

	/* ditto for the Path line */
	if (hlnblank(header.h_path))
	{
	    char	*cp;

	    fixaddress(header.h_path, bfr);
	    if (cp = strchr(bfr, ' '))
		*cp = '\0';
	    (void) hlcpy(header.h_path, bfr);
	}

	/*
	 * This kluge assumes that if we see mail headers *and* a Path: that
	 * what we're actually looking at is a mailed copy of someone's
	 * cross-posting to a moderated group -- so what we really want to do
	 * is drop the target user name on the mail path and concat the
	 * Path line to it.
	 */
	if (mailfrom[0])
	    if (hlblank(header.h_path))
		(void) hlcpy(header.h_path, mailfrom);
	    else
	    {
		char	*p = mailfrom + strlen(mailfrom) - 1, *q, *r, *s;
		int	bangcount = 0;

		while (p > mailfrom && bangcount < 2)
		    if (*p-- == PATHSEP)
			bangcount++;
		p += 2;
		*p = '\0';
		(void) strcat(mailfrom, header.h_path);

		/* try to optimize the path by pinching off loops */
		for (p = mailfrom; q = strchr(p, PATHSEP); p = q + 1)
		    for (r = q + 1; s = strchr(r, PATHSEP); r = s + 1)
		    {
			*q = *s = '\0';
			if (strcmp(p, r))
			    *q = *s = PATHSEP;
			else
			{
			    (void) strcpy(q + 1, s + 1);
			    *q = PATHSEP;
			    break;
			}
		    }

		(void) hlcpy(header.h_path, mailfrom);
	    }
	}

    /*
     * Generate some other headers from the environment
     * (and toss a few that the reader isn't permitted to specify)
     */
    if (originate(&header, Mflag) == FAIL)
    {
	(void) fprintf(stderr, "There was no subject on that article!\n");
	return(FAIL);
    }

    /* now write the news headers and trailing text to the given tempfile */
    header.h_intnumlines = linecount(stdin, (off_t)0);
    hwrite(&header, article = xfopen(fname, "w"), FALSE);
    while (fgets(bfr, LBUFLEN, stdin) != (char *)NULL)
	(void) fputs(bfr, article);
    (void) fclose(article);
    (void) unlink(rawfile);

    if (!Mflag && hlblank(header.h_approved) && checkincl(&header, fname) != 0)
    {
	(void) printf("Too much quoted text in that article!\n");
	return(FAIL);
    }

    return(SUCCEED);
}

main(argc, argv)
int	argc;
char	**argv;
{
    char	*msgid;
    char	optionsbuf[BUFLEN];

    newsinit();	/* set up defaults and initialize. */

    privileged = (uid == 0)
	|| strcmp(site.notify, username)
	|| strcmp(NEWSUSR, username);
   optionsbuf[0] = '\0';

    /* now process options according to the mode */
    mode = procopts(argc, argv, mode, postopts);
    if (mode == FAIL || (mode == INEWS && argc < 2))
    {
	(void) fputs(OPTFORM, stderr);
	xxit(1);
	/*NOTREACHED*/

    }

    /* following 2 branches are intended to be replaced by control messages */
    if (mode == CANCEL)
    {
	(void) sprintf(titlebuf, "cmsg cancel %s", filename);
	(void) sprintf(bfr, "cancel %s", filename);
	hlcpy(header.h_ctlmsg, bfr);
	/* fall through to transmission code */
    }
#ifndef FLEXGROUPS
    else if (mode == MAKENG)
    {
	(void) rdactive(NULLPRED);
	lcase(grouplist);

	/* only certain users are allowed to create newsgroups */
	if (!privileged)
	{
	    msg0("Please contact one of the local netnews people");
	    msg1("to create group \"%s\" for you", grouplist);
	    xxit(1);
	    /*NOTREACHED*/
	}

	(void) strcat(optionsbuf, "-P ");

	(void) sprintf(titlebuf, "cmsg newgroup %s", grouplist);
	(void) strcpy(appbuf, username);
	(void) strcpy(distbuf,
#ifdef ORGDISTRIB
	    ORGDISTRIB
#else /* !ORGDISTRIB */
	    "local"
#endif /* !ORGDISTRIB */
		);
	(void) sprintf(bfr, "newgroup %s", grouplist);
	hlcpy(header.h_ctlmsg, bfr);
	(void) strcat(grouplist, ".ctl");
	
	if (!bitbucket(stdin))
	{
	    (void) printf("Please enter a short description of the newsgroup.\n");
	    (void) printf("End with control D as usual.\n\n");
	}
	(void) fflush(stdout);
    }
#endif /* FLEXGROUPS */

    /*
     * include any options the user may have given in the generated header
     * these assume that we'll never hfree() the header
     */
    if (titlebuf[0])	hlcpy(header.h_subject, titlebuf);
    if (grouplist[0])	hlcpy(header.h_newsgroups, grouplist);
    if (distbuf[0])	hlcpy(header.h_distribution, distbuf);
    if (expbuf[0])	header.h_exptime= getdate(expbuf,(struct timeb*)NULL);
    if (followbuf[0])	hlcpy(header.h_followto, followbuf);
    if (orgbuf[0])	hlcpy(header.h_organization, orgbuf);
    if (forgedname[0])	hlcpy(header.h_from, forgedname);
    if (appbuf[0])	hlcpy(header.h_approved, appbuf);

    /* all modes fall through to here */
    if (mode != RNEWS)
    {
	if (signal(SIGHUP, SIGCAST(SIG_IGN)) != SIGCAST(SIG_IGN))
	    (void) signal(SIGHUP, ionsig);
	if (signal(SIGINT, SIGCAST(SIG_IGN)) != SIGCAST(SIG_IGN))
	    (void) signal(SIGINT, ionsig);

	artfile = savestr("/tmp/newsinXXXXXX"); (void) mktemp(artfile);

	/* accept a message and try to post it */
	if (accept(artfile) != SUCCEED)
	{
	    newssave(artfile);
	    (void) unlink(artfile);
	    xxit(1);
	    /*NOTREACHED*/
	}
    }

    /* hand it to rnews for posting */
    if (not_here && not_here[0] && not_here[0] != '-' && !strchr(not_here,' '))
    {
	(void) strcat(optionsbuf, "-x ");
	(void) strcat(optionsbuf, not_here);
    }
    if ((msgid = newpost(artfile, optionsbuf)) != (char *)NULL && vflag != 0)
	(void) fputs(msgid, stdout);
    xxit(0);
    /*NOTREACHED*/
}

catch_t xxit(status)
/* exit and cleanup */
int status;
{
#ifdef DEBUG
    if (debug == 0)
#endif /* DEBUG */
	(void) unlink(artfile);
    (void) unlink(rawfile);
    exit(status);
}

/* inews.c ends here */
