/****************************************************************************

NAME
   header.c - functions to read and write netnews's RFC-1036 headers

SYNOPSIS
   #include "header.h"

   void hfree(hp)			-- free store associated with a header
   hdr_t *hp;

   int hread(hp, maxsize, fp)		-- read a header into core
   hdr_t *hp; int maxsize; FILE *fp;

   void hwrite(hp, fp, wr)		-- write a header to a file, B format
   hdr_t *hp; FILE *fp; int wr;

   char *hlget(hname)			-- return contents of a named header
   char *hname;

   void happend(hp, cp)		-- concat cp to the header
   hdr_t *hp; char *cp;

   char *tailpath(hp)			-- extract a short name of the sender
   hdr_t *hp;

DESCRIPTION
   The hread() function returns an integer count of bytes in the header. This
count reflects the size *before* control-character interpretation and long-
line truncation rather than the size in core; it is intended for the use of
unbatching code.
   If the (long) second argument of hread() is nonzero, the read will abort
if it goes over that number of characters without recognizing a full header.

   The hwrite() function writes news headers to an output stream.
   If the (boolean) third argument of hwrite() is true, the Receipt-Date
header will be written out along with the others.
   A header that contains a non-blank To: field is presumed to have been
constructed for use in a mail reply. The path header, if present, will be
printed with a "News-Path:" label, and the references header will be printed
with an "In-Reply-To:" label.
   "To:", "Cc:" and "Bcc:" are all concatenated to the To: header on input.
   "Status: " headers (such as are generated by BSD Mail and AT&T mailx) are
ignored. So are the Relay-Version, Posting-Version, and Xref headers of older
news versions (though Xref can be re-enabled with the DOXREFS option).

   The interaction between the integer and string forms of the post-date and
expire-date fields obeys the following rules:

   1) When either of the string fields is picked up by an hread() the
corresponding time_t derived field is set.
   2) These fields are always printed from their time_t forms.
   3) If the date field is not in a format that getdate() recognizes, the
integer form will be FAIL, probably corresponding to 23:59 Dec 31 1969.
The hwrite() code recognizes this situation for the Date field, prints
out time now, and writes the original bad date as the content of a header
called 'X-Unparsable-Date'. This is a kluge but at least enables errors from
things like the broken BITNET mailer formats to get detected.

   Thus, the string forms are useful basically only for checking that a
valid value has been read, and it suffices to set the time_t fields
before writing a header.

   Note that the h_endoff field is set to (off_t)0 by hread(), not to the
article's actual length. Code that uses headers should assume that an (off_t)0
in that field means the message text runs to EOF.

   See comments in the code for more details.

NOTES
   The hread() function *does* mung read-in headers.  It discards the
old Relay-Version, Posting-Version, and (if DOXREFS is off) Xref headers.
It no longer forces message-id's to lower case.
   
   The hwrite() code prepends your local sitename to the Path if it's not
already there. For this to work properly, sitenames() must have been called.

   If the ALLOC_HDRS define is set in header.h, the implementation is
changed to use malloc()ed lines for header text rather than a bunch of
fixed-length buffers. Be warned that though other news modules generally use
an interface that hides the difference, the ALLOC_HDRS version has not been
tested in some time and may have bugs.

   Support for A and 2.9 news headers (non RFC-1036) has been removed.

REVISED BY
   Eric S. Raymond
   This software is Copyright (C) 1989 by Eric S. Raymond for the sole purpose
of protecting free redistribution; see the LICENSE file for details.

*****************************************************************************/
/*LINTLIBRARY*/
#include "news.h"
#include "header.h"

hdr_t header;	/* general-use header structure */

private int hdrsize;	/* track the byte length of the header being read */
private int hdrlineno;	/* track the current line of the header being read */

void hfree(hp)
/* Free the allocated storage in a header block. */
hdr_t	*hp;
{
    if (hlnblank(hp->h_to))		hlfree(hp->h_to);
    if (hlnblank(hp->h_path))		hlfree(hp->h_path);
    if (hlnblank(hp->h_from))		hlfree(hp->h_from);
    if (hlnblank(hp->h_newsgroups))	hlfree(hp->h_newsgroups);
    if (hlnblank(hp->h_subject))		hlfree(hp->h_subject);
    if (hlnblank(hp->h_ident))		hlfree(hp->h_ident);
    if (hlnblank(hp->h_postdate))		hlfree(hp->h_postdate);
    if (hlnblank(hp->h_expdate))		hlfree(hp->h_expdate);
    if (hlnblank(hp->h_references))	hlfree(hp->h_references);
    if (hlnblank(hp->h_ctlmsg))		hlfree(hp->h_ctlmsg);
    if (hlnblank(hp->h_sender))		hlfree(hp->h_sender);
    if (hlnblank(hp->h_replyto))		hlfree(hp->h_replyto);
    if (hlnblank(hp->h_followto))		hlfree(hp->h_followto);
    if (hlnblank(hp->h_distribution))	hlfree(hp->h_distribution);
    if (hlnblank(hp->h_organization))	hlfree(hp->h_organization);
    if (hlnblank(hp->h_keywords))		hlfree(hp->h_keywords);
    if (hlnblank(hp->h_summary))		hlfree(hp->h_summary);
    if (hlnblank(hp->h_approved))		hlfree(hp->h_approved);
    if (hlnblank(hp->h_supersedes))	hlfree(hp->h_supersedes);
    if (hlnblank(hp->h_contenttype))	hlfree(hp->h_contenttype);
#ifdef ZAPNOTES
    if (hlnblank(hp->h_nffrom))		hlfree(hp->h_nffrom);
    if (hlnblank(hp->h_nfid))		hlfree(hp->h_nfid);
#endif /* ZAPNOTES */
#ifdef DOXREFS
    if (hlnblank(hp->h_xref))		hlfree(hp->h_xref);
#endif /* DOXREFS */
    if (hlnblank(hp->h_backrefs))	hlfree(hp->h_backrefs);
    if (hp->h_other)
    {
	(void) free(hp->h_other);
	hp->h_other = (char *)NULL;
    }
    hp->h_exptime = hp->h_rectime = hp->h_posttime = (time_t) 0;
    hp->h_intnumlines = hp->h_intpriority = 0;
    hp->h_fp = (FILE *)NULL;
    hp->h_startoff = hp->h_endoff = hp->h_textoff = (off_t)0;
}

void happend(hp, cp)
/* add the given line to the pass-through part of the given header */
hdr_t	*hp;
char	*cp;
{
    char    *unrec;
    int     oldsize = hp->h_other ? strlen(hp->h_other) : 0;
    int     newsize = oldsize + strlen(cp) + 2;

    if ((unrec = malloc((unsigned)newsize)) != (char *)NULL)
    {
	if (oldsize == 0)
	    unrec[0] = '\0';
	else
	{
	    (void) strcpy(unrec, hp->h_other);
	    (void) free(hp->h_other);
	}
	(void) strcat(unrec, cp);
	(void) strcat(unrec, "\n");
	hp->h_other = unrec;
    }
}

/* ordinal values for header fields, returned by type() */
#define IGNORE		0
#define TO		1
#define CC		2
#define PATH		3
#define FROM 		4
#define NEWSGROUP 	5
#define SUBJECT 	6
#define MESSAGEID	7
#define POSTDATE	8
#define RECEIVED	9
#define EXPIREDATE	10
#define REFERENCES	11
#define CONTROL		12
#define SENDER		13
#define REPLYTO		14
#define FOLLOWTO	15
#define DISTRIBUTION	16
#undef ORGANIZATION
#define ORGANIZATION	17
#define NUMLINES	18
#define KEYWORDS	19
#define SUMMARY		20
#define PRIORITY	21
#define APPROVED	22
#define SUPERSEDES	23
#define CONTENTTYPE	24
#define NFID		25
#define NFFROM		26
#define XREF		27
#define BACKREFS	28
#define OTHER		99

typedef struct {char *hname; int hval; char *offset;} symbol;
static symbol headers[] =
{
	{"Approved:",		APPROVED,	header.h_approved},
	{"Back-References:",	BACKREFS,	header.h_backrefs},
	{"Bcc:",		TO,		(char *)NULL},
	{"Cc:",			TO,		(char *)NULL},
	{"Content-Type:",	CONTENTTYPE,	header.h_contenttype},
	{"Control:",		CONTROL,	header.h_ctlmsg},
	{"Date-Received:",	RECEIVED,	(char *)NULL},
	{"Date:",		POSTDATE,	header.h_postdate},
	{"Distribution:",	DISTRIBUTION,	header.h_distribution},
	{"Expires:",		EXPIREDATE,	(char *)NULL},
	{"Followup-To:",	FOLLOWTO,	header.h_followto},
	{"From:",		FROM,		header.h_from},
	{"Keywords:",		KEYWORDS,	header.h_keywords},
	{"Lines:",		NUMLINES,	(char *)NULL},
	{"Message-ID:",		MESSAGEID,	header.h_ident},
	{"News-Path:",		PATH,		header.h_path},
	{"Newsgroups:",		NEWSGROUP,	header.h_newsgroups},
	{"Nf-From:",		NFFROM,		header.h_nffrom},
	{"Nf-ID:",		NFID,		header.h_nfid},
	{"Organization:",	ORGANIZATION,	header.h_organization},
	{"Path:",		PATH,		header.h_path},
	{"Posted:",		POSTDATE,	header.h_postdate},
	{"Posting-Version:",	IGNORE,		(char *)NULL},
	{"Priority:",		PRIORITY,	(char *)NULL},
	{"Received:",		RECEIVED,	(char *)NULL},
	{"References:",		REFERENCES,	header.h_references},
	{"Relay-Version:",	IGNORE,		(char *)NULL},
	{"Reply-To:",		REPLYTO,	header.h_replyto},
	{"Sender:",		SENDER,		header.h_sender},
	{"Status:",		IGNORE,		(char *)NULL},
	{"Subject:",		SUBJECT,	header.h_subject},
	{"Summary:",		SUMMARY,	header.h_summary},
	{"Supersedes:",		SUPERSEDES,	header.h_supersedes},
	{"Title:",		SUBJECT,	header.h_subject},
	{"To:",			TO,		header.h_to},
#ifdef DOXREFS
	{"Xref:",		XREF,		header.h_xref},
#else
	{"Xref:",		IGNORE,		(char *)NULL},
#endif /* DOXREFS */
	{(char *)NULL,		OTHER,		(char *)NULL}
};

private int nheaders = (sizeof headers / sizeof *headers) - 1 ;

char *hlget(ptr)
/* return the value of a named header */
register char	*ptr;
{
    /* bug: this returns failure inappropriately on integer-valued headers */
    register symbol *ht, *ht1, *ht2;

    /*
     * ht1 is at the begining of the search area
     * ht2 is one past the end.
     */
    for (ht1 = headers, ht2 = headers + nheaders; ht1 < ht2;) {
	    register	i;

	    ht = ht1 + ((ht2 - ht1) / 2);
	    if ((i = *ptr - *ht->hname) == 0 &&
		(i = strncmp(ptr, ht->hname, strlen(ht->hname))) == 0)
		    return(ht->offset);
	    if (i < 0)
		    ht2 = ht;
	    else
		    ht1 = ht+1;
    }
    return((char *) NULL);
#ifdef notdef
    register symbol *ht;

    for (ht = headers; ht->hname; ht++)
	if (strncmp(ht->hname, ptr, strlen(ht->hname)) == 0)
	    return(ht->offset);
    return((char *)NULL);
#endif
}

private int type(ptr)
register char	*ptr;
{
    register symbol *ht, *ht1, *ht2;

    /*
     * ht1 is at the begining of the search area
     * ht2 is one past the end.
     */
    for (ht1 = headers, ht2 = headers + nheaders; ht1 < ht2;) {
	    register	i;

	    ht = ht1 + ((ht2 - ht1) / 2);
	    if ((i = *ptr - *ht->hname) == 0 &&
		(i = strncmp(ptr, ht->hname, strlen(ht->hname))) == 0)
		    return(ht->hval);
	    if (i < 0)
		    ht2 = ht;
	    else
		    ht1 = ht+1;
    }
    return(OTHER);
#ifdef notdef
    for (ht = headers; ht->hname; ht++)
	if (strncmp(ht->hname, ptr, strlen(ht->hname)) == 0)
	    return(ht->hval);
    return(OTHER);
#endif
}

private time_t cgtdate(date)
char *date;
{
    char    junk[40], month[40], day[30], tod[60], year[50];
    static  time_t lasttime = (time_t)0;
    static char lastdate[BUFLEN] = "";

    if (lastdate[0] && strcmp(date, lastdate) == 0)
	return lasttime;
#ifndef lint
    lasttime = getdate(date, (struct timeb *) 0);
#else
    lasttime = 0;		/* avoid 'struct timeb undefined' from lint */
#endif				/* lint */
    if (lasttime<0 &&sscanf(date,"%s %s %s %s %s",junk,month,day,tod,year)==5)
    {
	(void) sprintf(bfr, "%s %s, %s %s", month, day, year, tod);
#ifndef lint
	lasttime = getdate(bfr, (struct timeb *) 0);
#else
	lasttime = 0L;		/* avoid 'struct timeb undefined' from lint */
#endif				/* lint */
    }
    (void) strncpy(lastdate, date, BUFLEN);
    return(lasttime);
}

private int addhline(hp, cp)
/* add a new line to the given header */
register hdr_t	*hp;
char		*cp;
{
    char    *ptr, *colon;
    
    if (!isalpha(*cp) || (colon = strchr(cp, ':')) == (char *)NULL)
	return(FAIL);
    else if ((ptr = strchr(cp, ' ')) == (char *)NULL)
	ptr = colon + 1;
    else
	while (isspace(*++ptr))
	    continue;

    /* if there's nothing there, skip this field */
    if (*ptr == '\0' || *ptr == '\n')
	return(SUCCEED);

    (void) nstrip(ptr);	/* suppress any trailing \n */

    switch(type(cp))
    {
    case FAIL:		return(FAIL);
    case IGNORE:	break;
    case TO:		hlfree(hp->h_to);	/* fall through to CC case */
    case CC:		{
#ifdef ALLOCHDRS
			   hlcat(hp->to, ptr, " ");
#else
			   (void) strcat(hp->h_to, ptr);
			   (void) strcat(hp->h_to, " ");
#endif /* ALLOCHDRS */
			}
			break;
    case PATH:		if (hlblank(hp->h_path)) hlcpy(hp->h_path,ptr); break;
    case FROM:		if (hlblank(hp->h_from)) hlcpy(hp->h_from,ptr); break;
    case NEWSGROUP: 	if (hlblank(hp->h_newsgroups)) hlcpy(hp->h_newsgroups,ptr); break;
    case SUBJECT: 	if (hlblank(hp->h_subject))hlcpy(hp->h_subject,ptr);break;
    case MESSAGEID:	if (hlblank(hp->h_ident))hlcpy(hp->h_ident,ptr);break;
    case POSTDATE: 
	if (hlblank(hp->h_postdate)) hlcpy(hp->h_postdate, ptr);
	hp->h_posttime = cgtdate(hp->h_postdate);
	break;
    case RECEIVED:
	if (!islower(ptr[0]))	/* skip mailer Received headers */
	    hp->h_rectime = cgtdate(ptr);
	break;
    case EXPIREDATE:
	if (hlblank(hp->h_expdate)) hlcpy(hp->h_expdate, ptr);
	hp->h_exptime = cgtdate(hp->h_expdate);
	break;
    case REFERENCES:
	if (hlblank(hp->h_references)) hlcpy(cp = hp->h_references, ptr);
	break;
    case CONTROL: if (hlblank(hp->h_ctlmsg)) hlcpy(hp->h_ctlmsg,ptr); break;
    case SENDER:  if (hlblank(hp->h_sender)) hlcpy(hp->h_sender,ptr); break;
    case REPLYTO: if (hlblank(hp->h_replyto))hlcpy(hp->h_replyto,ptr);break;
    case FOLLOWTO:if (hlblank(hp->h_followto))hlcpy(hp->h_followto,ptr);break;
    case DISTRIBUTION:
	if (hlblank(hp->h_distribution)) hlcpy(hp->h_distribution, ptr);break;
    case ORGANIZATION: 
	if (hlblank(hp->h_organization)) hlcpy(hp->h_organization, ptr);break;
    case NUMLINES:  hp->h_intnumlines = atoi(ptr); break;
    case KEYWORDS:
	if (hlblank(hp->h_keywords)) hlcpy(hp->h_keywords, ptr); break;
    case SUMMARY:
	if (hlblank(hp->h_summary)) hlcpy(hp->h_summary, ptr); break;
    case PRIORITY:  hp->h_intpriority = atoi(ptr); break;
    case APPROVED:
	if (hlblank(hp->h_approved)) hlcpy(hp->h_approved, ptr); break;
    case SUPERSEDES:
	if (hlblank(hp->h_supersedes)) hlcpy(cp = hp->h_supersedes, ptr);
	break;
    case CONTENTTYPE:
	if (hlblank(hp->h_contenttype)) hlcpy(hp->h_contenttype, ptr); break;
#ifdef ZAPNOTES
    case NFID:	    if (hlblank(hp->h_nfid)) hlcpy(hp->h_approved, ptr);break;
    case NFFROM:
	if (hlblank(hp->h_nffrom)) hlcpy(hp->h_approved, ptr); break;
#endif /* ZAPNOTES */
    case BACKREFS:
	if (hlblank(hp->h_backrefs)) hlcpy(cp = hp->h_backrefs, ptr);
	break;
#ifdef DOXREFS
    case XREF:
	if (hlblank(hp->h_xref)) hlcpy(hp->h_xref, ptr); break;
#endif /* DOXREFS */
    case OTHER:
	happend(hp, cp);
	break;
    }
    hdrlineno++;
    return(SUCCEED);
}

private char *hfgets(buf, len, fp)
/*
 * hfgets() is like fgets(), but deals with continuation lines.
 * It also ensures that even if a line that is too long is
 * received, the remainder of the line is thrown away
 * instead of treated like a second line. Also, for each character
 * read it increments hdrsize.
 */
char *buf;
int len;
FILE *fp;
{
    register int    c;
    register int    n = 0;
    register char  *cp;

    /* first, fill the supplied buffer with characters */
    cp = buf;
    while (n < len && (c = getc(fp)) != EOF)
    {
	hdrsize++;
	if (c == '\n')
	    break;
	if (!iscntrl(c) || c == '\b' || c == '\t')
	{
	    *cp++ = c;
	    n++;
	}
    }
    if (c == EOF && cp == buf)
	return((char *)NULL);
    *cp = '\0';

    if (c != '\n')
    {
	/* Line too long - throw away the rest */
	while ((c = getc(fp)) != '\n' && c != EOF)
	    hdrsize++;
	if (c == '\n')
	    hdrsize++;
    }
    else if (cp == buf)
    {
	/* Don't look for continuation of blank lines */
	*cp++ = '\n';
	*cp = '\0';
	return buf;
    }

    while ((c = getc (fp)) == ' ' || c == '\t')	    /* for each cont line */
    {
	hdrsize++;

	/* Continuation line. */
	if ((n += 2) < len)
	{
	    *cp++ = '\n';
	    *cp++ = c;
	}
	while ((c = getc(fp)) != '\n' && c != EOF)
	{
	    hdrsize++;
	    if ((!iscntrl (c) || c == '\b' || c == '\t') && n++ < len)
		*cp++ = c;
	}
	if (c != EOF)
	    hdrsize++;
    }
/*
   if (c != EOF)
	hdrsize++;
*/
    if (n >= len - 1)
	cp = buf + len - 2;
    *cp++ = '\n';
    *cp = '\0';
    if (c != EOF)
    {
	(void) ungetc(c, fp);	/* push back first char of next header */
/*	hdrsize--;	*/
    }
    return(buf);
}

int hread(hp, maxsize, fp)
/*
 * Read header from file fp into *hp.
 * Return header length if header okay, else 0.
 */
register hdr_t *hp;
long maxsize;
FILE *fp;
{
    char lbuf[LBUFLEN];		/* sizeof(lbuf) = longest line we can handle */
    forward static int addhline();
    
    if (fp == (FILE *)NULL)
	return(0);
    else
	hp->h_fp = fp;

    hdrsize = 0;
    hdrlineno = 1;
    hp->h_startoff = ftell(fp);
    while (hfgets(lbuf, sizeof(lbuf) - 1, fp) != (char *)NULL)
	if (addhline(hp, lbuf) == FAIL)
	    break;
        else if (maxsize && (hdrsize > maxsize))
	    return(0);
    hp->h_textoff = ftell(fp);

    /* we don't check for all the RFC-1036 required headers here */
    if (hlblank(hp->h_from) || hlblank(hp->h_path) || hp->h_posttime == 0)
	return(0);

    return(hdrsize);
}

void hwrite(hp, fp, wr)
/*
 * Write header at 'hp' on stream 'fp' in B format.  Include received date
 * if wr is TRUE.
 */
register hdr_t *hp;
register FILE *fp;
bool	wr;
{
    if (hlnblank(hp->h_path))
    {
        if (hlnblank(hp->h_to))
	    (void) fputs("News-", fp);

	if (prefix(hp->h_path, site.pathname) &&
			strchr(NETCHRS, hp->h_path[strlen(site.pathname)]))
	    (void) fprintf(fp, "Path: %s\n", hp->h_path);
	else
	    (void) fprintf(fp, "Path: %s!%s\n", site.pathname, hp->h_path);
    }
    if (hlnblank(hp->h_from))
	(void) fprintf(fp, "From: %s\n", hp->h_from);
    if (hlnblank(hp->h_to))
	(void) fprintf(fp, "To: %s\n", hp->h_to);
    if (hlnblank(hp->h_newsgroups))
        (void) fprintf(fp, "Newsgroups: %s\n", hp->h_newsgroups);
    if (hlnblank(hp->h_subject))
    	(void) fprintf(fp, "Subject: %s\n", hp->h_subject);
    if (hlnblank(hp->h_ident))
	(void) fprintf(fp, "Message-ID: %s\n", hp->h_ident);
    if (hp->h_posttime)
	if (hp->h_posttime > 0)
	    (void) fprintf(fp, "Date: %s\n", arpadate(&hp->h_posttime));
	else
	{
	    time_t	now = time((time_t *)NULL);

	    (void) fprintf(fp, "Date: %s\n", arpadate(&now));
	    (void) fprintf(fp, "X-Unparsable-Date: %s\n", hp->h_postdate);
	}
    if (wr && hp->h_rectime)
	(void) fprintf(fp, "Date-Received: %s\n", arpadate(&hp->h_rectime));
    if (hp->h_exptime)
	(void) fprintf(fp, "Expires: %s\n", arpadate(&hp->h_exptime));
    if (hlnblank(hp->h_references))
	if (hlnblank(hp->h_to))
	    (void) fprintf(fp, "In-Reply-To: %s\n", hp->h_references);
	else
	    (void) fprintf(fp, "References: %s\n", hp->h_references);
    if (hlnblank(hp->h_ctlmsg))
	(void) fprintf(fp, "Control: %s\n", hp->h_ctlmsg);
    if (hlnblank(hp->h_sender))
	(void) fprintf(fp, "Sender: %s\n", hp->h_sender);
    if (hlnblank(hp->h_replyto))
	(void) fprintf(fp, "Reply-To: %s\n", hp->h_replyto);
    if (hlnblank(hp->h_followto))
	(void) fprintf(fp, "Followup-To: %s\n", hp->h_followto);
    if (hlnblank(hp->h_distribution))
	(void) fprintf(fp, "Distribution: %s\n", hp->h_distribution);
    if (hlnblank(hp->h_organization))
	(void) fprintf(fp, "Organization: %s\n", hp->h_organization);
    if (hlnblank(hp->h_keywords))
	(void) fprintf(fp, "Keywords: %s\n", hp->h_keywords);
    if (hlnblank(hp->h_summary))
	(void) fprintf(fp, "Summary: %s\n", hp->h_summary);
    if (hp->h_intpriority)
	(void) fprintf(fp, "Priority: %d\n", hp->h_intpriority);
    if (hlnblank(hp->h_approved))
	(void) fprintf(fp, "Approved: %s\n", hp->h_approved);
    if (hlnblank(hp->h_supersedes))
	(void) fprintf(fp, "Supersedes: %s\n", hp->h_supersedes);
    if (hlnblank(hp->h_contenttype))
	(void) fprintf(fp, "Content-Type: %s\n", hp->h_supersedes);
#ifdef ZAPNOTES
    if (hlnblank(hp->h_nfid))
	(void) fprintf(fp, "Nf-ID: %s\n", hp->h_nfid);
    if (hlnblank(hp->h_nffrom))
	(void) fprintf(fp, "Nf-From: %s\n", hp->h_nffrom);
#endif /* ZAPNOTES */
#ifdef DOXREFS
    if (hlnblank(hp->h_xref))
    {
	char *colon;

	if ((colon = strchr(hp->h_xref, ':')) != NULL
	 && (colon = strchr(colon + 1, ':')) != NULL)
	    (void) fprintf(fp, "Xref: %s\n", hp->h_xref);
    }
#endif /* DOXREFS */
    if (hlnblank(hp->h_backrefs))
	(void) fprintf(fp, "Back-References: %s\n", hp->h_backrefs);
    if (hp->h_other != (char *)NULL)
	(void) fputs(hp->h_other, fp);
    if (hp->h_intnumlines)
	(void) fprintf(fp, "Lines: %d\n", hp->h_intnumlines);
    (void) putc('\n', fp);
}

char *tailpath(hp)
/* return a "..!<site>!<name>" abbreviation for the message sender */
hdr_t *hp;
{
    char *spc, *sitebang;
    static char pathbuf[BUFLEN] = "...";

    if (spc = strchr(hp->h_path, ' '))	/* Chop off trailing " (name)" */
	*spc = '\0';
    if (sitebang = strrchr(hp->h_path, PATHSEP))
    {
	while (sitebang > hp->h_path && *--sitebang != PATHSEP)
	    continue;
    }
    else
	sitebang = hp->h_path;
    (void) strcpy(pathbuf + 3, sitebang);
    (void) nstrip(pathbuf);
    if (spc)
	*spc = ' ';
    return(pathbuf);
}

/* header.c ends here */
