/*
** This program is almost identical to Erik E. Fair's "nrecnews".  Most 
** changes were made just to make it fit in with the Notes config-file
** scheme.
** 
** It uses a companion sed script, ~notes/canon-addr.  Check it out 
** and change it for your mail system, if necessary.
** 
** Jacob Gore 88/11/06
*/

/*
** recnotes [inotes flags] -o Organization
**
** Recnotes is for gatewaying material from the mail into Notes.
** In order to do this, there are a number of interesting transformations
** that need to be made on the headers...
**
** WARNING:	recnotes disables the "recording" check - it has to because
**	by the time inotes is run, it's in the background and too late to
**	ask permission.  If you depend heavily on recordings you probably
**	should not allow recnotes (and thus the mail interface) to be used.
**
*/

#include "parms.h"
#include "structs.h"

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <signal.h>
#include <errno.h>
#include <sysexits.h>
#ifdef BSD4_2
#include <netdb.h>
#include <sys/wait.h>
#endif
#include "header.h"

typedef	SIGHNDLR	(*sighp)();	/* make the declarations readable */

static	int	ChildPid = (-1);	/* pid of inotes process */

char		*Pname;			/* program name */

#define	README	0	/* stuff for pipe(2) */
#define	WRITEME	1

/*
** If your mailer can deal with it, turning this on will cause
** return addresses of the form: foo@bar.dom to be changed in
** the Path: header to bar.dom!foo.
*/
#define	BANGCONV	1		/* change foo@bar to bar!foo */

/*
** This is a little tricky. If what is coming in is an original submission,
** we want to reject it if it doesn't have a subject header. However, if
** we're playing gateway, the incoming item has already been distributed
** elsewhere, and it is unreasonable to throw it out, so we supply a subject
** of `(none)'. The way we determine whether we're gatewaying is by checking
** for either of the `o' (change organization name) or `x' (don't send this
** to a specified site) options on the command line, because these indicate
** gatewaying activity.
*/
static	int	SubjReqd = TRUE;	/* require a subject? */

#ifdef EXPERIMENTAL
/*
** A Word to the Wise: don't turn this on, and don't use it.
** Your netnews neighbors will mung the header of articles using this
** option until they are unrecognizable. It was B news 2.9 compatability
** that got me... - Erik E. Fair, June 30, 1986
**
** More `tricky' stuff: the `Path:' header in netnews is for determining
** which sites have seen this article. It is also used for mail replies.
** We can't have it both ways when something is gatewayed through here
** with a UUCP path on it. So we will zap the address in the Path: header
** with the list name, for which, it is assumed, there is an alias on
** the gatewaying host, pointing to where the list lives so that the
** reply will go SOMEWHERE sensible...
*/
static char	*ListName = (char *)NULL;	/* name of mailing list */
#endif EXPERIMENTAL

SIGHNDLR childgone();
SIGHNDLR tempfail();
char	*errmsg();
char	*mung();
char	*snarf();
char	*parse_pattern();
char	*strsave();
char	*lcase();
char	*ucase();
char	*dotucase();
static	count();

extern	int	errno;
extern	int	re_exec();
extern	int	split();
extern	void	split_free();
extern	char	*index();
extern	char	*rindex();
extern	char	*strcat();
extern	char	*strncat();
extern	char	*strcpy();
extern	char	*strncpy();
extern	char	*sp_strip();
extern	char	*re_comp();
extern	char	*malloc();
extern	char	*calloc();
extern	sighp	signal();

main(argc, argv)
int argc;
char **argv;
{
    register int	i;
    register char	**av;
    char	*org, *error, **inotes_argv;
    char	buf[LBUFLEN], inotes_path[WDLEN];
    static char	*ufrom = "From ";
#ifdef notdef
    static char	*dash_h = "-h";
#endif
    struct hbuf	header;

    initenv();

    Pname = ((Pname = rindex(argv[0], '/')) ? Pname + 1 : argv[0]);

    /*
     ** so that cores will actually drop...
     */
    if (chdir("/tmp") < 0) {
	fprintf(stderr,"%s: can't chdir(/tmp) directory: %s\n",
		Pname, errmsg(errno));
	exit(EX_TEMPFAIL);
    }

    /*
     ** If someone wants to shut down the system, tell sendmail to
     ** try again later.
     */
    (void) signal(SIGTERM, tempfail);

    /* first read should fetch us the UNIX From line */
    if (fgets(buf, sizeof(buf), stdin) == (char *)NULL)
      exit(EX_NOINPUT);

#ifdef notdef
    /*
    ** Having a UNIX from line is unnecessary to the operation of
    ** the program, but not having it is indicative of operating in
    ** other than a {/bin/,deliver,send}mail environment, which *is*
    ** important.
    */
    if (strncmp(buf, ufrom, sizeof(ufrom)) != 0) {
	fprintf(stderr,
		"%s: first line of input was not UNIX From line.\n%s: %s\n",
		Pname, Pname, buf);
	exit(EX_DATAERR);	/* Ooops! */
    }
#else /*!notdef*/
    /*
    ** "UNIX From line" my foot.  MMDF, for instance, does not use a 
    ** "From " line.  Give the mail system, whatever it may be, the 
    ** benefit of the doubt; if it can't deal with whatever we are 
    ** doing here, it will probably let us know.
    */
#endif notdef

    (void) rfc822read(&header, stdin, buf);

    /* build inotes command */
    (void) sprintf(inotes_path, "%s/%s", libdir, "newsinput");

    /*
    ** allocate memory for the inotes argument vector, using the number of
    ** args that were passed to recnotes as a guide.
    */
    if ((av = inotes_argv = (char **)calloc((unsigned)argc+2, sizeof(char *)))
	== (char **)NULL) {
	fprintf(stderr, "%s: can't calloc(%d,%d)\n",
		Pname, argc + 2, sizeof(char *));
	exit(EX_OSERR);
    }
    *av++ = inotes_path;	/* program name in av[0] */
#ifdef notdef
    *av++ = dash_h;		/* always -h option to inews */
#endif

    /*
    ** Process the argument list, copying everything that we
    ** don't recognize into the new arg list for inotes 
    ** (they are assumed to be args to inotes)
    ** and changing things as we see fit.
    */
    for(i = 1; i < argc; i++) {
	if (argv[i][0] == '-') {
	    switch(argv[i][1]) {
	      case 'o':		/* add organization if not one */
		if (strlen(argv[i]) > 2) {
		    org = &argv[i][2];
		} else {
		    org = argv[++i];
		}
		if (header.organization[0] == '\0') {
		    (void) strcpy(header.organization, dotucase(org));
		}
		SubjReqd = FALSE;
		break;
#ifdef EXPERIMENTAL
	      case 'l':
		if (strlen(argv[i]) > 2) {
		    ListName = &argv[i][2];
		} else {
		    ListName = argv[++i];
		}
		(void) strcpy(header.path, ListName);
		SubjReqd = FALSE;
		break;
#endif EXPERIMENTAL
	      case 'x':
		*av++ = argv[i];
		SubjReqd = FALSE;
		break;
	      case 'n':		/* watch for notesgroups to go by */
#ifdef notdef
		*av++ = "-n";
#endif
		if (strlen(argv[i]) > 2) {
		    (void) strcpy(header.nbuf, &argv[i][2]);
#ifdef notdef
		    *av++ = &argv[i][2];
#endif
		} else {
		    (void) strcpy(header.nbuf, argv[++i]);
#ifdef NOTES
		    *av++ = argv[i];
#endif
		}
		break;
	      default:
		*av++ = argv[i];
		break;
	    }
	} else {
	    *av++ = argv[i];
	}
    }
    *av++ = (char *)NULL;	/* terminate */

    /*
    ** The mung() routine provides the consistency checks on the
    ** header.
    */
    if ((error = mung(&header)) != (char *)NULL) {
	fprintf(stderr,
		"%s: your message was not accepted by Notes, because:\n",
		Pname);
	fprintf(stderr, "%s: %s.\n", Pname, error);
	if (header.nbuf[0] != '\0') {
	    fprintf(stderr,"%s: it was going into the notesgroup%s%s\n",
		    Pname,
		    (index(header.nbuf, NGDELIM) ? "s " : " "),
		    header.nbuf);
	}
	exit(EX_DATAERR);
    }
    inotes(inotes_argv, &header);
    childgone(0);
}

/*
** conjure up an inotes and shove the message at it,
** checking for I/O errors as we go.
**
** I do the fork/exec directly, because I want to stop SIGPIPE,
** and I want to avoid having the shell reparse my arg list
** (already parsed by the shell which the mailer invoked to get me here...).
** It's also more efficient that way...
*/
inotes(argv, hp)
char	*argv[];
struct hbuf	*hp;
{
    register FILE	*pipe_fp = (FILE *)NULL;
    char	buf[BUFSIZ];
    int	fd[2];

    if (pipe(fd) < 0) {
	fprintf(stderr, "%s: pipe(2): %s\n", Pname, errmsg(errno));
	exit(EX_TEMPFAIL);
    }

/*    (void) signal(SIGCHLD, childgone); /* one shot */

    if ((ChildPid = fork()) < 0) {
	fprintf(stderr,"%s: fork(2): %s\n", Pname, errmsg(errno));
	exit(EX_TEMPFAIL);
    } else if (ChildPid == 0) {	/* CHILD PROCESS */
	if (fd[README] != 0) {
	    (void) close(fd[WRITEME]);
	    (void) dup2(fd[README], 0);
	    (void) close(fd[README]);
	}
	(void) execv(argv[0], argv);
	fprintf(stderr,"%s: could not exec %s: %s\n",
		Pname, argv[0], errmsg(errno));
	_exit(EX_OSERR);
    } else {			/* PARENT PROCESS */
	(void) close(fd[README]);
	(void) signal(SIGPIPE, childgone); /* one shot */
	if ((pipe_fp = fdopen(fd[WRITEME], "w")) == (FILE *)NULL)
	  exit(EX_SOFTWARE);
    }

    if (!rfc822write(hp, pipe_fp)) {
	fprintf(stderr,"%s: header write error: %s\n",
		Pname, errmsg(errno));
	exit(EX_IOERR);
    }

    while(fgets(buf, sizeof(buf), stdin) != (char *)NULL) {
	fputs(buf, pipe_fp);
	if (ferror(pipe_fp))
	  break;
    }

    (void) fflush(pipe_fp);
    if (ferror(pipe_fp))
      exit(EX_IOERR);
    (void) fclose(pipe_fp);	/* close down pipe */
    childgone(0);
}

/*
** Reap the inotes child properly, and exit with his return code,
** so that ultimate success or failure rests with inotes.
*/
#ifdef BSD4_2
/*ARGSUSED*/
SIGHNDLR
childgone(signo)
int	signo;
{
    register int	reaped_pid;
    union wait	status;

    if ((reaped_pid = wait(&status)) != ChildPid || reaped_pid == (-1)) {
	fprintf(stderr,
      "%s: Received signal %d\n\t Oh God! My child [%d] has been kidnapped!\n",
		Pname, signo, ChildPid);
	perror(Pname);
	exit(EX_OSERR);
    }
    if (WIFSIGNALED(status)) {
	fprintf(stderr,"%s: sub-process [%d] killed by signal %u\n",
		Pname, reaped_pid, status.w_termsig);
	if (status.w_coredump)
	  fprintf(stderr,"%s: sub-process [%d] dropped core on the floor...\n",
		  Pname, reaped_pid);
	exit(EX_SOFTWARE);
    }
    exit((int)status.w_retcode);
}
#else
/*ARGSUSED*/
childgone(signo)
int	signo;
{
    register int	reaped_pid;
    int		status;

#define	core(s)		((s) & 0200)
#define	exsig(s)	((s) & 0177)
#define	excode(s)	(((s) >> 8) & 0377)

    if ((reaped_pid = wait(&status)) != ChildPid || reaped_pid == (-1)) {
	exit(EX_OSERR);
    }
    if (exsig(status)) {
	fprintf(stderr,"%s: sub-process [%d] killed by signal %d\n",
		Pname, reaped_pid, exsig(status));
	if (core(status))
	  fprintf(stderr,"%s: sub-process [%d] dropped core on the floor...\n",
		  Pname, reaped_pid);
	exit(EX_SOFTWARE);
    }
    exit(excode(status));
}
#endif

/*ARGSUSED*/
SIGHNDLR
tempfail(signo)
int	signo;
{
    exit(EX_TEMPFAIL);		/* die restartably */
}

/*
** Check an RFC822 header for validity and mung it to RFC850 spec.
** returns NULL for everything OK, or a character pointer to an
** error message.
*/
char *
mung(hp)
register struct hbuf *hp;
{
    /*
    ** Message-ID
    */
    if (*hp->ident == '\0') {
	return("Message-ID header missing");
    } else {
	mung_msgid(hp->ident);
	if (!msgid_ok(hp->ident)) {
	    static char buf[LBUFLEN];

	    buf[0] = '\0';
	    strcat(buf, "Message-ID syntax error.\n");
	    strcat(buf, 
	    "*** Please refer to page 23, paragraph 4.6.1. and Appendix D\n");
	    strcat(buf,
	    "*** of NIC RFC #822 for the correct syntax, and fix your mailer");
	    return(buf);
	}
    }

    /*
    ** Newsgroups
    */
    if (*hp->nbuf == '\0') {
	return("Newsgroups header missing");
    } else {
	fixng(hp);
    }

    /*
    ** Subject
    */
    if (*hp->title == '\0') {
	if (SubjReqd)
	  return("Subject header missing");
	else
	  (void) strcat(hp->title, "(none)");
    }

    /*
    ** From
    */
    if (*hp->from == '\0') {
	return("From header missing");
    } else {
	fixfrom(hp);
    }

    /*
    ** References
    ** In-Reply-To
    */
    if (hp->followid[0] != '\0') 
      fixref(hp);

    return((char *)NULL);
}

/*
** This subroutine is a concession to the realities of the Internet
** and the USENET. Much as the idea is distasteful and likely
** to get me in trouble, I have to mung message-ids into a format
** that the USENET won't choke on. Pray that if we're doing multiple
** insertion point gatewaying that ALL the gateways mung exactly the
** same things.
**
** (Death to HERMES! Death to EAN!)
*/
mung_msgid(s)
register char	*s;
{
	register in_msgid = FALSE;

	if (s == (char *)NULL)
		return;

	for(; *s != '\0'; s++) {
		switch(*s) {
		case '<':
			in_msgid = TRUE;
			break;
		/* I hope no one is stupid enough to quote this */
		case '>':
			in_msgid = FALSE;
			*(s + 1) = '\0';	/* zap umass.bitnet */
			break;
		case ' ':	/* death & destruction if we let */
		case '\t':	/* tabs or spaces through */
		case '/':	/* reserved by UNIX Filesystem */
			if (in_msgid) *s = '.';
			break;
		}
	}
}

/*
** Canonicalize the "From:" line into the form
**
** From: local-part@domain (full-name)
**
** RFC822 doesn't require the comment to be at the end of the string
** like that.
*/
fixfrom(hp)
register struct hbuf *hp;
{
    char address[PATHLEN];
    char fullname[LBUFLEN];

    /*
     ** we should handle "Full-Name:" too
     ** but it doesn't get read by the header reader
     */
    crackfrom(address, fullname, hp->from);

    (void) canonicalize(address); /* mung for local conventions */

#if (defined(BSD4_2) || defined(BANGCONV))
    if (count(address, '@') == 1) {
	register char	*cp = index(address, '@');
#ifdef BSD4_2
	register struct hostent	*host;
#endif
	char	scratch[sizeof(address)];

	*cp++ = '\0';
#ifdef BSD4_2
	/*
	 ** If we can find the host's official name,
	 ** use that, instead of what they gave us,
	 ** because some people don't know how to
	 ** write sendmail.cf files right...
	 **
	 ** We should also check for MX records.
	 */
	if ((host = gethostbyname(lcase(strcpy(scratch, cp))))
	    != (struct hostent *)NULL) {
	    cp = ucase(host->h_name);
	} else if (index(scratch, '.') == (char *)NULL) {
	    /*
	     ** If it doesn't have a dot,
	     ** they must be using an alias.
	     ** We'll add .ARPA and try again...
	     */
	    if ((host = gethostbyname(strcat(scratch, ".arpa")))
		!= (struct hostent *)NULL)
	      cp = ucase(host->h_name);
	}
#endif
#ifdef BANGCONV
#ifdef EXPERIMENTAL
	if (ListName == (char *)NULL)
#endif EXPERIMENTAL
	  {
	      (void) sprintf(scratch, "%s!%s", cp, address);
	      (void) strncpy(hp->path, scratch, sizeof(hp->path));
	      hp->path[sizeof(hp->path) - 1] = '\0';
	  }
#endif
	(void) sprintf(scratch, "%s@%s", address, cp);
	(void) strncpy(address, scratch, sizeof(address));
    }
#endif

    if (fullname[0] != '\0') {
	(void) strcat(address, " (");
	(void) strcat(address, fullname);
	(void) strcat(address, ")");
    }
    /*
    ** else check for Full-Name: field in header and use that.
    ** At least, this is what I should be doing...
    */

    /* stick the canonicalized "from" back in */
    (void) strcpy(hp->from, address);
}

/*
** filter(wrfd, rdfd, cmd)
**
** will give back two file descriptors to a filter:
**	one to write into and close
**	one to read results from and close
**
** sort of a subroutine call to a program.
**
** Ah, the possibilities for deadlock are rife in here!
*/

filter(wrfd, rdfd, cmd)
int	*wrfd, *rdfd;
char	*cmd;
{
    register char	*name;
    int	tochild[2], frchild[2];

    if (access(cmd, X_OK) < 0)	/* if we can't execute, don't bother */
      return(FALSE);

    if (pipe(tochild) < 0 || pipe(frchild) < 0)
      return(FALSE);

    if ((ChildPid = fork()) < 0) { /* ERROR */
	(void) close(tochild[README]);
	(void) close(tochild[WRITEME]);
	(void) close(frchild[README]);
	(void) close(frchild[WRITEME]);
	return(FALSE);
    } else if (ChildPid) {	/* PARENT */
	*wrfd = tochild[WRITEME];
	*rdfd = frchild[README];
	(void) close(tochild[README]);
	(void) close(frchild[WRITEME]);
    } else {			/* CHILD */
	(void) close(tochild[WRITEME]);
	(void) close(frchild[README]);
	(void) dup2(tochild[README], 0);
	(void) dup2(frchild[WRITEME], 1);
	if (tochild[README] != 0)
	  (void) close(tochild[README]);
	if (frchild[WRITEME] != 1)
	  (void) close(frchild[WRITEME]);
	/* leaving stderr where it is now */
	name = ((name = rindex(cmd, '/')) != (char *)NULL ? name + 1 : cmd);
	(void) execl(cmd, name, (char *)NULL);
	_exit(EX_OSERR);	/* oh, dear me... */
    }
    return(TRUE);
}

/*
** Do local address canonicalization (i.e. mung it for the things you
** are prepared to accept back again). This is primarily for getting
** the addresses to unofficial networks (e.g. BITNET, MAILNET) correct
** when they're going out to USENET.
*/
canonicalize(address)
char	*address;
{
    register char	*cp;
    register int	len;
    char	buf[LBUFLEN];
    int	tochild, fromchild;
#ifdef BSD4_2
    union wait	status;
#else
    int		status;
#endif
    sighp	pipe_prev;

    pipe_prev = signal(SIGPIPE, SIG_IGN); /* bleah! */

    if (!filter(&tochild, &fromchild, address_canonizer)) {
	(void) signal(SIGPIPE, pipe_prev);
	return(FALSE);
    }

    (void) strcpy(buf, address);
    (void) strcat(buf, "\n");	/* sed(1) wants at least one line */
    if (write(tochild, buf, strlen(buf)) < 0 || close(tochild) < 0) {
	(void) close(tochild);	/* in case the write failed */
	(void) close(fromchild);
	(void) signal(SIGPIPE, pipe_prev);
	return(FALSE);
    }

    for(cp = buf; cp < &buf[sizeof(buf)]; ) {
	if ((len = read(fromchild, cp, sizeof(buf))) < 0) {
	    (void) close(fromchild);
	    (void) signal(SIGPIPE, pipe_prev);
	    return(FALSE);
	}
	if (len == 0) {		/* EOF */
	    (void) close(fromchild);
	    (void) wait(&status); /* can be the only one */
	    ChildPid = (-1);
	    /* and I don't care how it exited */
	    break;
	}
	cp += len;
	*cp = '\0';
    }
    (void) signal(SIGPIPE, pipe_prev);
    if (*(cp = sp_strip(buf)) != '\0') { /* anything come out? */
	(void) strcpy(address, cp);
	return(TRUE);
    }
    return(FALSE);
}

/*
** change notesgroups if the subject, keywords, or summary match
** a pattern found in the notesgroup remap file
*/
fixng(hp)
register struct hbuf *hp;
{
    char	**ng;
    register int	n = split(hp->nbuf, &ng, ',');
    register int	i = 0;
    register char	*cp, *notesgroup;
    char	cmdfile[LBUFLEN];
    char	*targets[5];

    if ((targets[i++] = lcase(strsave(hp->title))) == (char *)NULL)
      i--;
    if ((targets[i++] = lcase(strsave(hp->from))) == (char *)NULL)
      i--;
    if ((targets[i++] = lcase(strsave(hp->keywords))) == (char *)NULL)
      i--;
#ifdef notdef
    if ((targets[i++] = lcase(strsave(hp->summary))) == (char *)NULL)
      i--;
#endif
    targets[i] = (char *)NULL;

    if (i == 0) {
	if (n > 0)
	  split_free(&ng);
	return;			/* no point to it, nothing to match */
    }

    for(i = 0; i < n; i++) {
	if (ng[i] == (char *)NULL || ng[i][0] == '\0')
	  continue;

	for(cp = notesgroup = strsave(ng[i]); *cp != '\0'; cp++) {
	    if (*cp == '.')
	      *cp = '/';
	}

	(void) sprintf(cmdfile, "%s/%s/RECNOTES.CMD", spooldir, notesgroup);
	if ((cp = snarf(cmdfile)) == (char *)NULL)
	  continue;

	/* match & execute go here */
	match_n_mung(ng[i], cp, hp, targets);
	(void) free(notesgroup);	/* from strsave() */
	(void) free(cp);	/* from snarf() */
    }

    /*
     ** free space from targets - strsave()
     */
    for(i = 0; i < sizeof(targets) / sizeof(char *); i++) {
	if (targets[i] == (char *)NULL) break;
	(void) free(targets[i]);
	targets[i] = (char *)NULL;
    }
    if (n > 0)
      split_free(&ng);
}

match_n_mung(notesgroup, s, hp, targets)
char	*notesgroup, *s;
register struct hbuf *hp;
char *targets[];
{
    char	**patterns, *cmd;
    register int	t, ln, npat = split_d(s, &patterns, '\n');

    for(ln = 0; ln < npat; ln++) {
	/*
	 ** First, parse out the regular expression from the
	 ** command
	 */
	if ((cmd = parse_pattern(patterns[ln], ln)) == (char *)NULL)
	  continue;
	/*
	 ** OK, now do regular expression matching against
	 ** the target lines of the header.
	 */
	for(t = 0; targets[t] != (char *)NULL; t++) {
	    if (re_exec(targets[t]) == 1) {
		do_command(cmd, hp, notesgroup);
		break;
	    }
	}
    }
    split_free(&patterns);
}

char *
parse_pattern(p, lineno)
char *p;
int lineno;
{
    register char *cp, *RE, delim;
    char	*command;

    command = (char *)NULL;
    if ((delim = *(cp = RE = p)) == '#' || delim == '\0')
      return((char *)NULL);	/* comment or null line */

    while(*cp != '\0') {
	if (*cp == '\\' && *(cp + 1) == delim) {
	    cp++;
	    *RE++ = *cp++;
	} else if (*cp == delim) {
	    if (cp == RE)	/* first one */
	      cp++;		/* skip */
	    else {
		*RE = '\0';	/* last one */
		command = ++cp;	/* rest is command */
		RE = p;
		break;
	    }
	} else *RE++ = *cp++;
    }

    if (command == (char *)NULL || *command == '\0') {
	fprintf(stderr, "%s: incomplete regular expression, line %d\n",
		Pname, lineno);
	return((char *)NULL);
    }

    if ((cp = re_comp(RE)) != (char *)NULL) {
	fprintf(stderr, "%s: regular expression, line %d: %s\n",
		Pname, lineno, cp);
	return((char *)NULL);
    }
    return(command);
}

do_command(command, hp, notesgroup)
char *command, *notesgroup;
register struct hbuf *hp;
{
    char	**tokens, **ng;
    int	ntok = split_d(command, &tokens, '\0');
    int	nng = split(hp->nbuf, &ng, ',');

    if (ntok == 0) {
	split_free(&tokens);
	split_free(&ng);
	return;
    }

    switch(tokens[0][0]) {
      case 'a':			/* add */
	if (ntok > 1) {
	    register int i;

	    for(i = 1; i < ntok; i++) {
		(void) strcat(hp->nbuf, ",");
		(void) strcat(hp->nbuf, tokens[i]);
	    }
	}
	break;
      case 'd':			/* delete */
	if (nng == 1) {
	    (void) strcpy(hp->nbuf, "junk");
	} else {
	    register int i, n = TRUE;

	    for(i = 0; i < nng; i++) {
		if (strcmp(ng[i], notesgroup) == 0)
		  continue;
		if (n) {
		    hp->nbuf[0] = '\0';
		    (void) strcat(hp->nbuf, ng[i]);
		    n = FALSE;
		} else {
		    (void) strcat(hp->nbuf, ",");
		    (void) strcat(hp->nbuf, ng[i]);
		}
	    }
	}
	break;
      case 'm':			/* move */
	if (ntok > 1) {
	    if (nng == 1) {
		(void) strcpy(hp->nbuf, tokens[1]);
	    } else {
		register int i, n = TRUE;
		register char	*cp;

		for(i = 0; i < nng; i++) {
		    cp = (strcmp(ng[i], notesgroup) == 0 ? tokens[1] : ng[i]);
		    if (n) {
			hp->nbuf[0] = '\0';
			(void) strcat(hp->nbuf, cp);
			n = FALSE;
		    } else {
			(void) strcat(hp->nbuf, ",");
			(void) strcat(hp->nbuf, cp);
		    }
		}
	    }
	}
	break;
      case 'k':			/* kill */
	fprintf(stderr, "%s: your posting to %s was killed by %s\n",
		Pname, hp->nbuf, notesgroup);
	exit(EX_NOPERM);
    }

    split_free(&tokens);
    split_free(&ng);
}

/*
** Fix up the contents of In-Reply-To: fields and References: fields.
*/
fixref(hp)
register struct hbuf *hp;
{
    register char *cp = hp->followid;
    register char *ep;
    register char *max = cp + strlen(cp);
    int first = TRUE;
    char	scratch[LBUFLEN];

    scratch[0] = '\0';
    while((cp = index(cp, '<')) != (char *)NULL) {
	if ((ep = index(cp, '>')) == (char *)NULL)
	  break;
	if (((ep - cp) + 1) > sizeof(scratch) - (strlen(scratch) + 2))
	  break;		/* no more room */

	mung_msgid(cp);
	if (msgid_ok(cp)) {
	    if (first) {
		first = FALSE;
	    } else {
		/* delimit */
		(void) strcat(scratch, " ");
	    }
	    (void) strcat(scratch, cp);
	}
	if ((cp = ep + 2) >= max) /* advance pointer */
	  break;
    }
    (void) strcpy(hp->followid, scratch);
}

/*
** Convert the characters following dots to upper case, if they're
** lower case. Two dots in a row will leave one dot in their place.
** Modifies the argument.
*/
char *
dotucase(string)
char	*string;
{
    register char	*s = string;
    register char	*p = s;

    if (s == (char *)NULL)
      return(s);

    while(*p != '\0') {
	if (*p == '.') {
	    p++;
	    if (*p == '\0') {
		*s++ = '.';
		break;
	    }
	    if (islower(*p))
	      *p = toupper(*p);
	}
	*s++ = *p++;
    }
    *s = '\0';			/* terminate */
    return(string);
}

/*
** convert `s' to lower case
*/
char *
lcase(s)
register char	*s;
{
    register char	*cp;

    if (s == (char *)NULL)
      return(s);

    for(cp = s; *cp != '\0'; cp++)
      if (isupper(*cp))
	*cp = tolower(*cp);
    return(s);
}

/*
** convert `s' to upper case
*/
char *
ucase(s)
register char	*s;
{
    register char	*cp;

    if (s == (char *)NULL)
      return(s);

    for(cp = s; *cp != '\0'; cp++)
      if (islower(*cp))
	*cp = toupper(*cp);
    return(s);
}

/*
** count(s, c)
** count the number of chars in the string.
*/
static
count(s, c)
register char	*s, c;
{
    register int	n = 0;

    if (s == (char *)NULL)
      return(0);

    for(; *s != '\0'; s++)
      if (*s == c)
	n++;
    return(n);
}

char *
strsave(s)
char *s;
{
    register int	len;
    register char	*cp;

    if (s == (char *)NULL || *s == '\0') /* ugh */
      return((char *)NULL);

    if ((len = strlen(s)) <= 0)	/* don't bother */
      return((char *)NULL);

    if ((cp = malloc((unsigned)++len)) == (char *)NULL) {
	fprintf(stderr, "%s: strsave malloc(%d): %s\n",
		Pname, len, errmsg(errno));
	exit(EX_OSERR);
    }

    return(strcpy(cp, s));
}

char *
snarf(file)
char *file;
{
    register int	fd;
    register char	*p;
    struct stat	sb;

    if ((fd = open(file, O_RDONLY)) < 0) {
	if (errno != ENOENT) {
	    fprintf(stderr, "%s: open(%s): %s\n",
		    Pname, file, errmsg(errno));
	    exit(EX_OSERR);
	}
	return((char *)NULL);
    }
    if (fstat(fd, &sb) < 0) {
	fprintf(stderr, "%s: fstat(%s): %s\n",
		Pname, file, errmsg(errno));
	exit(EX_OSERR);
    }
    if ((sb.st_mode & S_IFMT) != S_IFREG) {
	fprintf(stderr,"%s: %s: not a regular file.\n", Pname, file);
	exit(EX_SOFTWARE);
    }
    if ((p = malloc((unsigned)sb.st_size + 1)) == (char *)NULL) {
	fprintf(stderr,"%s: %s: malloc(%ld): %s\n",
		Pname, file, sb.st_size, errmsg(errno));
	(void) close (fd);
	return((char *)NULL);	/* maybe able to continue */
    }
    if (read(fd, p, (int)sb.st_size) != (int)sb.st_size) {
	fprintf(stderr,"%s: read(%s, %ld): %s\n",
		Pname, file, sb.st_size, errmsg(errno));
	exit(EX_IOERR);
    }
    (void) close(fd);
    p[sb.st_size] = '\0';	/* terminate to be sure */
    return(p);
}
