/*
** See the procedure header for main (the first procedure) for an overview
** of the module
**
** INDEX
**	main
**	init_search_state
** 	next_note
*/

/*
** THINGS TO DO
**
** - if output is redirected, detatch from terminal after all input.
**
*/

/*
** Include the various include files.  
*/

#include "nfsmisc.h"
#include "parms.h"
#include "structs.h"
#include <stdio.h>
#include <ctype.h>
#include "nfsregexp.h"

/* Before including debug.h, we must include a file of definitions */
#include "nfsdebug.h"
#include "debug.h"

/* 
** Declare external functions and variables.
*/

extern char *strpbrk();		/* find occurance in str1 of char in str2 */
extern int optind;     		/* index of next argument to be processed*/

#ifdef __STDC__
char *mapnfname(char*);		/* map standard nf name (x.y) to directory
                                 * form (x/y).  */
#else
char *mapnfname();              /* map standard nf name (x.y) to directory
                                 * form (x/y).  */
#endif

/*
** Set various defines.
*/

#define MAXREGEXPR  	1000	/* max size of expanded regular expr.	*/
#define MAXLINE		1000    /* max len of a notes files line	*/
#define MAX_NUM_SEARCH  1000	/* max number of search strings		*/
#define MAX_SEARCH_LEN  1000	/* max chars in a search string.	*/
#define NFNAMELEN	80	/* notes file name length               */
#define NULL_PTR	0	/* value of a null pointer		*/
				
				/* charachters in reg. exprs		*/
#ifdef __STDC__
#define REGCHR 		"^$.\\[]-{}*+?"
#else
#define REGCHR 		"^$.\[]-{}*+?"
#endif
#define SEEK_FROM_START 0

/*
** Declared shared structures, types, etc.
*/

    /* 
    ** search_state is used to pass a set of information needed to carry 
    ** on the search, such as the options selected, etc.
    */

struct search_state_struct
{
    int argc;			/* copy of argc passed to main		   */
    int	author_flag;   		/* True if author string is to be searched */
    char ** argv;		/* copy of argv passed to main 		   */
    FILE *  file_names_file;	/* File containing list of names of notes  */
				/* files, NULL_PTR if none or all names    */
				/* have been read.			   */

				/* TRUE if parallel element is first of a  */
				/* group of strings to be matched together */
				/* An extra element at the end is set TRUE */
    int is_first[MAX_NUM_SEARCH];
				
				/* TRUE if parallel element of search_strs */
				/* contains regular expression characters. */
    int is_regexp[MAX_NUM_SEARCH];
    int matched[MAX_NUM_SEARCH];/* Set to TRUE when a string is matched.   */
    int num_search_strs;	/* Number of search strings in arrays.	   */

				/* Array of pointers to search strings	   */
    char * search_strs[MAX_NUM_SEARCH];
    int	text_flag;		/* True if the text of notes are searched  */
    int	title_flag;		/* True if the title strings are searched  */
};
typedef struct search_state_struct search_state_typ;


/*****************************
** main Usage: nfsearch [-atx -s stringfile -f filenamefile ] [ notesfile... ]
**
** See the manual page for nfsearch for a complete description.
** If -a is given, author names are searched.  If -t is given, text is 
** searched.  If -x is given, titles are searched.  If none are specified,
** all are searched.  The -s option allows a file to be given containing
** search strings.  The -f option allows a file to be given containing
** a list of notes files to be searched.  In addition, notesfiles may
** be specified on the command line.
**
** A notesfile name must be a simple name (not a path name). Only the last
** part, the basename, is used. (New notes change - S. Isaacs)
**
** In the search strings, regular expressions are used.  If in a particular
** string (line of the file), no regular expression features are used, then
** a case-independent search is implied, e.g. "CASE", "Case" and "case" all
** match each other.
**
** Parameters
**				INPUT PARAMETERS
** 	argc	The count of the arguments.  The first argument is the
**		command name.
**	
**	argv	A pointer to the first element in an array of pointers
**		to the arguements.
*/

main(argc,argv)
    char **	argv;
    int 	argc;
{
    /*
    ** Declare external procedures.
    */

    extern FILE *	fopen();
    extern char *	malloc();
    extern char *	strcpy();
#ifdef __STDC__
    extern char *	next_note(int, search_state_typ*, char*);
#else
    extern char *	next_note();
#endif
    extern FILE *	tmpfile();
    
				/* holds the name of a notes file       */
    char 	nf_name[NFNAMELEN];   	
    char *	nf_name_ptr;    /* pointer to a notes file              */
				
				/* contains state information for the   */
				/* search.  See the structure defn above*/
    search_state_typ search_state;
				

    /* Set debug stuff */
#ifdef DEBUG_ON
	/* level 1 = very broad level to find where problem area is     */
	debug_flags[1][4] = TRUE;

	/* level 2 = debug next_note					*/
	debug_flags[2][4] = TRUE;

	/* level 3 = debug match code 			*/
	/*	3,4 is for initial search strings 	*/
	/*      3,5 is for match_line			*/
	/*	3,6 is for match_found			*/
	/*	3,7 is for reset_matched		*/
	debug_flags[3][4] = FALSE;
	debug_flags[3][5] = FALSE;
	debug_flags[3][6] = FALSE;
	debug_flags[3][7] = FALSE;

	/* level 4 = debug gethrec			*/
	debug_flags[4][4] = FALSE;
#endif


    /* get options and set flags */
    
     initenv();   /* Use new notes settings */
    
#ifdef DEBUG_ON
    START_DEBUG(1,4)
	print_trace("call init_search_state");
    END_DEBUG
#endif
    init_search_state(argc, argv, &search_state);

    while ((nf_name_ptr = next_note(NFNAMELEN, &search_state, nf_name)) != NULL)
    {
#ifdef DEBUG_ON
	START_DEBUG(1,4)
	    fprintf(stderr, "The notes file name is ", nf_name_ptr);
	    fprintf(stderr, "%s\n.", nf_name_ptr);
	END_DEBUG
#endif
	search_file(nf_name_ptr, &search_state);
	(void) fflush(stdout);
    }
#ifdef DEBUG_ON
    START_DEBUG(1,4)
	fprintf(stderr, "Done with main.\n");
    END_DEBUG
#endif
    printf("\n\nEnd of search.\n");
    fflush(stdout);
}  /* end of main */


/*****************************
** Procedure get_search_strings
**	get the patterns to match from the user and place them in the file
**
** Parameters
**	search_fp	The file into which the search strings are placed.
**
*/

get_search_strings(search_fp)
    FILE *search_fp;
{
    char line[MAXLINE];   	/* holds one string              	*/
    int lineno=0;               /* number of strings in file search_fp	*/

    fprintf(stderr,"\nsearch arguments (control-d is end of input):\n");
    fprintf(stderr,"OR separator is blankline , AND separator is newline\n");
    while (fgets(line,MAXLINE,stdin) != NULL)
    {
	fputs(line,search_fp);
	lineno++;
    }
    clearerr(stdin);
    if (fseek(search_fp, 0L, SEEK_FROM_START) == EOF)
    {
        rpt_err(TRUE, "Unable to rewind search file.");
    }
    return(lineno);
}  /* end get_search_strings */


/*
** init_search_state  - Pare options and initialize search_state structure.
**
** 	
** Parameters
**	argc		- Standard UN*X count of arguments.
**	argv		- Standard UN*X pointer to array of char pointers
**			  to arguments.
**	search_state_ptr- A structure containing state information needed
**			  to perform the search.  It includes information 
**			  about which search options were selected and
**			  how to get the name of the next file to search.
** Externals
**	optind		- Index to next option.  Used by getopt.
*/

init_search_state(argc, argv, search_state_ptr)
    int 	argc;
    char **	argv;
    search_state_typ *	search_state_ptr;
{
    int chr;
    int opt_char;            	/* holds a character (option character)  */
    FILE * search_fp = NULL; 	/* file containing search strings	 */

    search_state_ptr->argc		= argc;
    search_state_ptr->argv		= argv;
    search_state_ptr->author_flag 	= FALSE;
    search_state_ptr->text_flag 	= FALSE;
    search_state_ptr->title_flag 	= FALSE;
    search_state_ptr->file_names_file 	= NULL;

    while ((opt_char=getopt(argc,argv,"afxst")) != EOF)
    { /* for each option character */
        switch(opt_char) 
	{  /* switch on option character */

            case 'a' : 
		search_state_ptr->author_flag = TRUE;
	        break;

       	    case 'x' : 
		search_state_ptr->title_flag = TRUE;
		break;

            case 't' : 
		search_state_ptr->text_flag = TRUE;
		break;

            case 'f' : 
		if(  (search_state_ptr->file_names_file=fopen(argv[optind],"r")) 		   == NULL
		  )
	        {
                    /* opens the file wich contains the notes files */
	        
		    fprintf(stderr,"can't open file %s \n",argv[optind]);
			exit(BAD);
		}
                else 
		{
		    optind++;  /* increment index,because filename read */
		}
		break;

            case 's' : 
		if((search_fp=fopen(argv[optind],"r")) == NULL)
		{
                             /* opens searchfile */
		    fprintf(stderr,"can't open searchfile %s \n",argv[optind]);
		    exit(BAD);
		}
                else 
		{
		    optind++; /* increment index,because filename red */
		}
		break;

            default  : 
		fprintf(stderr,"illegal option\n");
	  	exit(BAD);
	  	break;
	}  /* switch on option character */
    } /* for each option character */

    /* If none of the search options were chosen, turn them all on */

    if (    (search_state_ptr->author_flag == FALSE)
	 && (search_state_ptr->text_flag   == FALSE)
	 && (search_state_ptr->title_flag  == FALSE)
       )
    {
	search_state_ptr->author_flag = TRUE;
	search_state_ptr->text_flag   = TRUE;
	search_state_ptr->title_flag  = TRUE;
    }

    /*
    ** Initialize the search string information in search_state.  The user 
    ** will be prompted for the strings if a file was not provided.
    */

    init_search_strings(search_fp, search_state_ptr);

    /*
    ** If no file of notes file names was given and there are no other
    ** arguments, prompt the user for the list of names.
    */
    
    if ((search_state_ptr->file_names_file == NULL) && ((argc - optind) == 0))
    {
	search_state_ptr->file_names_file = tmpfile();
	fprintf(stderr, "\nEnter notes file names, one per line (control-d \
marks end of input):\n");
	while ( (chr = getchar()) != EOF)
	{
	    if (putc(chr, search_state_ptr->file_names_file) == EOF)
	    {
		fprintf(stderr,"\nUnable to write to file_names_file.\n");
		exit(BAD);
	    }
	}
	clearerr(stdin);

	/* position to start of the file_names_file */

	if (fseek(search_state_ptr->file_names_file, 0L, 0) == EOF)
	{
	    fprintf(stderr,"\nUnable to seek to start of file_names_file.\n");
	    exit(BAD);
	}
    }

    /* 
    ** If output has been redirected to a file, put the process in the
    ** background.
    */

    if (!isatty(1))
    {
	if (fork() > 0)
	{
	    /* We have launched a son process.  Let the parent die. */

	    exit(0);  
	}
    }
    printf("\n");
} /* end init_search_state */


/******************************
** init_search_strings - initialize search string info in search_state.
**
** Parameters
**	search_fp	- file containing search strings, if it was provided.
**	search_state_ptr- pointer to the search_state structure.
*/

init_search_strings(search_fp, search_state_ptr)
    FILE * 		search_fp;
    search_state_typ *	search_state_ptr;
{

    int		i;
    int		len;
    int		num_strs;
    char	temp_str[MAX_SEARCH_LEN];


    if (search_fp == NULL)
    {
	search_fp = tmpfile();
	if (get_search_strings(search_fp) <= 0)
	{
	    fprintf(stderr, "No search strings given.\n");
	    exit(BAD);
	}
    }

    /*
    ** now, initialize num_search_strs, search_strs[], is_rexexp[], and
    ** is_first[]
    */

    num_strs = 0;
    search_state_ptr->is_first[0] = TRUE;
    while ( (len = read_line(MAX_SEARCH_LEN, search_fp, temp_str)) >= 0)
    { 
	
	if (is_blank(temp_str))
	{
	    search_state_ptr->is_first[num_strs] = TRUE;
	}
	else
	{
	    if (   (search_state_ptr->search_strs[num_strs] 
		       = malloc((unsigned) len + 1))
		== NULL
	       )
	    {
		fprintf(stderr, "Unable to allocate memory for strings.\n");
		exit(BAD);
	    }

	    /*
	    ** if the string contains a regular expresion character, set
	    ** is_regexp[num_str] to TRUE.  If it is not a regular expression,
	    ** downshift all of the characters in the string.
	    */

	    search_state_ptr->is_regexp[num_strs] 
		= (!!strpbrk(temp_str, REGCHR));
	    if (!search_state_ptr->is_regexp[num_strs])
		for (i=0; (temp_str[i] != (char)NULL);  i++)
		    if (isupper(temp_str[i]))
			temp_str[i] = tolower(temp_str[i]);
	    strcpy(search_state_ptr->search_strs[num_strs], temp_str);

	    search_state_ptr->is_first[++num_strs] = FALSE;
	}
    }
    
    search_state_ptr->is_first[num_strs] = TRUE; /* to mark end of last group */
    search_state_ptr->num_search_strs = num_strs;
#ifdef DEBUG_ON
    START_DEBUG(3,4) 
	(void) fprintf(stderr,
		       "The initial search_state is: num_search_strs=%d.\n",
		       search_state_ptr->num_search_strs
		      );
	(void) fprintf(stderr,
		       "is_first, is_regexp, search_strs\n"
		      );
	for (j = 0;  (j < num_strs);  j++)
	{
	    (void) fprintf(stderr,
			   "%8d  %10d  %s\n",
			   search_state_ptr->is_first[j],
			   search_state_ptr->is_regexp[j],
			   search_state_ptr->search_strs[j]
			  );
	}
	(void) fprintf(stderr, 
		       "is_first[last]=%d", 
		       search_state_ptr->is_first[num_strs]
		      );
    END_DEBUG
#endif
} /* end init_search_strings */


/*************************
** Function is_blank
**	Return TRUE if the string is a blank line, otherwise FALSE.
**
** Parameters
**	line	String containing the line to be examined.
**
** Notes
**	A blank line contains only blankks and tabs before the end of line
**	or string.
*/

is_blank(line)
    char line[];

{
    int line_pos;

    for (line_pos=0;  
	 ((line[line_pos]==' ') || (line[line_pos] == '\t'));  
	 line_pos++
	)
    {
	;   /* keep incrementing line_pos */
    }
    if(line[line_pos] != '\n' && line[line_pos] != '\0')
	return(FALSE);
    else 
	return(TRUE);
}


/**************************
** match_found - Return TRUE if all strings in any set have matched.
**
** Parameters
**	search_state_ptr	- pointer to search_state
*/

int match_found(search_state_ptr)
    search_state_typ *	search_state_ptr;
{
    int 	matched;
    int 	idx;

#ifdef DEBUG_ON
    START_DEBUG(3,6)
	fprintf(stderr, "In match_found, matched[] array is:\n");
	print_matches(search_state_ptr);
    END_DEBUG
#endif
    idx   = 0;
    matched = FALSE;
    while (idx < search_state_ptr->num_search_strs) 
    {
	if (search_state_ptr->is_first[idx])
	{
	    if (matched)
	    {
		break;
	    }
	    matched = TRUE;
	}
	if (!search_state_ptr->matched[idx])
	{
	    matched = FALSE;
	}
	idx++;
    }
#ifdef DEBUG_ON
    START_DEBUG(3,6)
	fprintf(stderr, "match_found = %d\n", matched);
    END_DEBUG
#endif
    return (matched);
} /* end match_found */


/****************************
** match_line - set matched[] for all patterms which match the line.
**
** Parameters
**	line		- the line to be matched. (NULL terminated)
**	search_state_ptr- pointer to the structure search_state which contains
**			  various information relevant to the search
*/

match_line(line, search_state_ptr)
    char		line[];
    search_state_typ *	search_state_ptr;
{
#define UPSHIFT		'A' - 'a'

    int 	idx;
#if !defined(BSD4_2) && !defined(PDREGEX)
    char 	expbuf[MAXREGEXPR];
#endif
    int		line_chr;
    int 	line_len;
    int		match_pos;
    int		search_chr;
    int 	search_len;
    int		test_pos;

#ifdef DEBUG_ON
    START_DEBUG(3,5) 
	fprintf(stderr, "Line is '%s'.\n.  Initial matches are:\n", line);
	print_matches(search_state_ptr);
    END_DEBUG
#endif
    for (idx=0;  (idx < search_state_ptr->num_search_strs);  idx++)
    { /* for each search string */
	if (search_state_ptr->is_regexp[idx])
	{
#if (defined(BSD4_2) || defined(PDREGEX))
	    if (re_comp(search_state_ptr->search_strs[idx]))
	    {
		fprintf(stderr, "Bad regular expression\n");
		exit(BAD);
	    }
	    if (re_exec(line))
#else /*SYSV?*/
	    compile(search_state_ptr->search_strs[idx], 
		    expbuf,
	    	    &expbuf[MAXREGEXPR], 
		    NULL
		   );
	    if (step(line, expbuf))
#endif /*SYSV?*/
	    {
		search_state_ptr->matched[idx] = TRUE;
	    }
	}
	else
	{ /* do a case-independent match */
	    /* is the search string in the line independent of case? */

	    search_len = strlen(search_state_ptr->search_strs[idx]);
	    line_len   = strlen(line);
	    for (match_pos = 0;  
		 (match_pos < line_len - search_len + 1);  
		 match_pos ++
		)
	    {
		for (test_pos = 0;  (test_pos < search_len);  test_pos++)
		{
		    search_chr = search_state_ptr->search_strs[idx][test_pos];
		    line_chr   = line[match_pos + test_pos];
		    if (   (search_chr != line_chr)
			&& ( (search_chr + UPSHIFT) != line_chr)
		       )
		    {
			break;
		    }
		}
		if (test_pos >= search_len)
		{
		    search_state_ptr->matched[idx] = TRUE;
		    break;
		}
	    }
	} /* do a case-independent match */
    } /* for each search string */
#ifdef DEBUG_ON
    START_DEBUG(3,5) 
       fprintf(stderr, "At end, matches are:\n");
       print_matches(search_state_ptr);
    END_DEBUG
#endif
} /* end match_line */


/****************************
** match_text - search the note or response text and set matched[] for all
**		search patterns matched.
** 
** Parameters
**	io_ptr		- pointer to the io structure for the notes file
**	where_ptr	- pointer to the structure indicating the position
**			  of the text in the file
**	search_state_ptr- pointer to search_state
*/

match_text(io_ptr, where_ptr, search_state_ptr)
    struct io_f *	io_ptr;
    struct daddr_f *	where_ptr;
    search_state_typ *  search_state_ptr;
{
    int			chars_read;
    long		chars_to_read;
    int			first_chr_idx;
    int			i;
    int			last_chr_idx;
    char		line[MAXLINE + 1]; /* Note extra char for adding NULL */
    int			next_chr_idx = 0;
    int			num_chars;
    struct txthead_f 	text_head;

    
    line[MAXLINE] = (char)NULL;
    		/* to guarantee that the line is always terminated*/

    gethrec(io_ptr, where_ptr, &text_head);  /* get textheader and length */    
    chars_to_read = text_head.textlen;

    while (chars_to_read > 0)
    { /* while more to read */
	if ((num_chars = (MAXLINE - next_chr_idx)) > chars_to_read)
	{
	    num_chars = (int) chars_to_read;
	}
  /*    if ((chars_read = read(io_ptr->fidtxt,
			       &line[next_chr_idx], 
			       num_chars */
	if ((chars_read = fread(
			       &line[next_chr_idx],
			       1,
			       num_chars,
			       io_ptr->fidtxt
			      )
	    ) < 0
	   )
	{
	    fprintf(stderr, "Unable to read text.\n");
	    exit(BAD);
	}
	chars_to_read = chars_to_read - chars_read;
	if (chars_to_read == 0)
	{
	    /*
	    ** There is no more to read.  Make sure that the last buffer ends
	    ** with a newline character.
	    */

	    if (line[next_chr_idx + chars_read] != '\n')
	    {
		line[next_chr_idx + chars_read++] = '\n';
	    }
	}
	first_chr_idx = next_chr_idx;
	last_chr_idx  = next_chr_idx + chars_read - 1;
	while (next_chr_idx <= last_chr_idx)
	{
	    if (line[next_chr_idx] == '\n')
	    {
		line[next_chr_idx] = 0;
		match_line(&line[first_chr_idx], search_state_ptr);
		first_chr_idx = next_chr_idx + 1;
	    }
	    next_chr_idx++;
	}
	if (first_chr_idx == 0)
	{
	    /* no newline in full buffer, we must match against this */
	    
	    line[next_chr_idx] = 0;
	    match_line(&line[first_chr_idx], search_state_ptr);
	    first_chr_idx = next_chr_idx + 1;
	}

	/* Shift unmatched characters to start of buffer */

	for (i=first_chr_idx; (i < next_chr_idx); i++)
	{
	    line[i-first_chr_idx] = line[i];
	}
	next_chr_idx = next_chr_idx - first_chr_idx;
    } /* while more to read */
} /* end match text */


/****************************
** next_note - return a char pointer to the base name of the next note.
**             Uses "next_pat" to get a name OR pattern, but returns
**             only a name.
**
** Parameters
**	max_len		- the maximum lenght of the base note name
**	search_state_ptr- pointer to the structure search_state which contains
**			  various information relevant to the search
**	nf_name		- char array which will be pointed to if successful
**
** Method
**      Next_pat will return the next name or pattern in this call to
**         nfsearch.
**      Retpat will return the next name in the Active list of notesgroups
**          that matches the given pattern, or zero if none match.
**      Patcheck checks if the name is a pattern or not.
**
** Return - If success, &nf_name, a pointer to the returned string.  If
**	failure or no more names, NULL.
**    NOTE that nf_name will NOT necessarily contain the name.  It is not
**         used in main, or anyplace else I can see.  And the area pointed
**         to at the return is from malloc, and shouldn't go away.
*/

char * next_note(max_len, search_state_ptr, nf_name)
    int		max_len;
    search_state_typ *	search_state_ptr;
    char	nf_name[];
{
#ifdef __STDC__
    char *      next_pat(int, search_state_typ*, char*);
#else
    char *      next_pat();
#endif
    char * 	name_ptr;
    static char *pat_ptr;
    static int next = -1;
    
    for (;;) {  /* forever */
            if ( (next == -1) ||
		 ((next = retpat( next, pat_ptr, &name_ptr )) == 0) ) {
		    /* Get next potential pattern or notesname */
                    pat_ptr = next_pat(max_len, search_state_ptr, nf_name);
                    if (pat_ptr == NULL)      return NULL;
                    if (!patcheck(pat_ptr))   return pat_ptr;
		    next = 0;
	    }
	    else                              return name_ptr;
    } /* End of forever */
}
		    

/****************************
** next_pat - return a char pointer to the base name of the next note,
**            or a pattern with which to find the next note.
**
** Parameters
**	max_len		- the maximum lenght of the base note name
**	search_state_ptr- pointer to the structure search_state which contains
**			  various information relevant to the search
**	nf_name		- char array which will be pointed to if successful
**
** Return - If success, &nf_name, a pointer to the returned string.  If
**	failure or no more names, NULL.
*/

char * next_pat(max_len, search_state_ptr, nf_name)
    int		max_len;
    search_state_typ *	search_state_ptr;
    char	nf_name[];
{
    char 	tempname[NFNAMELEN];
    char * 	name_ptr;

#ifdef DEBUG_ON
    START_DEBUG(2,4)
	fprintf(stderr, "In next_note.\n");
	fprintf(stderr, "argc=%d, optind=%d\n",
		search_state_ptr->argc,
		optind
	       );
    END_DEBUG
#endif
    if (search_state_ptr->argc > optind)
    {   /* get the next name from the command line */
	name_ptr = search_state_ptr->argv[optind++];
	strncpy(nf_name, name_ptr, max_len);
	return ( &nf_name[0]);
    }
    else
    {   /* get the next name from the file of names */
	if (search_state_ptr->file_names_file == NULL)
	{
	    return (NULL);   /* There are no more names */
	}

	/* read file until a name is found */
	while (    read_string(tempname, 
			       NFNAMELEN, 
			       search_state_ptr->file_names_file
			      ) 
		>= 0
	      )
	{
#ifdef DEBUG_ON
	    START_DEBUG(2,4)
		fprintf(stderr, "The string read from the file is '%s'\n",
			tempname
		       );
	    END_DEBUG
#endif
	    if (!is_blank(tempname))
	    {
#ifdef DEBUG_ON
		START_DEBUG(2,4)
		    fprintf(stderr, "The name is not blank.\n");
		END_DEBUG
#endif
		name_ptr = tempname;
#ifdef DEBUG_ON
		START_DEBUG(2,4)
		    fprintf(stderr, "The base name is '%s'\n", name_ptr);
		END_DEBUG
#endif
		strncpy((char *) nf_name, name_ptr, max_len);
		return (&nf_name[0]);
	    }
	}

	/* no more names, return NULL */

	search_state_ptr->file_names_file = NULL;
	return (NULL);
    }
} /* end next_note */


/*************************
** print_header - if the header has not been printed, print it and set the flag
**		  to say it has been printed
**
** Parameters
**	file_name_str		- the name of the notes file.
**	header_printed_ptr	- pointer to the header_printed flag
*/

print_header(file_name_str, header_printed_ptr)
    char * 	file_name_str;
    int *	header_printed_ptr;
{
    if (!(*header_printed_ptr))
    {
	*header_printed_ptr = TRUE;
	fprintf(stdout,
		"\n\nThe following notes were found in %s :\n",
		file_name_str
	       );
    }
}


/***********************
** print_matches - print the matched[] array in search_state for debugging.
**
** Parameters
**	search_state_ptr	- pointer to search_state
*/

print_matches(search_state_ptr)
    search_state_typ *	search_state_ptr;
{
    int	idx;

    for (idx = 0;  (idx < search_state_ptr->num_search_strs);  idx++)
    {
	(void) fprintf(stderr, 
		       "    %3d     %d\n", 
		       idx,
		       search_state_ptr->matched[idx]
	              );
    }
}


/*********************************
** Function read_line
**	read a line from a file.  EOF is returned if no characters or error.
**	othewise the length of the line is returned, i.e, result >= 0.
**
** Parameters
**	max_chars	- number of chars that line can hold.
**	in_fptr		- pointer to FILE from which characters are read
**	line		- character array in which the line is returned.
**
*/

int read_line(max_chars, in_fptr, line)
	int	max_chars;
	FILE *  in_fptr;
	char	line[];
{
    int		i = 0;
    int		chr;

    while (   (i < (max_chars - 1)) 
           && ((chr = getc(in_fptr)) != EOF)
	   && (chr != '\n')
	  )
    {
	line[i++] = chr;
    }
    if ( (i == 0) && (chr == EOF))
    {
	line[i] = 0;      /* mark end of string */
	return (EOF);
    }
    else
    {
	line[i] = 0;    /* mark end of string */
	return (i);     /* return success     */
    }
}


/* Function read_string
**	read in a string from the given file.  Return -1 or string len.
**
** Parameters
**	string		A character string into which the string is placed
**	max_str_len	The maximum number of characters before the string
**			terminator.
**	string_file	The file from which the string is read
**
** Notes:
** 	A string read from the file is terminated by \n, \t or ' '.
**	If no characters read before EOF, -1 is returned, otherwise
** 	the length of the string is returned.
*/

int read_string(string,max_str_len,string_file)
    char string[];
    int max_str_len;
    FILE *string_file;
{
    int c;
    int string_len=0;
    while (--max_str_len 
	   && (c=getc(string_file)) != EOF 
	   && c != '\n' 
	   && c != ' ' 
	   && c != '\t'
	  )
    {
	  string[string_len++]=c;
    }
    if((c == EOF) && (string_len == 0))
    return(-1);
    string[string_len] = '\0';
    return(string_len);
}


/********************************************
** Procedure regerr
**	Error exit procedure used by the regexp(7) UN*X facility.
**
*/

/* error exit for the "regexp" facility */

regerr()
{
    printf("RE error\n");
}


/**************************
** reset_matches - reset the array matched[] in search_state
**
** Parameters
**	search_state_ptr	- pointer to search_state
*/

reset_matches(search_state_ptr)
    search_state_typ *	search_state_ptr;
{
    int 	i;

    for (i = 0;  i < search_state_ptr->num_search_strs;  i++)
    {
	search_state_ptr->matched[i] = FALSE;
    }
#ifdef DEBUG_ON
    START_DEBUG(3,7)
	fprintf(stderr, "After matched reset:\n");
	print_matches(search_state_ptr);
    END_DEBUG
#endif
}


/********************** 
** rpt_err - report an error and then terminate.  If no error, just return.
** 
** Parameters 
**      failure         - TRUE if a failure has occured, else FALSE  
**	err_str		- string containing the message to report.
*/

rpt_err (failure,err_str) 
    int 	failure; 
    char *	err_str; 
{
    if (failure)
    { 
	printf("Error %s\n",err_str);
	exit(BAD);
    }
} 


/******************************
** search_file - search the named file for the notes indicated by search_state
**
** Parameters
** 	file_name_str	- the name of the notes file (basename not full path)
**	search_state_ptr- pointer to search_state.
*/

search_file(file_name_str, search_state_ptr)
    char *		file_name_str;
    search_state_typ *	search_state_ptr;
{
    struct io_f 	io;
    int			j;
    int			header_printed = FALSE;
    struct note_f 	note;
    int 		note_cnt;
    int			rblock;
    int			resp_idx;
    struct resp_f 	rsprec;
    int			roffset;
    char 		title[TITLEN+1];	/* holds notes title */
    char                author[PATHLEN];      /* holds authors name */

    /* opens the notesfiles ../text,.../note.indx and .../rsp.indx */
    	
    if(init(&io,file_name_str) < 0)
    {
  	fprintf(stderr,"can't open %s \n",file_name_str);
  	return;
    }

    /* checks whether the user is allowed to read the given notesfile */

    if(allow(&io,READOK) == 0)
    {
	fprintf(stderr,"You are not allowed to read %s\n",file_name_str);
        closenf(&io);
        return;
    }


    /* do for every note */
    
    reset_matches(search_state_ptr);
    for(note_cnt=1;  note_cnt <= io.descr.d_nnote;  note_cnt++)
    { /* for each note */
        getnrec(&io, note_cnt, &note);      /* get the description of note i */

        if(note.n_stat & DELETED)           /* note deleted ?                */
        {
		continue;		    /* then go to next note.  	     */
	}
	
	/* copy the title of the note to title[]			*/

	for(j=0;  j<TITLEN;  j++)          
            title[j]=note.n_title[j];
        title[j]='\0';

	if (search_state_ptr->title_flag)
	{
	    match_line(title, search_state_ptr);
	}
	if (search_state_ptr->author_flag)
	{
	    gethdr(&io, &note.n_msg, (int)note.n_msg.m_from, author);
	    match_line(author, search_state_ptr);
	}
	if (search_state_ptr->text_flag)
	{
#ifdef DEBUG_ON
	    START_DEBUG(4,4)
		fprintf(stderr, "Note %d.\n", note_cnt);
	    END_DEBUG
#endif
	    match_text(&io, &note.n_msg.m_addr, search_state_ptr);
	}
	if (match_found(search_state_ptr))
	{
	    print_header(file_name_str, &header_printed); 
	    fprintf(stdout,"   Note %3d              %s\n",note_cnt,title);
	}
	reset_matches(search_state_ptr);

        
	/* 
	** If text is to be searched, search each response.
	*/

	if (search_state_ptr->text_flag)
	{ /* search responses if text is searched */
	    for(resp_idx=1;  resp_idx <= (int) note.n_nresp;resp_idx++)
	    { /* for each response */
		/* 
		** Get the address of the resp_idx'th response to note
		** note_cnt in the text file.
		*/

		if(lrsp(&io,note_cnt,resp_idx,&rsprec,&roffset,&rblock) == -1)    
		{
		    break;
		}
#ifdef DEBUG_ON
		START_DEBUG(4,4)
		    fprintf(stderr, 
			    "Note %d response %d.\n", 
			    note_cnt,
			    resp_idx
			   );
		END_DEBUG
#endif
		match_text(&io, &rsprec.r_msg[roffset].m_addr, search_state_ptr);
		if (match_found(search_state_ptr))
		{
		    print_header(file_name_str, &header_printed); 
		    fprintf(stdout,
			"   Note %3d Response %3d %s\n",
			note_cnt,
			resp_idx,
			title
		       );
		}
		reset_matches(search_state_ptr);
	    } /* for each response */
	} /* search responses if text is checked */
    } /* for each note */

    if (header_printed)
    {
	printf("\n");
    }
    else
    {
	printf("...%s", file_name_str);
    }

    closenf(&io);                  
}  /* end search_file */
