/*
 *---------------------------------------------------------------------------
 *
 *		nestor -- NEws STatistics ORganizer
 *
 *  nestor.c -- summarizes NNTP-based Usenet News traffic statistics
 *  Copyright (C) 1991  Edwin H. Kremer <edwin@cs.ruu.nl>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *---------------------------------------------------------------------------
 */

static	char	*Version = "1.0 PL%d\n";

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

#define EXTERN	/* empty */
#include "nestor.h"
#include "patchlevel.h"

/*
 *	The keyword mapping table. See also in the file "nestor.h"
 *	for an explanation of the 'KEY_*' constants. The '*_E_??H'
 *	codes mean: failure due configuration of either local ('L')
 *	or remote ('R') host.
 */

EXTERN keytab_t keytable[] = {
	{ "connect"		, KEY_CONNECT	},
	{ "ihave_stats"		, KEY_IHAVE	},
	{ "times"		, KEY_TIMES	},
	{ "group"		, KEY_GROUP	},	/* groupname	*/
	{ "exit"		, KEY_EXIT	},
	{ "unrecognized"	, KEY_UNRECOG	},	/* command	*/
	{ "post"		, KEY_POST	},	/* succeeded	*/
	{ "newnews"		, KEY_NEWNEWS	},
	{ "newnews_stats"	, KEY_NEWNEWS	},
	{ "refused"		, KEY_E_RLH	},	/* connection	*/
	{ "no"			, KEY_E_RLH	},	/* space/groups	*/
	{ "cpstdin:"		, KEY_E_RNET	},	/* EOF before..	*/
	{ "disconnect:"		, KEY_E_RNET	},	/* Broken pipe	*/
	{ "transfer_timeout" 	, KEY_E_RNET	},
	{ "timeout" 		, KEY_E_RNET	},

	{ "stats"		, KEY_STATS	},
	{ "xmit"		, KEY_XMIT	},
	{ "greeted"		, KEY_E_XRH	},	/* us with ...	*/
	{ "hello:"		, KEY_E_XNET	},	/* no route to	*/
	{ "xfer:"		, KEY_E_XNET	},	/* timeout	*/
	{ NIL_CHAR		, -1		}
};


/*
 *---------------------------------------------------------------------------
 *	Usage			- how to use this program
 *---------------------------------------------------------------------------
 */
int
Usage(prog)
	char	*prog;
{
   fprintf(stderr, "Usage: %s   -ainorvxV   [ filename ]\n", prog);
   fprintf(stderr, "\
      -i  report Incoming statistics\n\
      -o  report Outgoing statistics only\n\
      -n  Network connections overview only\n\
      -r  Remote readerships (read/post) statistics\n\
      -x  eXtended report, quality of certain links etc.\n\
      -a  select All of the above to be reported\n\
      -v  be a bit more Verbose, e.g. scream loud on errors in syslog\n\
      -V  show Version and exit\n");

   exit(-1);
}

/*
 *---------------------------------------------------------------------------
 *				M A I N
 *---------------------------------------------------------------------------
 */
main(argc, argv)
	int	argc;
	char	*argv[];
{
	extern int	optind;		/* for getopt() support		*/
	register FILE	*logfp;		/* open logfile filepointer	*/
	register char	*service_match;	/* service name  in logentry	*/	
	register short	buf_idx;	/* buffer index; only be 0 or 1	*/
	register int	ch,		/* options parsing character	*/
			n_input_lines;	/* number of lines read in log	*/
	char		*logfile;	/* logfile we're parsing	*/

	opts = num_nntp_hosts = n_input_lines = buf_idx = 0;
	service_match = start_date = NIL_CHAR;

	while ( (ch = getopt(argc, argv, "ainorvxV")) != EOF )
		switch ( ch )
		{
			case 'a':	/* everything we can get	*/
				opts = OPT_ALL;
				break;
			case 'i':	/* incoming stats report	*/
				opts |= OPT_INCOMING;
				break;
			case 'n':	/* network (trouble) report	*/
				opts |= OPT_NETWORK;
				break;
			case 'o':	/* outgoing stats report	*/
				opts |= OPT_OUTGOING;
				break;
			case 'r':	/* remote read/post stats	*/
				opts |= OPT_REMOTE;
				break;
			case 'x':	/* smart extended report	*/
				opts |= OPT_EXTENDED;
				break;
			case 'v':	/* scream load on errors	*/
				++verbose;
				break;
			case 'V':	/* print version and exit	*/
				printf(Version, PATCHLEVEL);
				exit(-1);
			default	:
				Usage(argv[0]);
		}


	if ( !opts || argc - optind > 1 )
		Usage(argv[0]);

	if ( argc - optind > 0 )
		logfile = argv[optind];
	else
		logfile = NNTP_SYSLOG;

	if ( !strcmp(logfile, "-") )
		logfp = stdin;
	else
		if ( (logfp = fopen(logfile, "r")) == NULL )
			perror("fopen_logfile"), exit(-1);

	while ( fgets(logbuff[buf_idx], BUFSIZ, logfp) != NULL )
	{
		service_match = (char *) strstr(logbuff[buf_idx], " nntp");
		if ( service_match != NIL_CHAR )
		{
			if ( n_input_lines == 0 )
				start_date = (char *) StrSave(logbuff[buf_idx]);
			if ( ScanLine(logbuff[buf_idx], service_match) != -1 )
				++n_input_lines;

			buf_idx ^= 0x01;
		}
	}

	fclose(logfp);

	/* we want the statistics sorted on hostname, so here we go...	*/
	SortNntps(nntps);

	/* now its time to report to the user what we've found		*/

	if ( start_date )
	{
		printf("\n\
   -- Statistics (%d lines) from %15.15s till %15.15s --\n",
			n_input_lines,
			start_date,
			logbuff[buf_idx]);
	}

	if ( opts & OPT_INCOMING )
		Report_nntpd();

	if ( opts & OPT_OUTGOING )
	{
		Report_nntpxmit();
		Report_nntpxfer();
	}

	if ( opts & OPT_NETWORK )
		Report_network();

	if ( opts & OPT_REMOTE )
		Report_remote();

	if ( opts & OPT_EXTENDED )
		Report_smart();

	exit(0);
}
