/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 *
 */
/*
 *	Lots of modifications (new guts, more or less..) by
 *	Matti Aarnio <mea@nic.funet.fi>  (copyright) 1992-1995
 */

#include "hostenv.h"
#include <stdio.h>
#include <ctype.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "mail.h"
#include "scheduler.h"
#include NDIR_H

#include "prototypes.h"

static struct vertex *findvertex __((long, long, int*));
static int ctlowner __((struct ctlfile *));
static void vtxupdate __((struct vertex *, int, int));
static void expaux __((struct vertex *, int, char *));

extern char *strchr();
extern char *strrchr();
extern int ermdir();
extern int rmdir();
extern int eunlink();
extern struct spblk *sp_fhead(), *sp_fnext();
extern time_t time();
extern time_t now;
extern int global_wrkcnt;

/*
 * Parse a diagnostic string from a transport agent, and do whatever is
 * necessary to update our model of the world to reflect the new reality.
 *
 *	#hungry
 *	# other debug comment
 *	18453/3527\tIETF-NOTARY\tok Quick SMTP connection!
 *	18453/3527\tIETF-NOTARY\tdeferred Unable to contact host!
 *	18453/3527\tIETF-NOTARY\terror Unknown user name "fred".
 *	18453/3527\tIETF-NOTARY\tretryat [+]NNNN circuit unavailable.
 */

/* dispatch table for diagnostic types */

static int u_ok       __((struct vertex *, long, long, long, char*, char*));
static int u_deferred __((struct vertex *, long, long, long, char*, char*));
static int u_error    __((struct vertex *, long, long, long, char*, char*));
static int u_retryat  __((struct vertex *, long, long, long, char*, char*));

static struct diagcodes {
	char	*name;
	int	(*fcn)();
} diags[] = {
		{	"ok",		u_ok		},
		{	"deferred",	u_deferred	},
		{	"error",	u_error		},
		{	"retryat",	u_retryat	},
		{	NULL,		NULL		}
};

void
update(fd, diagnostic)
	int fd;
	char	*diagnostic;
{
	register char	*cp;
	char	*type, *message, *notary;
	long	offset;
	long	inum;
	int	index;
	struct vertex *vp;
	struct diagcodes *dcp;
	struct procinfo *proc = &cpids[fd];

	if (*diagnostic == 0) {
	  /* Lone newline.. old-style indications from the transporter */
	  if (proc->tofd >= 0 &&
	      proc->hungry == 0) {
	    proc->hungry = 1;
	    time(&now);
	    proc->hungertime = now;
	    ++hungry_childs;
	  }
	  /* Things are known to DIE from under us! */
	  /* .. propably not a good idea to try to pick any next.. */
	  if (proc->overfed > 0)
	    proc->overfed -= 1;
	  pick_next_vertex(proc, 1);
	  if (proc->hungry)
	    feed_child(proc);
	  return;
	}
	if (*diagnostic == '#' /* DEBUG comment.. */) {
	  if (strncmp(diagnostic,"#hungry",7)==0) {
	    /* This is an "actor model" behaviour,
	       where actor tells, when it needs a new
	       job fed to it. */
	    if (proc->tofd   >= 0) {
	      proc->hungry = 1;
	      time(&proc->hungertime);
	      if (proc->overfed > 0)
		/* It was overfed, decrement that counter first.. */
		proc->overfed -= 1;
	      if (!proc->overfed) {
		++hungry_childs;
		/* Unless it is still overfed,
		   Pick next, and feed it! */
		pick_next_vertex(proc,1);
		/* While we have a thread, and things to feed.. */
		while (!proc->fed && proc->thread) {
		  if (proc->hungry)
		    feed_child(proc);
		  if (proc->fed)
		    proc->overfed += 1;
		  /* See if we should, and can feed more! */
		  if (proc->thg == NULL ||
		      proc->pid == 0    ||
		      proc->thread == NULL)
		    break;	/* No new active threads/vertices/proc.. */
		  if (proc->cmdbuf[0] != 0)
		    break;	/* it is already overstuffed */
		  if (proc->overfed >= proc->thg->ce.overfeed)
		    break;	/* if the limit is zero, don't overfeed ever.*/
		  /* Ok, increment the counter, and loop back.. */
		  proc->hungry = 1; /* Simulate hunger.. */
		  proc->fed = 1; /* tell them, it has been fed.. */
		  pick_next_vertex(proc,1);
		  /* If it got next,  ``proc->fed'' is now zero.. */
		}
		proc->hungry = 0; /* ... satiated.. */
	      } else {
		if (verbose)
		  printf("... child pid %d overfed=%d\n",
			 proc->pid,proc->overfed);
	      }
	    } else
	      if (verbose)
		printf("'#hungry' from child without forward-channel\n");
	    return;
	  } /* end of '#hungry' processing */
	  if (strncmp(diagnostic,"#resync",7)==0) {
	    /* The transporter has noticed that the scheduler
	       gave it a job spec, which does not have anything
	       left for processing, it is time for the scheduler
	       to recheck the job file. */
	    char *p, *s = diagnostic + 7;
	    while (*s == ' ' || *s == '\t') ++s;
	    p = strchr(s,'\n');
	    if (p) *p = 0; /* newline AFTER filename */
	    if (*s != 0)
	      resync_file(s);
	    return;
	  }
	  fprintf(stderr, "%s DBGdiag: %s\n", timestring(), diagnostic);
	  return;
	} /* end of debug-diagnostics processing */
	inum = atol(diagnostic);
	if ((cp = strchr(diagnostic, '/')) == NULL) {
	  fprintf(stderr, "%s Misformed diagnostic: %s\n",
		  timestring(), diagnostic);
	  return;
	}
	offset = atol(++cp);
	if ((cp = strchr(cp, '\t')) == NULL) {
	  fprintf(stderr, "%s Misformed diagnostic: %s\n",
		  timestring(), diagnostic);
	  return;
	}
	notary = ++cp;
	if ((cp = strchr(cp, '\t')) == NULL) {
	  fprintf(stderr, "%s Misformed diagnostic: %s\n",
		  timestring(), diagnostic);
	  return;
	}
	*cp = 0; /* Trailing TAB after notary string, NULL it */
	type = ++cp;
	while (*cp != '\0' && isascii(*cp) && !isspace(*cp))
	  ++cp;
	if (*cp == '\0') {
	  message = NULL;
	} else {
	  *cp++ = '\0';
	  message = cp;
	}
	if (verbose)
	  printf("diagnostic: %ld/%ld\t%s\t%s\n",inum,offset,notary,type);

	if ((vp = findvertex(inum, offset, &index)) == NULL)
	  return;

	if (vp->notary != NULL)
	  free(vp->notary);
	vp->notary = NULL;
	if (*notary)
	  vp->notary = strsave(notary);

	/* Select function by the type name: ok/error/retryat/deferred */

	for (dcp = &diags[0]; dcp->name != NULL; ++dcp) {
	  /* XX: replace strcmp() with cistrcmp() ??  Should not need,
	     unless something is wrong with the transporters. */
	  if (strcmp(dcp->name, type) == 0) {
	    (dcp->fcn)(vp, index, inum, offset, notary, message);
	    break;
	  }
	}
	if (dcp->name == NULL)
	  fprintf(stderr, "%s Unknown diagnostic type ignored: %s\n",
		  timestring(), type);
}

/*
 * Deallocate a control file descriptor.
 */

void
unctlfile(cfp, no_unlink)
	struct ctlfile *cfp;
	int no_unlink;
{
	struct spblk *spl;
	char	path[MAXPATHLEN+1];

	if (!no_unlink) {
	  reporterrs(cfp);
#ifdef	LOG_INFO
	  syslog(LOG_INFO, "%s: done", cfp->mid);
#endif	/* LOG_INFO */
	  eunlink(cfp->mid);
	  --global_wrkcnt;
	  if (verbose)
	    printf("%s: unlink %s (mid=0x%p)", cfp->logident, cfp->mid, cfp->mid);

	  sprintf(path, "../%s/%s", QUEUEDIR, cfp->mid);
	  eunlink(path);
	  if (verbose)
	    printf("   and %s/\n", path);

	  if (cfp->vfp != NULL) {
	    fseek(cfp->vfp, 0, 2);
	    fprintf(cfp->vfp,
		    "scheduler done processing %s\n", cfp->mid);
	    fclose(cfp->vfp);
	  }
	}
	spl = sp_lookup(cfp->id, spt_mesh[L_CTLFILE]);
	if (spl != NULL)
	  sp_delete(spl, spt_mesh[L_CTLFILE]);
	free_cfp_memory(cfp);
}

void unvertex(vp, justfree, ok)
	register struct vertex *vp;
	int justfree, ok;
{
	int	i, remove;

	if (vp->ngroup > 0)
	  return;

	if (vp->proc && vp->proc->vertex == vp) {
	  vp->proc->fed = 1;     /* Mark it fed just in case.. */
	  vp->proc->overfed = 0; /* .. and clear this .. */
	  /* Pick next, but don't feed it (yet)! */
	  pick_next_vertex(vp->proc, ok);
	  if (vp->proc && vp->proc->vertex == vp){
	    fprintf(stderr,
		    "unvertex(vtx=0x%p,%d,%d) failed to pick_next_vertex()!\n",
		    vp,justfree,ok);
	    abort();
	  }
	}

	for (i = 0; i < SIZE_L; ++i) {
	  if (vp->next[i] != NULL)
	    vp->next[i]->prev[i] = vp->prev[i];
	  if (vp->prev[i] != NULL)
	    vp->prev[i]->next[i] = vp->next[i];
	  if (i == L_CTLFILE)
	    continue;
	  remove = 0;
	  if (vp->orig[i]->link == vp)
	    if ((vp->orig[i]->link = vp->next[i]) == NULL)
	      remove = 1;
	  if (vp->orig[i]->lastlink == vp)
	    if ((vp->orig[i]->lastlink = vp->prev[i]) == NULL)
	      remove = 1;
	  if (remove) {
	    vp->orig[i]->linkcnt -= 1;
	    unweb(i, vp->orig[i]);
	    vp->orig[i] = NULL;
	  }
	}

	if (vp->cfp->head == vp)
	  if ((vp->cfp->head = vp->next[L_CTLFILE]) == NULL) 
	    unctlfile(vp->cfp, justfree);

	web_disentangle(vp, ok); /* does also unthread() */

	if (vp->message != NULL) free(vp->message);
	if (vp->notary  != NULL) free(vp->notary);
	/* if (vp->sender != NULL) free(vp->sender); */ /* XX: cache !! ?? */
	free((char *)vp);
}

static struct vertex *findvertex(inum, offset, idx)
	long	inum;
	long	offset;
	int	*idx;
{
	struct spblk *spl;
	struct ctlfile *cfp;
	struct vertex *vp;
	int	i;

	/* It is NOT POSSIBLE to cache cfp, based on the inum */
	spl = sp_lookup((u_long)inum, spt_mesh[L_CTLFILE]);
	if (spl == NULL || (cfp = (struct ctlfile *)spl->data) == NULL) {
	  fprintf(stderr, "%s: cannot find control file for %ld!\n",
		  progname, inum);
	  return NULL;
	}
	for (i = 0; i < cfp->nlines; ++i) {
	  if (cfp->offset[i] == offset) {
	    *idx = i;
	    break;
	  }
	}
	if (i >= cfp->nlines) {
	  fprintf(stderr,
		  "%s: unknown address offset %ld in control file %ld!\n",
		  progname, offset, inum);
	  return NULL;
	}
	for (vp = cfp->head; vp != NULL; vp = vp->next[L_CTLFILE])
	  for (i = 0; i < vp->ngroup; ++i)
	    if (vp->index[i] == *idx)
	      return vp;
	fprintf(stderr,
		"%s: multiple processing of address at %ld in control file %ld!\n",
		progname, offset, inum);
	return NULL;
}

/*
 * To implement the CF_OBSOLETES command, we need to map a message-id to
 * a control file structure, unlink the unprocessed addresses from the web,
 * and physically get rid of the message.  We also need to implement some
 * form of security here; same file ownership and error return address might
 * be a good approximation.
 */

static int ctlowner(cfp)
	struct ctlfile *cfp;
{
	char *path;
	struct stat stbuf;
	static int nope = -9;

	if (cfp->mid == NULL)
	  abort();
	path = emalloc(5+strlen(cfp->mid)+sizeof QUEUEDIR);
	sprintf(path, "../%s/%s", QUEUEDIR, cfp->mid);
	if (stat(path, &stbuf) < 0)
	  return --nope;
	free(path);
	return stbuf.st_uid;
}

void
deletemsg(msgid, curcfp)
	char *msgid;
	struct ctlfile *curcfp;
{
	struct ctlfile *cfp = NULL;
	struct spblk *spl;

	/* map message id to ctl structure */
	for (spl = sp_fhead(spt_mesh[L_CTLFILE]); spl != NULL ;
	     spl = sp_fnext(spl)) {
	  cfp = (struct ctlfile *)spl->data;
	  /* XX: message-id comparison is a Hard Problem. Approximate. */
	  if (strcmp(cfp->logident, msgid) == 0)
	    break;
	}
	if (spl == NULL)
		return;
	/* security checks */
	/* XX: address comparison is also a Hard Problem... sigh */
	if ((cfp->erroraddr == NULL && curcfp->erroraddr != NULL)
	    || (cfp->erroraddr != NULL && curcfp->erroraddr == NULL)
	    || (cfp->erroraddr != curcfp->erroraddr
		&& strcmp(cfp->erroraddr, curcfp->erroraddr) != 0))
		return;
	if (ctlowner(cfp) != ctlowner(curcfp))
		return;
	/*
	 * It might be useful to return a report about what happened, but
	 * for mailing lists this is dangerous.  Let's not, until we can
	 * test for some 'return-receipt-requested' flag.
	 */

#ifdef	LOG_INFO
	syslog(LOG_INFO, "%s: obsoleted by %s", cfp->mid, curcfp->mid);
#endif	/* LOG_INFO */

	/*
	 * unvertex() will do unctlfile() on the last vertex, hence
	 * this strange way of doing the unlink.
	 */
	while (cfp->head->next[L_CTLFILE] != NULL) {
	  cfp->head->ngroup = 0;
	  unvertex(cfp->head,0,1);
	}
	cfp->head->ngroup = 0;
	unvertex(cfp->head,0,1);
}


/* Lifted from BIND res/res_debug.c */
/*
 * Return a mnemonic for a time to live
 */
char *
saytime(value, buf, shortform)
	long value;
	char *buf;
	int shortform;
{
	int secs, mins, hours, fields = 0;
	register char *p;

	p = buf;

	while (*p) ++p;
	if (value < 0) {
	  *p++ = '-'; *p = 0;
	  value = -value;
	}

	secs = value % 60;
	value /= 60;
	mins = value % 60;
	value /= 60;
	hours = value % 24;
	value /= 24;

#define	PLURALIZE(x)	x, (x == 1) ? "" : "s"
	if (value) {
	  if (shortform)
	    sprintf(p, "%ldd", value);
	  else
	    sprintf(p, "%ld day%s", PLURALIZE(value));
	  ++fields;
	  while (*++p);
	}
	if (hours) {
	  if (shortform)
	    sprintf(p, "%dh", hours);
	  else {
	    if (value && p != buf)
	      *p++ = ' ';
	    sprintf(p, "%d hour%s", PLURALIZE(hours));
	  }
	  ++fields;
	  while (*++p);
	}
	if (mins && fields < 2) {
	  if (shortform)
	    sprintf(p, "%dm", mins);
	  else {
	    if ((hours || value) && p != buf)
	      *p++ = ' ';
	    sprintf(p, "%d min%s", PLURALIZE(mins));
	  }
	  while (*++p);
	}
	if (secs && fields < 2) {
	  if (shortform)
	    sprintf(p, "%ds", secs);
	  else {
	    if ((mins || hours || value) && p != buf)
	      *p++ = ' ';
	    sprintf(p, "%d sec%s", PLURALIZE(secs));
	  }
	  while (*++p);
	}
	*p = '\0';
	return buf;
}

/*
 * vtxupdate() -- delete the vertex matching our (sub)index
 */

static void vtxupdate(vp, index, ok)
	struct vertex *vp;
	int index, ok;
{
	int i;

	for (i = 0; i < vp->ngroup; ++i)
	  if (vp->index[i] == index) {
	    /* remove us from the vertex indices */
	    vp->ngroup -= 1;
	    /* compact the index array */
	    for (++i; i <= vp->ngroup; ++i)
	      vp->index[i-1] = vp->index[i];
	    /* if none are left, unvertex() it.. */
	    if (vp->ngroup <= 0)
	      unvertex(vp, 0, ok);
	    break;
	  }
}


static void logstat(fp,vp,reason)
	FILE *fp;
	struct vertex *vp;
	char *reason;
{
	time(&now);
	fprintf(fp,"%ld %s %ld %ld %s %s/%s\n",
		(long)vp->cfp->ctime, vp->cfp->mid,
		(long)(vp->cfp->envctime - vp->cfp->ctime),
		(long)(now - vp->cfp->envctime), reason,
		vp->orig[L_CHANNEL]->name,vp->orig[L_HOST]->name);
	fflush(fp);
}


static void expaux(vp, index, buf)
	struct vertex *vp;
	int index;
	char *buf;
{
	int i;

	/* Report expiry */
	for (i = 0 ; i < vp->ngroup; ++i)
	  if (vp->index[i] == index) {
	    msgerror(vp, vp->cfp->offset[index], buf);
	    break;
	  }

	/* Log something into the scheduler log */
	fprintf(stderr, "%s %s: %s/%s from %s %s\n", timestring(), progname,
		vp->orig[L_CHANNEL]->name, vp->orig[L_HOST]->name,
		vp->cfp->erroraddr == NULL ? "?" : vp->cfp->erroraddr, buf);

	if (statuslog)
	  logstat(statuslog,vp,"expire");

	/* Delete this vertex from scheduling datasets */
	vtxupdate(vp, index, 0);
}

void
expire(vp, index)
	struct vertex *vp;
	int index;
{
	int i;
	char *emsg, buf[BUFSIZ];
	static char *fmt = "\r%s, problem was:\r%s";

	if (vp->notary == NULL) {
	  /* addres / action / status / diagnostic / wtt */
	  sprintf(buf,"%s\001%s\001%s\001%s",
		  "\003", /* XX: recipient address! XX: MAGIC INFO! */
		  "failure",
		  "4.0.0 (unspecified timeout failure)",
		  "smtp; 400 (Expired after ");
	  saytime((u_long)(vp->ce_expiry - vp->cfp->ctime), buf, 0);
	  strcat(buf,")\001");
	  vp->notary = strsave(buf);
	}

	strcpy(buf, "expired after ");
	saytime((u_long)(vp->ce_expiry - vp->cfp->ctime), buf, 0);

	if (vp->message != NULL && *(vp->message) != '\0') {
	  emsg = emalloc(strlen(buf) + strlen(vp->message) + strlen(fmt));
	  sprintf(emsg, fmt, buf, vp->message);
	} else
	  emsg = buf;

	if (index < 0) {
	  /* Expire from the LAST index to the first, this way
	     we won't do mistake of calling indixes after they
	     have been deleted.. */
	  for (i = vp->ngroup -1; i >= 0; --i)
	    expaux(vp, vp->index[i], emsg);
	} else
	  expaux(vp, index, emsg);

	if (emsg != buf)
	  free(emsg);
}

/*ARGSUSED*/
static int u_ok(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	/* printf("%s: %d/%d/%s/ok %s\n", vp->cfp->logident, inum, offset,
	   notary, message ? message : "-"); */
	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: ok %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  message == NULL ? "(sent)" : message);
	}

	/* XX: Save/process info regarding delivery receipts! */

	if (statuslog)
	  logstat(statuslog,vp,"ok");

	/* Delete this vertex from scheduling datasets */
	vtxupdate(vp, index, 1);
	return 1;
}

static int u_deferred(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	/* printf("%s: %d/%d/%s/deferred %s\n", vp->cfp->logident,
	   inum, offset, notary, message ? message : "-"); */
	if (message != NULL) {
	  if (vp->message != NULL)
	    free(vp->message);
	  /* fprintf(stderr, "add message '%s' to node %s/%s\n",
	     message, vp->orig[L_CHANNEL]->name,
	     vp->orig[L_HOST]->name); */
	  vp->message = strsave(message);
	}

	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: deferred %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  message == NULL ? "(unknown)" : message);
	}
	/*
	 * Even though we may get several of these per web entry,
	 * the heuristic in reschedule() to ignore the request if
	 * the time is already in the future should help out.
	 */
	reschedule(vp, -1, index);
	return 1;
}

static int u_error(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	if (message == NULL)
	  message = "(unknown)";
	fprintf(stderr,
		"%s %s: %ld/%ld/%s/error %s\n", timestring(),
		vp->thgrp->ce.command, inum, offset, notary, message);

	msgerror(vp, offset, message);

	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: error %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  message);
	}

	if (statuslog)
	  logstat(statuslog,vp,"error");

	/* Delete this vertex from scheduling datasets */
	vtxupdate(vp, index, 0);
	return 1;
}

/*
 * specify relative (w/ leading +) or absolute (w/o leading +) retry time.
 */

static int u_retryat(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	time_t	retrytime;
	char	*cp;

	if (*message == '+') {
	  time(&retrytime);
	  ++message;
	} else
	  retrytime = 0;
	for (cp = message; *cp != '\0' && isdigit(*cp); ++cp)
	  continue;
	if (*cp != '\0')
	  *cp++ = '\0';
	retrytime += atoi(message);
	message = cp;
	if (*message == '\0')
	  message = NULL;
	
	if (message != NULL) {
	  if (vp->message != NULL)
	    free(vp->message);
	  /* fprintf(stderr, "add message '%s' to node %s/%s\n",
	     message, vp->orig[L_CHANNEL]->name,
	     vp->orig[L_HOST]->name); */
	  vp->message = strsave(message);
	}

	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: retryat %d %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  (int)retrytime, message == NULL ? "(unknown)" : message);
	}
	/*
	 * Even though we may get several of these per web entry,
	 * the heuristic in reschedule() to ignore the request if
	 * the time is already in the future should help out.
	 */

	/* ``vp'' might become expired by  thread_reschedule() .. */
	if (vp->proc && vp->proc->vertex == vp)
	  /* Pick next, but don't feed it (yet)! */
	  pick_next_vertex(vp->proc, 0);

	thread_reschedule(vp->thread, retrytime, index);

	return 1;
}
