/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/*
 * Interface routines for the generic database mechanism.
 *
 * This is a scheme by which one can define particular unary or binary
 * relations, so they can be used by the functions of the configuration
 * file in a manner that is independent of the particular implementation
 * or lookup mechanism. This allows flexible autoconfiguration for the
 * needs or capabilities of each host.
 */

#include "mailer.h"
#include <ctype.h>
#include <sys/file.h>
#include "libdb/search.h"

struct sptree *spt_databases;

/*
 * The following tables describes the kinds of lookups we support.
 */

typedef enum { Nul, Boolean, Pathalias, Indirect, NonNull } postprocs;

struct cache {
	u_char		*key;
	struct conscell	*value;
	time_t		expiry;
};

struct db_info {
	char		*file;			/* a file parameter */
	char		*subtype;		/* optional selector */
	int		flags;			/* miscellaneous options */
	int		cache_size;		/* default cache size */
	time_t		ttl;			/* default time to live */
	struct conscell	*(*driver)();		/* completion routine */
	struct conscell	*(*lookup)();		/* low-level lookup routine */
	void		(*close)();		/* flush buffered data, close */
	int		(*add)();		/* add key/value pair */
	int		(*remove)();		/* remove key */
	void		(*print)();		/* print database */
	void		(*owner)();		/* print database owner */
	int		(*modcheckp)();		/* should we reopen database? */
	postprocs	postproc;		/* post-lookup applicator */
	struct cache	*cache;			/* cache entry array */
};

/* bits in the flags field */
#define	DB_MAPTOLOWER	01
#define	DB_MAPTOUPPER	02
#define	DB_MODCHECK	04

/* drivers */
extern struct conscell *find_domain();
/* lookups */
extern struct conscell *search_hosts(), *search_seq(), *search_bin();
extern struct conscell *search_res(), *search_dbm(), *search_ndbm();
extern struct conscell *search_core(), *search_yp(), *search_gdbm();
extern struct conscell *search_yp(), *search_header();
/* closes */
extern void close_core(), close_seq(), close_ndbm(), close_dbm(), close_gdbm();
extern void close_header();
/* adds */
extern int add_core(), add_seq(), add_ndbm(), add_dbm(), add_gdbm();
extern int add_header();
/* removes */
extern int remove_core(), remove_ndbm(), remove_dbm(), remove_gdbm();
extern int remove_header();
/* prints */
extern void print_core(), print_hosts(), print_seq();
extern void print_ndbm(), print_dbm(), print_gdbm();
extern void print_header();
/* owners */
extern void owner_core(), owner_seq(), owner_ndbm(), owner_dbm(), owner_gdbm();
extern void owner_header();
/* modchecks */
extern int modp_seq(), modp_ndbm(), modp_gdbm();

struct db_kind {
	char		*name;		/* database type identification */
	struct db_info	config;		/* default configuration information */
} db_kinds[] = {
{ "incore",	{ NULL, NULL, 0, 0, 0, NULL, search_core, close_core,
		  add_core, remove_core, print_core, owner_core, NULL, Nul } },
{ "header",	{ NULL, NULL, 0, 0, 0, NULL, search_header, close_header,
		  add_header, remove_header, print_header, owner_header, NULL, Nul } },
{ "unordered",	{ NULL, NULL, 0, 10, 0, NULL, search_seq, close_seq,
		  add_seq, NULL, print_seq, owner_seq, modp_seq, Nul } },
{ "ordered",	{ NULL, NULL, 0, 10, 0, NULL, search_bin, close_seq,
		  NULL, NULL, print_seq, owner_seq, modp_seq, Nul }	},

#ifdef	USE_HOSTS
{ "hostsfile",	{ HOSTS_FILE, NULL, 0, 10, 0, NULL, search_hosts, NULL,
		  NULL, NULL, print_hosts, NULL, NULL, Nul } },
#endif	/* USE_HOSTS */
#ifdef	USE_RESOLV
{ "bind",	{ RESOLV_CONF, NULL, 0, 10, 0, NULL,
		  search_res, NULL, NULL, NULL, NULL, NULL, NULL, Nul }	},
#endif	/* USE_RESOLV */
#ifdef	USE_NDBM
{ "ndbm",	{ NULL, NULL, 0, 10, 0, NULL, search_ndbm, close_ndbm,
		  add_ndbm, remove_ndbm, print_ndbm, owner_ndbm,
		  modp_ndbm, Nul } },
#endif	/* USE_NDBM */
#ifdef	USE_GDBM
{ "gdbm",	{ NULL, NULL, 0, 10, 0, NULL, search_gdbm, close_gdbm,
		  add_gdbm, remove_gdbm, print_gdbm, owner_gdbm,
		  modp_gdbm, Nul } },
#endif	/* USE_GDBM */
#ifdef	USE_DBM
{ "dbm",	{ NULL, NULL, 0, 10, 0, NULL, search_dbm, close_dbm,
		  add_dbm, remove_dbm, print_dbm, owner_dbm,
		  NULL, Nul } },
#endif	/* USE_DBM */
#ifdef	USE_YP
{ "yp",		{ NULL, NULL, 0, 10, 0, NULL, search_yp, NULL,
		  NULL, NULL, NULL, NULL, NULL, Nul }	},
#endif	/* USE_YP */
{ 0 }	/* proto_config is initialized from this entry */
};

/*
 * Define a new relation according to the command line arguments.
 * (This is a built-in C-coded function.)
 */

int
run_relation(argc, argv)
	int	argc;
	char	**argv;
{
	struct db_info proto_config, *dbip;
	struct db_kind *dbkp;
	struct spblk *spl;
	int c, symid, errflg, set_cache_size, oval;
	char *cp, *dbtype;
	extern int optind, stickymem;
	extern char *optarg;
	extern struct sptree *spt_builtins, *spt_funclist;

	errflg = 0;
	dbtype = NULL;
	set_cache_size = 0;
	optind = 0;
	proto_config
		= db_kinds[sizeof db_kinds/(sizeof (struct db_kind))-1].config;
	while ((c = getopt(argc, argv, "bilmnpud:f:s:t:e:")) != EOF)
		switch (c) {
		case 'b':	/* boolean postprocessor */
			proto_config.postproc = Boolean;
			break;
		case 'd':	/* driver routine */
			if (strcmp(optarg, "pathalias") == 0)
				proto_config.driver = find_domain;
			else
				++errflg;
			break;
		case 'f':	/* file name */
			oval = stickymem;
			stickymem = MEM_PERM;
			proto_config.file = strnsave(optarg, strlen(optarg));
			stickymem = oval;
			break;
		case 'i':	/* indirect reference postprocessor */
			proto_config.postproc = Indirect;
			break;
		case 'l':	/* map all keys to lower-case */
			proto_config.flags |= DB_MAPTOLOWER;
			break;
		case 'm':
			proto_config.flags |= DB_MODCHECK;
			break;
		case 'n':	/* ensure non-null return value postprocessor */
			proto_config.postproc = NonNull;
			break;
		case 'p':	/* pathalias postprocessor */
			proto_config.postproc = Pathalias;
			break;
		case 's':	/* cache size */
			proto_config.cache_size = atoi(optarg);
			set_cache_size = 1;
			break;
		case 't':	/* database type */
			dbtype = optarg;
			if ((cp = strchr(optarg, ',')) != NULL
			    || (cp = strchr(optarg, '/')) != NULL) {
				*cp++ = '\0';
				oval = stickymem;
				stickymem = MEM_PERM;
				cp = strnsave(cp, strlen(cp));
				stickymem = oval;
			}
			proto_config.subtype = cp;
			break;
		case 'e':	/* expiry - cache data time to live */
			proto_config.ttl = atol(optarg);
			break;
		case 'u':	/* map all keys to uppercase */
			proto_config.flags |= DB_MAPTOUPPER;
			break;
		case 'F':	/* find routine */
		case 'S':	/* search routine */
		default:
			++errflg;
			break;
		}
	if (errflg || optind != argc - 1 || dbtype == NULL) {
		(void) fprintf(stderr,
		"Usage: %s -t dbtype [-f file -e# -s# -blnu -d driver] name\n",
			argv[0]);
		return 1;
	}
	if ((proto_config.flags & (DB_MAPTOLOWER|DB_MAPTOUPPER)) ==
	    (DB_MAPTOLOWER|DB_MAPTOUPPER)) {
		fprintf(stderr,
		    "relation: the -l and -u flags are mutually exclusive\n");
		return 2;
	}
	for (dbkp = &db_kinds[0];
	     dbkp < &db_kinds[(sizeof db_kinds)/sizeof (struct db_kind)];
	     ++dbkp) {
		if (dbkp->name == NULL || strcmp(dbkp->name, dbtype) == 0)
			break;
	}
	/* the -1 in the following compensates for the terminating null entry */
	if (dbkp >= &db_kinds[((sizeof db_kinds)/sizeof (struct db_kind))-1]) {
		fprintf(stderr,
			"relation: I don't know about the %s database type!\n",
			dbtype);
		return 3;
	}
	symid = symbol((u_char *)argv[optind]);
	if (sp_lookup(symid, spt_databases) != NULL) {
		fprintf(stderr, "%s: %s is already a defined database!\n",
				argv[0], argv[optind]);
		return 4;
	}
	if (spt_builtins != NULL && sp_lookup(symid, spt_builtins) != NULL) {
		fprintf(stderr, "%s: %s is already a built in function!\n",
				argv[0], argv[optind]);
		return 5;
	}
	if (spt_funclist != NULL && sp_lookup(symid, spt_funclist) != NULL) {
		fprintf(stderr, "%s: %s is already a defined function!\n",
				argv[0], argv[optind]);
		return 6;
	}
	if (dbkp->config.lookup == search_core
	    || dbkp->config.lookup == search_header) {
		/* The database name is used as a key by the incore routines */
		oval = stickymem;
		stickymem = MEM_PERM;
		proto_config.file = strnsave(argv[optind],
					     strlen(argv[optind]));
		stickymem = oval;
		/* and the subtype is used to stash the splay tree */
		proto_config.subtype = (char *)sp_init();
	} else {
		if (proto_config.file == NULL)
			proto_config.file = dbkp->config.file;
		if (proto_config.subtype == NULL)
			proto_config.subtype = dbkp->config.subtype;
	}
	if (proto_config.flags == 0)
		proto_config.flags = dbkp->config.flags;
	if (!set_cache_size || proto_config.cache_size < 0)
		proto_config.cache_size = dbkp->config.cache_size;
	if (proto_config.ttl < 0)
		proto_config.ttl = dbkp->config.ttl;
	if (proto_config.driver == NULL)
		proto_config.driver = dbkp->config.driver;
	if (proto_config.lookup == NULL)
		proto_config.lookup = dbkp->config.lookup;
	if (proto_config.close == NULL)
		proto_config.close = dbkp->config.close;
	if (proto_config.add == NULL)
		proto_config.add = dbkp->config.add;
	if (proto_config.remove == NULL)
		proto_config.remove = dbkp->config.remove;
	if (proto_config.print == NULL)
		proto_config.print = dbkp->config.print;
	if (proto_config.owner == NULL)
		proto_config.owner = dbkp->config.owner;
	if (proto_config.modcheckp == NULL)
		proto_config.modcheckp = dbkp->config.modcheckp;
	if (proto_config.postproc == Nul)
		proto_config.postproc = dbkp->config.postproc;
	dbip = (struct db_info *)smalloc(MEM_PERM, sizeof (struct db_info));
	*dbip = proto_config;
	if (dbip->cache_size > 0) {
		dbip->cache = (struct cache *)
			smalloc(MEM_PERM, (u_int)(dbip->cache_size
					* (sizeof (struct cache))));
		dbip->cache->key = NULL;	/* search terminator */
	} else
		dbip->cache = NULL;	/* superfluous, but why not ... */
	(void) sp_install(symid, (u_char *)dbip, 0, spt_databases);
	if ((spl = sp_lookup(symbol((u_char *)DBLOOKUPNAME), spt_builtins)) == NULL) {
		fprintf(stderr, "%s: '%s' isn't built in\n",
				argv[0], DBLOOKUPNAME);
		return 1;
	}
	(void) sp_install(symid, spl->data, 0, spt_builtins);
	return 0;
}

/* return the splay tree associated with the named incore database */

struct sptree *
icdbspltree(name)
	char *name;
{
	struct db_info *dbip;
	struct spblk *spl;

	if ((spl = sp_lookup(symbol((u_char *)name), spt_databases)) == NULL)
		return NULL;
	dbip = (struct db_info *)spl->data;
	return (struct sptree *)dbip->subtype;
}

/*
 * Database maintenance.
 *
 * Usage: 	db add name key value
 *		db remove name key
 *		db flush name
 *		db print name
 *		db index
 *		db owner name
 */

int
iclistdbs(spl)
	struct spblk *spl;
{
	struct db_kind *dbkp;
	struct db_info *dbip;
	struct cache *cachep;
	char *cp;
	int i;

	(void) printf("%-13s", pname(spl->key));

	dbip = (struct db_info *)spl->data;
	cp = NULL;
	for (dbkp = &db_kinds[0];
	     dbkp < &db_kinds[(sizeof db_kinds)/sizeof (struct db_kind)];
	     ++dbkp) {
		if (dbkp->config.lookup == dbip->lookup) {
			cp = dbkp->name;
			break;
		}
	}
	printf(" %s", cp);
	i = strlen(cp);
	if (dbip->lookup != search_core && dbip->lookup != search_header
	    && dbip->postproc != Indirect && dbip->subtype != NULL) {
		printf(",%s", dbip->subtype);
		i += 1 + strlen(dbip->subtype);
	}
	i = (i > 10) ? 1 : 11 - i;

	printf("%*s", i, " ");
	
	if (dbip->cache_size == 0 || dbip->cache == NULL)
		printf("%*s", 11, " ");
	else {
		for (i = 0, cachep = dbip->cache;
		     i < dbip->cache_size && cachep->key != NULL;
		     ++cachep)
			++i;
		printf("%2d/%-2d ", i, dbip->cache_size);
		if (dbip->ttl > 0)
			printf("%4d ", dbip->ttl);
		else
			printf("     ");
	}

	i = 0;
	if (dbip->flags & DB_MAPTOLOWER)
		putchar('l'), ++i;
	if (dbip->flags & DB_MAPTOUPPER)
		putchar('U'), ++i;
	switch (dbip->postproc) {
	case NonNull: putchar('n'), ++i; break;
	case Boolean: putchar('B'), ++i; break;
	case Pathalias: putchar('p'), ++i; break;
	case Indirect: putchar('@'), ++i; break;
	default: break;
	}
	if (i == 0)
		putchar('-'), ++i;
	i = (i > 2) ? 1 : 3 - i;

	printf("%*s", i, " ");
	
	if (dbip->lookup != search_core && dbip->lookup != search_header
	    && dbip->file != NULL) {
		printf("%s", dbip->file);
		if (dbip->postproc == Indirect)
			printf(" -> %s", dbip->subtype);
	}

	putchar('\n');
	return 0;
}

int
run_db(argc, argv)
	int argc;
	char *argv[];
{
	int errflag;
	char *cp;
	struct db_info *dbip;
	struct search_info si;
	struct spblk *spl;
	extern void cacheflush();

	if (argc == 2 && (argv[1][0] == 'i' || argv[1][0] == 't')) {
		/* print an index/toc of the databases */
		sp_scan(iclistdbs, (struct spblk *)NULL, spt_databases);
		return 0;
	}

	errflag = 0;
	if (argc == 1)
		errflag = 1;
	else
		switch (argv[1][0]) {
		case 'a':	errflag = (argc != 5); break;
		case 'd':
		case 'r':	errflag = (argc != 4); break;
		case 'f':
		case 'n':
		case 'o':
		case 'p':	errflag = (argc != 3); break;
		default:	errflag = 1; break;
		}
	if (errflag) {
		(void) fprintf(stderr,
"Usage: %s { add|remove|flush|owner|print|toc } [ database [ key [ value ] ] ]\n",
			argv[0]);
		return 1;
	}

	if ((spl = sp_lookup(symbol((u_char *)argv[2]), spt_databases)) == NULL) {
		fprintf(stderr, "%s: unknown database \"%s\"!\n",
				argv[0], argv[2]);
		return 2;
	}
	dbip = (struct db_info *)spl->data;
	if (dbip == NULL) {
		fprintf(stderr, "%s: null database definition for \"%s\"!\n",
				argv[0], argv[2]);
		return 3;
	}

	if (argv[3] != NULL) {
		if (dbip->flags & DB_MAPTOLOWER) {
			for (cp = argv[3]; *cp != NULL; ++cp)
				if (isascii(*cp) && isupper(*cp))
					*cp = tolower(*cp);
		} else if (dbip->flags & DB_MAPTOUPPER) {
			for (cp = argv[3]; *cp != NULL; ++cp)
				if (isascii(*cp) && islower(*cp))
					*cp = toupper(*cp);
		}
	}

	si.file = dbip->file;
	si.key = (u_char *)argv[3];
	si.subtype = dbip->subtype;
	si.ttl = dbip->ttl;

	switch (argv[1][0]) {
	case 'a':	/* add db key value */
		if (dbip->add == NULL) {
			fprintf(stderr, "%s: %s: no add capability!\n",
					argv[0], argv[2]);
			return 1;
		}
		if ((*dbip->add)(&si, argv[4]) == EOF) {
			fprintf(stderr, "%s: %s: didn't add (\"%s\",\"%s\")!\n",
					argv[0], argv[2], argv[3], argv[4]);
			return 1;
		}
		break;
	case 'd':	/* delete db key */
	case 'r':	/* remove db key */
		if (dbip->remove == NULL) {
			fprintf(stderr, "%s: %s: no remove capability!\n",
					argv[0], argv[2]);
			return 1;
		}
		if ((*dbip->remove)(&si) == EOF) {
			fprintf(stderr, "%s: %s: didn't remove \"%s\"!\n",
					argv[0], argv[2], argv[3]);
			return 1;
		}
		break;
	case 'n':	/* null/nuke db */
	case 'f':	/* flush */
		cacheflush(dbip);
		if (dbip->close == NULL) {
			fprintf(stderr, "%s: %s: no flush capability!\n",
					argv[0], argv[2]);
			return 1;
		}

		/*
		 * Note that the close/flush routine should not remove the
		 * file/db name entry from the splay tree, because there may
		 * be multple references to it.  For example: /dev/null.
		 */
		(*dbip->close)(&si);

		/*
		 * Close auxiliary (indirect) data file as well (aliases.dat).
		 * There may be a subtle assumption here that the innards of
		 * the close routine only cares about si.file...
		 */
		if (dbip->postproc == Indirect) {
			si.file = dbip->subtype;
			(*dbip->close)(&si);
		}
		break;
	case 'o':	/* owner */
		if (dbip->owner == NULL) {
			fprintf(stderr,
				"%s: %s: no ownership information available!\n",
				argv[0], argv[2]);
			return 1;
		}
		(*dbip->owner)(&si, stdout);
		break;
	case 'p':	/* print db */
		if (dbip->print == NULL) {
			fprintf(stderr, "%s: %s: no printing capability!\n",
					argv[0], argv[2]);
			return 1;
		}
		(*dbip->print)(&si, stdout);
		break;
	default:
		fprintf(stderr, "%s: unknown command '%s'\n", argv[0], argv[1]);
		return 5;
	}
	return 0;
}


#define	CACHE(x,f)	((dbip->cache+(x))->f)

/*
 * This is the basic interface to the database lookup routines.
 * It uses the configuration information for each relation properly,
 * implements caching, etc.
 */

struct conscell *
db(dbname, key)
	u_char *dbname, *key;
{
	register int i, j;
	int oval;
	struct conscell *l, *ll, *tmp;
	struct spblk *spl;
	struct db_info *dbip;
	u_char *cp, *realkey;
	struct search_info si;
	struct cache tce;
	u_char kbuf[BUFSIZ];	/* XX: */
	extern int stickymem, D_db, deferit;
	extern time_t now;
	extern struct conscell *readchunk();
	extern void freeTokens(), cacheflush();
	extern int cistrcmp();

	if (key == NULL || *key == '\0') {
		(void) fprintf(stderr,
			       "Null key looked up in %s relation!\n", dbname);
		return NULL;
	}
	/* Is caching dbip worth it? I'm not so sure */
	spl = sp_lookup(symbol(dbname), spt_databases);
	dbip = (struct db_info *)spl->data;
	if (dbip == NULL) {
		fprintf(stderr, "Undefined database %s!\n", dbname);
		return NULL;
	}
	if (D_db)
		fprintf(stderr, "%s(%s)\n", dbname, key);
	/* apply flags */
	realkey = NULL;
	if (dbip->flags & DB_MAPTOLOWER) {
		for (cp = kbuf, realkey = key; *realkey; ++realkey) {
			if (isascii(*realkey) && isupper(*realkey))
				*cp++ = tolower(*realkey);
			else
				*cp++ = *realkey;
		}
		*cp = '\0';
		key = kbuf;
	} else if (dbip->flags & DB_MAPTOUPPER) {
		for (cp = kbuf, realkey = key; *realkey; ++realkey) {
			if (isascii(*realkey) && islower(*realkey))
				*cp++ = toupper(*realkey);
			else
				*cp++ = *realkey;
		}
		*cp = '\0';
		key = kbuf;
	}
	si.file = dbip->file;
	si.key = key;
	si.subtype = dbip->subtype;
	si.ttl = dbip->ttl;
	if ((dbip->flags & DB_MODCHECK)
	    && dbip->modcheckp != NULL && (*dbip->modcheckp)(&si)) {
		if (dbip->close != NULL) {
			cacheflush(dbip);
			(*dbip->close)(&si);
			if (dbip->postproc == Indirect) {
				si.file = dbip->subtype;
				(*dbip->close)(&si);
				si.file = dbip->file;
			}
		}
	}
	/* look for the desired result in the cache first */
	oval = j = 0;
	if (dbip->cache_size > 0) {
		for (i=j=0; i < dbip->cache_size && CACHE(i,key) != NULL; ++i) {
			if (CACHE(i,expiry) > 0 && CACHE(i,expiry) < now) {
				if (D_db)
					fprintf(stderr,
						"... expiring %s from cache\n",
						CACHE(i,key));
				if ((ll = CACHE(i,value)) != NULL)
					s_free_tree(ll);
				if (CACHE(i,key) != NULL) /* always true */
					(void) free((char *)CACHE(i,key));
				continue;
			}
			if (i > j) { /* expiry */
				*(dbip->cache+j) = *(dbip->cache+i);
				if (D_db)
					fprintf(stderr,
						"... setting cache[%d]: %s\n",
						j, CACHE(j,key));
			}
			if (D_db)
				fprintf(stderr,
					"... comparing %s and %s in cache\n",
					CACHE(i,key), key);
			if (CISTREQ(CACHE(i,key), key)) {
				/* cache maintenance */
				tce = *(dbip->cache+i);
				if (j > 0 && D_db)
					fprintf(stderr,
						"... ripple down from %d\n",j);
				/* ripple down to beginning */
				for (oval = j; j > 0; --j)
					*(dbip->cache+j) = *(dbip->cache+j-1);
				*(dbip->cache) = tce;
				if (i > oval) {	/* ripple down to end */
					for (j = ++oval, ++i;
					     i < dbip->cache_size
					      && CACHE(i,key) != NULL;
					     ++i, ++j)
						*(dbip->cache+j) =
							*(dbip->cache+i);
					CACHE(j,key) = NULL;
					if (D_db)
						fprintf(stderr,
						  "... ripple up to %d of %d\n",
						  i, j);
				}
				if (D_db)
					fprintf(stderr, "... found in cache\n");
				/* return a scratch value */
				return s_copy_tree(dbip->cache->value);
			}
			++j;
		}
		if (j < i)	/* stomp on expired entry */
			CACHE(j,key) = NULL;	/* mark end of cache entries */
		oval = stickymem;	/* cannot return before this is reset */
		stickymem = MEM_MALLOC;	/* this is reset down below */
		/* key gets clobbered somewhere, so save it here */
		realkey = (u_char *)strnsave((char *)key, strlen((char *)key));
	}
	l = NULL;
	if ((dbip->driver == NULL && (l = (*dbip->lookup)(&si)) != NULL)
	    || (dbip->driver != NULL
		&& (l = (*dbip->driver)(dbip->lookup, &si)))) {

		switch (dbip->postproc) {
		case Boolean:
			if (stickymem == MEM_MALLOC)
				s_free_tree(l);
			l = newstring((u_char *)strnsave((char *)key,
							 strlen(key)));
			break;
		case NonNull:
			if (STRING(l) && *(l->string) == '\0') {
				if (stickymem == MEM_MALLOC)
					s_free_tree(l);
				l = newstring((u_char *)strnsave((char *)key,
								 strlen(key)));
			} else if (LIST(l) && car(l) == NULL) {
				car(l) = newstring((u_char *)
							strnsave((char *)key,
							      strlen(key)));
			}
			break;
		case Indirect:
			/*
			 * If we got anything, it should be a byte offset into
			 * the file named by the subtype.  Used for aliases.
			 */
			if (LIST(l) || !isascii(*(l->string))
				    || !isdigit(*(l->string))) {
				if (stickymem == MEM_MALLOC)
					s_free_tree(l);
				l = NULL;
				break;
			}
			i = atoi(l->string);	/* file offset */
			if (stickymem == MEM_MALLOC)
				s_free_tree(l);
			l = readchunk(dbip->subtype, (long)i);
			break;
		case Pathalias:
			/* X: fill this out */
			break;
		default:
			break;
		}
		if (D_db) {
			fprintf(stderr, "%s(%s) = ", dbname, key);
			s_grind(l, stderr);
			putc('\n', stderr);
		}
	} else if (dbip->postproc == NonNull) {
		if (D_db)
			fprintf(stderr, "%s(%s) = %s\n", dbname, key, key);
		l = newstring((u_char *)strnsave((char *)key, strlen(key)));
	} else {
		if (dbip->cache_size > 0) {
			stickymem = oval;
			(void) free((char *)realkey);
		}
		return NULL;
	}
	if (!deferit && dbip->cache_size > 0) {
		i = j;
		/* insert new cache entry at head of cache */
		if (i >= dbip->cache_size) {
			i = dbip->cache_size - 1;
			if (D_db)
				fprintf(stderr, "... freeing cache[%d]: %s\n",
						i, CACHE(i,key));
			if ((ll = CACHE(i,value)) != NULL)
				s_free_tree(ll);
			(void) free((char *)CACHE(i,key));
		} else if (i < dbip->cache_size - 1) {
			if (D_db)
				fprintf(stderr, "... terminating cache[%d]\n",
						i);
			CACHE(i,key) = NULL;		/* terminator */
			++i;
		}
		/* ripple old entries up */
		while (--i >= 0)
			*(dbip->cache+i+1) = *(dbip->cache+i);
		dbip->cache->key = realkey;
		dbip->cache->value = l;
		if (D_db)
			fprintf(stderr, "... added %s to cache", realkey);
		if (si.ttl > 0) {
			dbip->cache->expiry = now + si.ttl;
			if (D_db)
				fprintf(stderr, " (ttl=%ld)\n", si.ttl);
		} else {
			dbip->cache->expiry = 0;
			if (D_db)
				fprintf(stderr, "\n", si.ttl);
		}
		stickymem = oval;
		ll = s_copy_tree(l);	/* scratch version returned to shell */
	} else if (dbip->cache_size > 0) {
		stickymem = oval;
		ll = s_copy_tree(l);	/* scratch version returned to shell */
		(void) free((char *)realkey);
		s_free_tree(l);
	} else
		ll = l;

	return ll;
}

char *
dbfile(dbname)
	u_char *dbname;
{
	struct db_info *dbip;
	struct spblk *spl;

	spl = sp_lookup(symbol(dbname), spt_databases);
	dbip = (struct db_info *)spl->data;
	if (dbip == NULL) {
		(void) fprintf(stderr, "Undefined database '%s'!\n", dbname);
		return NULL;
	}
	return dbip->file;
}

/*
 * Flush all cache entries from a database definition.
 */

void
cacheflush(dbip)
	struct db_info *dbip;
{
	int i;
	struct conscell *ll;

	if (dbip == NULL || dbip->cache_size == 0 | dbip->cache == NULL)
		return;

	/* flush cache */
	for (i = 0; i < dbip->cache_size && CACHE(i,key) != NULL; ++i) {
		if ((ll = CACHE(i,value)) != NULL)
			s_free_tree(ll);
		if (CACHE(i,key) != NULL) /* always true */
			(void) free((char *)CACHE(i,key));
	}
	dbip->cache->key = NULL;	/* ensure terminator */
}

#ifdef	MALLOC_TRACE

/*
 * Flush everything; back to original state
 */

void
_sptdbreset(spl)
	struct spblk *spl;
{
	char *av[4];
	extern char *pname();

	av[0] = "db"; av[1] = "flush"; av[2] = pname((u_int)(spl->key));
	av[3] = NULL;
	printf("flushing cache of %s\n", av[2]);
	run_db(3, av);
}

void
dbfree()
{
	extern struct sptree *spt_funclist;

	sp_scan(_sptdbreset, (struct spblk *)NULL, spt_databases);
}
#endif	/* MALLOC_TRACE */

char *
dbtype(dbname)
	u_char *dbname;
{
	struct db_info *dbip;
	struct db_kind *dbkp;
	struct spblk *spl;

	spl = sp_lookup(symbol(dbname), spt_databases);
	if (spl == NULL) {
		(void) fprintf(stderr, "Undefined database '%s'!\n", dbname);
		return NULL;
	}
	dbip = (struct db_info *)spl->data;
	if (dbip == NULL) {
		(void) fprintf(stderr, "Undefined database '%s'!\n", dbname);
		return NULL;
	}
	for (dbkp = &db_kinds[0];
	     dbkp < &db_kinds[(sizeof db_kinds)/sizeof (struct db_kind)];
	     ++dbkp) {
		if (dbkp->config.lookup == dbip->lookup)
			return dbkp->name;
	}
	return NULL;
}

/*
 * This routine is good for looking up foo.bar.edu in e.g. a gateway list.
 * It is typically used for pathalias database lookup.
 */

struct conscell *
find_domain(lookupfn, sip)
	struct conscell *(*lookupfn)();
	struct search_info *sip;
{
	register u_char *cp;
	struct conscell *l;
	u_char buf[BUFSIZ], *realkey;

	/* check the key as given */
	if ((l = (*lookupfn)(sip)) != NULL)
		return l;
	/* iterate over the subdomains of the key */
	for (cp = realkey = sip->key; *cp;) {
		while (*cp && *cp != '.')
			++cp;
		while (*cp == '.')
			++cp;
		if (*(cp-1) == '.') {
			sip->key = cp-1;
			if ((l = (*lookupfn)(sip)) != NULL)
				return l;
		}
	}
	/* if all else failed, try prepending a dot and look for subdomains */
	if (*realkey != '.' && strlen(realkey) < sizeof buf - 1) {
		buf[0] = '.';
		(void) strcpy((char *)&buf[1], realkey);
		sip->key = buf;
		if ((l = (*lookupfn)(sip)) != NULL)
			return l;
	}
	return NULL;
}

struct conscell *
readchunk(file, offset)
	char *file;
	long offset;
{
	FILE *fp;
	register unsigned char *cp;
	unsigned char *as;
	struct conscell *tmp, *l;
	struct spblk *spl;
	int retry, symid, flag, len;
	char buf[BUFSIZ];
	extern int deferit;
	extern void v_set();

	if (file == NULL || offset < 0)
		return NULL;
	
	retry = 0;
	symid = symbol((u_char *)file);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (fp = (FILE *)spl->data) == NULL) {
reopen:
		fp = fopen(file, "r");
		if (fp == NULL) {
			++deferit;
			v_set(DEFER, DEFER_IO_ERROR);
			fprintf(stderr, "search_seq: cannot open %s!\n", file);
			return NULL;
		}
		if (spl == NULL)
			sp_install(symid, (u_char *)fp, O_RDONLY, spt_files);
		else
			spl->data = (u_char *)fp;
	}

	if (fseek(fp, (long)offset, 0) != 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		(void) fprintf(stderr,
			"indirect postprocessor: bad seek (%ld) on '%s'!\n",
			(long)offset, file);
		return NULL;
	}

	len = 0;
	flag = 0;
	buf[sizeof buf - 1] = '\0';
	while (fgets(buf, sizeof buf, fp) != NULL) {
		/* tab and space are valid continuation characters */
		if (flag && buf[0] != '\t' && buf[0] != ' ')
			break;
		if (buf[sizeof buf - 1] == '\0')
			len += strlen(buf)-1;
		else if (buf[sizeof buf - 1] == '\n')
			len += sizeof buf - 1;
		else
			len += sizeof buf;
		flag = 1;
	}

	if (fseek(fp, (long)offset, 0) != 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr,
			"indirect postprocessor: bad seek (%ld) on '%s'!\n",
			(long)offset, file);
		return NULL;
	}

	cp = as = (u_char *)tmalloc(len+1);
	l = NULL;
	flag = 0;
	buf[sizeof buf - 1] = '\0';
	while (fgets(buf, sizeof buf, fp) != NULL) {
		/* printaliases (actually hdr_print()) prints lines < 80 char */

		if (buf[0] == '\t')
			buf[0] = ' ';
		else if (flag && buf[0] != ' ')
			break;

		if (buf[sizeof buf - 1] == '\0') {
			len = strlen(buf)-1;
			strncpy(cp, buf, len);
			cp += len;
		} else if (buf[sizeof buf - 1] == '\n') {
			strncpy(cp, buf, sizeof buf - 1);
			cp += sizeof buf - 1;
		} else {
			strncpy(cp, buf, sizeof buf);
			cp += sizeof buf;
		}
		flag = 1;
	}
	if (cp > as) {
		*cp = 0;
		l = newstring(as);
	}
	if (!retry && ferror(fp)) {
		if (l != NULL) {
			if (stickymem == MEM_MALLOC)
				s_free_tree(l);
			l = NULL;
		}
		(void) fclose(fp);
		++retry;
		goto reopen;
	}

	return l;
}
