/*
#ident	"@(#)smail/src:RELEASE-3_2_0_120:addr.c,v 1.69 2004/08/27 06:46:17 woods Exp"
 */

/*
 *    Copyright (C) 1987, 1988 Ronald S. Karr and Landon Curt Noll
 *    Copyright (C) 1992  Ronald S. Karr
 * 
 * See the file COPYING, distributed with smail, for restriction
 * and warranty information.
 */

/*
 * addr.c:
 *	routines to parse addresses
 *
 *	external functions:  preparse_address, preparse_address_1, parse_address,
 *			     mixed_address, build_uucp_route,
 *			     build_partial_uucp_route,
 *			     strip_rfc822_comments, strip_rfc822_whitespace,
 *			     address_token, back_address_token, alloc_addr,
 *			     insert_addr_list, remove_addr, addr_sort, addr_cmp
 *			     note_error, free_error
 */

#include "defs.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif

#ifdef HAVE_STRING_H
# if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#  include <memory.h>
# endif
# include <string.h>
#endif
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif

#ifdef __STDC__
# include <stdarg.h>
#else
# include <varargs.h>
#endif

#include <pcre.h>

#include "smail.h"
#include "alloc.h"
#include "list.h"
#include "smailsock.h"
#include "main.h"
#include "config.h"
#include "parse.h"
#include "addr.h"
#include "field.h"
#include "route.h"
#include "direct.h"
#include "smailstring.h"
#include "dys.h"
#include "match.h"
#include "exitcodes.h"
#include "transport.h"
#include "smailconf.h"
#include "bindlib.h"
#include "transports/smtplib.h"			/* rfc821_is_*() decls. */
#include "debug.h"
#include "extern.h"
#include "smailport.h"

/* functions local to this file */
static int check_target_and_remainder __P((char **, char **, int));
static char *escaped __P((char *, char *));
static char *internal_build_uucp_route __P((char *, char **, int, int));
static int addrcmp __P((const void *, const void *));


/*
 * preparse_address - do preliminary parsing that might be needed for address
 *
 * this routine should be used when an address is first extracted from a
 * source.  It transforms some mutant addressing forms into something more
 * managable.
 *
 * Transformations:
 *
 *	<string>		becomes just  string (recursively)
 *	host!(host!)*@route	becomes a pure !-route
 *
 * NOTE:  We don't handle @route:host!(host!)*@route, for now.  Maybe later.
 *
 * input:
 *	address	- address to be preparsed
 *	error	- error message
 *
 * output:
 *	parsed address, or NULL for parsing error, message returned in error
 *	output is guarranteed to not be a pointer to the input
 */
char *
preparse_address(address, error)
    char *address;			/* address to be preparsed */
    char **error;			/* return error message here */
{
    char *ignore;

    return preparse_address_1(address, error, &ignore);
}

char *
preparse_address_1(address, error, rest)
    char *address;
    char **error;
    char **rest;
{
    register char *ap;			/* temp for scanning address */
    char *mark_start = NULL;		/* marked position of < */
    char *mark_end = NULL;		/* marked position of > */
    int nest_cnt = 0;			/* nesting count for angle brackets */

    DEBUG1(DBG_ADDR_HI, "preparse_address(%v) entry:\n", address);
    /*
     * scan for < and > pairs and find the last or innermost matching
     * pair.
     */
    for (ap = address; ap && *ap; ap = address_token(ap)) {
	if (*ap == '<') {
	    nest_cnt++;
	    mark_start = ap + 1;
	    mark_end = NULL;
	} else if (*ap == '>') {
	    nest_cnt--;
	    if (mark_end == NULL) {
		mark_end = ap;
	    }
	}
    }
    if (ap == NULL) {
	*error = "bad address token";
	DEBUG1(DBG_ADDR_LO,
	       "preparse_address found error %s: returns (null)\n",
	       *error);
	return NULL;
    }
    if (mark_start && mark_end == NULL) {
	/* hmm, no match for the < token */
	*error = "no match for `<' in address";
	DEBUG1(DBG_ADDR_LO,
	       "preparse_address found error %s: returns (null)\n",
	       *error);
	return NULL;
    }
    if (nest_cnt != 0) {
	if (nest_cnt < 0) {
	    *error = "no match for > in address";
	} else {
	    *error = "no match for < in address";
	}
	DEBUG1(DBG_ADDR_LO,
	       "preparse_address found error %s: returns (null)\n",
	       *error);
	return NULL;
    }
    /* narrow to the route-addr */
    if (mark_end) {
	*mark_end = '\0';
	address = ap = mark_start;
    }

    /*
     * now search for the mutant form: path!@route-addr
     */
    if (*ap == '@') {
	/* valid route-addr, not a mutant one */
	ap = xmalloc((size_t) (strlen(address) + 1));
	strcpy(ap, address);
	if (mark_end) {
	    *mark_end++ = '>';		/* widen the original address */
	}
	DEBUG1(DBG_ADDR_HI, "preparse_address returns: %s\n", ap);
	*rest = mark_end;
	return ap;			/*  no transformations */
    }

    while (*ap) {
	ap = address_token(ap);
	if (ap == NULL) {
	    *error = "bad address token";
	    DEBUG1(DBG_ADDR_LO,
		   "preparse_address found error %s: returns (null)\n",
		   *error);
	    return NULL;
	}
	if (*ap != '!') {
	    ap = xmalloc((size_t) (strlen(address) + 1));
	    strcpy(ap, address);
	    if (mark_end) {
		*mark_end++ = '>';	/* widden the original address */
	    }
	    DEBUG1(DBG_ADDR_HI, "preparse address returns: %v\n", ap);
	    *rest = mark_end;
	    return ap;		/* address should be okay */
	}
	ap++;
	if (*ap == '@') {
	    /* matched host!(host!)*@route -- build the !-route */
	    register char *p = xmalloc((size_t) strlen(address));
	    DEBUG(DBG_ADDR_MID, "found host!(host!)*@route form--ugh!\n");
	    /* first part already !-route */
	    strncpy(p, address, (size_t) (ap - address));
	    if (mark_end) {
		*mark_end++ = '>';	/* widden the original address */
	    }
	    ap = build_uucp_route(ap, error, 0); /* build !-route */
	    if (ap == NULL) {
		DEBUG1(DBG_ADDR_LO,
		       "preparse_address(): build_uucp_route() failed: %s: returns: (null)\n",
		       *error);
		return NULL;
	    }
	    strcat(p, ap);		/* concatenate together */
	    xfree(ap);
	    DEBUG1(DBG_ADDR_HI, "preparse_address returns: %v\n", p);
	    *rest = mark_end;
	    return p;			/* transformed */
	}
    }
    ap = xmalloc((size_t) (strlen(address) + 1));
    strcpy(ap, address);
    if (mark_end) {
	*mark_end++ = '>';	/* widden the original address */
    }
    DEBUG1(DBG_ADDR_HI, "preparse address returns: %v\n", ap);
    *rest = mark_end;
    return ap;				/* no transformations */
}


/*
 * parse_address - destructively "extract" a target and remainder from an address
 *
 * using the rules in section 3.2 of the mailer.design document,
 * extract a target and a remainder from an address.
 *
 * The target is defined as the first destination host in an address,
 * the remainder is defined as the remaining parat of the address
 * after extracting the target.
 *
 * A short form of the rules for extraction is the following table
 * of addressing forms in order of lowest to highest precedence:
 *
 *	+---------------------------------------------------------------+
 *	| form			| description		| return	|
 *	|-----------------------|-----------------------|---------------|
 *	| @target,remainder	| route from route-addr	| RFC_ROUTE	|
 *	| @target:remainder	| route from route-addr	| RFC_ENDROUTE	|
 *	| remainder@target	| standard mailbox	| MAILBOX	|
 *	| target!remainder	| UUCP !-route		| UUCP_ROUTE	|
 *	| remainder%target	| obsolete mailbox hack	| PCT_MAILBOX	|
 *	| remainder		| local address form	| LOCAL		|
 *	+---------------------------------------------------------------+
 * If USE_BERKNET or USE_DECNET are defined:
 *	+---------------------------------------------------------------+
 *	| target::remainder	| decnet route		| DECNET	|
 *	| target:remainder	| obsolete berkenet	| BERKNET	|
 *	+---------------------------------------------------------------+
 *
 * The precedence of the % and ! operators can be switched for
 * addresses of the form a!b%c@d.  This switch will happen if the
 * variable switch_percent_and_bang is TRUE.
 *
 * inputs:
 *	address	- string containing the address to be parsed
 *	target	- where to store pointer to computed destination host
 *	remainder - where to store pointer to computed remainder (or error)
 *
 * outut:
 *	return the address form as described in the above table.  Also,
 *	return in target a pointer to to the target and return in
 *	remainder a pointer to the remainder.  If an error is detected
 *	return FAIL and load the remainder with an error message.
 *	If target is NULL, then only a form is returned, a target and
 *	remainder are not returned, though an error message may still
 *	be loaded into remainder.
 *
 * in-out:
 *	*flagp - flagp is used to maintain state between invocations
 *		 of parse_address() that are used to parse successive
 *		 remainder components.  It is used to manage the
 *		 variant rules used for RFC1123 compliance for the %
 *		 operator in the presense of a user@host address.
 *
 *		 When parse_address() is called to parse a complete
 *		 address, *flagp should be 0.  If parse_address is
 *		 used (perhaps successively) to parse generated
 *		 remainder strings, then the previous *flagp value should
 *		 be re-passed.  FOUND_MAILBOX will be or'd into *flagp
 *		 if a user@host form is encountered, in which case further
 *		 parses of remainder addresses may use the RFC1123
 *		 precedence interpretation of the % operator.
 *
 * NOTE:  address will be modified unless it is in local form, or
 *	  unless an error occurs.
 *
 * calls: address_token, back_address_token
 * called by: build_uucp_route
 */
int
parse_address(address, target, remainder, flagp)
    char *address;			/* address to parse (destructively) */
    char **target;			/* store pointer to target host here */
    char **remainder;			/* store pointer to remainder here, or error msg txt on failure */
    int *flagp;				/* flag passed between invocations */
{
    char *ep;				/* pointer to end of address */
    register char *last_tokens;		/* start of second to last token */
    register char *ap;			/* pointer for scanning address */
    register char *p;			/* temp */
    int switch_flag;

    DEBUG1(DBG_ADDR_HI, "parse_address called: address=%v\n", address);

    if (target) {
	*target = NULL;
    }
    /*
     * make sure we have an address
     */
    ap = address;
    if (*ap == '\0') {
	/* nothing to do with a zero-length address */
	*remainder = "(null address)";
	DEBUG1(DBG_ADDR_MID, "parse_address: %s\n", *remainder);
	return FAIL;
    }

    switch_flag = flagp && *flagp & FOUND_MAILBOX && switch_percent_and_bang;

    /*
     * does the address begin with @target[,:] ?
     */
    if (*ap == '@') {
	if (target) {
	    *target = ap + 1;			/* mark the target */
	}
	ap = address_token(ap + 1);		/* skip target */
	if (ap == NULL) {
	    *remainder = "bad address token";
	    DEBUG1(DBG_ADDR_MID, "parse_address: %s\n", *remainder);
	    return FAIL;
	}

	/* ensure that the `,' or `:' is in the address */
	if (!ap) {
	    /* interesting, address just contained '@' */
	    *remainder = "syntax error:  no target host";
	    DEBUG1(DBG_ADDR_MID, "parse_address: %s\n", *remainder);
	    return FAIL;
	}
	if (*ap == ',' || *ap == ':') {
	    int retval = (*ap == ',' ? RFC_ROUTE : RFC_ENDROUTE);

	    if (target) {
		*ap++ = '\0';			/* NUL-terminate target */
		*remainder = ap;
		if (check_target_and_remainder(target, remainder, retval) == FAIL) {
		    return FAIL;
		}
		DEBUG3(DBG_ADDR_HI,
		       "parse_address: %s: target=%v, remainder=%v\n",
		       retval == RFC_ROUTE ? "RFC_ROUTE" : "RFC_ENDROUTE",
		       *target, *remainder);
	    } else {
		DEBUG(DBG_ADDR_HI, "parse_address: RFC_ROUTE.\n");
	    }
	    return retval;
	}
	/* we have a syntax error, missing `,' or `:' */
	*remainder = "syntax error: , or : missing in route-addr";
	DEBUG1(DBG_ADDR_MID, "parse_address: %s\n", *remainder);
	return FAIL;
    }

    /*
     * is the address a standard mailbox ?
     * i.e., does the address end in @target ?
     */
    ep = address + strlen(address);
    last_tokens = back_address_token(ap, ep);
    if (last_tokens && last_tokens > ap) {
	last_tokens = back_address_token(ap, last_tokens);
    }
    if (last_tokens == NULL) {
	*remainder = "bad address token";
	DEBUG1(DBG_ADDR_MID, "parse_address: %s\n", *remainder);
	return FAIL;
    }
    if (last_tokens > ap && *last_tokens == '@') {
	/*
	 * it matches @token, null terminate the remainder and finish up;
	 * also set FOUND_MAILBOX to turn on RFC1123-compliant parsing
	 * of %
	 */
	if (flagp) {
	    *flagp |= FOUND_MAILBOX;
	}
	if (target) {
	    *last_tokens = '\0';	/* NUL-terminate previous tokens */
	    *target = last_tokens+1;
	    *remainder = ap;
	    if (check_target_and_remainder(target, remainder, MAILBOX) == FAIL) {
		return FAIL;
	    }
	    DEBUG2(DBG_ADDR_HI,
		   "parse_address: MAILBOX: target=%v, remainder=%v\n",
		   *target, *remainder);
	} else {
	    DEBUG(DBG_ADDR_HI, "parse_address: MAILBOX\n");
	}
	return MAILBOX;
    }

    /*
     * HACK!!  goto percent processing if we are using RFC1123-compliant
     * % parsing
     */

    if (switch_flag) {
	goto switch_order_percent;
    }
 switch_order_bang:
    /*
     * is the address a UUCP !-route ?
     * i.e., does the address begin with target! ?
     */
    p = address_token(ap);
    if (p && *p == '!') {
	/* it matches target!, null terminate target and finish up */
	if (target) {
	    *p = '\0';
	    *target = ap;
	    *remainder = p+1;
	    if (check_target_and_remainder(target, remainder, UUCP_ROUTE) == FAIL) {
		return FAIL;
	    }
	    DEBUG2(DBG_ADDR_HI,
		   "parse_address: UUCP_ROUTE: target=%v, remainder=%v\n",
		   *target, *remainder);
	} else {
	    DEBUG(DBG_ADDR_HI, "parse_address: UUCP_ROUTE\n");
	}
	return UUCP_ROUTE;
    }

    /*
     * is the address a BERKENET or DECNET syntax?
     */
#if defined(USE_DECNET) || defined(USE_BERKENET)
    if (p && *p == ':') {
# if defined(USE_DECNET)
	if (*(p + 1) == ':') {
	    /* DECNET syntax */
	    if (target) {
		*p = '\0';
		*target = ap;
		*remainder = p + 2;
		if (check_target_and_remainder(target, remainder, DECNET) == FAIL) {
		    return FAIL;
		}
		DEBUG2(DBG_ADDR_HI,
		       "parse_address: DECNET: target=%v, remainder=%v\n",
		       *target, *remainder);
	    } else {
		DEBUG(DBG_ADDR_HI, "parse_address: DECNET\n");
	    }
	    return DECNET;
	}
# endif /* USE_DECNET */
# if defined(USE_BERKENET)
	/* Berkenet syntax */
	if (target) {
	    *p = '\0';
	    *target = ap;
	    *remainder = p + 1;
	    if (check_target_and_remainder(target, remainder, BERKNET) == FAIL) {
		return FAIL;
	    }
	    DEBUG2(DBG_ADDR_HI,
		   "parse_address: BERKENET: target=%v, remainder=%v\n",
		   *target, *remainder);
	} else {
	    DEBUG(DBG_ADDR_HI, "parse_address: BERKENET\n");
	}
	return BERKENET;
# endif /* USE_BERKENET */
    }
#endif /* USE_DECNET || USE_BERKENET */

    if (switch_flag) {
	goto switch_order_local;
    }
 switch_order_percent:
    /*
     * is the address a non-standard mailbox ?
     * i.e., does the address end in %target ?
     */
    if (last_tokens && last_tokens - ap > 0 && *last_tokens == '%') {
	/* it matches @target, null terminate the remainder and finish up */
	if (target) {
	    *last_tokens = '\0';
	    *target = last_tokens+1;
	    *remainder = ap;
	    if (check_target_and_remainder(target, remainder, PCT_MAILBOX) == FAIL) {
		return FAIL;
	    }
	    DEBUG2(DBG_ADDR_HI,
		   "parse_address: PCT_MAILBOX: target=%v, remainder=%v\n",
		   *target, *remainder);
	} else {
	    DEBUG(DBG_ADDR_HI, "parse_address: PCT_MAILBOX\n");
	}
	return PCT_MAILBOX;
    }

    if (switch_flag) {
	goto switch_order_bang;
    }
 switch_order_local:
    /*
     * we have a local form address
     */
    if (target) {
	*remainder = ap;
	DEBUG2(DBG_ADDR_HI, "parse_address: LOCAL: target=%v, remainder=%v\n",
	       *target ? *target : "(no-domain)", *remainder);
    } else {
	DEBUG(DBG_ADDR_HI, "parse_address: LOCAL\n");
    }
    return LOCAL;
}

/*
 * check_target_and_remainder - check for glaring problems
 *
 * Returns SUCCEED if all is well, FAIL otherwise.
 *
 * A pointer to the error message related to any problems found is returned in
 * the (*remainderp) pointer.
 */
static int
check_target_and_remainder(targetp, remainderp, form)
    char **targetp;				/* ptr to hostname str */
    char **remainderp;				/* ptr to mailbox str */
    int form;
{
    int c;
    char *p;

    DEBUG3(DBG_ADDR_HI, "check_target_and_remainder(): hostname='%v', mailbox='%v', form=%s\n",
	   *targetp,
	   *remainderp,
	   (form == RFC_ROUTE) ? "RFC_ROUTE" :
	   (form == RFC_ENDROUTE) ? "RFC_ENDROUTE" :
	   (form == MAILBOX) ? "MAILBOX" :
	   (form == UUCP_ROUTE) ? "UUCP_ROUTE" :
	   (form == PCT_MAILBOX) ? "PCT_MAILBOX" :
	   (form == LOCAL) ? "LOCAL" :
	   (form == BERKENET) ? "BERKENET" :
	   (form == DECNET) ? "DECNET" : "<bad-form!>");
    /*
     * first check the remainder...
     *
     * We should be checking for full RFC [2]822 conformance (i.e. the
     * following syntax) at a minimum, but we would need
     * rfc822_is_quoted_string() to do that.  Also note that much of the
     * parsing and removal of extraneous parts such as CFWS and FWS has already
     * been done by our callers.
     *
     *	local-part      =       dot-atom / quoted-string / obs-local-part
     *
     *	dot-atom        =       [CFWS] dot-atom-text [CFWS]
     *
     *	dot-atom-text   =       1*atext *("." 1*atext)
     *
     *	atext           =       ALPHA / DIGIT / ; Any character except controls,
     *	                        "!" / "#" /     ;  SP, and specials.
     *	                        "$" / "%" /     ;  Used for atoms
     *	                        "&" / "'" /
     *	                        "*" / "+" /
     *	                        "-" / "/" /
     *	                        "=" / "?" /
     *	                        "^" / "_" /
     *	                        "`" / "{" /
     *	                        "|" / "}" /
     *	                        "~"
     *
     *	quoted-string   =       [CFWS]
     *	                        DQUOTE *([FWS] qcontent) [FWS] DQUOTE
     *	                        [CFWS]
     *
     *	qcontent        =       qtext / quoted-pair
     *
     *	qtext           =       NO-WS-CTL /     ; Non white space controls
     *	                        %d33 /          ; The rest of the US-ASCII
     *	                        %d35-91 /       ;  characters not including "\"
     *	                        %d93-126        ;  or the quote character
     *
     *	quoted-pair     =       ("\" text) / obs-qp
     *
     *	text            =       %d1-9 /         ; Characters excluding CR and LF
     *	                        %d11 /
     *	                        %d12 /
     *	                        %d14-127 /
     *	                        obs-text
     *
     *	NO-WS-CTL       =       %d1-8 /         ; US-ASCII control characters
     *	                        %d11 /          ;  that do not include the
     *	                        %d12 /          ;  carriage return, line feed,
     *	                        %d14-31 /       ;  and white space characters
     *	                        %d127
     *
     *	ctext           =       NO-WS-CTL /     ; Non white space controls
     *	                        %d33-39 /       ; The rest of the US-ASCII
     *	                        %d42-91 /       ;  characters not including "(",
     *	                        %d93-126        ;  ")", or "\"
     *
     *	ccontent        =       ctext / quoted-pair / comment
     *
     *	comment         =       "(" *([FWS] ccontent) [FWS] ")"
     *
     *	CFWS            =       *([FWS] comment) (([FWS] comment) / FWS)
     *
     *	FWS             =       ([*WSP CRLF] 1*WSP) /   ; Folding white space
     *	                        obs-FWS
     *
     * Instead we will simply check for strict RFC 821 compatability as will be
     * needed for any addresses used in SMTP.  Other transports may not be
     * quite so restrictive as this (or may even be more restrictive!).
     */
    p = *remainderp;
    if (*p == '\0') {
	*remainderp = "no remainder address";
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, ''): %s\n", *targetp, *remainderp);
	return FAIL;
    }
    if (form == MAILBOX && !rfc821_is_dot_string(*remainderp) && !rfc821_is_quoted_string(*remainderp)) {
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): mailbox is not a valid RFC-821 local-part.", *targetp, *remainderp);
	*remainderp = "mailbox is not a valid RFC-821 local-part";
	return FAIL;
    } else if (form != MAILBOX) {
	/*
	 *	XXX some validation really should be done on the remainer in
	 *	non-MAILBOX forms too!!!
	 */
	DEBUG2(DBG_ADDR_LO, "check_target_and_remainder(%v, %v): not checking non-MAILBOX form in localpart.\n", *targetp, *remainderp);
     }
    /*
     * now check the target...
     */
    p = *targetp;
#ifdef HAVE_BSD_NETWORKING
    if (*p == '[') {
	in_addr_t inet;			/* IP address */
	char *p2;			/* pointer to closing bracket (]) */

	p2 = strchr(p, ']');
	*p2 = '\0';
	inet = get_inet_addr(&(p[1]));
	*p2 = ']';
	DEBUG4(DBG_ADDR_HI, "check_target_and_remainder(%v, %v): inet addr given: [0x%lx] aka [%s]\n",
	       *targetp, *remainderp,
	       ntohl(inet), inet_ntoa(inet_makeaddr((in_addr_t) ntohl(inet), (in_addr_t) 0)));
	if (inet == INADDR_NONE) {
	    DEBUG2(DBG_ADDR_LO, "check_target_and_remainder(%v, %v): get_inet_addr() failed: Invalid host address literal form\n", *targetp, *remainderp);
	    *remainderp = "Invalid host address literal form";
	    return FAIL;
	}
	return SUCCEED;
    }
#endif
    if (*p == '-') {
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): domain name cannot begin with `-'\n", *targetp, *remainderp);
	*remainderp = "domain name cannot begin with `-'";
	return FAIL;
    }
    if (*p == '.') {
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): domain name cannot begin with `.'\n", *targetp, *remainderp);
	*remainderp = "domain name cannot begin with `.'";
	return FAIL;
    }
#ifdef HAVE_BSD_NETWORKING		/* XXX having to ifdef this is bogus... */
    if (get_inet_addr(p) != INADDR_NONE) {
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): domain name cannot be a bare IP address\n", *targetp, *remainderp);
	*remainderp = "domain name cannot be a bare IP address";
	return FAIL;
    }
#endif
    /*
     * this test is more in line with RFC-821's restrictions, which are based
     * on RFC-1035's hostname restrictions.  Other transports may not be quite
     * so restrictive as this (or may even be more restrictive!).
     *
     * WARNING: we should probably also ensure the total length, the length of
     * each label, the start char of each label, etc., are all valid.
     *
     * There's somewhat more careful code in smtprecv.c:verify_host(), but at
     * this point we don't really care if the name is fully qualified or not,
     * nor if it's a bare IP address (i.e. missing the square brackets), as
     * eventually we'll find out if it's routable or not one way or another.
     */
    while ((c = *p++)) {
	if (! isascii(c) || ! (isalnum(c) || c == '.' || c == '-')) {
	    DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): invalid character in domain name\n", *targetp, *remainderp);
	    *remainderp = "invalid character in domain name";
	    return FAIL;
	}
    }
    --p;				/* backup to the last char */
    if (*p == '-') {
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): domain name cannot end with `-'\n", *targetp, *remainderp);
	*remainderp = "domain name cannot end with `-'";
	return FAIL;
    }
    if (*p == '.') {
	DEBUG2(DBG_ADDR_MID, "check_target_and_remainder(%v, %v): domain name cannot end with `.'\n", *targetp, *remainderp);
	*remainderp = "domain name cannot end with `.'";
	return FAIL;
    }

    return SUCCEED;
}



/*
 * mixed_address - check for mixed operators in an address
 *
 * Return TRUE if the given address contains both a % operator and
 * some set of !-like operators (i.e., !, :, or ::); otherwise,
 * return FALSE.
 */
int
mixed_address(address)
    char *address;
{
    int fndpct = 0;
    int fndbang = 0;
    char *p;

    for (p = address; p; p = address_token(p)) {
	switch (*p) {
	case ':':
	case '!':
	    if (fndpct)
		return TRUE;
	    fndbang = TRUE;
	    break;

	case '%':
	    if (fndbang)
		return TRUE;
	    fndpct = TRUE;
	    break;
	}
    }

    return FALSE;
}

/*
 * build_uucp_route - convert an address into a UUCP route.
 *
 * Given an address using any of the addressing forms known to the
 * parse_address() routine, convert that address into a pure uucp
 * !-route.  The return value is always freeable with xfree().
 *
 * If there is an error, return NULL.
 *
 * inputs:
 *	address	- the address to transform into a UUCP !-route
 *	error	- on error, set this to error message, if non-NULL
 *
 * output:
 *	transformed address, or NULL if a syntax error occured
 */
char *
build_uucp_route(address, error, flag)
    char *address;			/* address to transform into !-route */
    char **error;			/* return an error message here */
    int flag;				/* flag returned by parse_address() */
{
    return internal_build_uucp_route(address, error, FALSE, flag);
}

/*
 * build_partial_uucp_route - convert an address into a partial UUCP route.
 *
 * Given an address using any of the addressing forms known to the
 * parse_address routine, convert that address into a uucp !-route,
 * possibly with %-forms left at the end.  The return value is always
 * freeable with xfree().
 *
 * If there is an error, return NULL.
 *
 * inputs:
 *	address	- the address to transform into a UUCP !-route
 *	error	- on error, set this to error message, if non-NULL
 *
 * output:
 *	transformed address, or NULL if a syntax error occured
 */
char *
build_partial_uucp_route(address, error, flag)
    char *address;			/* address to transform into !-route */
    char **error;			/* return an error message here */
    int flag;				/* flag from parse_address() */
{
    return internal_build_uucp_route(address, error, TRUE, flag);
}

/*
 * internal_build_uucp_route - internal form for uucp-route building
 *
 * called from build_uucp_route and build_partial_uucp_route.  If the
 * `partial' flag is TRUE then the latter style is used, otherwise a
 * pure !-route is built.
 */
static char *
internal_build_uucp_route(address, error, partial, flag)
    char *address;			/* address to transform into !-route */
    char **error;			/* return an error message here */
    int partial;			/* TRUE to allow %-form in route */
    int flag;
{
    struct str str;
    register struct str *sp = &str;	/* dynamic string region */
    int uucp_route = TRUE;		/* TRUE if already pure !-route */
    char *target = NULL;		/* target returned by parse_address */
    char *remainder;			/* remainder from parse_address */
    char *storage;			/* malloc region for old address */

    DEBUG1(DBG_ADDR_HI, "internal_build_uucp_route entry: address=%s\n",
	   address);
    /*
     * allocate a new copy of the address so it can be examined destructively.
     * XXX this seems to be bogus....
     */
    storage = remainder = xmalloc((size_t) (strlen(address) + 1));
    strcpy(storage, address);

    /* initialize for copy into string region */
    STR_INIT(sp);

    /* loop until we have a local form or a %-form an error occurs */
    for (;;) {
	int form = parse_address(remainder, &target, &remainder, &flag);

	switch (form) {

	case FAIL:			/* something went wrong, somewhere */
	    *error = remainder;
	    DEBUG(DBG_ADDR_MID, "internal_build_uucp_route returns: (null)\n")
	    return NULL;

	case UUCP_ROUTE:		/* okay, this part is a !-route */
	    STR_CAT(sp, target);	/* add target! to route */
	    STR_NEXT(sp, '!');
	    break;

	case PCT_MAILBOX:		/* matched something%host... */
	    /*
	     * If we are building a pure uucp route, then a%b is just
	     * another remote form.  Otherwise, finding this form ends
	     * the parsing process.
	     */
	    if (!partial) {
		goto remote_form;
	    }
	    /* FALLTHRU */

	case LOCAL:			/* local form, we are done */
	    /* if address was already a pure !-route, return the old one */
	    if (uucp_route) {
		/* free garbage */
		xfree(storage);
		STR_FREE(sp);
		DEBUG1(DBG_ADDR_HI,
		      "internal_build_uucp_route returns: %s (unchanged)\n",
		      address);
		return COPY_STRING(address);
	    } else {
		/* append final local-part */
		STR_CAT(sp, remainder);
		if (form == PCT_MAILBOX) {
		    /* %-form requires the target to be included */
		    STR_NEXT(sp, '%');
		    STR_CAT(sp, target);
		}
		STR_NEXT(sp, '\0');
		xfree(storage);		/* free garbage */
		STR_DONE(sp);
		DEBUG1(DBG_ADDR_HI, "internal_build_uucp_route returns: %s\n",
		       STR(sp));
		return STR(sp);		/* return completed !-route */
	    }
	    /*NOTREACHED*/

	default:			/* not pure !-route, other form */
	remote_form:
	    STR_CAT(sp, target);	/* add target! to route */
	    STR_NEXT(sp, '!');
	    uucp_route = FALSE;
	}
    }
}

/*
 * strip_rfc822_comments - destructively strip RFC822 comments from a string
 */
void
strip_rfc822_comments(s)
    char *s;
{
    char *p, *q;
    int c;
    int level;

    p = q = s;
    while ((c = *p++)) {
	if (c == '(') {
	    level = 1;

	    while ((c = *p)) {
		p++;
		if (c == '(') {
		    level++;
		    continue;
		}
		if (c == ')') {
		    --level;
		    if (level == 0)
			break;
		    continue;
		}
		if (c == '\\') {
		    if (*p)
			p++;
		}
	    }
	    continue;
	}
	if (c == '\\') {
	    *q++ = c;
	    if ((c = *p)) {
		*q++ = c;
		p++;
	    }
	    continue;
	}
	if (c == '"') {
	    *q++ = c;
	    while ((c = *p)) {
		p++;
		*q++ = c;
		if (c == '"')
		    break;
		if (c == '\\') {
		    if ((c = *p)) {
			*q++ = c;
			p++;
		    }
		}
	    }
	    continue;
	}
	*q++ = c;
    }
    *q++ = '\0';
}

/*
 * strip_rfc822_whitespace - destructively strip *extra* whitespace from an
 * RFC822 address
 */
void
strip_rfc822_whitespace(s)
    char *s;
{
    char *p, *q;
    int c;
    int level;
    static char delims[] = "@:;<>().,";
    int space = 0;

    p = q = s;
    while ((c = *p++)) {
	if (isspace((int) c)) {
	    space = 1;
	    continue;
	}
	if (space) {
	    space = 0;
	    if (q > s && !strchr(delims, *(q - 1)) && !strchr(delims, c)) {
		*q++ = ' ';
	    }
	}
	if (c == '(') {
	    level = 1;

	    while ((c = *p++)) {
		*q++ = c;
		if (c == '(') {
		    level++;
		    continue;
		}
		if (c == ')') {
		    --level;
		    if (level == 0)
			break;
		    continue;
		}
		if (c == '\\') {
		    if (*p) {
			*q++ = c;
			p++;
		    }
		}
	    }
	    continue;
	}
	if (c == '\\') {
	    *q++ = c;
	    if ((c = *p)) {
		*q++ = c;
		p++;
	    }
	    continue;
	}
	if (c == '"') {
	    *q++ = c;
	    while ((c = *p)) {
		p++;
		*q++ = c;
		if (c == '"')
		    break;
		if (c == '\\') {
		    if ((c = *p)) {
			*q++ = c;
			p++;
		    }
		}
	    }
	    continue;
	}
	*q++ = c;
    }
    *q++ = '\0';
}


/*
 * address_token - scan forward one token in an address
 *
 * an address token is delimited by a character from the set [@!%:,]
 * a token can also be a domain literal between [ and ], or
 * a quoted literal between double quotes.  \ can precede a character
 * to take away its special properties.
 * domain literals and quoted literals and other tokens can be strung
 * together into one single token if they are separated by `.'.  Otherwise
 * a domain literal or quoted literal represents one token.
 *
 * input:
 *	ap	- pointer to start of a token
 *
 * output:
 *	the end of the input token.  Return NULL on error.
 *
 * called by: parse_address
 */
char *
address_token(ap)
    register char *ap;			/* address to be scanned */
{
    static enum state {			/* states for the state machine */
	s_normal,			/* not in a literal or \ escape */
	s_cquote,			/* previous char was \ */
	s_quote,			/* scanning quoted literal */
	s_domlit			/* scanning domain literal */
    } state;
    enum state save_state = s_normal;	/* previous state for \ escape */
    int dot = FALSE;			/* TRUE if last char was unescaped . */

    /* setup initial state */
    switch (*ap++) {
    case '\0':				/* no tokens */
	return NULL;			/* error */

    case '@':				/* delimiters are one token a piece */
    case '!':
    case '%':
    case ':':
    case ',':
    case '>':
    case '<':
	return ap;			/* so return that single token */

    case '"':				/* start in a quoted literal */
	state = s_quote;
	break;

    case '[':				/* start in a domain literal */
	state = s_domlit;
	break;

    case '.':				/* start with an initial dot */
	state = s_normal;
	dot = TRUE;
	break;

    case '\\':				/* start initially with \ escape */
	save_state = s_normal;
	state = s_cquote;
	break;

    default:				/* otherwise begin in normal state */
	state = s_normal;
	break;
    }

    /*
     * scan until end of token
     */
    while (*ap) {
	switch (state) {

	case s_normal:			/* scan for token delimeter */
	    switch (*ap) {

	    case '\\':			/* \ escape, save state, then cquote */
		save_state = s_normal;
		state = s_cquote;
		break;

	    case '[':			/* domain continue if last char is . */
		if (dot) {
		    state = s_domlit;
		} else {
		    return ap;
		}
		break;

	    case '"':			/* quote continue if last char is . */
		if (dot) {
		    state = s_quote;
		} else {
		    return ap;
		}
		break;

	    case '@':
	    case '!':
	    case '%':
	    case ':':
	    case ',':
	    case '<':
	    case '>':
		return ap;		/* found the end of a token */
	    }
	    /* dot is TRUE if this char was a dot */
	    dot = ('.' == *ap++);
	    break;

	case s_quote:			/* scan for end of a quote */
	    if (*ap == '\\') {
		/* \ escape in quote */
		ap++;
		save_state = s_quote;
		state = s_cquote;
	    } else if (*ap++ == '"') {
		/* end of quote -- check for . after it */
		if (*ap == '.') {
		    /* if exists, continue scanning */
		    state = s_normal;
		} else {
		    /* otherwise we have a complete token */
		    return ap;
		}
	    }
	    break;

	case s_domlit:			/* scan for end of domain literal */
	    if (*ap == '\\') {
		/* \ escape in domain literal */
		ap++;
		save_state = s_domlit;
		state = s_cquote;
	    } else if (*ap++ == ']') {
		/* end of domain literal -- check for . after it */
		if (*ap == '.') {
		    /* if exists, continue scanning */
		    state = s_normal;
		} else {
		    /* otherwise we have a complete token */
		    return ap;
		}
	    }
	    break;

	case s_cquote:			/* process \ escape */
	    ap++;			/* just skip the char */
	    state = save_state;		/* and return to previous state */
	    break;
	}
    }

    /*
     * fell through -- error if we are not in the normal state
     */
    if (state != s_normal) {
	return NULL;
    }

    return ap;				/* all done, return the token */

}


/*
 * back_address_token - scan backward one token in an address
 *
 * see the rules in address_token for how to delimit an address token.
 * This procedure does it going backwards.
 *
 * Note:  this routine is more complex than address_token, because
 *	  addresses are intended to be scanned forward.
 *
 * inputs:
 *	ba	- beginning of an address (firewall)
 *	ap	- pointer to character past end of token
 *
 * output:
 *	return start of token that ap points past.  Return NULL on error.
 *
 * called by: parse_address
 * calls: escaped
 */
char *
back_address_token(ba, ap)
    register char *ba;			/* beginning of address (firewall) */
    register char *ap;			/* character past end of token */
{
    static enum state {			/* states for the state machine */
	s_normal,			/* not in a literal */
	s_quote,			/* scanning quoted literal */
	s_domlit			/* scanning domain literal */
    } state;
    int dot = FALSE;			/* TRUE if next char is unescaped . */
    register char *p;			/* temp */

    /*
     * trap no tokens
     */
    if (ba == ap) {
	return NULL;
    }

    /*
     * setup initial state
     */
    --ap;				/* backup to end of token */
    if ((p = escaped(ba, ap))) {
	/* if last char is escaped, we are in the normal state */
	state = s_normal;
	ap = p;
    } else {
	switch (*ap) {
	case '@':			/* delimiters are one token a piece */
	case '!':
	case '%':
	case ':':
	case ',':
	case '>':
	case '<':
	    return ap;			/* so return that single token */

	case '"':			/* start in a quoted literal */
	    state = s_quote;
	    break;

	case ']':			/* start in a domain literal */
	    state = s_domlit;
	    break;

	case '.':			/* start with an initial dot */
	    state = s_normal;
	    dot = TRUE;
	    break;

	default:			/* otherwise begin in normal state */
	    state = s_normal;
	    break;
	}
	--ap;				/* this char already processed */
    }

    /*
     * scan until beginning of token
     */
    while (ap - ba >= 0) {
	switch (state) {

	case s_normal:			/* scan for token delimeter */
	    /* trap escaped character */
	    if ((p = escaped(ba, ap))) {
		ap = p;
	    } else {
		/* not escaped, process it */
		switch (*ap) {

		case ']':		/* domain okay if next char is . */
		    if (dot) {
			state = s_domlit;
		    } else {
			return ap+1;
		    }
		    break;

		case '"':		/* quote okay if next char is . */
		    if (dot) {
			state = s_quote;
		    } else {
			return ap+1;
		    }
		    break;

		case '@':
		case '!':
		case '%':
		case ':':
		case ',':
		case '>':
		case '<':
		    return ap+1;	/* found the end of a token */
		}
		/* dot is TRUE if this char was a dot */
		dot = ('.' == *ap--);
	    }
	    break;

	case s_quote:			/* scan for end of a quote */
	    if ((p = escaped(ba, ap))) {
		/* trap \ escape */
		ap = p;
	    } else if (*ap-- == '"') {
		/* end of quote -- check for . before it */
		if (ap - ba >= 0 && *ap == '.' && !escaped(ba, ap)) {
		    /* if exists, continue scanning */
		    state = s_normal;
		} else {
		    /* otherwise we have a complete token */
		    return ap+1;
		}
	    }
	    break;

	case s_domlit:			/* scan for end of domain literal */
	    if ((p = escaped(ba, ap))) {
		/* trap \ escape */
		ap = p;
	    } else if (*ap-- == '[') {
		/* end of domain literal -- check for . before it */
		if (ap - ba >= 0 && *ap == '.' && !escaped(ba, ap)) {
		    /* if exists, continue scanning */
		    state = s_normal;
		} else {
		    /* otherwise we have a complete token */
		    return ap+1;
		}
	    }
	    break;
	}
    }

    /*
     * fell through -- error if we are not in the normal state
     */
    if (state != s_normal) {
	return NULL;
    }

    return ap+1;			/* all done, return the token */
}

/*
 * escaped - determine if a character is \ escaped, scanning backward
 *
 * given the beginning of a string and a character positition within
 * it, determine if that character is \ escaped or not, tracing through
 * multiple \ chars if necessary.  Basically, if the character position
 * is preceded by an odd number of \ chars, the current character is
 * \ escaped.
 *
 * inputs:
 *	ba	- beginning of string
 *	ap	- character position in string
 *
 * output:
 *	beginning of set of \ chars previous to ap, or NULL if the
 *	character at ap is not backslash escaped.
 *
 * called by: back_address_token
 */
static char *
escaped(ba, ap)
    register char *ba;			/* beginning of string */
    register char *ap;			/* character position in string */
{
    register size_t i = 0;		/* count of \ characters */

    /*
     * count the number of preceding \ characters, but don't go past
     * the beginning of the string.
     */
    --ap;
    while (ap - ba >= 0 && *ap == '\\') {
	i++; --ap;
    }

    /* if odd number of \ chars, then backslash escaped */
    return (i%2==1)? ap: NULL;
}


/*
 * alloc_addr - allocate a struct addr
 *
 * NOTE: the caller must setup the addr fields correctly.  This routine
 *	 marks certain fields with improper values, which unless changed,
 *	 will results in other routines doing a panic().
 */
struct addr *
alloc_addr()
{
    register struct addr *new;		/* our new address */

    /* grab it */
    new = (struct addr *) xmalloc(sizeof(*new));

    /* preset the proper values */
    (void) memset((char *) new, '\0', sizeof(*new)); /* XXX hope NULL pointers are all zeros! */
    new->match_count = -1;
    new->uid = (unsigned int) BOGUS_USER;	/* the identity is not known yet */
    new->gid = (unsigned int) BOGUS_GROUP;	/* the identity is not known yet */

    return new;
}


/*
 * free_addr - free a struct addr
 */
void
free_addr(done)
    struct addr *done;			/* addr struct to free */
{
    if (done->parent) {
	free_addr(done->parent);
	done->parent = NULL;
    }
    if (done->true_addr) {
	free_addr(done->true_addr);
	done->true_addr = NULL;
    }
    if (done->in_addr) {
	xfree((char *) done->in_addr);
	done->in_addr = NULL;
    }
    if (done->target) {
	xfree((char *) done->target);
	done->target = NULL;
    }
    if (done->remainder) {
	xfree((char *) done->remainder);
	done->remainder = NULL;
    }
    if (done->rem_prefix) {
	xfree((char *) done->rem_prefix);
	done->rem_prefix = NULL;
    }
    if (done->rem_suffix) {
	xfree((char *) done->rem_suffix);
	done->rem_suffix = NULL;
    }
    if (done->work_addr) {
	xfree((char *) done->work_addr);
	done->work_addr = NULL;
    }
    if (done->local_name) {
	xfree((char *) done->local_name);
	done->local_name = NULL;
    }
    if (done->owner) {
	xfree((char *) done->owner);
	done->owner = NULL;
    }
    if (done->route) {
	xfree((char *) done->route);
	done->route = NULL;
    }
    if (done->next_host) {
	xfree((char *) done->next_host);
	done->next_host = NULL;
    }
    if (done->next_addr) {
	xfree((char *) done->next_addr);
	done->next_addr = NULL;
    }
    /* XXX home is not always uniquely allocated storage, but sometimes it is */
    /* XXX do multiple addresses point to the same tphint_list? */
    if (done->error) {
	/* XXX what about error->message ??? */
	xfree((char *) done->error);
	done->error = NULL;
    }
    xfree((char *) done);

    return;
}


/*
 * free_addr_list - free a list of addresses
 */
void
free_addr_list(donelst)
    struct addr *donelst;		/* list of addrs to free */
{
    struct addr *cur;
    struct addr *next;

    for (cur = donelst; cur; cur = next) {
	next = cur->succ;
	free_addr(cur);
    }

    return;
}


/*
 * insert_addr_list - insert a list of addrs into another list
 *
 * insert each addr in an input list at the beginning of an output list.
 * In the process or in some addr flags and (possibly) set next_addr
 * to an error message.
 */
void
insert_addr_list(in, out, error)
    register struct addr *in;		/* input list */
    register struct addr **out;		/* output list */
    register struct error *error;	/* error structure (if non-NULL) */
{
    struct addr *next;

    DEBUG(DBG_ADDR_HI, "insert_addr_list() called:\n");
#ifndef NODEBUG
    if (error) {
	DEBUG2(DBG_ADDR_HI, "\tERR%ld: %s\n",
	       error->info & ERR_MASK, error->message);
    }
#endif	/* NODEBUG */
    /* loop over all of the input addrs */
    for (; in; in = next) {
	next = in->succ;

	DEBUG1(DBG_ADDR_HI, "\t%s\n", in->in_addr);
	if (error) {
	    in->error = error;		/* set the error message, if given */
	}
	in->succ = *out;
	*out = in;
    }
}


/*
 * remove_addr - remove any matched addresses from an input list
 *
 * given an address string and (perhaps) a parent address string and
 * an input address list, remove any occurance of an address in the
 * input list whose in_addr matches the specified address string and
 * whose parent in_addr string matches the specified parent string.
 * If parent is NULL then there must not be a parent address, otherwise
 * there must be a matching parent address.
 */
struct addr *
remove_addr(in, address, parent)
    struct addr *in;			/* input addr list */
    char *address;			/* address to match against */
    char *parent;			/* ultimate parent of address to match */
{
    register struct addr *cur;		/* current address to process */
    struct addr *next;			/* next address to process */
    struct addr *out = NULL;		/* output address list */

    DEBUG2(DBG_ADDR_HI, "remove_addr(in, %v, %v) called ...\n", address, parent);

    for (cur = in; cur; cur = next) {
	register struct addr *top;	/* the ultimate parent address to compare */

	next = cur->succ;

	/* find the top parent to log the original in_addr */
	for (top = cur; top->parent && top->parent->in_addr; top = top->parent) {
	    ;
	}
	if (top == cur) {
	    top = NULL;
	}
	if (EQ(cur->in_addr, address)) {
	    /* the address does match */
	    if (parent) {
		/* a matching parent is also required for a match */
		if (top && EQ(parent, top->in_addr)) {
		    /* match, don't put it on the output queue */
		    DEBUG2(DBG_ADDR_MID, "remove_addr(): %v ... (with parent address %v) already delivered\n",
			   cur->in_addr, top->in_addr);
#ifdef not_yet
		    free_addr(cur);
#endif
		    continue;
		}
	    } else if (top == NULL) {
		/* match, don't put it on the output queue */
		DEBUG1(DBG_ADDR_MID, "remove_addr(): %v ... already delivered\n",
		       cur->in_addr);
#ifdef not_yet
		free_addr(cur);
#endif
		continue;
	    }
	}
	DEBUG1(DBG_ADDR_HI, "remove_addr(): %v ... not delivered -- re-queue\n",
	       cur->in_addr);

	/* no match, put the address on the output queue */
	cur->succ = out;
	out = cur;
    }

    return out;				/* return the new list */
}

/*
 * return only those addrs from 'in' which have an in_addr field that matches
 * one of the REs in 're_list'
 */
struct addr *
keep_matching_addrs(in, re_list)
    struct addr *in;
    char *re_list;
{
    struct addr *cur;
    struct addr *next;
    struct addr *keep = NULL;

    for (cur = in; cur; cur = next) {
	char *reason;			/* XXX ignored... */

	next = cur->succ;

	/* XXX ignores errors.... */
	if (match_re_list(cur->in_addr, re_list, FALSE, &reason) == MATCH_MATCHED) {
	    DEBUG1(DBG_ADDR_MID, "remove_nonmatching_addrs():  keeping '%s'\n", cur->in_addr);
	    cur->succ = keep;
	    keep = cur;
	}
	/* else XXX free_addr(cur) */
    }

    return keep;
}


/*
 * offset passed through the heap to the compare function....
 */
static int sort_offset;

/*
 * addr_sort - sort an input list of addrs and return the new sorted list
 *
 * calling sequence is:
 *	sorted_list = addr_sort(input_list, OFFSET(addr, tag_name)
 *
 * where tag_name is the (char *) element name in the addr structure to
 * sort on.
 */
struct addr *
addr_sort(in, offset)
    struct addr *in;
    int offset;				/* XXX should be unsigned? */
{
    struct addr **addrv;		/* array of addresses */
    register size_t addrc;		/* count of addresses */
    register struct addr **addrp;	/* temp addr pointer */
    register struct addr *a;		/* address list or current address */

    /* pass offset value to addrcmp() by setting file local variable */
    sort_offset = offset;

    /* count the input addresses */
    addrc = 0;
    for (a = in; a; a = a->succ) {
	addrc++;
    }

    /* allocate space for an array for that many pointers */
    addrv = (struct addr **) xmalloc(addrc * sizeof(*addrv));

    /* build the array from the input list */
    for (addrp = addrv, a = in; a; a = a->succ) {
	*addrp++ = a;
    }

    /* sort the array */
    qsort((char *)addrv, addrc, sizeof(*addrv), addrcmp);

    /*
     * turn the sorted array into a sorted list
     * Start from the end of the array so the generated list will start
     * from the beginning.
     */
    for (addrp = addrv + addrc, a = NULL; addrc > 0; --addrc) {
	(*--addrp)->succ = a;
	a = *addrp;
    }

    return a;
}

/*
 * addrcmp - compare two addr structures based on a field at sort_offset.
 */
static int
addrcmp(x, y)
    const void *x;
    const void *y;
{
    const char *a = *((const char * const *) x);
    const char *b = *((const char * const *) y);

    return strcmp((a + sort_offset), (b + sort_offset));
}

/*
 * note_error - create an error structure for inclusion in an addr structure
 */
struct error *
note_error(info, message)
    unsigned long int info;
    char *message;
{
    struct error *ret = (struct error *)xmalloc(sizeof(*ret));

    DEBUG2(DBG_ADDR_MID, "note_error(ERR_%ld, %s)\n", (info & ERR_MASK), message);

    ret->info = info;
    ret->message = message;

    return ret;
}

void
free_error(err)
	struct error *err;
{
	/* XXX Note we cannot free the message -- may be a constant! */
	xfree((char *) err);

	return;
}

#ifndef NDEBUG

void
dump_addr_list(in)
    struct addr *in;			/* input list */
{
    register struct addr *cur;		/* current address to process */

    for (cur = in; cur; cur = cur->succ) {
	dump_addr(cur, "");
    }

    return;
}

void
dump_addr(cur, prefix)
    struct addr *cur;			/* a single address (succ should be NULL) */
    char *prefix;
{
#define X_SHOW_XALLOC(p)	(p ? (X_IS_XALLOC(p) ? "(XALLOC) " : "") : "")

    dprintf(errfile, "in_addr = %v\n", cur->in_addr);
    if (cur->flags || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    flags = 0x%lx\n", prefix, cur->flags);
    if (cur->parseflags || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    parseflags = 0x%x\n", prefix, cur->parseflags);
    if (cur->work_addr || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    work_addr = %s%v\n", prefix, X_SHOW_XALLOC(cur->work_addr), cur->work_addr);
    if (cur->next_host || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    next_host = %s%v\n", prefix, X_SHOW_XALLOC(cur->next_host), cur->next_host);
    if (cur->next_addr || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    next_addr = %s%v\n", prefix, X_SHOW_XALLOC(cur->next_addr), cur->next_addr);
    if (cur->target || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    target = %s%v\n", prefix, X_SHOW_XALLOC(cur->target), cur->target);
    if (cur->route || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    route = %s%v\n", prefix, X_SHOW_XALLOC(cur->route), cur->route);
    if (cur->remainder || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    remainder = %s%v\n", prefix, X_SHOW_XALLOC(cur->remainder), cur->remainder);
    if (cur->rem_prefix || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    rem_prefix = %s%v\n", prefix, X_SHOW_XALLOC(cur->rem_prefix), cur->rem_prefix);
    if (cur->rem_suffix || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    rem_suffix = %s%v\n", prefix, X_SHOW_XALLOC(cur->rem_suffix), cur->rem_suffix);
    if (cur->local_name || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    local_name = %s%v\n", prefix, X_SHOW_XALLOC(cur->local_name), cur->local_name);
    if (cur->owner || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    owner = %s%v\n", prefix, X_SHOW_XALLOC(cur->owner), cur->owner);
    /* XXX if debug >= DBG_ADDR_MID then dump home, uid, and gid */
    if (cur->director || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    director = %s%v\n", prefix, X_SHOW_XALLOC(cur->director), cur->director ? cur->director->name : "[none]");
    if (cur->router || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    router = %s%v\n", prefix, X_SHOW_XALLOC(cur->router), cur->router ? cur->router->name : "[none]");
    if (cur->match_count != -1 || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    match_count = %d\n", prefix, cur->match_count);
    if (cur->transport || debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    transport = %s%v\n", prefix, X_SHOW_XALLOC(cur->transport), cur->transport ? cur->transport->name : "[none]");
    /* XXX if debug >= DBG_ADDR_MID then dump transport hints */
    if (cur->parent) {
	char *more_indent = xprintf("%s    ", prefix);

	dprintf(errfile, "%sparent->", more_indent);
	dump_addr(cur->parent, more_indent);
	xfree(more_indent);
    }
    if (cur->true_addr) {
	char *more_indent = xprintf("%s    ", prefix);

	dprintf(errfile, "%strue_addr->", more_indent);
	dump_addr(cur->true_addr, more_indent);
	xfree(more_indent);
    }
    if (cur->error) {
	dprintf(errfile, "%s    error->info[code] = ERR_%ld\n", prefix, cur->error->info & ERR_MASK);
	/* XXX shoudl decode flags into the C constants */
	dprintf(errfile, "%s    error->info[flags] = 0x%lx\n", prefix, (cur->error->info & ~ERR_MASK));
	dprintf(errfile, "%s    error->message = %s%s\n", prefix, X_SHOW_XALLOC(cur->error->message), cur->error->message);
    } else if (debug >= DBG_ADDR_HI)
	dprintf(errfile, "%s    error = [none]\n", prefix);

    return;
}

#endif /* NDEBUG */

#ifdef STANDALONE

int return_to_sender = FALSE;
int exitvalue = 0;
FILE *errfile;

#ifdef DEBUG_LEVEL
int debug = DEBUG_LEVEL;
#else /* DEBUG_LEVEL */
int debug = 0;
#endif /* DEBUG_LEVEL */

/*
 * test the functions in addr by calling parse_address for each
 * argument given to the program.
 */
void
main(argc, argv)
    int argc;				/* count of arguments */
    char **argv;			/* vector of arguments */
{
    char *s;				/* temp string */
    char *addr;				/* preparsed address */
    char *error;			/* error message */
    int form;				/* form from parse_address */
    char *target = NULL;		/* target returned by parse_address */
    char *remainder = NULL;		/* remainder from parse_address */
    int i;

    errfile = stderr;

    /*
     * if first argument is a number, change the debug level
     */
    if (argc > 1 && isdigit((int) argv[1][0])) {
	debug = atoi(*++argv);
	argc--;
    }

    /*
     * loop over all arguments or read from standard input if none
     */
    if (argc > 1) {
	while (*++argv) {
	    fprintf(stderr, "input:  <%s>\n", *argv);

	    /* preparse the address to get rid of mutant forms */
	    addr = preparse_address(*argv, &error);
	    if (addr) {
		fprintf(stderr, "preparse_address: %s\n", addr);
	    } else {
		fprintf(stderr, "preparse_address: %s\n", error);
		break;
	    }

	    /* see what build_uucp_route yields */
	    s = build_uucp_route(addr, &error, 0);
	    if (s) {
		fprintf(stderr, "build_uucp_route: %s\n", s);
	    } else {
		fprintf(stderr, "build_uucp_route: %s\n", error);
	    }

	    /* see what parse_address yields */
	    form = parse_address(addr, &target, &remainder, (int *) NULL);
	    if (form == LOCAL) {
		printf("LOCAL %s\n", remainder);
	    } else if (form == FAIL) {
		fprintf(stderr, "parse_address: %s\n", remainder);
	    } else {
		printf("REMOTE %s@%s\n", remainder, target);
	    }
	}
    } else {
	char *line;

	while ((line = read_line(stdin))) {
	    int len;

	    /* trim the trailing newline, if any */
	    len = strlen(line);
	    if (line[len - 1] == '\n') {
		line[len - 1] = '\0';
	    }
	    fprintf(stderr, "input:  <%s>\n", line);

	    /* preparse the address to get rid of mutant forms */
	    addr = preparse_address(line, &error);
	    if (addr) {
		fprintf(stderr, "preparse_address: %s\n", addr);
	    } else {
		fprintf(stderr, "preparse_address: %s\n", error);
		break;
	    }

	    /* see what build_uucp_route yields */
	    s = build_uucp_route(addr, &error, 0);
	    if (s) {
		fprintf(stderr, "build_uucp_route: %s\n", s);
	    } else {
		fprintf(stderr, "build_uucp_route: %s\n", error);
	    }

	    /* see what parse_address yields */
	    form = parse_address(addr, &target, &remainder, (int *) NULL);
	    if (form == LOCAL) {
		printf("LOCAL %s\n", remainder);
	    } else if (form == FAIL) {
		fprintf(stderr, "parse_address: %s\n", remainder);
	    } else {
		printf("REMOTE %s@%s\n", remainder, target);
	    }
	}
    }

    exit(exitvalue);
}

#endif	/* STANDALONE */

/* 
 * Local Variables:
 * c-file-style: "smail"
 * End:
 */
