/****************************************************************
 *	Copyright 1986, Gene H. Olson, Quest Research, Burnsville   *
 *	Minnesota.   Permission to copy and distribute this         *
 *	program, all associated code in source and binary form,     *
 *	for any purpose, so long as this notice is preserved.       *
 ****************************************************************/

#include <stdio.h>
#include <signal.h>
#include <termio.h>

#include "dial.h"

extern int nolock ;					/* See dial(3C) */
extern int optind ;					/* See getopt(3) */
extern char *optarg ;				/* See getopt(3) */

extern unsigned sleep() ;
extern void perror() ;
extern void exit() ;
extern char *malloc() ;

static struct termio myterm ;		/* My terminal flags */
static int dialfd ;					/* Dial file descriptor */


/*****
 *	quit - exit gracefully.
 */

quit(status)
{
	(void) signal(SIGHUP, SIG_IGN) ;
	(void) signal(SIGINT, SIG_IGN) ;
	(void) signal(SIGQUIT, SIG_IGN) ;
	(void) signal(SIGPIPE, SIG_IGN) ;
	(void) signal(SIGTERM, SIG_IGN) ;

	if (dialfd >= 0) undial(dialfd) ;
	(void) ioctl(0, TCSETA, &myterm) ;

	exit(status) ;
	}


/*****
 *	catch - catch interrupt.
 */

catch()
{
	quit(INTRPT) ;
	}



/*****
 *	main - Main program.
 */

main(argc,argv)
int argc ;
char **argv ;
{
	register short opt ;
	register FILE *dialfile ;
	register short ch ;
	register int err ;
	register int i ;
	unsigned waitfree ;
	unsigned redial ;
	short copy ;
	struct termio tio ;
	CALL call ;

	waitfree = 0 ;
	redial = 0 ;
	copy = 0 ;

	/*
	 *	Set up tty options to be used when
	 *	modem control is enabled.
	 */

	tio.c_iflag = IGNPAR | IGNBRK | ISTRIP | IXON ;
	tio.c_oflag = 0 ;
	tio.c_cflag = CS8 | CREAD | HUPCL | CLOCAL ;
	tio.c_lflag = NOFLSH ;

	tio.c_cc[VMIN] = 10 ;
	tio.c_cc[VTIME] = 1 ;

	/*
	 *	Initialize call structure.
	 */

	call.attr = &tio ;
	call.baud = 0 ;
	call.speed = 0 ;
	call.line = 0 ;
	call.telno = 0 ;
	call.modem = 0 ;
	call.device = 0 ;

	/*
	 *	Decode options.
	 */

	while ((opt = getopt(argc,argv,"ceonrwb:l:s:")) != EOF) {
		switch (opt) {

			/*
			 *	Copy standard input to remote.
			 */

			case 'c':
				copy = 1 ;
				break ;

			/*
			 *	Even parity after carrier detect.
			 */

			case 'e':
				tio.c_cflag &= ~(CSIZE | PARENB | PARODD) ;
				tio.c_cflag |= CS7 | PARENB ;
				break ;

			/*
			 *	Odd parity after carrier detect.
			 */

			case 'o':
				tio.c_cflag &= ~(CSIZE | PARENB | PARODD) ;
				tio.c_cflag |= CS7 | PARENB | PARODD ;
				break ;

			/*
			 *	No lock file.  Caller allocates it for us.
			 */

			case 'n':
				nolock = 1 ;
				break ;

			/*
			 *	Redial if no answer.
			 */

			case 'r':
				redial = 60 ;
				break ;

			/*
			 *	Wait for free outgoing line.
			 */

			case 'w':
				waitfree = 30 ;
				break ;

			/*
			 *	Device name for outgoing line.
			 */

			case 'l':
				call.line = optarg ;
				break ;

			/*
			 *	Baud rate for initial dial.
			 */

			case 'b':
				call.baud = atoi(optarg) ;
				break ;

			/*
			 *	Baud rate after connect.
			 */

			case 's':
				call.speed = atoi(optarg) ;
				break ;
					
			/*
			 *	Bad parameter.
			 */

			default:
				exit(2) ;
			}
		}

	/*
	 *	Get phone number.
	 */

	if (optind != argc) call.telno = argv[optind] ;

	/*
	 *	Setup signal catching stuff.
	 */

	dialfd = -1 ;

	(void) ioctl(0, TCGETA, &myterm) ;

	if (signal(SIGHUP, SIG_IGN) == SIG_DFL) (void) signal(SIGHUP, catch) ;
	if (signal(SIGINT, SIG_IGN) == SIG_DFL) (void) signal(SIGINT, catch) ;
	if (signal(SIGQUIT, SIG_IGN) == SIG_DFL) (void) signal(SIGQUIT, catch) ;
	if (signal(SIGPIPE, SIG_IGN) == SIG_DFL) (void) signal(SIGPIPE, catch) ;
	if (signal(SIGTERM, SIG_IGN) == SIG_DFL) (void) signal(SIGTERM, catch) ;

	/*
	 *	Attempt the dial.
	 */

	for (;;) {
		dialfd = dial(call) ;

		if (dialfd == NO_ANS && redial) {
			dialmsg("No answer, redial in %d seconds.\n", redial) ;
			(void) sleep(redial) ;
			}

		else if ((dialfd == DV_NT_A || dialfd == NO_BD_A) && waitfree) {
			dialmsg("No outgoing line, will retry in %d seconds.\n", waitfree) ;
			(void) sleep(waitfree) ;
			}

		else break ;
		}

	if (dialfd < 0) quit(-dialfd) ;

	/*
	 *	If a copy operation is selected, copy standard
	 *	input to the dialed-up device.
	 */

	if (copy) {

		dialfile = fdopen(dialfd, "w") ;
		if (dialfile == 0) {
			perror("fdopen") ;
			quit(-L_PROB) ;
			}

		(void) setvbuf(dialfile, malloc(BUFSIZ), _IONBF, BUFSIZ) ;

		err = 0 ;

		while (err >= 0 && (ch = getchar()) != EOF)
			err = putc(ch, dialfile) ;

		for (i = 0 ; i < 10 ; i++)
			err = putc('\000',dialfile) ;

		if (err >= 0) err = fflush(dialfile) ;

		if (err < 0) {
			perror("copy write error") ;
			quit(-L_PROB) ;
			}
		}

	quit(0) ;
	return(0) ;
	}
