
/*
 * folder_realloc.c -- realloc a folder/msgs structure
 *
 * $Id$
 */

#include <h/mh.h>

/*
 * Reallocate some of the space in the folder
 * structure (currently just message status array).
 *
 * Return pointer to new folder structure.
 * If error, return NULL.
 */

struct msgs *
folder_realloc (struct msgs *mp, int lo, int hi)
{
    int msgnum;

    /* sanity checks */
    if (lo < 1)
	adios (NULL, "BUG: calling folder_realloc with lo < 1");
    if (lo > mp->lowmsg)
	adios (NULL, "BUG: calling folder_realloc with lo > mp->lowmsg");
    if (hi < mp->hghmsg)
	adios (NULL, "BUG: calling folder_realloc with hi < mp->hghmsg");

    /* Check if we really need to reallocate anything */
    if (lo == mp->lowoff && hi == mp->hghoff)
	return mp;

    if (lo != mp->lowoff) {
	/*
	 * We are changing the offset of the message status
	 * array.  So we will need to shift everything.
	 */
	seqset_t *tmpstats;

	/* first allocate the new message status space */
	if (!(tmpstats = malloc MSGSTATSIZE(mp, lo, hi))) {
	    advise (NULL, "unable to reallocate message storage");
	    return NULL;
	}

	/* then copy messages status array with shift */
	for (msgnum = mp->lowmsg; msgnum <= mp->hghmsg; msgnum++)
	    tmpstats[msgnum - lo] = mp->msgstats[msgnum - mp->lowoff];
	free(mp->msgstats);
	mp->msgstats = tmpstats;
    } else {
	/*
	 * We are just extending (or shrinking) the end of message
	 * status array.  So we don't have to move anything and can
	 * just realloc the message status array.
	 */
	if (!(mp->msgstats = realloc(mp->msgstats, MSGSTATSIZE(mp, lo, hi)))) {
	    advise (NULL, "unable to reallocate message storage");
	    return NULL;
	}
    }

    mp->lowoff = lo;
    mp->hghoff = hi;

    /*
     * Clear all the flags for entries outside
     * the current message range for this folder.
     */
    for (msgnum = mp->lowoff; msgnum < mp->lowmsg; msgnum++)
	clear_msg_flags (mp, msgnum);
    for (msgnum = mp->hghmsg + 1; msgnum <= mp->hghoff; msgnum++)
	clear_msg_flags (mp, msgnum);

    return mp;
}
