
/*
 * seq_save.c -- 1) synchronize sequences
 *            -- 2) save public sequences
 *
 * $Id$
 */

#include <h/mh.h>
#include <h/signals.h>


/*
 * 1.  If sequence is public and folder is readonly,
 *     then change it to be private
 * 2a. If sequence is public, then add it to the sequences file
 *     in folder (name specified by mh-sequences profile entry).
 * 2b. If sequence is private, then add it to the
 *     context file.
 */

void
seq_save (struct msgs *mp)
{
    int i;
    char flags, *cp, attr[BUFSIZ], seqfile[BUFSIZ * 2];
    FILE *fp;
    sigset_t set, oset;

    if (!(mp->msgflags & SEQMOD))
	return;
    mp->msgflags &= ~SEQMOD;

    context_read ();
    fp = NULL;

    flags = mp->msgflags;

    /*
     * If no mh-sequences file is defined, or if a mh-sequences file
     * is defined but empty (*mh_seq == '\0'), then pretend folder
     * is readonly.  This will force all sequences to be private.
     */
    if (mh_seq && *mh_seq != '\0')
	sprintf (seqfile, "%s/%s", mp->foldpath, mh_seq);
    else
	set_readonly (mp);

    for (i = 0; mp->msgattrs[i]; i++) {
	sprintf (attr, "atr-%s-%s", mp->msgattrs[i], mp->foldpath);
	if (is_readonly(mp) || is_seq_private(mp, i)) {
priv:;
	    /*
	     * handle private sequences
	     */
	    if ((cp = seq_list(mp, mp->msgattrs[i])))
		context_replace (attr, cp);	/* update sequence in context   */
	    else
		context_del (attr);	/* delete sequence from context */
	} else {
	    /*
	     * handle public sequences
	     */
	    context_del (attr);	/* delete sequence from context */

	    /* get space separated list of sequence ranges */
	    if (!(cp = seq_list(mp, mp->msgattrs[i])))
		continue;

	    if (!fp) {
		/*
		 * Attempt to open file for public sequences.
		 * If that fails (probably because folder is readonly,
                 * then make sequence private.
		 */
		if ((fp = fopen (seqfile, "w")) == NULL
			&& (unlink (seqfile) == NOTOK ||
			    (fp = fopen (seqfile, "w")) == NULL)) {
		    admonish (attr, "unable to write");
		    goto priv;
		}
		sigemptyset (&set);
		sigaddset(&set, SIGHUP);
		sigaddset(&set, SIGINT);
		sigaddset(&set, SIGQUIT);
		sigaddset(&set, SIGTERM);
		SIGPROCMASK (SIG_BLOCK, &set, &oset);
	    }
	    fprintf (fp, "%s: %s\n", mp->msgattrs[i], cp);
	}
    }

    if (fp) {
	fclose (fp);
	SIGPROCMASK (SIG_SETMASK, &oset, &set);  /* reset signal mask */
    } else {
	/*
	 * If folder is not readonly, and we didn't save any
	 * public sequences, then remove that file.
	 */
	if (!is_readonly(mp))
	    unlink (seqfile);
    }

    /*
     * Reset folder flag, since we may be
     * pretending that folder is readonly.
     */
    mp->msgflags = flags;
}
